package LISM::Handler;

use strict;
use Sys::Syslog;
use Sys::Syslog qw(:macros);
use Data::Dumper;

=head1 NAME

LISM::Handler - an base class for LISM handler implementations

=head1 DESCRIPTION

This class is meant as an interface of handler called when the LDAP operation is done.

=head1 CONSTRUCTOR

This is a plain constructor.

=cut

sub new
{
    my $class = shift;

    my $this = {};
    bless $this, $class;

    return $this;
}

=head1 METHODS

=head2 config($conf)

Set configuration data.

=cut

sub config
{
    my $self = shift;
    my ($conf) = @_;

    $self->{_config} = $conf;

    return 0;
}

=pod

=head2 init

Initailize the storage object.
Returns 0 if it complete successfully.

=cut

sub init
{
    my $self = shift;
    my $conf = $self->{_config};

    $conf->{numloglevel} = {debug => LOG_DEBUG,
                            info => LOG_INFO,
                            notice => LOG_NOTICE,
                            warning => LOG_WARNING,
                            error => LOG_ERR,
                            critical => LOG_CRIT,
                            alert => LOG_ALERT,
                            emergency => LOG_EMERG
                           };

    # check configuration
    if ($self->_checkConfig()) {
        $self->log(level => 'alert', message => "Configuration error");
        return -1;
    }

    return 0;
}

=pod

=head2 pre_bind

This method is called bofore L<LISM> do the bind operation.
Returns 0 if it completes successfully.

=cut

sub pre_bind
{
    return 0;
}

=pod

=head2 post_bind

This method is called after L<LISM> do the bind operation.
Returns 0 if it completes successfully.

=cut

sub post_bind
{
    return 0;
}

=pod

=head2 pre_compare

This method is called bofore L<LISM> do the compare operation.
Returns 0 if it completes successfully.

=cut

sub pre_compare
{
    return 0;
}

=pod

=head2 post_compare

This method is called after L<LISM> do the compare operation.
Returns 0 if it completes successfully.

=cut

sub post_compare
{
    return 0;
}

=pod

=head2 pre_search

This method is called bofore L<LISM> do the search operation.
Returns 0 if it completes successfully.

=cut

sub pre_search
{
    return 0;
}

=pod

=head2 post_search

This method is called after L<LISM> do the search operation.
Returns 0 if it completes successfully.

=cut

sub post_search
{
    return 0;
}

=pod

=head2 pre_modify

This method is called bofore L<LISM> do the modify operation.
Returns 0 if it completes successfully.

=cut

sub pre_modify
{
    return 0;
}

=pod

=head2 post_modify

This method is called after L<LISM> do the modify operation.
Returns 0 if it completes successfully.

=cut

sub post_modify
{
    return 0;
}

=pod

=head2 pre_add

This method is called before L<LISM> do the add operation.
Returns 0 if it completes successfully.

=cut

sub pre_add
{
    return 0;
}

=head2 post_add

This method is called after L<LISM> do the add operation.
Returns 0 if it completes successfully.

=cut

sub post_add
{
    return 0;
}

=pod

=head2 pre_modrdn

This method is called before L<LISM> do the modrdn operation.
Returns 0 if it completes successfully.

=cut

sub pre_modrdn
{
    return 0;
}

=pod

=head2 post_modrdn

This method is called after L<LISM> do the modrdn operation.
Returns 0 if it completes successfully.

=cut

sub post_modrdn
{
    return 0;
}

=pod

=head2 pre_delete

This method is called before L<LISM> do the delete operation.
Returns 0 if it completes successfully.

=cut

sub pre_delete
{
    return 0;
}

=pod

=head2 post_delete

This method is called before L<LISM> do the delete operation.
Returns 0 if it completes successfully.

=cut

sub post_delete
{
    return 0;
}

=pod

=head2 log(level, message)

log message to syslog.

=cut

sub log
{
    my $self = shift;
    my $conf = $self->{_config};
    my %p = @_;

    openlog('LISM', 'pid', 'local4');
    setlogmask(Sys::Syslog::LOG_UPTO($conf->{numloglevel}{$conf->{sysloglevel}}));
    syslog($conf->{numloglevel}{$p{'level'}}, $p{'message'});
    closelog
}

sub _checkConfig
{
    return 0;
}

=head1 SEE ALSO

L<LISM>

=head1 AUTHOR

Kaoru Sekiguchi, <sekiguchi.kaoru@secioss.co.jp>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2006 by Kaoru Sekiguchi

This library is free software; you can redistribute it and/or modify
it under the GNU LGPL.

=cut

1;
