/*
 * Copyright (C) 2003-2006 Kouji Sugisawa. All rights reserved.
 */

package jp.sourceforge.livez.mail;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Properties;

import javax.mail.Address;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.FolderNotFoundException;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.NoSuchProviderException;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.mail.search.SearchTerm;

import org.apache.commons.lang.StringUtils;

import jp.sourceforge.livez.lang.JaStringUtils;

/**
 * [M@\񋟂NXłB
 * 
 * @author V _
 */
public final class MailRetriever {

	/**
	 * [vt@Cێ܂B
	 */
	protected MailProfile profile = null;

	/**
	 * [ZbVێ܂B
	 */
	protected Session session = null;

	/**
	 * XgAێ܂B
	 */
	private Store store;

	/**
	 * JgtH_ێ܂B
	 */
	private Folder currentFolder = null;


	/**
	 * [vt@C񂩂NX\zRXgN^łB
	 * 
	 * @param profile [vt@C
	 * @param debug fobO[h
	 * @throws NoSuchProviderException vt@CŎw肳ꂽMvgR̃voC_Ȃꍇ
	 */
	public MailRetriever(final MailProfile profile, final boolean debug) throws NoSuchProviderException {
		this.profile = profile;
		Properties prop = profile.getProperties();
		if (debug)
			prop.put("mail.debug", "true");
		session = Session.getInstance(prop, null);
		store = session.getStore(profile.receive.protocol);
	}

	/**
	 * [vt@C񂩂NX\zRXgN^łB
	 * 
	 * @param profile [vt@C
	 * @throws NoSuchProviderException vt@CŎw肳ꂽMvgR̃voC_Ȃꍇ
	 */
	public MailRetriever(final MailProfile profile) throws NoSuchProviderException {
		this(profile, false);
	}

	/**
	 * [MT[o֐ڑ܂B<p>
	 * ڑɎgp郆[UyуpX[h̓[vt@CɊ܂߂ꂽgp܂B
	 * 
	 * @throws MessagingException
	 */
	public void connect() throws MessagingException {
		connect(profile.receive.account, profile.receive.password);
	}

	/**
	 * [MT[o֐ڑ܂B
	 * 
	 * @param account [U
	 * @param password pX[h
	 * @throws MessagingException
	 */
	private void connect(final String account, final String password) throws MessagingException {
		disconnect();
		store.connect(profile.receive.hostname, profile.receive.port, account, password);

		try {
			setCurrentFolder("INBOX");
		} catch (MessagingException e) {
			disconnect();
			throw e;
		}
	}

	/**
	 * JgtH_ݒ肵܂B
	 * 
	 * @param expunge ݂̃JgtH_ɑ΂
	 * @param name tH_
	 * @param readOnly ǎpŊJǂ
	 * @throws MessagingException
	 */
	public void setCurrentFolder(final boolean expunge, final String name, final boolean readOnly) throws MessagingException {
		closeCurrentFolder(expunge);

		if (StringUtils.isEmpty(name))
			currentFolder = store.getFolder("INBOX");
		else
			currentFolder = store.getFolder(JaStringUtils.microsoftUnicodeToUnicode(name));

		if (currentFolder == null)
			throw new FolderNotFoundException();

		if (readOnly)
			currentFolder.open(Folder.READ_ONLY);
		else
			currentFolder.open(Folder.READ_WRITE);
	}

	/**
	 * JgtH_ݒ肵܂B
	 * 
	 * @param expunge
	 * @param name
	 * @throws MessagingException
	 */
	public void setCurrentFolder(final boolean expunge, final String name) throws MessagingException {
		setCurrentFolder(expunge, name, true);
	}

	/**
	 * JgtH_ݒ肵܂B
	 * 
	 * @param path
	 * @param readOnly
	 * @throws MessagingException
	 */
	public void setCurrentFolder(final String path, final boolean readOnly) throws MessagingException {
		setCurrentFolder(false, path, readOnly);
	}

	/**
	 * JgtH_ݒ肵܂B
	 * 
	 * @param path
	 * @throws MessagingException
	 */
	public void setCurrentFolder(final String path) throws MessagingException {
		setCurrentFolder(false, path);
	}

	/**
	 * JgtH_JĂꍇ͕܂B
	 */
	private void closeCurrentFolder(boolean expunge) {
		if (currentFolder != null && currentFolder.isOpen()) {
			try {
				currentFolder.close(expunge);
			} catch (MessagingException e) {}
			currentFolder = null;
		}
	}

	/**
	 * [MT[oؒf܂B
	 */
	public void disconnect() {
		disconnect(false);
	}

	/**
	 * [MT[oؒf܂B
	 * 
	 * @param expunge 
	 */
	public void disconnect(final boolean expunge) {
		closeCurrentFolder(expunge);
		try {
			store.close();
		} catch (MessagingException e) {}
	}

	/**
	 * [MT[oւ̐ڑ󋵂Ԃ܂B
	 * 
	 * @return T[o֐ڑĂꍇ <code>true</code> AȊȌꍇ <code>false</code> Ԃ܂B
	 */
	public boolean isConnected() {
		return ((store != null) && (store.isConnected()));
	}

	/**
	 * JgtH_JĂ邩ǂԂ܂B
	 * 
	 * @return JgtH_JĂꍇ̂ <code>true</code> Ԃ܂BȊȌꍇ <code>false</code> Ԃ܂B
	 */
	public boolean isOpen() {
		return ((currentFolder != null) && (currentFolder.isOpen()));
	}

	/**
	 * TutH_̈ꗗԂ܂B
	 * 
	 * @throws MessagingException
	 */
	public String[] getFolderNames() throws MessagingException {
		Folder[] folders = currentFolder.listSubscribed();//.list("*");
		String[] results = new String[folders.length];
		for (int index = 0; index < folders.length; index++) {
			results[index] = JaStringUtils.unicodeToMicrosoftUnicode(folders[index].getFullName());
			//if (StringUtils.isEmpty(results[index])) results[index] = "INBOX";
		}
		return results;
	}

	/**
	 * tH_̈ꗗԂ܂B
	 * 
	 * @throws MessagingException
	 */
	public String[] getAllFolderNames() throws MessagingException {
		Folder[] folders = store.getDefaultFolder().list("%");
		String[] results = new String[folders.length];
		for (int index = 0; index < folders.length; index++) {
			results[index] = JaStringUtils.unicodeToMicrosoftUnicode(folders[index].getFullName());
			//if (StringUtils.isEmpty(results[index])) results[index] = "INBOX";
		}
		return results;
	}


	// ---------------------------------------- Message count

	/**
	 * ݂̃tH_̐VKbZ[W̐擾܂B<p>
	 * POP3 vgȐꍇ͕K 0 Ԃ܂B
	 * 
	 * @throws MessagingException
	 */
	public int getNewMessageCount() throws MessagingException {
		return currentFolder.getNewMessageCount();
	}

	/**
	 * ݂̃tH_̖ǃbZ[W̐擾܂B<p>
	 * POP3 vgȐꍇ́AVłĂǂƂăJEg܂B
	 * 
	 * @throws MessagingException
	 */
	public int getUnreadMessageCount() throws MessagingException {
		return currentFolder.getUnreadMessageCount();
	}

	public int getMessageCount() throws MessagingException {
		return currentFolder.getMessageCount();
	}

	/**
	 * w肳ꂽtH_݂邩ǂԂ܂B
	 * 
	 * @param path
	 * @throws MessagingException
	 */
	public boolean isExistsFolder(final String path) throws MessagingException {
		Folder folder = getFolder(path);
		return folder.exists();
	}
	
	/**
	 * path Ŏw肳ꂽtH_쐬Lɂ܂B<p>
	 * w肳ꂽtH_ɑ݂ꍇ͖܂B
	 * 
	 * @param path pX
	 * @throws MessagingException
	 */
	public boolean createFolder(final String path) throws MessagingException {
		Folder folder = getFolder(path);
		if (!folder.exists()) {
			if (folder.create(Folder.HOLDS_FOLDERS | Folder.HOLDS_MESSAGES)) {
				folder.setSubscribed(true);
				return true;
			}
		}
		return false;
	}

	/**
	 * tH_ύX܂B
	 * 
	 * @param path1 ύXÕtH_
	 * @param path2 ύX̃tH_
	 * @return ɃtH_ύXꂽꍇ̂ true Ԃ܂B
	 * @throws MessagingException
	 */
	public boolean renameFolder(final String path1, final String path2) throws MessagingException {
		Folder folder1 = getFolder(path1);
		Folder folder2 = getFolder(path2);
		if (folder1.exists() && !folder2.exists()) {
			if (folder1.isOpen())
				folder1.close(false);
			return folder1.renameTo(folder2);
		}
		return false;
	}

	/**
	 * path Ŏw肳ꂽtH_폜܂B
	 * 
	 * @param path pX
	 * @throws MessagingException
	 */
	public boolean deleteFolder(final String path) throws MessagingException {
		Folder folder = getFolder(path);
		if (folder.exists())
			return folder.delete(true);
		return false;
	}

	/**
	 * path Ŏw肳ꂽtH_擾܂B
	 * 
	 * @param path pX
	 * @return tH_IuWFNg
	 * @throws MessagingException
	 */
	private Folder getFolder(final String path) throws MessagingException {
		String separator = String.valueOf(currentFolder.getSeparator());
		Folder folder;

		if (path.startsWith(separator))
			// Ύw̏ꍇ
			folder = store.getFolder(currentFolder.getFullName() + separator + path);
		else
			// Ύw̏ꍇ
			folder = store.getFolder(path);

		return folder;
	}


	// ---------------------------------------- getMessages

	public Message[] getMessages() throws MessagingException {
		return currentFolder.getMessages();
	}

	/**
	 * [MAMimeMessage ̔zƂĕԂ܂B
	 * 
	 * @return MimeMessage ̔z
	 * @throws MessagingException
	 */
	public MimeMessage[] getMimeMessages() throws MessagingException {
		Message[] messages = currentFolder.getMessages();
		MimeMessage[] mails = new MimeMessage[messages.length];

		for (int index = 0; index < messages.length; index++)
			mails[index] = (MimeMessage) messages[index];

		return mails;
	}

	/**
	 * [MAMailMessage ̔zƂĕԂ܂B
	 * 
	 * @return MailMessage ̔z
	 * @throws MessagingException
	 */
	public MailMessage[] getMailMessages() throws MessagingException {
		Message[] messages = currentFolder.getMessages();
		MailMessage[] mails = new MailMessage[messages.length];

		for (int index = 0; index < messages.length; index++)
			mails[index] = new MailMessageImpl((MimeMessage)messages[index]);

		return mails;
	}

	/**
	 * w肳ꂽtOɊ֘A郁bZ[W̔zԂ܂B
	 * 
	 * @param flag tO
	 * @param match true ̏ꍇ̓tOƈv郁bZ[WAȊȌꍇ̓tOƈvȂbZ[W𒊏o܂
	 * @return MailMessage ̔z
	 * @throws MessagingException
	 */
	public MailMessage[] getMailMessages(final Flags.Flag flag, final boolean match) throws MessagingException {
		Message[] messages = currentFolder.getMessages();

		List<MailMessage> list = new ArrayList<MailMessage>(messages.length);
		for (int index = 0; index < messages.length; index++) {
			Flags.Flag[] sf = messages[index].getFlags().getSystemFlags();
			boolean proccess = !match;

			for (int i = 0; i < sf.length; i++) {
				if (sf[i] == flag) {
					proccess = match;
					break;
				}
			}

			if ((match && proccess) || (!match && proccess))
				list.add(new MailMessageImpl((MimeMessage)messages[index]));
		}

		MailMessage[] mails = new MailMessage[list.size()];
		for (int index = 0; index < list.size(); index++)
			mails[index] = list.get(index);

		return mails;
	}


	// ---------------------------------------- APPEND

	/**
	 * w肳ꂽ MailMessage ݂̃tH_ɒǉ܂B<p>
	 * 
	 * @param message MailMessage
	 * @throws MessagingException ǉsꍇ
	 */
	public void appendMessages(final MailMessage message) throws MessagingException {
		appendMessages(((MailMessageImpl)message).message);
	}

	/**
	 * w肳ꂽ MimeMessage ݂̃tH_ɒǉ܂B<p>
	 * 
	 * @param message MimeMessage
	 * @throws MessagingException ǉsꍇ
	 */
	public void appendMessages(final MimeMessage message) throws MessagingException {
		message.setFrom(null);
		Address[] address = new InternetAddress[1];
		try {
			address[0] = new InternetAddress(profile.address, profile.username, "ISO-2022-JP");
		} catch (UnsupportedEncodingException e) {
			throw new MessagingException();
		}
		message.addFrom(address);
		message.setSentDate(new Date());
		message.setHeader("Content-Transfer-Encoding", "7bit");//iG̏ꍇ8bit
		message.saveChanges();

		currentFolder.appendMessages(new Message[] { message });
	}


	// ---------------------------------------- COPY

	/**
	 * ݂̃tH_w肳ꂽ MailMessage ʂ̃tH_ɃRs[܂B<p>
	 * 
	 * @param mail MailMessage
	 * @param folder bZ[W̃Rs[tH_
	 * @throws MessagingException
	 */
	public void copyMessage(final MailMessage mail, final String folder) throws MessagingException {
		copyMessage(((MailMessageImpl)mail).message, folder);
	}

	/**
	 * ݂̃tH_w肳ꂽ MimeMessage ʂ̃tH_ɃRs[܂B<p>
	 * 
	 * @param message MimeMessage
	 * @param folder bZ[W̃Rs[tH_
	 * @throws MessagingException
	 */
	protected void copyMessage(final MimeMessage message, final String folder) throws MessagingException {
		currentFolder.copyMessages(new Message[] { message }, getFolder(folder));
	}


	// ---------------------------------------- SEARCH

	/**
	 * w肳ꂽɈv郁bZ[WɂČ݂̃tH_܂B<p>
	 * 
	 * @param term 
	 * @return v郁bZ[W̔z
	 * @throws MessagingException
	 */
	public Message[] search(final SearchTerm term) throws MessagingException {
		return currentFolder.search(term);
	}

	/**
	 * w肳ꂽɈv郁bZ[WɂČ݂̃tH_܂B<p>
	 * 
	 * @param term 
	 * @return v郁bZ[W̔z
	 * @throws MessagingException
	 */
	public MimeMessage[] searchMimeMessages(final SearchTerm term) throws MessagingException {
		Message[] messages = currentFolder.search(term);
		MimeMessage[] results = new MimeMessage[messages.length];

		for (int i = 0; i < messages.length; i++)
			results[i] = (MimeMessage) messages[i];

		return results;
	}

	/**
	 * w肳ꂽɈv郁bZ[WɂČ݂̃tH_܂B<p>
	 * 
	 * @param term 
	 * @return v郁bZ[W̔z
	 * @throws MessagingException
	 */
	public MailMessage[] searchMailMessages(final SearchTerm term) throws MessagingException {
		Message[] messages = currentFolder.search(term);
		MailMessage[] results = new MailMessage[messages.length];

		for (int i = 0; i < messages.length; i++)
			results[i] = new MailMessageImpl((MimeMessage)messages[i]);

		return results;
	}

}
