/*
 * Copyright 1997-2004 Sun Microsystems, Inc. All Rights Reserved.
 */

package com.sun.mail.imap;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.Date;
import java.util.Hashtable;
import java.util.NoSuchElementException;
import java.util.Vector;

import javax.mail.FetchProfile;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.FolderClosedException;
import javax.mail.FolderNotFoundException;
import javax.mail.Message;
import javax.mail.MessageRemovedException;
import javax.mail.MessagingException;
import javax.mail.Quota;
import javax.mail.ReadOnlyFolderException;
import javax.mail.StoreClosedException;
import javax.mail.UIDFolder;
import javax.mail.event.ConnectionEvent;
import javax.mail.event.FolderEvent;
import javax.mail.event.MessageChangedEvent;
import javax.mail.internet.MimeMessage;
import javax.mail.search.FlagTerm;
import javax.mail.search.SearchException;
import javax.mail.search.SearchTerm;

import com.sun.mail.iap.BadCommandException;
import com.sun.mail.iap.CommandFailedException;
import com.sun.mail.iap.ConnectionException;
import com.sun.mail.iap.Literal;
import com.sun.mail.iap.ProtocolException;
import com.sun.mail.iap.Response;
import com.sun.mail.iap.ResponseHandler;
import com.sun.mail.imap.protocol.FetchResponse;
import com.sun.mail.imap.protocol.IMAPProtocol;
import com.sun.mail.imap.protocol.IMAPResponse;
import com.sun.mail.imap.protocol.ListInfo;
import com.sun.mail.imap.protocol.MailboxInfo;
import com.sun.mail.imap.protocol.MessageSet;
import com.sun.mail.imap.protocol.Status;
import com.sun.mail.imap.protocol.UID;
import com.sun.mail.util.CRLFOutputStream;


/**
 * ̃NX IMAP tH_܂B<p>
 * 
 * Ă IMAPFolder IuWFNg IMAPStore IuWFNgƂ̃vgRڑL܂B<p>
 * 
 * Applications that need to make use of IMAP-specific features may cast
 * a <code>Folder</code> object to an <code>IMAPFolder</code> object and
 * use the methods on this class. The {@link #getQuota getQuota} and
 * {@link #setQuota setQuota} methods support the IMAP QUOTA extension.
 * Refer to <A HREF="http://www.ietf.org/rfc/rfc2087.txt">RFC 2087</A>
 * for more information. <p>
 * 
 * The {@link #getACL getACL}, {@link #addACL addACL},
 * {@link #removeACL removeACL}, {@link #addRights addRights},
 * {@link #removeRights removeRights}, {@link #listRights listRights}, and
 * {@link #myRights myRights} methods support the IMAP ACL extension.
 * Refer to <A HREF="http://www.ietf.org/rfc/rfc2086.txt">RFC 2086</A>
 * for more information. <p>
 * 
 * The {@link #doCommand doCommand} method and
 * {@link IMAPFolder.ProtocolCommand IMAPFolder.ProtocolCommand}
 * interface support use of arbitrary IMAP protocol commands. <p>
 * 
 * See the <a href="package-summary.html">jp.sourceforge.livez.mail.imap</a> package
 * documentation for further information on the IMAP protocol provider. <p>
 * 
 * <strong>WARNING:</strong> The APIs unique to this class should be
 * considered <strong>EXPERIMENTAL</strong>.  They may be changed in the
 * future in ways that are incompatible with applications using the
 * current APIs.
 */
/*
 * The folder object itself serves as a lock for the folder's state
 * EXCEPT for the message cache (see below), typically by using
 * synchronized methods.  When checking that a folder is open or
 * closed, the folder's lock must be held.  It's important that the
 * folder's lock is acquired before the messageCacheLock (see below).
 * Thus, the locking hierarchy is that the folder lock, while optional,
 * must be acquired before the messageCacheLock, if it's acquired at
 * all.  Be especially careful of callbacks that occur while holding
 * the messageCacheLock into (e.g.) superclass Folder methods that are
 * synchronized.  Note that methods in IMAPMessage will acquire the
 * messageCacheLock without acquiring the folder lock. <p>
 * 
 * When a folder is opened, it creates a messageCache (a Vector) of 
 * empty IMAPMessage objects. Each Message has a messageNumber - which
 * is its index into the messageCache, and a sequenceNumber - which is
 * its IMAP sequence-number. All operations on a Message which involve
 * communication with the server, use the message's sequenceNumber. <p>
 * 
 * The most important thing to note here is that the server can send
 * unsolicited EXPUNGE notifications as part of the responses for "most"
 * commands. Refer RFC2060, sections 5.3 &  5.5 for gory details. Also, 
 * the server sends these  notifications AFTER the message has been 
 * expunged. And once a message is expunged, the sequence-numbers of 
 * those messages after the expunged one are renumbered. This essentially
 * means that the mapping between *any* Message and its sequence-number 
 * can change in the period when a IMAP command is issued and its responses
 * are processed. Hence we impose a strict locking model as follows: <p>
 * 
 * We define one mutex per folder - this is just a Java Object (named 
 * messageCacheLock). Any time a command is to be issued to the IMAP
 * server (i.e., anytime the corresponding IMAPProtocol method is
 * invoked), follow the below style:
 *		
 *	synchronized (messageCacheLock) { // ACQUIRE LOCK
 *	    issue command ()
 *	    
 *	    // The response processing is typically done within
 *	    // the handleResponse() callback. A few commands (Fetch,
 *	    // Expunge) return *all* responses and hence their
 *	    // processing is done here itself. Now, as part of the
 *	    // processing unsolicited EXPUNGE responses, we renumber
 *	    // the necessary sequence-numbers. Thus the renumbering
 *	    // happens within this critical-region, surrounded by
 *	    // locks.
 *	    process responses ()
 *	} // RELEASE LOCK
 * 
 * This technique is used both by methods in IMAPFolder and by methods
 * in IMAPMessage and other classes that operate on data in the folder.
 * Note that holding the messageCacheLock has the side effect of
 * preventing the folder from being closed, and thus ensuring that the
 * folder's protocol object is still valid.  The protocol object should
 * only be accessed while holding the messageCacheLock (except for calls
 * to IMAPProtocol.isREV1(), which don't need to be protected because it
 * doesn't access the server).
 * 
 * Note that interactions with the Store's protocol connection do
 * not have to be protected as above, since the Store's protocol is
 * never in a "meaningful" SELECT-ed state.
 */
public class IMAPFolder extends Folder implements UIDFolder, ResponseHandler {

	protected String fullName;					// tH_(tl[)
	protected String name;						// tH_
	protected int type;						// tH_ ^Cv 
	protected char separator;					// Zp[^(؂蕶)
	protected Flags availableFlags;			// p\ȃtOQ
	protected Flags permanentFlags;			// iIȃtOQ
	protected boolean exists = false;		// ̃tH_{ɑ݂邩ǂ
	protected boolean isNamespace = false;	// tH_OԖǂ
	protected String attributes[];

	protected IMAPProtocol protocol;			// ̃tH_g̃vgRIuWFNg
	protected Vector messageCache;				// bZ[WLbV
	protected Object messageCacheLock;			// bZ[WLbṼANZbTbN

	protected Hashtable uidTable;				// UID -> bZ[W nbVe[u

	/* IMAP ؂蕶 7 rbg US-ASCII łB(NUL )
	 * We use '\uffff' (a non 7bit character) to indicate that we havent
	 * yet determined what the separator character is.
	 * We use '\u0000' (NUL) to indicate that no separator character
	 * exists, i.e., a flat hierarchy
	 */
	protected static final char UNKNOWN_SEPARATOR = '\uffff';

	private boolean opened = false;		// ̃tH_JĂ邩

	/* This field tracks the state of this folder. If the folder is closed
	 * due to external causes (i.e, not thru the close() method), then
	 * this field will remain false. If the folder is closed thru the
	 * close() method, then this field is set to true.
	 * 
	 * If reallyClosed is false, then a FolderClosedException is
	 * generated when a method is invoked on any Messaging object
	 * owned by this folder. If reallyClosed is true, then the
	 * IllegalStateException runtime exception is thrown.
	 */
	private boolean reallyClosed = true;

	private int total = -1;				// bZ[WLbṼbZ[W̍v
	private int recent = -1;				// RECENT bZ[W̐
	private int realTotal = -1;			// T[õbZ[W̍v
	private int uidvalidity = -1;			// UIDValidity
	private int uidnext = -1;
	private boolean doExpungeNotification = true;	// used in expunge handler

	private Status cachedStatus = null;
	private long cachedStatusTime = 0;

	private boolean debug = false;
	private PrintStream out;				// fobOpo̓Xg[

	private boolean connectionPoolDebug;

// 2004/06/30 Sugisawa added.
private String charset = null;

	/**
	 * tFb`wb_Q̃tFb`vtB[ڂłB
	 * ̓NX IMAPFolder ɓ̐V FetchProfile ڃ^Cvׂ <code>FetchProfile.Item</code> NXg܂B
	 * 
	 * @see FetchProfile
	 */
	public static final class FetchProfileItem extends FetchProfile.Item {

		protected FetchProfileItem(final String name) {
			super(name);
		}

		/**
		 * HEADERS ̓tFb`v̊ <code>FetchProfile</code>  Folder Ɋ܂ގłtFb`vtB[ڂłB
		 * This item indicates that the headers for messages in the specified 
		 * range are desired to be prefetched. <p>
		 * 
		 * NCAgł̎gpȉɎ܂:<p>
		 * <blockquote><pre>
		 * 
		 * 	FetchProfile fp = new FetchProfile();
		 *	fp.add(IMAPFolder.FetchProfileItem.HEADERS);
		 *	folder.fetch(msgs, fp);
		 * 
		 * </pre></blockquote><p>
		 */ 
		public static final FetchProfileItem HEADERS = new FetchProfileItem("HEADERS");

		/**
		 * SIZE ̓tFb`v̊ <code>FetchProfile</code>  Folder Ɋ܂ގłtFb`vtB[ڂłB
		 * This item indicates that the sizes of the messages in the specified 
		 * range are desired to be prefetched. <p>
		 * 
		 * SIZE should move to FetchProfile.Item in JavaMail 1.3.
		 */
		public static final FetchProfileItem SIZE = new FetchProfileItem("SIZE");
	}

	/**
	 * Constructor used to create a possibly non-existent folder.
	 * 
	 * @param fullName	̃tH_̃tl[
	 * @param separator ̃tH_̖OԂ̊̋؂蕶
	 * @param store Store
	 */
	protected IMAPFolder(final String fullName, final char separator, final IMAPStore store) {
		this(fullName, separator, store, false);
	}

	/**
	 * Constructor used to create a possibly non-existent folder.
	 * 
	 * @param fullName	̃tH_̃tl[
	 * @param separator ̃tH_̖OԂ̊̋؂蕶
	 * @param store Store
	 */
	protected IMAPFolder(
		final String fullName,
		final char separator,
		final IMAPStore store,
		final boolean isNamespace) {

		super(store);

		if (fullName == null)
			throw new NullPointerException("Folder name is null");

		this.fullName = fullName;
		this.separator = separator;
		this.isNamespace = isNamespace;
		messageCacheLock = new Object();
		debug = store.getSession().getDebug();
		connectionPoolDebug = store.getConnectionPoolDebug();

		out = store.getSession().getDebugOut();
		if (out == null)	// should never happen
			out = System.out;
// Sugisawa added. 2005/05/07
this.charset = store.getSession().getProperty("mail.imap.folder.charset");
	}

	/**
	 * Constructor used to create an existing folder.
	 */
	protected IMAPFolder(final ListInfo li, final IMAPStore store) {
		this(li.name, li.separator, store);
		if (li.hasInferiors)
		    type |= HOLDS_FOLDERS;
		if (li.canOpen)
		    type |= HOLDS_MESSAGES;
		exists = true;
		attributes = li.attrs;
	}

	/* Ensure that this folder exists. If 'exists' has been set to true,
	 * we don't attempt to validate it with the server again. Note that
	 * this can result in a possible loss of sync with the server.
	 */
	private void checkExists() throws MessagingException {
		// If the boolean field 'exists' is false, check with the
		// server by invoking exists() ..
		if (!exists && !exists())
			throw new FolderNotFoundException(this, fullName + " not found");
	}

	/*
	 * tH_Ă鎖ۏ؂܂B
	 * ASSERT: Must be called with this folder's synchronization lock held.
	 */
	private void checkClosed() {
		if (opened)
			throw new IllegalStateException("This operation is not allowed on an open folder");
	}

	/*
	 * tH_JĂ鎖ۏ؂܂B
	 * ASSERT: Must be called with this folder's synchronization lock held.
	 */
	private void checkOpened() throws FolderClosedException {
		if (!opened) {
			if (reallyClosed)
				throw new IllegalStateException("This operation is not allowed on a closed folder");
			// Folder was closed "implicitly"
			throw new FolderClosedException(this, "Lost folder connection to server");
		}
	}

	/* Check that the given message number is within the range
	 * of messages present in this folder. If the message
	 * number is out of range, we ping the server to obtain any
	 * pending new message notifications from the server.
	 */
	private void checkRange(final int msgno) throws MessagingException {
		if (msgno < 1) // message-numbers start at 1
			throw new IndexOutOfBoundsException();

		if (msgno <= total)
			return;

		// Out of range, let's ping the server and see if
		// the server has more messages for us.

		synchronized(messageCacheLock) { // Acquire lock
			try {
				keepConnectionAlive(false);
			} catch (ConnectionException cex) {
				// Oops, lost connection
				throw new FolderClosedException(this, cex.getMessage());
			} catch (ProtocolException pex) { 
				throw new MessagingException(pex.getMessage(), pex);
			}
		} // Release lock

		if (msgno > total) // Still out of range ? Throw up ...
			throw new IndexOutOfBoundsException();
	}

	/* Check whether the given flags are supported by this server,
	 * and also verify that the folder allows setting flags.
	 */
	private void checkFlags(final Flags flags) throws MessagingException {
		if (mode != READ_WRITE)
			throw new IllegalStateException("Cannot change flags on READ_ONLY folder: " + fullName);
		if (!availableFlags.contains(flags))
			throw new MessagingException("These flags are not supported by this implementation");
	}

	/**
	 *  Folder ̖OԂ܂B
	 */
	public String getName() {
		/* Return the last component of this Folder's full name.
		 * Folder components are delimited by the separator character.
		 */
		if (name == null) {
			try {
				name = 	fullName.substring(fullName.lastIndexOf(getSeparator()) + 1);
			} catch (MessagingException mex) {}
		}
		return name;
	}

	/**
	 *  Folder ̊SȖOԂ܂B
	 */
	public String getFullName() {
		return fullName;	
	}

	/**
	 * ̃tH_̐etH_Ԃ܂B
	 */
	public Folder getParent() throws MessagingException {
		char c = getSeparator();
		int index;
		if ((index = fullName.lastIndexOf(c)) != -1)
			return new IMAPFolder(fullName.substring(0, index), c, (IMAPStore)store);
		return new DefaultFolder((IMAPStore)store);
	}

	/**
	 * ̃tH_{ɃT[oɑ݂邩ǂ܂B
	 */
	public final boolean exists() throws MessagingException {
		// Check whether this folder exists ..
		ListInfo[] li = null;
		final String lname;
		if(isNamespace && separator != '\0')
			lname = fullName + separator;
		else
			lname = fullName;

		li = (ListInfo[]) doCommand(new ProtocolCommand() {
			public Object doCommand(IMAPProtocol p) throws ProtocolException {
				return p.list("", lname);
			}
		});

		if (li != null) {
			int i = findName(li, lname);
			fullName = li[i].name;
			separator = li[i].separator;
			int len = fullName.length();
			if(separator != '\0' && len > 0 && fullName.charAt(len - 1) == separator)
				fullName = fullName.substring(0, len - 1);
			type = 0;
			if (li[i].hasInferiors)
				type |= HOLDS_FOLDERS;
			if (li[i].canOpen)
				type |= HOLDS_MESSAGES;
			exists = true;
			attributes = li[i].attrs;
		} else
			exists = false;

		return exists;
	}

	/**
	 * Which entry in <code>li</code> matches <code>lname</code>?
	 * If the name contains wildcards, more than one entry may be
	 * returned.
	 */
	private int findName(final ListInfo li[], final String lname) {
		int i;
		// if the name contains a wildcard, there might be more than one
		for (i = 0; i < li.length; i++)
			if (li[i].name.equals(lname))
				break;

		if (i >= li.length) {	// nothing matched exactly
			// XXX - possibly should fail?  But what if server
			// is case insensitive and returns the preferred
			// case of the name here?
			i = 0;	// use first one
		}
		return i;
	}

	/**
	 * w肳ꂽp^[ɈvSẴTutH_̈ꗗԂ܂B
	 */
	public Folder[] list(final String pattern) throws MessagingException {
		return doList(pattern, false);
	}

	/**
	 * w肳ꂽp^[ɈvTuXNCuꂽSẴTutH_̈ꗗԂ܂B
	 */
	public Folder[] listSubscribed(final String pattern) throws MessagingException {
		return doList(pattern, true);
	}

	private Folder[] doList(final String pattern, final boolean subscribed) 
		throws MessagingException {

		checkExists(); // insure that this folder does exist.

		if (!isDirectory()) // Why waste a roundtrip to the server ?
			return new Folder[0];

		final char c = getSeparator();

		ListInfo[] li = (ListInfo[]) doCommandIgnoreFailure(
			new ProtocolCommand() {
				public Object doCommand(IMAPProtocol p) throws ProtocolException {
					if (subscribed)
						return p.lsub("", fullName + c + pattern);
					return p.list("", fullName + c + pattern);
				}
			}
		);

		if (li == null)
			return new Folder[0];

		/*
		 * The UW based IMAP4 servers (e.g. SIMS2.0) include
		 * current folder (terminated with the separator), when
		 * the LIST pattern is '%' or '*'. i.e, <LIST "" mail/%> 
		 * returns "mail/" as the first LIST response.
		 * 
		 * Doesn't make sense to include the current folder in this
		 * case, so we filter it out. Note that I'm assuming that
		 * the offending response is the *first* one, my experiments
		 * with the UW & SIMS2.0 servers indicate that .. 
		 */
		int start = 0;
		// ŏ LIST X|X܂B
		if (li.length > 0 && li[0].name.equals(fullName + c)) 
			start = 1; // start from index = 1

		IMAPFolder[] folders = new IMAPFolder[li.length - start];
		for (int i = start; i < li.length; i++)
			folders[i - start] = new IMAPFolder(li[i], (IMAPStore)store);

		return folders;
	}

	/**
	 * ؂蕶Ԃ܂B
	 */
	public synchronized char getSeparator() throws MessagingException {
		if (separator == UNKNOWN_SEPARATOR) {
			ListInfo[] li = null;

			li = (ListInfo[]) doCommand(new ProtocolCommand() {
				public Object doCommand(IMAPProtocol p) throws ProtocolException {
					// REV1 allows the following LIST format to obtain
					// the hierarchy delimiter of non-existent folders
					if (p.isREV1()) // IMAP4rev1
						return p.list(fullName, "");
					// IMAP4, note that this folder must exist for this to work :(
					return p.list("", fullName);
				}
			});

			if (li != null) 
				separator = li[0].separator;
			else
				separator = '/'; // punt !
		}
		return separator;
	}

	/**
	 * ̃tH_̌^Ԃ܂B
	 */
	public final int getType() throws MessagingException {
		checkExists();
		return type;
	}

	/**
	 * ̃tH_TuXNCuꂩ܂B<p>
	 */
	public final boolean isSubscribed() {
		ListInfo[] li = null;
		final String lname;
		if(isNamespace && separator != '\0')
			lname = fullName + separator;
		else
			lname = fullName;

		try {
			li = (ListInfo[]) doProtocolCommand(new ProtocolCommand() {
				public Object doCommand(IMAPProtocol p) throws ProtocolException {
					return p.lsub("", lname);
				}
			});
		} catch (ProtocolException pex) {}

		if (li != null) {
			int i = findName(li, lname);
			return li[i].canOpen;
		}
		return false;
	}

	/**
	 * ̃tH_TuXNCu̓TuXNCu܂B
	 */
	public final void setSubscribed(final boolean subscribe) throws MessagingException {
		doCommandIgnoreFailure(new ProtocolCommand() {
			public Object doCommand(IMAPProtocol p) throws ProtocolException {
				if (subscribe)
					p.subscribe(fullName);
				else
					p.unsubscribe(fullName);
				return null;
			}
		});
	}

	/**
	 * w肳ꂽ^ł̃tH_쐬܂B
	 */
	public synchronized boolean create(final int type) throws MessagingException {
		char c = 0;
		if ((type & HOLDS_MESSAGES) == 0)	// only holds folders
			c = getSeparator();
		final char sep = c;
		Object ret = doCommandIgnoreFailure(new ProtocolCommand() {
			public Object doCommand(IMAPProtocol p) throws ProtocolException {
				if ((type & HOLDS_MESSAGES) == 0)	// only holds folders
					p.create(fullName + sep);
				else {
					p.create(fullName);

					// Certain IMAP servers do not allow creation of folders
					// that can contain messages *and* subfolders. So, if we
					// were asked to create such a folder, we should verify
					// that we could indeed do so.
					if ((type & HOLDS_FOLDERS) != 0) {
						// we want to hold subfolders and messages. Check
						// whether we could create such a folder.
						ListInfo[] li = p.list("", fullName);
						if (li != null && !li[0].hasInferiors) {
							// Hmm ..the new folder doesn't support Inferiors ? Fail
							p.delete(fullName);
							throw new ProtocolException("Unsupported type");
					    }
					}
				}
				return Boolean.TRUE;
			}
		});

		if (ret == null)
			return false; // CREATE failure, maybe this folder already exists ?

		boolean exists = exists();
		// Notify listeners on self and our Store
		notifyFolderListeners(FolderEvent.CREATED);
		return exists;
	}

	/**
	 * ̃tH_ɐVKbZ[W݂邩܂B
	 */
	public synchronized boolean hasNewMessages() throws MessagingException {
		checkExists();

		if (opened) {	// If we are open, we already have this information
			// Folder is open, make sure information is up to date
			synchronized(messageCacheLock) {
				// tickle the folder and store connections.
				try {
					keepConnectionAlive(true);
				} catch (ConnectionException cex) {
					throw new FolderClosedException(this, cex.getMessage());
				} catch (ProtocolException pex) {
					throw new MessagingException(pex.getMessage(), pex);
				}
			}
			return (recent > 0);
		}

		// First, the cheap way - use LIST and look for the \Marked
		// or \Unmarked tag

		Boolean b = (Boolean) doCommandIgnoreFailure(new ProtocolCommand() {
			public Object doCommand(IMAPProtocol p) throws ProtocolException {
				ListInfo[] li = p.list("", fullName);
				if (li != null) {
					if (li[0].changeState == ListInfo.CHANGED)
						return Boolean.TRUE;
					else if (li[0].changeState == ListInfo.UNCHANGED)
						return Boolean.FALSE;
				}

				// LIST didn't work. Try the hard way, using STATUS
				Status status = getStatus();
				if (status.recent > 0)
					return Boolean.TRUE;
				return Boolean.FALSE;
			}
		});
		if (b == null)
			// Probably doesn't support STATUS, tough luck.
			return false;
		return b.booleanValue();
	}

	/**
	 * w肳ꂽOɑΉTutH_擾܂B<p>
	 */
	public Folder getFolder(final String name) throws MessagingException {
		// If we know that this folder is *not* a directory, don't
		// send the request to the server at all ...
		if (exists && !isDirectory())
			throw new MessagingException("Cannot contain subfolders");

		char c = getSeparator();
		return new IMAPFolder(fullName + c + name, c, (IMAPStore)store);
	}

	/**
	 * ̃tH_폜܂B
	 */
	public synchronized boolean delete(final boolean recurse) throws MessagingException {  
		checkClosed(); // insure that this folder is closed.

		if (recurse) {
			// Delete all subfolders.
			Folder[] f = list();
			for (int i = 0; i < f.length; i++)
				f[i].delete(recurse); // ignore intermediate failures
		}

		// Attempt to delete this folder

		Object ret = doCommandIgnoreFailure(new ProtocolCommand() {
			public Object doCommand(IMAPProtocol p) throws ProtocolException {
				p.delete(fullName);
				return Boolean.TRUE;
			}
		});

		if (ret == null)
			// Non-existent folder/No permission ??
			return false;

		// DELETE succeeded.
		exists = false;

		// Notify listeners on self and our Store
		notifyFolderListeners(FolderEvent.DELETED);
		return true;
	}

	/**
	 * ̃tH_̖OύX܂B<p>
	 */
	public synchronized boolean renameTo(final Folder f) throws MessagingException {
		checkClosed(); // insure that we are closed.
		checkExists();
		if (f.getStore() != store)
		    throw new MessagingException("Can't rename across Stores");

		Object ret = doCommandIgnoreFailure(new ProtocolCommand() {
		    public Object doCommand(IMAPProtocol p) throws ProtocolException {
				p.rename(fullName, f.getFullName());
				return Boolean.TRUE;
		    }
		});

		if (ret == null)
			return false;

		exists = false;
		notifyFolderRenamedListeners(f);
		return true;
	}

	/**
	 * w肳ꂽ[hł̃tH_J܂B
	 */
	public synchronized void open(final int mode) throws MessagingException {
		checkClosed(); // insure that we are not already open

		MailboxInfo mi = null;
		// Request store for our own protocol connection.
		protocol = ((IMAPStore)store).getProtocol(this);
// Sugisawa added. 2005/06/13
protocol.setCharset(this.charset);

		CommandFailedException cfex = null;

		synchronized(messageCacheLock) { // Acquire messageCacheLock
label0:{
			/*
			 * Add response handler right away so we get any alerts or
			 * notifications that occur during the SELECT or EXAMINE.
			 * Have to be sure to remove it if we fail to open the
			 * folder.
			 */
			protocol.addResponseHandler(this);

			try {
				if (mode == READ_ONLY)
					mi = protocol.examine(fullName);
				else 
					mi = protocol.select(fullName);
		    } catch (CommandFailedException cex) {
				// got a NO; connection still good, return it
				releaseProtocol(true);
				protocol = null;
				cfex = cex;
				break label0;
		    } catch (ProtocolException pex) {
				// got a BAD or a BYE; connection may be bad, close it
				try {
					protocol.logout();
				} catch (ProtocolException pex2) {
					// 
				} finally {
					releaseProtocol(false);
					protocol = null;
				}
				throw new MessagingException(pex.getMessage(), pex);
			}

			if (mi.mode != mode) {
				if (mode == READ_WRITE && mi.mode == READ_ONLY &&
					((IMAPStore)store).allowReadOnlySelect()) {
					;	// all ok, allow it
				} else {	// otherwise, it's an error
					try {
						// close mailbox and return connection
						protocol.close();
						releaseProtocol(true);
				    } catch (ProtocolException pex) {
						// something went wrong, close connection
						try {
							protocol.logout();
						} catch (ProtocolException pex2) {
							// 
						} finally {
							releaseProtocol(false);
						}
					} finally {
						protocol = null;
					}
					throw new ReadOnlyFolderException(this, "Cannot open in desired mode");
				}
			}

			// ϐ
			opened = true;
			reallyClosed = false;
			this.mode = mi.mode;
			availableFlags = mi.availableFlags;
			permanentFlags = mi.permanentFlags;
			total = realTotal = mi.total;
			recent = mi.recent;
			uidvalidity = mi.uidvalidity;
			uidnext = mi.uidnext;

			// K؂ȃTCỸbZ[WLbVz쐬܂
			messageCache = new Vector(total);
			// yʂ IMAPMessage IuWFNgŃLbV𖄂߂܂
			for (int i = 0; i < total; i++)
				messageCache.addElement(new IMAPMessage(this, i+1, i+1));

		}
		} // bN܂

		if (cfex != null) {
			checkExists(); // ݃`FbN
			if ((type & HOLDS_MESSAGES) == 0)
				throw new MessagingException("folder cannot contain messages");
			throw new MessagingException(cfex.getMessage(), cfex);
		}

		// Xi[ɒʒm܂
		notifyConnectionListeners(ConnectionEvent.OPENED);
	}

	/**
	 * vtFb` (w肳ꂽ FetchProfile ɊÂ)B
	 */
	public synchronized void fetch(final Message[] msgs, final FetchProfile fp) throws MessagingException {
		checkOpened();
		IMAPMessage.fetch(this, msgs, fp);
	}

	/**
	 * bZ[Wz񒆂̃bZ[WɎw肳ꂽtOݒ肵܂B
	 */
	public synchronized void setFlags(final Message[] msgs, final Flags flag, final boolean value) throws MessagingException {
		checkOpened();
		checkFlags(flag); // tOQ1؂܂

		if (msgs.length == 0) // boundary condition
			return;

		synchronized(messageCacheLock) {
			try {
				MessageSet ms[] = Utility.toMessageSet(msgs, null);
				if(ms == null)
					throw new MessageRemovedException("Messages have been removed");
				protocol.storeFlags(ms, flag, value);
			} catch (ConnectionException cex) {
				throw new FolderClosedException(this, cex.getMessage());
			} catch (ProtocolException pex) {
				throw new MessagingException(pex.getMessage(), pex);
			}
		}
	}

	/**
	 * ̃tH_܂B
	 */
	public synchronized void close(final boolean expunge) throws MessagingException {
		close(expunge, false);
	}

	/**
	 * T[o[҂ƂȂÃtH_܂B<p>
	 * 
	 * @since JavaMail 1.3.3
	 */
	public synchronized void forceClose() throws MessagingException {
		close(false, true);
	}

	private void close(final boolean expunge, final boolean flag1) throws MessagingException {
		synchronized(messageCacheLock) {
			/*
			 * If we already know we're closed, this is illegal.
			 * Can't use checkOpened() because if we were forcibly
			 * closed asynchronously we just want to complete the
			 * closing here.
			 */
			if (!opened && reallyClosed)
				throw new IllegalStateException("This operation is not allowed on a closed folder");

			reallyClosed = true; // Ok, lets reset

			// Maybe this folder is already closed, or maybe another
			// thread which had the messageCacheLock earlier, found
			// that our server connection is dead and cleaned up
			// everything ..
			if (!opened)
				return;

			try {
				if (flag1) {
					if (debug)
						out.println("DEBUG: forcing folder " + fullName + " to close");
					if (protocol != null)
						protocol.disconnect();
				} else

				// If the connection pool is full logout the connection,
				// otherwise  
				if (((IMAPStore)store).isConnectionPoolFull()) {
					if (debug)
						out.println("DEBUG: pool is full, not adding an Authenticated connection");

					// If the expunge flag is set, close the folder first.
					if (expunge)
						protocol.close();

					if (protocol != null)
						protocol.logout();
				} else {
					// If the expunge flag is set or we're open read-only we
					// can just close the folder, otherwise open it read-only
					// before closing.
					if (!expunge && mode == READ_WRITE) {
						try {
							protocol.examine(fullName);
						} catch (ProtocolException pex2) {
							if (protocol != null)
								protocol.disconnect();
						}
					}
					if (protocol != null)
						protocol.close();
				}
			} catch (ProtocolException pex) {
				throw new MessagingException(pex.getMessage(), pex);
			} finally {
				// cleanup if we haven't already
				if (opened)
					cleanup(true);
			}
		}
	}

	// NOTE: this method can currently be invoked from close() or
	// from handleResponses(). Both invocations are conditional,
	// based on the "opened" flag, so we are sure that multiple
	// Connection.CLOSED events are not generated. Also both
	// invocations are from within messageCacheLock-ed areas.
	private void cleanup(final boolean returnToPool) {
		releaseProtocol(returnToPool);
		protocol = null;
		messageCache = null;
		uidTable = null;
		exists = false; // to force a recheck in exists().
		opened = false;
		notifyConnectionListeners(ConnectionEvent.CLOSED);
	}

	/**
	 * ̐ڑ{ɃI[vł邩܂B
	 */
	public synchronized boolean isOpen() {
		synchronized(messageCacheLock) {
			// Probe the connection to make sure its really open.
			if (opened) {
				try {
					keepConnectionAlive(false);
				} catch (ProtocolException pex) {}
			}
		}
		return opened;
	}

	/**
	 * T[oT|[gitOԂ܂B
	 */
	public final Flags getPermanentFlags() {
		return permanentFlags;
	}

	/**
	 * bZ[W擾܂B
	 */
	public synchronized int getMessageCount() throws MessagingException {
		checkExists();
		if (!opened) {
			IMAPProtocol p;
			// If this folder is not yet open, we use STATUS to
			// get the total message count
			try {
				Status status = getStatus();
				return status.total;
			} catch (BadCommandException bex) {
				// doesn't support STATUS, probably vanilla IMAP4 ..
				// lets try EXAMINE
				p = null;
			} catch (ConnectionException cex) {
				throw new StoreClosedException(store, cex.getMessage());
			} catch (ProtocolException pex) {
				throw new MessagingException(pex.getMessage(), pex);
			}
			try {
				p = getStoreProtocol();
				MailboxInfo minfo = p.examine(fullName);
				p.close();
				return minfo.total;
			} catch (ProtocolException pex) {
				// Give up.
				throw new MessagingException(pex.getMessage(), pex);
			} finally {
				releaseStoreProtocol(p);
			}
		}

		// Folder is open, we know what the total message count is ..
		synchronized(messageCacheLock) {
			// tickle the folder and store connections.
			try {
				keepConnectionAlive(true);
				return total;
			} catch (ConnectionException cex) {
				throw new FolderClosedException(this, cex.getMessage());
			} catch (ProtocolException pex) {
				throw new MessagingException(pex.getMessage(), pex);
			}
		}
	}

	/**
	 * VKbZ[W̐擾܂B
	 */
	public synchronized int getNewMessageCount() throws MessagingException {
		checkExists();
		if (!opened) {
			IMAPProtocol p;
			// If this folder is not yet open, we use STATUS to
			// get the new message count
			try {
				Status status = getStatus();
				return status.recent;
			} catch (BadCommandException bex) {
				// doesn't support STATUS, probably vanilla IMAP4 ..
				// lets try EXAMINE
				p = null;
			} catch (ConnectionException cex) {
				throw new StoreClosedException(store, cex.getMessage());
			} catch (ProtocolException pex) {
				throw new MessagingException(pex.getMessage(), pex);
			}

			try {
				p = getStoreProtocol();
				MailboxInfo minfo = p.examine(fullName);
				p.close();
				return minfo.recent;
			} catch (ProtocolException pex) {
				// MuAbv
				throw new MessagingException(pex.getMessage(), pex);
			} finally {
				releaseStoreProtocol(p);
			}
		}

		// Folder is open, we know what the new message count is ..
		synchronized(messageCacheLock) {
			// tickle the folder and store connections.
			try {
				keepConnectionAlive(true);
				return recent;
			} catch (ConnectionException cex) {
				throw new FolderClosedException(this, cex.getMessage());
			} catch (ProtocolException pex) {
				throw new MessagingException(pex.getMessage(), pex);
			}
		}
	}

	/**
	 * ǃbZ[W̐擾܂B
	 */
	public synchronized int getUnreadMessageCount() throws MessagingException {
		checkExists();
		if (!opened) {
			// If this folder is not yet open, we use STATUS to
			// get the unseen message count
			try {
				Status status = getStatus();
				return status.unseen;
			} catch (BadCommandException bex) {
				// doesn't support STATUS, probably vanilla IMAP4 ..
				// Could EXAMINE, SEARCH for UNREAD messages and
				// return the count .. bah, not worth it.
				return -1;
			} catch (ConnectionException cex) {
				throw new StoreClosedException(store, cex.getMessage());
			} catch (ProtocolException pex) {
				throw new MessagingException(pex.getMessage(), pex);
			}
		}

		// if opened, issue server-side search for messages that do
		// *not* have the SEEN flag.
		Flags f = new Flags();
		f.add(Flags.Flag.SEEN);
		try {
			synchronized(messageCacheLock) {
				int[] matches = protocol.search(new FlagTerm(f, false));
				return matches.length; // NOTE: 'matches' is never null
			}
		} catch (ConnectionException cex) {
			throw new FolderClosedException(this, cex.getMessage());
		} catch (ProtocolException pex) {
			// Shouldn't happen
			throw new MessagingException(pex.getMessage(), pex);
		}
	}

	/**
	 * 폜ς݃bZ[W̐擾܂B
	 */
	public synchronized int getDeletedMessageCount() throws MessagingException {
		checkExists();
		if (!opened) {
			// no way to do this on closed folders
			return -1;
		}

		// if opened, issue server-side search for messages that do
		// have the DELETED flag.
		Flags f = new Flags();
		f.add(Flags.Flag.DELETED);
		try {
			synchronized(messageCacheLock) {
				int[] matches = protocol.search(new FlagTerm(f, true));
				return matches.length; // NOTE: 'matches' is never null
			}
		} catch (ConnectionException cex) {
			throw new FolderClosedException(this, cex.getMessage());
		} catch (ProtocolException pex) {
			// Shouldn't happen
			throw new MessagingException(pex.getMessage(), pex);
		}
	}

	/*
	 * ŏɃLbVÃtH_ׂ̈ STATUS R}ȟʂ擾܂B
	 * ASSERT: ̃tH_̓bNĂԂŁAĂ΂Ȃ΂Ȃ܂B
	 * ASSERT: tH_͕ĂȂ΂Ȃ܂B
	 */
	private Status getStatus() throws ProtocolException {
		int statusCacheTimeout = ((IMAPStore)store).getStatusCacheTimeout();

		// if allowed to cache and our cache is still valid, return it
		if (statusCacheTimeout > 0 && cachedStatus != null &&
			System.currentTimeMillis() - cachedStatusTime < statusCacheTimeout)
			return cachedStatus;

		IMAPProtocol p = null;

		try {
			p = getStoreProtocol();	// XXX
			Status s = p.status(fullName, null); 
			// if allowed to cache, do so
			if (statusCacheTimeout > 0) {
				cachedStatus = s;
				cachedStatusTime = System.currentTimeMillis();
			}
			return s;
		} finally {
			releaseStoreProtocol(p);
		}
	}

	/**
	 * w肳ꂽbZ[W擾܂B
	 */
	public synchronized Message getMessage(final int msgnum) throws MessagingException {
		checkOpened();
		checkRange(msgnum);
		return (Message) messageCache.elementAt(msgnum - 1);
	}

	/**
	 * w肳ꂽbZ[W̃tH_ɒǉ܂B
	 */
	public void appendMessages(final Message[] msgs) throws MessagingException {
		checkExists(); // ̃tH_݂鎖mF܂

		// XXX - have to verify that messages are in a different
		// store (if any) than target folder, otherwise could
		// deadlock trying to fetch messages on the same connection
		// we're using for the append.

		int maxsize = ((IMAPStore)store).getAppendBufferSize();

		for (int i = 0; i < msgs.length; i++) {
			final Message m = msgs[i];
			final MessageLiteral mos;

			try {
				// bZ[W傫߂̂mꍇAobt@O܂B
				mos = new MessageLiteral(m, m.getSize() > maxsize ? 0 : maxsize);
			} catch (IOException ex) {
				throw new MessagingException("IOException while appending messages", ex);
			} catch (MessageRemovedException mrex) {
				continue; // ꂽbZ[WXLbv܂B
			}

			Date d = m.getReceivedDate(); // tێ܂
			if (d == null)
				d = m.getSentDate();
			final Date dd = d;
			final Flags f = m.getFlags();
			doCommand(new ProtocolCommand() {
				public Object doCommand(IMAPProtocol p) throws ProtocolException {
					p.append(fullName, f, dd, mos);
					return null;
				}
			});
		}
	}

	/**
	 * Append the given messages into this folder.
	 * Return array of AppendUID objects containing UIDs of these messages in the destination folder.
	 * Each element of the returned array corresponds to an element of the msgs array.
	 * A null element means the server didn't return UID information for the appended message.
	 * <p>Depends on the APPENDUID response code defined by the UIDPLUS extension - RFC 2359.
	 * 
	 * @throws MessagingException
	 */
	public AppendUID[] appendUIDMessages(final Message[] msgs) throws MessagingException {
		checkExists();

		// XXX - have to verify that messages are in a different
		// store (if any) than target folder, otherwise could
		// deadlock trying to fetch messages on the same connection
		// we're using for the append.

		int maxsize = ((IMAPStore)store).getAppendBufferSize();
		AppendUID aappenduid[] = new AppendUID[msgs.length];

		for (int j = 0; j < msgs.length; j++) {
			Message m = msgs[j];
			final MessageLiteral mos;
			try {
				// bZ[W傫߂̂mꍇAobt@O܂B
				mos = new MessageLiteral(m, m.getSize() > maxsize ? 0 : maxsize);
			} catch (IOException ex) {
				throw new MessagingException("IOException while appending messages", ex);
			} catch (MessageRemovedException mrex) {
				continue; // ꂽbZ[WXLbv܂B
			}

			Date d = m.getReceivedDate();
			if (d == null)
				d = m.getSentDate();
			final Date dd = d;
			final Flags f = m.getFlags();
			AppendUID appenduid = (AppendUID) doCommand(new ProtocolCommand() {
				public Object doCommand(IMAPProtocol p) throws ProtocolException {
					return p.appenduid(fullName, f, dd, mos);
				}
			});
			aappenduid[j] = appenduid;
		}

		return aappenduid;
	}

	/**
	 * Append the given messages into this folder.
	 * Return array of Message objects representing the messages in the destination folder.
	 * Note that the folder must be open. Each element of the returned array corresponds to an element of the msgs array.
	 * A null element means the server didn't return UID information for the appended message.
	 * <p>Depends on the APPENDUID response code defined by the UIDPLUS extension - RFC 2359.
	 * 
	 * @throws MessagingException
	 */
	public Message[] addMessages(final Message[] msgs) throws MessagingException {
		checkOpened();

		MimeMessage[] amimemessage = new MimeMessage[msgs.length];
		AppendUID[] aappenduid = appendUIDMessages(msgs);
		for (int i = 0; i < aappenduid.length; i++) {
			AppendUID uid = aappenduid[i];
			if (uid != null && uid.uidvalidity == (long)uidvalidity) {
				try {
					amimemessage[i] = (MimeMessage) getMessageByUID(uid.uid);
				} catch (MessagingException mex) {}
			}
		}

		return amimemessage;
	}

	/**
	 * ̃tH_w肳ꂽʂ̃tH_֎w肳ꂽbZ[WRs[܂B
	 */
	public synchronized void copyMessages(final Message[] msgs, final Folder folder) throws MessagingException {
		checkOpened();

		if (msgs.length == 0) // boundary condition
			return;

		// If the destination belongs to our same store, optimize
		if (folder.getStore() == store) {
			synchronized(messageCacheLock) {
				try {
					MessageSet ms[] = Utility.toMessageSet(msgs, null);
					if (ms == null)
						throw new MessageRemovedException("Messages have been removed");
					protocol.copy(ms, folder.getFullName());
				} catch (CommandFailedException cfx) {
					if (cfx.getMessage().indexOf("TRYCREATE") != -1)
						throw new FolderNotFoundException(folder, folder.getFullName() + " does not exist");
					throw new MessagingException(cfx.getMessage(), cfx);
				} catch (ConnectionException cex) {
					throw new FolderClosedException(this, cex.getMessage());
				} catch (ProtocolException pex) {
					throw new MessagingException(pex.getMessage(), pex);
				}
			}
		} else // destination is a different store.
		    super.copyMessages(msgs, folder);
	}

	/**
	 * Expunge all messages marked as DELETED.
	 */
	public synchronized Message[] expunge() throws MessagingException {
		return expunge(null);
	}

	/**
	 * Expunge the indicated messages, which must have been marked as DELETED.
	 * 
	 * @throws MessagingException
	 */
	public synchronized Message[] expunge(final Message[] msgs) throws MessagingException {
		checkOpened();

		Vector v = new Vector(); // to collect expunged messages

		if (msgs != null) {
			FetchProfile fetchprofile = new FetchProfile();
			fetchprofile.add(javax.mail.UIDFolder.FetchProfileItem.UID);
			fetch(msgs, fetchprofile);
		}

		synchronized(messageCacheLock) {
			doExpungeNotification = false; // We do this ourselves later
			try {
				if (msgs != null)
					protocol.uidexpunge(Utility.toUIDSet(msgs));
				else
					protocol.expunge();
			} catch (CommandFailedException cfx) {
				// expunge not allowed, perhaps due to a permission problem?
				if (mode != READ_WRITE)
					throw new IllegalStateException("Cannot expunge READ_ONLY folder: " + fullName);
				throw new MessagingException(cfx.getMessage(), cfx);
			} catch (ConnectionException cex) {
				throw new FolderClosedException(this, cex.getMessage());
			} catch (ProtocolException pex) {
				// Bad bad server ..
				throw new MessagingException(pex.getMessage(), pex);
			} finally {
				doExpungeNotification = true;
			}

			// Cleanup expunged messages and sync messageCache with reality.
			for (int i = 0; i < messageCache.size(); ) {
				IMAPMessage m = (IMAPMessage) messageCache.elementAt(i);
				if (m.isExpunged()) {
				    v.addElement(m); // add into vector of expunged messages

					/* remove this message from the messageCache.
					 *
					 * Note that this also causes all succeeding messages
					 * in the cache to shifted downward in the vector,
					 * therby decrementing the vector's size. (and hence
					 * we need to do messageCache.size() at the top of
					 * this loop.
					 */
					messageCache.removeElementAt(i);

					/* remove this message from the UIDTable */
					if (uidTable != null) {
						long uid = m.getUID();
						if (uid != -1)
							uidTable.remove(new Long(uid));
					}
				} else {
					/* Valid message, sync its message number with its sequence number.
					 */
					m.setMessageNumber(m.getSequenceNumber());
					i++; // done; increment index, go check next message
				}
			}
		}

		// Update 'total'
		total = messageCache.size();

		// Notify listeners. This time its for real, guys.
		Message[] messages = new Message[v.size()];
		v.copyInto(messages);
		if (messages.length > 0)
			notifyMessageRemovedListeners(true, messages);
		return messages;
	}

	/**
	 * w肳ꂽɈv郁bZ[WɂătH_܂B
	 */
	public synchronized Message[] search(final SearchTerm term) throws MessagingException {
		checkOpened();

		try {
			Message[] matchMsgs = null;
			synchronized(messageCacheLock) {
				int[] matches = protocol.search(term);
				if (matches != null) {
					matchMsgs = new IMAPMessage[matches.length];
					// Map seq-numbers into actual Messages.
					for (int i = 0; i < matches.length; i++)	
						matchMsgs[i] = getMessageBySeqNumber(matches[i]);
				}
			}
			return matchMsgs;
		} catch (CommandFailedException cfx) {
			// unsupported charset or search criterion
			return super.search(term);
		} catch (SearchException sex) {
			// too complex for IMAP
			return super.search(term);
		} catch (ConnectionException cex) {
			throw new FolderClosedException(this, cex.getMessage());
		} catch (ProtocolException pex) {
			// bug in our IMAP layer ?
			throw new MessagingException(pex.getMessage(), pex);
		}
	}

	/**
	 * Search the folder for messages matching the given term. Returns
	 * array of matching messages. Returns an empty array if no matching
	 * messages are found.
	 */
	public synchronized Message[] search(final SearchTerm term, final Message[] msgs)  throws MessagingException {
		checkOpened();

		if (msgs.length == 0)
			// need to return an empty array (not null!)
			return msgs;

		try {
			Message[] matchMsgs = null;
			synchronized(messageCacheLock) {
				MessageSet ms[] = Utility.toMessageSet(msgs, null);
				if (ms == null)
					throw new MessageRemovedException("Messages have been removed");
				int[] matches = protocol.search(ms, term);
				if (matches != null) {
					matchMsgs = new IMAPMessage[matches.length];
					for (int i = 0; i < matches.length; i++)	
						matchMsgs[i] = getMessageBySeqNumber(matches[i]);
				}
			}
			return matchMsgs;
		} catch (CommandFailedException cfx) {
			// unsupported charset or search criterion
			return super.search(term, msgs);
		} catch (SearchException sex) {
			// too complex for IMAP
			return super.search(term, msgs);
		} catch (ConnectionException cex) {
			throw new FolderClosedException(this, cex.getMessage());
		} catch (ProtocolException pex) {
			// bug in our IMAP layer ?
			throw new MessagingException(pex.getMessage(), pex);
		}
	}

    /***********************************************************
     *		UIDFolder C^[tFCX \bh
     **********************************************************/

	/**
	 * ̃tH_ UIDValidity Ԃ܂B
	 */
	public synchronized long getUIDValidity() throws MessagingException {
		if (opened) // we already have this information
			return uidvalidity;

		IMAPProtocol p = null;
		Status status = null;

		try {
			p = getStoreProtocol();	// XXX
			String[] item = { "UIDVALIDITY" };
			status = p.status(fullName, item);
		} catch (BadCommandException bex) {
			// Probably a RFC1730 server
			throw new MessagingException("Cannot obtain UIDValidity", bex);
		} catch (ConnectionException cex) {
			// Oops, the store or folder died on us.
			throwClosedException(cex);
		} catch (ProtocolException pex) {
			throw new MessagingException(pex.getMessage(), pex);
		} finally {
			releaseStoreProtocol(p);
		}

		return status.uidvalidity;
	}

	/**
	 * Returns the predicted UID that will be assigned to the next message that is appended to this folder.
	 * If the folder is closed, the STATUS command is used to retrieve this value.
	 * If the folder is open, the value returned from the SELECT or EXAMINE command is returned.
	 * Note that messages may have been appended to the folder while it was open and thus this value may be out of date.<p>
	 * Servers implementing RFC2060 likely won't return this value when a folder is opened.
	 * Servers implementing RFC3501 should return this value when a folder is opened.<p>
	 * 
	 * @return UIDNEXT lA͕sȏꍇ -1
	 * @since JavaMail 1.3.3
	 */
	public synchronized long getUIDNext() throws MessagingException {
		if (opened)
			return uidnext;

		IMAPProtocol p = null;
		Status status = null;

		try {
			p = getStoreProtocol();
			String item[] = { "UIDNEXT" };
			status = p.status(fullName, item);
		} catch (BadCommandException bex) {
			throw new MessagingException("Cannot obtain UIDNext", bex);
		} catch (ConnectionException cex) {
			throwClosedException(cex);
		} catch (ProtocolException pex) {
			throw new MessagingException(pex.getMessage(), pex);
		} finally {
			releaseStoreProtocol(p);
		}
		return status.uidnext;
	}

	/**
	 * Get the Message corresponding to the given UID.
	 * If no such message exists, <code>null</code> is returned.
	 */
	public synchronized Message getMessageByUID(final long uid) throws MessagingException {
		checkOpened(); // insure folder is open

		Long l = new Long(uid);
		IMAPMessage m = null;

		if (uidTable != null) {
		    // Check in uidTable
		    m = (IMAPMessage) uidTable.get(l);
		    if (m != null) // found it
		    	return m;
		} else
		    uidTable = new Hashtable();

		// Check with the server
		try {
			synchronized(messageCacheLock) {
				// Issue UID FETCH command
				UID u = protocol.fetchSequenceNumber(uid);

				if (u != null && u.msgno <= total) { // Valid UID 
					m = (IMAPMessage) messageCache.elementAt(u.msgno - 1);
					m.setUID(u.uid); // set this message's UID ..
					// .. and put this into the hashtable
					uidTable.put(l, m);
				}
			}
		} catch(ConnectionException cex) {
			throw new FolderClosedException(this, cex.getMessage());
		} catch (ProtocolException pex) {
			throw new MessagingException(pex.getMessage(), pex);
		}

		return m;
	}

	/**
	 * Get the Messages specified by the given range. <p>
	 * Returns Message objects for all valid messages in this range.
	 * Returns an empty array if no messages are found.
	 */
	public synchronized Message[] getMessagesByUID(final long start, final long end) throws MessagingException {
		checkOpened(); // insure that folder is open

		if (uidTable == null)
		    uidTable = new Hashtable();

		Message[] msgs; // array of messages to be returned
		try {
			synchronized(messageCacheLock) {
				// Issue UID FETCH for given range
				UID[] ua = protocol.fetchSequenceNumbers(start, end);

				msgs = new Message[ua.length];
				// NOTE: Below must be within messageCacheLock region
				for (int i = 0; i < ua.length; i++) {
					IMAPMessage m = (IMAPMessage) messageCache.elementAt(ua[i].msgno-1);
					m.setUID(ua[i].uid);
					msgs[i] = m;
					uidTable.put(new Long(ua[i].uid), m);
				}
			}
		} catch(ConnectionException cex) {
			throw new FolderClosedException(this, cex.getMessage());
		} catch (ProtocolException pex) {
			throw new MessagingException(pex.getMessage(), pex);
		}

		return msgs;
	}

	/**
	 * Get the Messages specified by the given array. <p>
	 *
	 * <code>uids.length()</code> elements are returned.
	 * If any UID in the array is invalid, a <code>null</code> entry
	 * is returned for that element.
	 */
	public synchronized Message[] getMessagesByUID(final long[] uids) throws MessagingException {
		checkOpened(); // insure that folder is open
		long[] unavailUids = uids;

		if (uidTable != null) {
			Vector v = new Vector(); // to collect unavailable UIDs
			for (int i = 0; i < uids.length; i++) {
				Long l;
				if (!uidTable.containsKey(l = new Long(uids[i])))
					// This UID has not been loaded yet.
					v.addElement(l);
			}

			int vsize = v.size();
			unavailUids = new long[vsize];
			for (int i = 0; i < vsize; i++)
				unavailUids[i] = ((Long)v.elementAt(i)).longValue();
		} else
			uidTable = new Hashtable();

		if (unavailUids.length > 0) {
			try {
				synchronized(messageCacheLock) {
					// Issue UID FETCH request for given uids
					UID[] ua = protocol.fetchSequenceNumbers(unavailUids);
					for (int i = 0; i < ua.length; i++) {
						IMAPMessage m = (IMAPMessage)messageCache.elementAt(ua[i].msgno-1);
						m.setUID(ua[i].uid);
						uidTable.put(new Long(ua[i].uid), m);
				    }
				}
			} catch(ConnectionException cex) {
				throw new FolderClosedException(this, cex.getMessage());
			} catch (ProtocolException pex) {
				throw new MessagingException(pex.getMessage(), pex);
			}
		}

		// Return array of size = uids.length
		Message[] msgs = new Message[uids.length];
		for (int i = 0; i < uids.length; i++)
			msgs[i] = (Message) uidTable.get(new Long(uids[i]));
		return msgs;
	}

	/**
	 * w肳ꂽbZ[W UID 擾܂B
	 */
	public synchronized long getUID(final Message message) throws MessagingException {
		if (message.getFolder() != this)
			throw new NoSuchElementException("Message does not belong to this folder");

		checkOpened(); // tH_JĂ鎖ۏ؂܂

		IMAPMessage m = (IMAPMessage) message;
		// If the message already knows its UID, great ..
		long uid;
		if ((uid = m.getUID()) != -1)
			return uid;

		UID u = null;
		synchronized(messageCacheLock) { // Acquire Lock
			m.checkExpunged(); // insure that message is not expunged
			try {
				u = protocol.fetchUID(m.getSequenceNumber());
			} catch (ConnectionException cex) {
				throw new FolderClosedException(this, cex.getMessage());
			} catch (ProtocolException pex) {
				throw new MessagingException(pex.getMessage(), pex);
			}
		}

		if (u != null) {
			uid = u.uid;
			m.setUID(uid); // bZ[W UID ݒ肵܂

			// insert this message into uidTable
			if (uidTable == null)
				uidTable = new Hashtable();
			uidTable.put(new Long(uid), m);
		}

		return uid;
	}

	/**
	 * Get the quotas for the quotaroot associated with this
	 * folder.  Note that many folders may have the same quotaroot.
	 * Quotas are controlled on the basis of a quotaroot, not
	 * (necessarily) a folder.  The relationship between folders
	 * and quotaroots depends on the IMAP server.  Some servers
	 * might implement a single quotaroot for all folders owned by
	 * a user.  Other servers might implement a separate quotaroot
	 * for each folder.  A single folder can even have multiple
	 * quotaroots, perhaps controlling quotas for different
	 * resources.
	 * 
	 * @return array of Quota objects for the quotaroots associated with this folder
	 * @throws MessagingException if the server doesn't support the QUOTA extension
	 */
	public final Quota[] getQuota() throws MessagingException {
		return (Quota[]) doOptionalCommand("QUOTA not supported", new ProtocolCommand() {
			public Object doCommand(IMAPProtocol p) throws ProtocolException {
			    return p.getQuotaRoot(fullName);
			}
	    });
	}

	/**
	 * Set the quotas for the quotaroot specified in the quota argument.
	 * Typically this will be one of the quotaroots associated with this
	 * folder, as obtained from the <code>getQuota</code> method, but it
	 * need not be.
	 * 
	 * @param quota the quota to set
	 * @throws MessagingException if the server doesn't support the QUOTA extension
	 */
	public final void setQuota(final Quota quota) throws MessagingException {
		doOptionalCommand("QUOTA not supported", new ProtocolCommand() {
			public Object doCommand(IMAPProtocol p) throws ProtocolException {
				p.setQuota(quota);
				return null;
			}
		});
	}

	/**
	 * Get the access control list entries for this folder.
	 * 
	 * @return array of access control list entries
	 * @throws MessagingException if the server doesn't support the ACL extension
	 */
	public final ACL[] getACL() throws MessagingException {
		return (ACL[]) doOptionalCommand("ACL not supported", new ProtocolCommand() {
			public Object doCommand(IMAPProtocol p) throws ProtocolException {
				return p.getACL(fullName);
			}
		});
	}

	/**
	 * Add an access control list entry to the access control list
	 * for this folder.
	 * 
	 * @param acl the access control list entry to add
	 * @throws MessagingException if the server doesn't support the ACL extension
	 */
	public final void addACL(final ACL acl) throws MessagingException {
		setACL(acl, '\0');
	}

	/**
	 * Remove any access control list entry for the given identifier
	 * from the access control list for this folder.
	 * 
	 * @param name the identifier for which to remove all ACL entries
	 * @throws MessagingException if the server doesn't support the ACL extension
	 */
	public final void removeACL(final String name) throws MessagingException {
		doOptionalCommand("ACL not supported", new ProtocolCommand() {
			public Object doCommand(IMAPProtocol p) throws ProtocolException {
				p.deleteACL(fullName, name);
				return null;
			}
		});
	}

	/**
	 * Add the rights specified in the ACL to the entry for the
	 * identifier specified in the ACL.  If an entry for the identifier
	 * doesn't already exist, add one.
	 * 
	 * @param acl the identifer and rights to add
	 * @throws MessagingException if the server doesn't support the ACL extension
	 */
	public final void addRights(final ACL acl) throws MessagingException {
		setACL(acl, '+');
	}

	/**
	 * Remove the rights specified in the ACL from the entry for the
	 * identifier specified in the ACL.
	 * 
	 * @param acl the identifer and rights to remove
	 * @throws MessagingException if the server doesn't support the ACL extension
	 */
	public final void removeRights(final ACL acl) throws MessagingException {
		setACL(acl, '-');
	}

	/**
	 * Get all the rights that may be allowed to the given identifier.
	 * Rights are grouped per RFC 2086 and each group is returned as an
	 * element of the array.  The first element of the array is the set
	 * of rights that are always granted to the identifier.  Later
	 * elements are rights that may be optionally granted to the
	 * identifier. <p>
	 * 
	 * Note that this method lists the rights that it is possible to
	 * assign to the given identifier, <em>not</em> the rights that are
	 * actually granted to the given identifier.  For the latter, see
	 * the <code>getACL</code> method.
	 * 
	 * @param name the identifier to list rights for
	 * @return array of Rights objects representing possible rights for the identifier
	 * @throws MessagingException if the server doesn't support the ACL extension
	 */
	public final Rights[] listRights(final String name) throws MessagingException {
		return (Rights[]) doOptionalCommand("ACL not supported", new ProtocolCommand() {
			public Object doCommand(IMAPProtocol p) throws ProtocolException {
				return p.listRights(fullName, name);
			}
		});
	}

	/**
	 * Get the rights allowed to the currently authenticated user.
	 * 
	 * @return the rights granted to the current user
	 * @throws MessagingException if the server doesn't support the ACL extension
	 */
	public final Rights myRights() throws MessagingException {
		return (Rights) doOptionalCommand("ACL not supported", new ProtocolCommand() {
			public Object doCommand(IMAPProtocol p) throws ProtocolException {
				return p.myRights(fullName);
			}
		});
	}

	private void setACL(final ACL acl, final char mod) throws MessagingException {
		doOptionalCommand("ACL not supported", new ProtocolCommand() {
			public Object doCommand(IMAPProtocol p) throws ProtocolException {
				p.setACL(fullName, mod, acl);
				return null;
			}
		});
	}

	/**
	 * IMAP T[o LIST X|XƋɕԂ擾܂B<p>
	 * 
	 * @since JavaMail 1.3.3
	 */
	public String[] getAttributes() throws MessagingException {
		if (attributes == null)
			exists();
		return (String[]) attributes.clone();
	}

	/**
	 * X|XnhB
	 * ̓vgRwɂČĂяoR[obN[`łB
	 */
	/*
	 * ASSERT: messageCacheLock Ăꍇ̃\bhďoĉB
	 * ASSERT: This method must *not* invoke any other method that
	 * might grab the 'folder' lock or 'message' lock (i.e., any 
	 * synchronized methods on IMAPFolder or IMAPMessage)
	 * since that will result in violating the locking hierarchy.
	 */
	public final void handleResponse(final Response r) {
		/*
		 * ŏɁA\ ALERT ʒm Store ɈϔC܂B
		 */
		if (r.isOK() || r.isNO() || r.isBAD() || r.isBYE())
			((IMAPStore)store).handleResponseCode(r);

		/*
		 * x́Aꂪ BYE  OK X|Xł邩K؂ɑ삵`FbN܂B
		 */
		if (r.isBYE()) {
			if (opened)
				cleanup(false); 
			return;
		} else if (r.isOK()) {
			return;
		} else if (!r.isUnTagged()) {
			return;	// XXX - ċNĂ͂Ȃ܂
		}

		/* x́Aꂪ IMAP ̓̃X|Xǂ`FbN܂B */
		if (!(r instanceof IMAPResponse)) {
			// 炭R[hoOI
			// XXX - should be an assert
			out.println("UNEXPECTED RESPONSE : " + r.toString());
			out.println("CONTACT javamail@sun.com");
			return;
		}

		IMAPResponse ir = (IMAPResponse) r;

		if (ir.keyEquals("EXISTS")) { // EXISTS
			int exists = ir.getNumber();
			if (exists <= realTotal) 
				// Could be the EXISTS following EXPUNGE, ignore 'em
				return;

			int count = exists - realTotal; // number of new messages
			Message[] msgs = new Message[count];

			// Add 'count' new IMAPMessage objects into the messageCache
			for (int i = 0; i < count; i++) {
				// Note that as a side-effect, we also increment
				// total & realTotal
				IMAPMessage msg = new IMAPMessage(this, ++total, ++realTotal);
				msgs[i] = msg;
				messageCache.addElement(msg);
			}

			// Notify listeners.
			notifyMessageAddedListeners(msgs);
		} else if (ir.keyEquals("EXPUNGE")) {
			// EXPUNGE response.

			IMAPMessage msg = getMessageBySeqNumber(ir.getNumber());
			msg.setExpunged(true); // mark this message expunged.

			// Renumber the cache, starting from just beyond 
			// the expunged message.
			for (int i = msg.getMessageNumber(); i < total; i++) {
				// Note that 'i' actually indexes the message
				// beyond the expunged message.
				IMAPMessage m = (IMAPMessage) messageCache.elementAt(i);
				if (m.isExpunged()) // an expunged message, skip
				    continue;

				// Decrement this message's seqnum
				m.setSequenceNumber(m.getSequenceNumber() - 1);
			} // Whew, done.

			// decrement 'realTotal'; but leave 'total' unchanged
			realTotal--;

			if (doExpungeNotification) {
				// Do the notification here.
				Message[] msgs = { msg };
				notifyMessageRemovedListeners(false, msgs);
		    }
		} else if (ir.keyEquals("FETCH")) {
			// The only unsolicited FETCH response that makes sense
			// to me (for now) is FLAGS updates. Ignore any other junk.
			FetchResponse f = (FetchResponse) ir;
			// Get FLAGS response, if present
			Flags flags = (Flags) f.getItem(Flags.class);

			if (flags != null) {
				IMAPMessage msg = getMessageBySeqNumber(f.getNumber());
				if (msg != null) {	// should always be true
					msg._setFlags(flags);
					notifyMessageChangedListeners(MessageChangedEvent.FLAGS_CHANGED, msg);
				}
			}
		} else if (ir.keyEquals("RECENT")) {
			// update 'recent'
			recent = ir.getNumber();
		}
	}

	/**
	 * w肳ꂽ Response ̔z𑀍삵܂B
	 * 
	 * ASSERT: messageCacheLock Ăꍇ̃\bhďoĉ
	 */
	final void handleResponses(final Response[] r) {
		for (int i = 0; i < r.length; i++) {
			if (r[i] != null)
				handleResponse(r[i]);
		}
	}

	/**
	 * ̃tH_̃XgÃvgRڑ擾܂B
	 * 
	 * When acquiring a store protocol object, it is important to
	 * use the following steps:
	 * 
	 *     IMAPProtocol p = null;
	 *     try {
	 *         p = getStoreProtocol();
	 *         // R}hs܂
	 *     } catch (WhateverException ex) {
	 *         // OnhO܂
	 *     } finally {
	 *         releaseStoreProtocol(p);
	 *     }
	 */
	protected synchronized IMAPProtocol getStoreProtocol() throws ProtocolException {
		if (connectionPoolDebug)
			out.println("DEBUG: getStoreProtocol() - borrowing a connection");
		return ((IMAPStore)store).getStoreProtocol();
	}

	/**
	 * K؂ 'closed' O𓊂܂B
	 */
	private synchronized void throwClosedException(final ConnectionException cex) 
		throws FolderClosedException, StoreClosedException {
		// If it the folder's protocol object, throw a FolderClosedException;
		// otherwize, throw a StoreClosedException.
		if (protocol != null && cex.getProtocol() == protocol || protocol == null && !reallyClosed)
			throw new FolderClosedException(this, cex.getMessage());
		throw new StoreClosedException(store, cex.getMessage());
	}

	/**
	 * ̃tH_ IMAPProtocol IuWFNgԂ܂B<p>
	 * 
	 * <strong>NOTE:</strong>
	 *		This method was intended to allow experimentation with
	 *		simple extension commands that can use the low level
	 *		Protocol object APIs to send commands and process
	 *		responses. <p>
	 * 
	 * <strong>NOTE:</strong> Using this protocol object is completely
	 *		<strong>UNSAFE</strong>	because there's no way to aquire
	 *		the required locks.  See the <code>doCommand</code> method
	 *		for a safe alternative. <p>
	 * 
	 * @return tH_JĂꍇɎgp IMAPProtocol IuWFNg
	 * @see #doCommand
	 */
	public final IMAPProtocol getProtocol() {
		return protocol;
	}

	/**
	 * [U`p IMAP vgRR}h̃VvȃC^[tFCXłB
	 */
	public static interface ProtocolCommand {

		/**
		 * ꂽ IMAPProtocol IuWFNggpāA[U[`̃R}hs܂B
		 */
		public Object doCommand(IMAPProtocol protocol) throws ProtocolException;

	}

	/**
	 * [Uɂċꂽ IMAP R}hs܂B
	 * The command is executed
	 * in the appropriate context with the necessary locks held and
	 * using the appropriate <code>IMAPProtocol</code> object. <p>
	 * 
	 * This method returns whatever the <code>ProtocolCommand</code>
	 * object's <code>doCommand</code> method returns.  If the
	 * <code>doCommand</code> method throws a <code>ConnectionException</code>
	 * it is translated into a <code>StoreClosedException</code> or
	 * <code>FolderClosedException</code> as appropriate.  If the
	 * <code>doCommand</code> method throws a <code>ProtocolException</code>
	 * it is translated into a <code>MessagingException</code>. <p>
	 * 
	 * The following example shows how to execute the IMAP NOOP command.
	 * Executing more complex IMAP commands requires intimate knowledge
	 * of the <code>com.sun.mail.iap</code> and
	 * <code>com.sun.mail.imap.protocol</code> packages, best acquired by
	 * reading the source code. <p>
	 * 
	 * <blockquote><pre>
	 * import com.sun.mail.iap.*;
	 * import com.sun.mail.imap.*;
	 * import com.sun.mail.imap.protocol.*;
	 * 
	 * ...
	 * 
	 * IMAPFolder f = (IMAPFolder)folder;
	 * Object val = f.doCommand(new IMAPFolder.ProtocolCommand() {
	 * 	public Object doCommand(IMAPProtocol p) throws ProtocolException {
	 *		p.simpleCommand("NOOP", null);
	 * 		return null;
	 * 	}
	 * });
	 * </pre></blockquote>
	 * <p>
	 * 
	 * Here's a more complex example showing how to use the proposed
	 * IMAP SORT extension: <p>
	 * 
	 * <pre><blockquote>
	 * import com.sun.mail.iap.*;
	 * import com.sun.mail.imap.*;
	 * import com.sun.mail.imap.protocol.*;
	 * 
	 * ...
	 * 
	 * IMAPFolder f = (IMAPFolder)folder;
	 * Object val = f.doCommand(new IMAPFolder.ProtocolCommand() {
	 * 	public Object doCommand(IMAPProtocol p) throws ProtocolException {
	 * 		// Issue command
	 * 		Argument args = new Argument();
	 * 		Argument list = new Argument();
	 * 		list.writeString("SUBJECT");
	 * 		args.writeArgument(list);
	 * 		args.writeString("UTF-8");
	 * 		args.writeString("ALL");
	 * 		Response[] r = p.command("SORT", args);
	 * 		Response response = r[r.length-1];
	 * 
	 * 		// Grab response
	 * 		Vector v = new Vector();
	 * 		if (response.isOK()) { // command succesful 
	 * 			for (int i = 0, len = r.length; i < len; i++) {
	 * 				if (!(r[i] instanceof IMAPResponse))
	 * 					continue;
	 * 
	 * 				IMAPResponse ir = (IMAPResponse) r[i];
	 * 				if (ir.keyEquals("SORT")) {
	 * 					String num;
	 * 					while ((num = ir.readAtomString()) != null)
	 * 						System.out.println(num);
	 * 					r[i] = null;
	 * 				}
	 * 			}
	 * 		}
	 * 
	 * 		// dispatch remaining untagged responses
	 * 		p.notifyResponseHandlers(r);
	 * 		p.handleResult(response);
	 * 
	 * 		return null;
	 * 	}
	 * });
	 * </pre></blockquote>
	 */
	public final Object doCommand(final ProtocolCommand cmd) throws MessagingException {
		try {
			return doProtocolCommand(cmd);
		} catch (ConnectionException cex) {
			// Oops, the store or folder died on us.
			throwClosedException(cex);
		} catch (ProtocolException pex) {
			throw new MessagingException(pex.getMessage(), pex);
		}

		return null;
	}

	public final Object doOptionalCommand(final String err, final ProtocolCommand cmd) throws MessagingException {
		try {
			return doProtocolCommand(cmd);
		} catch (BadCommandException bex) {
			throw new MessagingException(err, bex);
		} catch (ConnectionException cex) {
			// Oops, the store or folder died on us.
			throwClosedException(cex);
		} catch (ProtocolException pex) {
			throw new MessagingException(pex.getMessage(), pex);
		}
		return null;
	}

	public final Object doCommandIgnoreFailure(final ProtocolCommand cmd) throws MessagingException {
		try {
			return doProtocolCommand(cmd);
		} catch (CommandFailedException cfx) {
			return null;
		} catch (ConnectionException cex) {
			// Oops, the store or folder died on us.
			throwClosedException(cex);
		} catch (ProtocolException pex) {
			throw new MessagingException(pex.getMessage(), pex);
		}
		return null;
	}

	protected final Object doProtocolCommand(final ProtocolCommand cmd) throws ProtocolException {
		synchronized (this) {
			if (opened && !((IMAPStore)store).hasSeparateStoreConnection()) {
				synchronized (messageCacheLock) {
					return cmd.doCommand(getProtocol());
				}
			}
		}

		// only get here if using store's connection
		IMAPProtocol p = null;

		try {
			p = getStoreProtocol();
//Sugisawa added. 2004/6/30
p.setCharset(charset);
			return cmd.doCommand(p);
		} finally {
// Sugisawa added. 2004/6/30
if (p != null)
	p.setCharset(charset);

			releaseStoreProtocol(p);
		}
	}

	/**
	 * XgAvgRIuWFNg܂B
	 * If we borrowed a protocol object from the connection pool, give it back.
	 * If we used our own protocol object, nothing to do.
	 */
	protected synchronized void releaseStoreProtocol(final IMAPProtocol p) {
		if (p != protocol)
			((IMAPStore)store).releaseStoreProtocol(p);
	}

	/**
	 * vgRIuWFNg܂B
	 * 
	 * ASSERT: messageCacheLock Ăꍇ̃\bhďoĉ
	 */
	private void releaseProtocol(final boolean returnToPool) {
		if (protocol != null) {
			protocol.removeResponseHandler(this);

			if (returnToPool)
				((IMAPStore)store).releaseProtocol(this, protocol);
			else
				((IMAPStore)store).releaseProtocol(this, null);
		}
	}

	/**
	 * ڑ1bԈȏgpĂȂꍇAڑ̈ێׂ̈ noop R}h𔭍s܂B
	 * A<code>keepStoreAlive</code>  true ̏ꍇAXgA̐ڑɂ noop o܂B
	 */
	private void keepConnectionAlive(final boolean keepStoreAlive) throws ProtocolException {
		if (System.currentTimeMillis() - protocol.getTimestamp() > 1000)
			protocol.noop(); 

		if (keepStoreAlive && ((IMAPStore)store).hasSeparateStoreConnection()) {
			IMAPProtocol p = null;
			try {
				p = ((IMAPStore)store).getStoreProtocol();
				if (System.currentTimeMillis() - p.getTimestamp() > 1000)
					p.noop();
			} finally {
				((IMAPStore)store).releaseStoreProtocol(p);
			}
		}
	}

	/**
	 * w肳ꂽV[PXio[̃bZ[WIuWFNg擾܂B
	 * Ȃꍇ null Ԃ܂B
	 * 
	 * ASSERT: messageCacheLock Ăꍇ̃\bhďoĉ
	 */
	final IMAPMessage getMessageBySeqNumber(final int seqnum) {
		/* Check messageCache for message matching the given
		 * sequence number. We start the search from position (seqnum - 1)
		 * and continue down the vector till we get a match.
		 */
		for (int i = seqnum-1; i < total; i++) {
			IMAPMessage msg = (IMAPMessage) messageCache.elementAt(i);
			if (msg.getSequenceNumber() == seqnum)
				return msg;
		}
		return null;
	}

	private boolean isDirectory() {
		return ((type & HOLDS_FOLDERS) != 0);
	}

}

/**
 * An object that holds a Message object and reports its size and writes it to another OutputStream on demand.
 * Used by appendMessages to avoid the need to
 * buffer the entire message in memory in a single byte array
 * before sending it to the server.
 */
final class MessageLiteral implements Literal {

	private Message msg;
	private int msgSize = -1;
	private byte[] buf;		// obt@OꂽbZ[WA null

	public MessageLiteral(final Message msg, final int maxsize) throws MessagingException, IOException {
		this.msg = msg;
		// ɗOԂlɁAŃTCYvZ܂B
		LengthCounter lc = new LengthCounter(maxsize);
		OutputStream os = new CRLFOutputStream(lc);
		msg.writeTo(os);
		os.flush();
		msgSize = lc.getSize();
		buf = lc.getBytes();
	}

	public int size() {
		return msgSize;
	}

	public void writeTo(OutputStream os) throws IOException {
		// the message should not change between the constructor and this call
		try {
			if (buf != null)
				os.write(buf, 0, msgSize);
			else {
				os = new CRLFOutputStream(os);
				msg.writeTo(os);
			}
		} catch (MessagingException mex) {
			// exceptions here are bad, "should" never happen
			throw new IOException("MessagingException while appending message: " + mex);
		}
	}

}

/**
 * Xg[ɏ܂ꂽoCg𐔂܂B
 * Aēxf[^Ȃ΂ȂȂׂɁAȃbZ[W̃Rs[ۑ܂B
 */
final class LengthCounter extends OutputStream {

	private int size = 0;
	private byte[] buf;
	private int maxsize;

	public LengthCounter(final int maxsize) {
		buf = new byte[8192];
		this.maxsize = maxsize;
	}

	public void write(final int b) {
		int newsize = size + 1;
		if (buf != null) {
			if (newsize > maxsize && maxsize >= 0) {
				buf = null;
			} else if (newsize > buf.length) {
				byte newbuf[] = new byte[Math.max(buf.length << 1, newsize)];
				System.arraycopy(buf, 0, newbuf, 0, size);
				buf = newbuf;
				buf[size] = (byte) b;
			} else {
				buf[size] = (byte) b;
			}
		}
		size = newsize;
	}

	public void write(final byte b[], final int off, final int len) {
		if ((off < 0) || (off > b.length) || (len < 0) || ((off + len) > b.length) || ((off + len) < 0))
			throw new IndexOutOfBoundsException();
		else if (len == 0)
			return;
		int newsize = size + len;
		if (buf != null) {
			if (newsize > maxsize && maxsize >= 0)
				buf = null;
			else if (newsize > buf.length) {
				byte newbuf[] = new byte[Math.max(buf.length << 1, newsize)];
				System.arraycopy(buf, 0, newbuf, 0, size);
				buf = newbuf;
				System.arraycopy(b, off, buf, size, len);
			} else
				System.arraycopy(b, off, buf, size, len);
		}
		size = newsize;
	}

	public void write(final byte[] b) throws IOException {
		write(b, 0, b.length);
	}

	public int getSize() {
		return size;
	}

	public byte[] getBytes() {
		return buf;
	}

}
