/*
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved.
 */

package com.sun.mail.imap;

import java.util.Vector;

/**
 * Rights NX͔F؎ʎq(Ⴆ΁A[UO[v)ւ̌Zbg\܂B<p>
 * 
 * right  <code>Rights.Right</code> ̃Ci[NXɂĕ\܂B<p>
 * 
 * A set of standard rights are predefined (see RFC 2086).  Most folder
 * implementations are expected to support these rights.  Some
 * implementations may also support site-defined rights. <p>
 * 
 * ȉ̃R[h̃Tv̓tH_ւ̂Ȃ̌𒲂ׂ@܂B<p>
 * <pre>
 * 
 * Rights rights = folder.myRights();
 * 
 * // ̃tH_ɏ߂邩ǂ܂B
 * if (rights.contains(Rights.Right.WRITE))
 *	System.out.println("Can write folder");
 * 
 * // tH_̏݌ȊO̎̑SĂ̌W[ɗ^܂B
 * // Now give Joe all my rights, except the ability to write the folder
 * rights.remove(Rights.Right.WRITE);
 * ACL acl = new ACL("joe", rights);
 * folder.setACL(acl);
 * </pre>
 * <p>
 */
public final class Rights implements Cloneable {

	private boolean[] rights = new boolean[128];	// XXX

	/**
	 * ̃Ci[NX͌X̌\܂B
	 * 1Zbg̕W̌IuWFNg͂ŎOɒ`܂B
	 */
	public static final class Right {

		private static Right[] cache = new Right[128];

		// XXX - initialization order?
		/**
		 * Lookup - [{bNX LIST/LSUB R}hŌ܂B
		 */
		public static final Right LOOKUP = getInstance('l');

		/**
		 * Read - [{bNX SELECT ACHECKAFETCHAPARTIALASEARCHACOPY sł܂B
		 */
		public static final Right READ = getInstance('r');

		/**
		 * Keep seen/unseen information across sessions - STORE \SEEN flag.
		 */
		public static final Right KEEP_SEEN = getInstance('s');

		/**
		 * Write - STORE flags other than \SEEN and \DELETED.
		 */
		public static final Right WRITE = getInstance('w');

		/**
		 * Insert - perform APPEND, COPY into mailbox.
		 */
		public static final Right INSERT = getInstance('i');

		/**
		 * Post - send mail to submission address for mailbox,
		 * not enforced by IMAP4 itself.
		 */
		public static final Right POST = getInstance('p');

		/**
		 * Create - CREATE new sub-mailboxes in any implementation-defined
		 * hierarchy, RENAME or DELETE mailbox.
		 */
		public static final Right CREATE = getInstance('c');

		/**
		 * Delete - STORE \DELETED flag, perform EXPUNGE.
		 */
		public static final Right DELETE = getInstance('d');

		/**
		 * Administer - perform SETACL.
		 */
		public static final Right ADMINISTER = getInstance('a');

		char right;	// the right represented by this Right object

		/**
		 * Private constructor used only by getInstance.
		 */
		private Right(final char right) {
			if (right >= 128)
				throw new IllegalArgumentException("Right must be ASCII");
			this.right = right;
		}

		/**
		 * Get a Right object representing the specified character.
		 * Characters are assigned per RFC 2086.
		 */
		public static synchronized Right getInstance(final char right) {
			if (right >= 128)
				throw new IllegalArgumentException("Right must be ASCII");
			if (cache[right] == null)
				cache[right] = new Right(right);
			return cache[right];
		}

		public String toString() {
			return String.valueOf(right);
		}

	}

	/**
	 * Construct an empty Rights object.
	 */
	public Rights() {}

	/**
	 * Construct a Rights object initialized with the given rights.
	 * 
	 * @param rights	the rights for initialization
	 */
	public Rights(final Rights rights) {
		System.arraycopy(rights.rights, 0, this.rights, 0, this.rights.length);
	}

	/**
	 * Construct a Rights object initialized with the given rights.
	 * 
	 * @param rights	the rights for initialization
	 */
	public Rights(final String rights) {
		for (int i = 0; i < rights.length(); i++)
		    add(Right.getInstance(rights.charAt(i)));
	}

	/**
	 * Construct a Rights object initialized with the given right.
	 * 
	 * @param right	the right for initialization
	 */
	public Rights(final Right right) {
		this.rights[right.right] = true;
	}

	/**
	 * Add the specified right to this Rights object.
	 * 
	 * @param right	the right to add
	 */
	public void add(final Right right) {
		this.rights[right.right] = true;
	}

	/**
	 * Add all the rights in the given Rights object to this
	 * Rights object.
	 * 
	 * @param rights Rights object
	 */
	public void add(final Rights rights) {
		for (int i = 0; i < rights.rights.length; i++)
			if (rights.rights[i])
				this.rights[i] = true;
	}

	/**
	 * Remove the specified right from this Rights object.
	 * 
	 * @param right the right to be removed
	 */
	public void remove(final Right right) {
		this.rights[right.right] = false;
	}

	/**
	 * Remove all rights in the given Rights object from this Rights object.
	 * 
	 * @param rights the right to be removed
	 */
	public void remove(final Rights rights) {
		for (int i = 0; i < rights.rights.length; i++)
			if (rights.rights[i])
				this.rights[i] = false;
	}

	/**
	 * Check whether the specified right is present in this Rights object.
	 * 
	 * @return true of the given right is present, otherwise false.
	 */
	public boolean contains(final Right right) {
		return this.rights[right.right];
	}

	/**
	 * Check whether all the rights in the specified Rights object are
	 * present in this Rights object.
	 * 
	 * @return true if all rights in the given Rights object are present, otherwise false.
	 */
	public boolean contains(final Rights rights) {
		for (int i = 0; i < rights.rights.length; i++)
			if (rights.rights[i] && !this.rights[i])
				return false;

		// If we've made it till here, return true
		return true;
	}

	/**
	 * Check whether the two Rights objects are equal.
	 * 
	 * @return true if they're equal
	 */
	public boolean equals(final Object obj) {
		if (!(obj instanceof Rights))
			return false;

		Rights rights = (Rights) obj;

		for (int i = 0; i < rights.rights.length; i++)
			if (rights.rights[i] != this.rights[i])
				return false;

		return true;
	}

	/**
	 * Compute a hash code for this Rights object.
	 * 
	 * @return nbVR[h
	 */
	public int hashCode() {
		int hash = 0;
		for (int i = 0; i < this.rights.length; i++)
			if (this.rights[i])
				hash++;
		return hash;
	}

	/**
	 * Return all the rights in this Rights object.  Returns
	 * an array of size zero if no rights are set.
	 * 
	 * @return array of Rights.Right objects representing rights
	 */
	public Right[] getRights() {
		Vector v = new Vector();
		for (int i = 0; i < this.rights.length; i++)
			if (this.rights[i])
				v.addElement(Right.getInstance((char)i));
		Right[] rights = new Right[v.size()];
		v.copyInto(rights);
		return rights;
	}

	/**
	 * Returns a clone of this Rights object.
	 */
	public Object clone() {
		return new Rights(this);
	}

	public String toString() {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < this.rights.length; i++)
			if (this.rights[i])
				sb.append((char)i);
		return sb.toString();
	}

}
