/*
 * Copyright 1997-2004 Sun Microsystems, Inc. All Rights Reserved.
 */

package com.sun.mail.imap.protocol;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;
import java.util.Vector;

import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Quota;
import javax.mail.UIDFolder;
import javax.mail.internet.MimeUtility;
import javax.mail.search.SearchException;
import javax.mail.search.SearchTerm;

import com.sun.mail.iap.Argument;
import com.sun.mail.iap.BadCommandException;
import com.sun.mail.iap.CommandFailedException;
import com.sun.mail.iap.ConnectionException;
import com.sun.mail.iap.Literal;
import com.sun.mail.iap.ParsingException;
import com.sun.mail.iap.Protocol;
import com.sun.mail.iap.ProtocolException;
import com.sun.mail.iap.Response;
import com.sun.mail.imap.ACL;
import com.sun.mail.imap.AppendUID;
import com.sun.mail.imap.Rights;
import com.sun.mail.util.ASCIIUtility;
import com.sun.mail.util.BASE64EncoderStream;

/**
 * ̃NX́Aiap.Protocol IuWFNggAIMAP Z}eBbNX܂B
 * In general, there is a method corresponding to each
 * IMAP protocol command. The typical implementation issues the
 * appropriate protocol command, collects all responses, processes
 * those responses that are specific to this command and then
 * dispatches the rest (the unsolicited ones) to the dispatcher
 * using the <code>notifyResponseHandlers(r)</code>.
 */
public final class IMAPProtocol extends Protocol {

	private boolean rev1 = false;		// REV1 server ?
	private boolean authenticated;		// authenticated?
	// WARNING: authenticated may be set to true in superclass
	//		constructor, don't initialize it here.

	private Hashtable capabilities = null;
	private Vector authmechs = null;
	private String[] searchCharsets; 	// array of search charsets

	private String name;
	private Properties props;
	private SaslAuthenticator saslAuthenticator;	// if SASL is being used

// Sugisawa added. 2004/6/30
private String charset = null;
public void setCharset(final String charset) {
	this.charset = charset;
}

	/**
	 * RXgN^łB
	 * Opens a connection to the given host at given port.
	 * 
	 * @param host ڑ̃zXg
	 * @param port ڑ̃|[gԍ
	 * @param debug fobO[h
	 * @param props ̃vgRŎgp Properties IuWFNg
	 */
	public IMAPProtocol(
		final String name,
		final String host,
		final int port, 
		final boolean debug,
		final PrintStream out,
		final Properties props,
		final boolean isSSL)
		throws IOException, ProtocolException {

		super(host, port, debug, out, props, "mail." + name, isSSL);

		this.name = name;
		this.props = props;

		// CAPABILITY `FbN
		if (capabilities == null)
			capability();

		if(hasCapability("IMAP4rev1"))
			rev1 = true;

		searchCharsets = new String[2]; // 2, for now.
		searchCharsets[0] = "UTF-8";
		searchCharsets[1] = MimeUtility.mimeCharset(MimeUtility.getDefaultJavaCharset());
	}

	/**
	 * CAPABILITY R}h
	 * 
	 * @see "RFC2060, section 6.1.1"
	 */
	public void capability() throws ProtocolException {
		// Check CAPABILITY
		Response r[] = command("CAPABILITY", null);

		if (!r[r.length - 1].isOK())
			throw new ProtocolException(r[r.length - 1].toString());

		capabilities = new Hashtable(10);
		authmechs = new Vector(5);
		for (int i = 0, len = r.length; i < len; i++) {
			if (!(r[i] instanceof IMAPResponse))
				continue;

			IMAPResponse ir = (IMAPResponse) r[i];
			// Handle *all* untagged CAPABILITY responses.
			//   Though the spec seemingly states that only
			// one CAPABILITY response string is allowed (6.1.1),
			// some server vendors claim otherwise.
			if (ir.keyEquals("CAPABILITY"))
				parseCapabilities(ir);
		}
	}

	/**
	 * If the response contains a CAPABILITY response code, extract
	 * it and save the capabilities.
	 */
	protected void setCapabilities(final Response r) {
		byte b;
		while ((b = r.readByte()) > 0 && b != (byte)'[')
			;
		if (b == 0)
			return;
		String s = r.readAtom();
		if (!s.equalsIgnoreCase("CAPABILITY"))
			return;

		capabilities = new Hashtable(10);
		authmechs = new Vector(5);
		parseCapabilities(r);
	}

	/**
	 * Parse the capabilities from a CAPABILITY response or from
	 * a CAPABILITY response code attached to (e.g.) an OK response.
	 */
	protected void parseCapabilities(final Response r) {
		String s;
		while ((s = r.readAtom(']')) != null) 
			if (s.length() == 0) {
				if (r.peekByte() == (byte)']')
					break;
				/*
				 * Probably found something here that's not an atom.
				 * Rather than loop forever or fail completely, we'll
				 * try to skip this bogus capability.  This is known
				 * to happen with:
				 *   Netscape Messaging Server 4.03 (built Apr 27 1999)
				 * that returns:
				 *   * CAPABILITY * CAPABILITY IMAP4 IMAP4rev1 ...
				 * The "*" in the middle of the capability list causes
				 * us to loop forever here.
				 */
				r.skipToken();
			} else {
				capabilities.put(s.toUpperCase(), s);
				if (s.regionMatches(true, 0, "AUTH=", 0, 5)) {
					authmechs.addElement(s.substring(5));
					if (debug)
						out.println("IMAP DEBUG: AUTH: " + s.substring(5));
				}
			}
	}

	/**
	 * Check the greeting when first connecting; look for PREAUTH response.
	 */
	protected void processGreeting(final Response r) throws ProtocolException {
		super.processGreeting(r);	// BAD `FbN܂
		if (r.isOK()) {
			setCapabilities(r);
			return;	// OK `FbN܂
		}
		// ̗B̑I PREAUTH ł
		IMAPResponse ir = (IMAPResponse) r;
		if (ir.keyEquals("PREAUTH")) {
			authenticated = true;
			setCapabilities(r);
		} else
			throw new ConnectionException(this, r);
	}

	/**
	 * Returns <code>true</code> if the connection has been authenticated,
	 * either due to a successful login, or due to a PREAUTH greeting response.
	 */
	public boolean isAuthenticated() {
		return authenticated;
	}

	/**
	 * ꂪ IMAP4rev1 T[ołꍇA<code>true</code> Ԃ܂B
	 */
	public boolean isREV1() {
		return rev1;
	}

	/**
	 * Returns whether this Protocol supports non-synchronizing literals.
	 */
	protected boolean supportsNonSyncLiterals() {
		return hasCapability("LITERAL+");
	}

	/**
	 * T[õX|XǍ݂܂B
	 */
	public Response readResponse() throws IOException, ProtocolException {
		return IMAPResponse.readResponse(this);
	}

	/**
	 * Check whether the given capability is supported by
	 * this server. Returns <code>true</code> if so, otherwise
	 * returns false.
	 */
	public boolean hasCapability(final String c) {
		return capabilities.containsKey(c.toUpperCase());
	}

	/**
	 * \Pbgڑ܂B
	 * 
	 * ̃\bh Protocol.disconnect() \bhĂяołB
	 */
	public void disconnect() {
		super.disconnect();
		authenticated = false;
	}

	/**
	 * NOOP R}h
	 * 
	 * @see "RFC2060, section 6.1.2"
	 */
	public void noop() throws ProtocolException {
		if (debug)
			out.println("IMAP DEBUG: IMAPProtocol noop");
		simpleCommand("NOOP", null);
	}

	/**
	 * LOGOUT R}h
	 * 
	 * @see "RFC2060, section 6.1.3"
	 */
	public void logout() throws ProtocolException {
		// XXX - what happens if exception is thrown?
		Response[] r = command("LOGOUT", null);

		authenticated = false;
		// dispatch any unsolicited responses.
		//  NOTE that the BYE response is dispatched here as well
		notifyResponseHandlers(r);
		disconnect();
	}

	/**
	 * LOGIN R}h
	 * 
	 * @see "RFC2060, section 6.2.2"
	 */
	public void login(final String u, final String p) throws ProtocolException {
		Argument args = new Argument();
		args.writeASCII(u);
		args.writeASCII(p);

		simpleCommand("LOGIN", args);
		// if we get this far without an exception, we're authenticated
		authenticated = true;
	}

	/**
	 * AUTHENTICATE ́AAUTH=LOGIN ƋɃXL[F؂悤ɖ߂܂B
	 * 
	 * @see "RFC2060, section 6.2.1"
	 */
	public void authlogin(final String u, final String p) throws ProtocolException {
		Vector v = new Vector();
		String tag = null;
		Response r = null;
		boolean done = false;

		try {
			tag = writeCommand("AUTHENTICATE LOGIN", null);
		} catch (Exception ex) {
			// Convert this into a BYE response
			r = Response.byeResponse(ex);
			done = true;
		}

		OutputStream os = getOutputStream(); // stream to IMAP server

		/* Wrap a BASE64Encoder around a ByteArrayOutputstream
		 * to craft b64 encoded username and password strings
		 *
		 * Note that the encoded bytes should be sent "as-is" to the
		 * server, *not* as literals or quoted-strings.
		 *
		 * Also note that unlike the B64 definition in MIME, CRLFs 
		 * should *not* be inserted during the encoding process. So, I
		 * use Integer.MAX_VALUE (0x7fffffff (> 1G)) as the bytesPerLine,
		 * which should be sufficiently large !
		 *
		 * Finally, format the line in a buffer so it can be sent as
		 * a single packet, to avoid triggering a bug in SUN's SIMS 2.0
		 * server caused by patch 105346.
		 */

		ByteArrayOutputStream bos = new ByteArrayOutputStream();
		OutputStream b64os = new BASE64EncoderStream(bos, Integer.MAX_VALUE);
		byte[] CRLF = { (byte)'\r', (byte)'\n'};
		boolean first = true;

		while (!done) { // loop till we are done
			try {
				r = readResponse();
				if (r.isContinuation()) {
					// Server challenge ..
					String s;
					if (first) { // Send encoded username
						s = u;
						first = false;
					} else 	// Send encoded password
						s = p;

					// obtain b64 encoded bytes
					b64os.write(ASCIIUtility.getBytes(s));
					b64os.flush(); 	// complete the encoding

					bos.write(CRLF); 	// CRLF termination
					os.write(bos.toByteArray()); // write out line
					os.flush(); 	// flush the stream
					bos.reset(); 	// reset buffer
				} else if (r.isTagged() && r.getTag().equals(tag))
					// Ah, our tagged response
					done = true;
				else if (r.isBYE()) // outta here
					done = true;
				else // hmm .. unsolicited response here ?!
					v.addElement(r);
			} catch (Exception ioex) {
				// convert this into a BYE response
				r = Response.byeResponse(ioex);
				done = true;
			}
		}

		/* Dispatch untagged responses.
		 * NOTE: in our current upper level IMAP classes, we add the
		 * responseHandler to the Protocol object only *after* the 
		 * connection has been authenticated. So, for now, the below
		 * code really ends up being just a no-op.
		 */
		Response[] responses = new Response[v.size()];
		v.copyInto(responses);
		notifyResponseHandlers(responses);

		// Handle the final OK, NO, BAD or BYE response
		handleResult(r);
		// If the response includes a CAPABILITY response code, process it
		if (r.isOK())
			setCapabilities(r);
		// if we get this far without an exception, we're authenticated
		authenticated = true;
	}

	/**
	 * The AUTHENTICATE command with AUTH=PLAIN authentication scheme.
	 * This is based heavly on the {@link #authlogin} method.
	 * 
	 * @param  authzid the authorization id
	 * @param  u the username
	 * @param  p the password
	 * @throws ProtocolException as thrown by {@link Protocol#handleResult}.
	 * @see "RFC3501, section 6.2.2"
	 * @see "RFC2595, section 6"
	 * @since JavaMail 1.3.2
	 */
	public void authplain(final String authzid, final String u, final String p) throws ProtocolException {
		Vector v = new Vector();
		String tag = null;
		Response r = null;
		boolean done = false;

		try {
			tag = writeCommand("AUTHENTICATE PLAIN", null);
		} catch (Exception ex) {
			r = Response.byeResponse(ex);
			done = true;
		}

		OutputStream os = getOutputStream();	// stream to IMAP server

		/* Wrap a BASE64Encoder around a ByteArrayOutputstream
		 * to craft b64 encoded username and password strings
		 *
		 * Note that the encoded bytes should be sent "as-is" to the
		 * server, *not* as literals or quoted-strings.
		 *
		 * Also note that unlike the B64 definition in MIME, CRLFs
		 * should *not* be inserted during the encoding process. So, I
		 * use Integer.MAX_VALUE (0x7fffffff (> 1G)) as the bytesPerLine,
		 * which should be sufficiently large !
		 *
		 * Finally, format the line in a buffer so it can be sent as
		 * a single packet, to avoid triggering a bug in SUN's SIMS 2.0
		 * server caused by patch 105346.
		 */

		ByteArrayOutputStream bos = new ByteArrayOutputStream();
		BASE64EncoderStream b64os = new BASE64EncoderStream(bos, Integer.MAX_VALUE);

		while (!done) {	// loop till we are done
			try {
				r = readResponse();
				if (r.isContinuation()) {
					// Server challenge ..
					final String nullByte = "\0";
					String s = authzid + nullByte + u + nullByte + p;

					// obtain b64 encoded bytes
					b64os.write(ASCIIUtility.getBytes(s));
					b64os.flush(); 	// complete the encoding

					bos.write(CRLF); 	// CRLF termination
					os.write(bos.toByteArray());	// write out line
					os.flush();	// flush the stream
					bos.reset();	// reset buffer
				} else if (r.isTagged() && r.getTag().equals(tag))
					// Ah, our tagged response
					done = true;
				else if (r.isBYE())	// outta here
					done = true;
				else	// hmm .. unsolicited response here ?!
					v.addElement(r);
			} catch (Exception ioex) {
				// convert this into a BYE response
				r = Response.byeResponse(ioex);
				done = true;
			}
		}

		/* Dispatch untagged responses.
		 * NOTE: in our current upper level IMAP classes, we add the
		 * responseHandler to the Protocol object only *after* the
		 * connection has been authenticated. So, for now, the below
		 * code really ends up being just a no-op.
		 */
		Response aresponse[] = new Response[v.size()];
		v.copyInto(aresponse);
		notifyResponseHandlers(aresponse);

		// Handle the final OK, NO, BAD or BYE response
		handleResult(r);
		// If the response includes a CAPABILITY response code, process it
		if (r.isOK())
			setCapabilities(r);

		// if we get this far without an exception, we're authenticated
		authenticated = true;
	}

	/**
	 * SASL-based login.
	 */
	public void sasllogin(final String allowed[], final String realm, final String authzid, final String u, final String p) throws ProtocolException {
		if (saslAuthenticator == null) {
			try {
				Class sac = Class.forName("com.sun.mail.imap.protocol.IMAPSaslAuthenticator");
				Constructor c = sac.getConstructor(new Class[] {
					IMAPProtocol.class,
					String.class,
					Properties.class,
					Boolean.TYPE,
					PrintStream.class,
					String.class
				});
				saslAuthenticator = (SaslAuthenticator) c.newInstance(new Object[] {
					this, name, props,
					debug ? Boolean.TRUE : Boolean.FALSE,
					out, host
				});
			} catch (Exception ex) {
				if (debug)
					out.println("IMAP DEBUG: Can't load SASL authenticator: " + ex);
				// probably because we're running on a system without SASL
				return;	// not authenticated, try without SASL
			}
		}

		// were any allowed mechanisms specified?
		Vector v;
		if (allowed != null && allowed.length > 0) {
			// remove anything not supported by the server
			v = new Vector(allowed.length);
			for (int i = 0; i < allowed.length; i++)
				if (authmechs.contains(allowed[i]))	// XXX - case must match
					v.addElement(allowed[i]);
		} else
			// everything is allowed
			v = authmechs;

		String mechs[] = new String[v.size()];
		v.copyInto(mechs);
		if (saslAuthenticator.authenticate(mechs, realm, authzid, u, p))
			authenticated = true;
	}

	// XXX - for IMAPSaslAuthenticator access to protected method
	OutputStream getIMAPOutputStream() {
		return getOutputStream();
	}

	/**
	 * PROXYAUTH R}h
	 * 
	 * @see "Netscape/iPlanet/SunONE Messaging Server extension"
	 */
	public void proxyauth(final String u) throws ProtocolException {
		Argument args = new Argument();
		args.writeASCII(u);

		simpleCommand("PROXYAUTH", args);
	}

	/**
	 * STARTTLS R}h
	 * 
	 * @see "RFC3501, section 6.2.1"
	 */
	public void startTLS() throws ProtocolException {
		try {
			startTLS("STARTTLS");
		} catch (ProtocolException pex) {
			// ProtocolException just means the command wasn't recognized,
			// or failed.  This should never happen if we check the
			// CAPABILITY first.
			throw pex;
		} catch (Exception ex) {
			// any other exception means we have to shut down the connection
			// generate an artificial BYE response and disconnect
			Response r[] = {
				Response.byeResponse(ex)
			};
			notifyResponseHandlers(r);
			disconnect();
		}
	}

	/**
	 * SELECT R}h
	 * 
	 * @see "RFC2060, section 6.3.1"
	 */
	public MailboxInfo select(String mbox) throws ProtocolException {
		// RFC2060 ɏ] mbox GR[h܂
// Sugisawa added. 2004/6/30
if (charset == null)
		mbox = BASE64MailboxEncoder.encode(mbox);

// Sugisawa added. 2004/8/17
if (mbox.indexOf(' ') > -1) mbox = '"' + mbox + '"';

		Argument args = new Argument();	
// Sugisawa changed. 2004/6/30
//		args.writeString(mbox);
// Sugisawa added. 2004/6/30
if (charset == null)
	args.writeASCII(mbox);
else {
	try {
		args.writeRawString(mbox, charset);
	} catch (UnsupportedEncodingException e) {
		args.writeASCII(mbox);
	}
}

		Response[] r = command("SELECT", args);

		// Note that MailboxInfo also removes those responses 
		// it knows about
		MailboxInfo minfo = new MailboxInfo(r);

		// dispatch any remaining untagged responses
		notifyResponseHandlers(r);

		Response response = r[r.length - 1];

		if (response.isOK()) // command succesful 
			if (response.toString().indexOf("READ-ONLY") != -1)
				minfo.mode = Folder.READ_ONLY;
			else
				minfo.mode = Folder.READ_WRITE;

		handleResult(response);
		return minfo;
	}

	/**
	 * EXAMINE R}h
	 * 
	 * @see "RFC2060, section 6.3.2"
	 */
	public MailboxInfo examine(String mbox) throws ProtocolException {
		// RFC2060 ɏ] mbox GR[h܂
// Sugisawa added. 2004/7/1
if (charset == null)
		mbox = BASE64MailboxEncoder.encode(mbox);

// Sugisawa added. 2004/8/17
if (mbox.indexOf(' ') > -1) mbox = '"' + mbox + '"';

		Argument args = new Argument();	
// Sugisawa changed. 2004/7/1
//		args.writeString(mbox);
// Sugisawa added. 2004/7/1
if (charset == null)
	args.writeASCII(mbox);
else {
	try {
		args.writeRawString(mbox, charset);
	} catch (UnsupportedEncodingException e) {
		args.writeASCII(mbox);
	}
}

		Response[] r = command("EXAMINE", args);

		// Note that MailboxInfo also removes those responses
		// it knows about
		MailboxInfo minfo = new MailboxInfo(r);
		minfo.mode = Folder.READ_ONLY; // Obviously

		// dispatch any remaining untagged responses
		notifyResponseHandlers(r);

		handleResult(r[r.length - 1]);
		return minfo;
	}

	/**
	 * STATUS R}h
	 * 
	 * @see "RFC2060, section 6.3.10"
	 */
	public Status status(String mbox, String[] items) throws ProtocolException {
		if (!isREV1() && !hasCapability("IMAP4SUNVERSION")) 
			// STATUS is rev1 only, however the non-rev1 SIMS2.0 
			// does support this.
			throw new BadCommandException("STATUS not supported");

		// RFC2060 ɏ] mbox GR[h܂
// Sugisawa added. 2004/7/1
if (charset == null)
		mbox = BASE64MailboxEncoder.encode(mbox);

// Sugisawa added. 2004/8/17
if (mbox.indexOf(' ') > -1) mbox = '"' + mbox + '"';

		Argument args = new Argument();	
// Sugisawa changed. 2004/7/1
//		args.writeString(mbox);
// Sugisawa added. 2004/7/1
if (charset == null)
	args.writeASCII(mbox);
else {
	try {
		args.writeRawString(mbox, charset);
	} catch (UnsupportedEncodingException e) {
		args.writeASCII(mbox);
	}
}

		Argument itemArgs = new Argument();
		if (items == null)
			items = Status.standardItems;

		for (int i = 0, len = items.length; i < len; i++)
			itemArgs.writeAtom(items[i]);
		args.writeArgument(itemArgs);

		Response[] r = command("STATUS", args);

		Status status = null;
		Response response = r[r.length - 1];

		// SĂ STATUS X|X݂͂܂
		if (response.isOK()) { // command succesful 
			for (int i = 0, len = r.length; i < len; i++) {
				if (r[i] instanceof IMAPResponse) {
					IMAPResponse ir = (IMAPResponse) r[i];
					if (ir.keyEquals("STATUS")) {
						if (status == null)
							// Sugisawa changed. 2005/04/14
							status = new Status(ir, charset);
						else // collect 'em all
							// Sugisawa changed. 2005/04/14
							Status.add(status, new Status(ir, charset));
						r[i] = null;
					}
				}
		    }
		}

		// dispatch remaining untagged responses
		notifyResponseHandlers(r);
		handleResult(response);
		return status;
	}

	/**
	 * CREATE R}h
	 * 
	 * @see "RFC2060, section 6.3.3"
	 */
	public void create(String mbox) throws ProtocolException {
		// RFC2060 ɏ] mbox GR[h܂
// Sugisawa added. 2004/7/1
if (charset == null)
		mbox = BASE64MailboxEncoder.encode(mbox);

// Sugisawa added. 2004/8/17
if (mbox.indexOf(' ') > -1) mbox = '"' + mbox + '"';

		Argument args = new Argument();	
// Sugisawa changed. 2004/7/1
//		args.writeString(mbox);
// Sugisawa added. 2004/7/1
if (charset == null)
	args.writeASCII(mbox);
else {
	try {
		args.writeRawString(mbox, charset);
	} catch (UnsupportedEncodingException e) {
		args.writeASCII(mbox);
	}
}

		simpleCommand("CREATE", args);
	}

	/**
	 * DELETE R}h
	 * 
	 * @see "RFC2060, section 6.3.4"
	 */
	public void delete(String mbox) throws ProtocolException {
		// RFC2060 ɏ] mbox GR[h܂
// Sugisawa added. 2004/7/1
if (charset == null)
		mbox = BASE64MailboxEncoder.encode(mbox);

// Sugisawa added. 2004/8/17
if (mbox.indexOf(' ') > -1) mbox = '"' + mbox + '"';

		Argument args = new Argument();	
// Sugisawa changed. 2004/7/1
//		args.writeString(mbox);
// Sugisawa added. 2004/7/1
if (charset == null)
	args.writeASCII(mbox);
else {
	try {
		args.writeRawString(mbox, charset);
	} catch (UnsupportedEncodingException e) {
		args.writeASCII(mbox);
	}
}

		simpleCommand("DELETE", args);
	}

	/**
	 * RENAME R}h
	 * 
	 * @see "RFC2060, section 6.3.5"
	 */
	public void rename(String o, String n) throws ProtocolException {
		// RFC2060 ɏ] mbox GR[h܂
// Sugisawa changed. 2004/7/1
//		o = BASE64MailboxEncoder.encode(o);
//		n = BASE64MailboxEncoder.encode(n);
// Sugisawa added. 2004/7/1
if (charset == null) {
	o = BASE64MailboxEncoder.encode(o);
	n = BASE64MailboxEncoder.encode(n);
}

//Sugisawa added. 2004/12/21
if (o.indexOf(' ') > -1) o = '"' + o + '"';
if (n.indexOf(' ') > -1) n = '"' + n + '"';

		Argument args = new Argument();	
// Sugisawa changed. 2004/7/1
//		args.writeString(o);
//		args.writeString(n);
// Sugisawa added. 2004/7/1
if (charset == null) {
	args.writeASCII(o);
	args.writeASCII(n);
} else {
	try {
		args.writeRawString(o, charset);
		args.writeRawString(n, charset);
	} catch (UnsupportedEncodingException e) {
		args.writeASCII(o);
		args.writeASCII(n);
	}
}

		simpleCommand("RENAME", args);
	}

	/**
	 * SUBSCRIBE R}h
	 * 
	 * @see "RFC2060, section 6.3.6"
	 */
	public void subscribe(String mbox) throws ProtocolException {
		Argument args = new Argument();	
		// RFC2060 ɏ] mbox GR[h܂
// Sugisawa added. 2004/7/1
if (charset == null)
		mbox = BASE64MailboxEncoder.encode(mbox);

// Sugisawa added. 2004/8/17
if (mbox.indexOf(' ') > -1) mbox = '"' + mbox + '"';

// Sugisawa changed. 2004/7/1
//		args.writeString(mbox);
// Sugisawa added. 2004/7/1
if (charset == null)
	args.writeASCII(mbox);
else {
	try {
		args.writeRawString(mbox, charset);
	} catch (UnsupportedEncodingException e) {
		args.writeASCII(mbox);
	}
}

		simpleCommand("SUBSCRIBE", args);
	}

	/**
	 * UNSUBSCRIBE R}h
	 * 
	 * @see "RFC2060, section 6.3.7"
	 */
	public void unsubscribe(String mbox) throws ProtocolException {
		Argument args = new Argument();	
		// RFC2060 ɏ] mbox GR[h܂
// Sugisawa added. 2004/7/1
if (charset == null)
		mbox = BASE64MailboxEncoder.encode(mbox);

// Sugisawa added. 2004/8/17
if (mbox.indexOf(' ') > -1) mbox = '"' + mbox + '"';

// Sugisawa changed. 2004/7/1
//		args.writeString(mbox);
// Sugisawa added. 2004/7/1
if (charset == null)
	args.writeASCII(mbox);
else {
	try {
		args.writeRawString(mbox, charset);
	} catch (UnsupportedEncodingException e) {
		args.writeASCII(mbox);
	}
}

		simpleCommand("UNSUBSCRIBE", args);
	}

	/**
	 * LIST R}h
	 * 
	 * @see "RFC2060, section 6.3.8"
	 */
	public ListInfo[] list(final String ref, final String pattern) throws ProtocolException {
		return doList("LIST", ref, pattern);
	}

	/**
	 * LIST R}h
	 * 
	 * @since Sugisawa added. 2004/06/30
	 * @deprecated
	 */
	public ListInfo[] list(final String ref, final String pattern, final String charset) throws ProtocolException {
		this.charset = charset;
		return list(ref, pattern);
	}

	/**
	 * LSUB R}h
	 * 
	 * @see "RFC2060, section 6.3.9"
	 */
	public ListInfo[] lsub(final String ref, final String pattern) throws ProtocolException {
		return doList("LSUB", ref, pattern);
	}

	/**
	 * LSUB R}h
	 * 
	 * @since Sugisawa added. 2004/06/30
	 * @deprecated
	 */
	public ListInfo[] lsub(final String ref, final String pattern, final String charset) throws ProtocolException {
		this.charset = charset;
		return lsub(ref, pattern);
	}

	private ListInfo[] doList(final String cmd, String ref, String pat) throws ProtocolException {
		// RFC2060 ɏ] mbox GR[h܂

		ref = BASE64MailboxEncoder.encode(ref);
// Sugisawa changed. 2004/08/17
//		pat = BASE64MailboxEncoder.encode(pat);
// Sugisawa added. 2004/08/17
		if (charset == null)
			pat = BASE64MailboxEncoder.encode(pat);

// Sugisawa added. 2004/8/17
if (pat.indexOf(' ') > -1) pat = '"' + pat + '"';

		Argument args = new Argument();	
		args.writeASCII(ref);
// Sugisawa changed. 2004/6/30
//		args.writeString(pat);

// Sugisawa added. 2004/6/30
if (charset == null)
	args.writeASCII(pat);
else {
	try {
		args.writeRawString(pat, charset);
	} catch (UnsupportedEncodingException e) {
		args.writeASCII(pat);
	}
}

		Response[] r = command(cmd, args);

		ListInfo[] linfo = null;
		Response response = r[r.length - 1];

		if (response.isOK()) { // command succesful 
			Vector v = new Vector(1);
			for (int i = 0, len = r.length; i < len; i++) {
				if (r[i] instanceof IMAPResponse) {
					IMAPResponse ir = (IMAPResponse) r[i];
					if (ir.keyEquals(cmd)) {
// Sugisawa changed. 2004/6/30
//						v.addElement(new ListInfo(ir));
// Sugisawa added. 2004/6/30
if (charset == null)
	v.addElement(new ListInfo(ir));
else
	v.addElement(new ListInfo(ir, charset));

						r[i] = null;
					}
				}
			}
			if (v.size() > 0) {
				linfo = new ListInfo[v.size()];
				v.copyInto(linfo);
			}
		}

		// Dispatch remaining untagged responses
		notifyResponseHandlers(r);
		handleResult(response);
		return linfo;
	}

	/**
	 * APPEND R}h
	 * 
	 * @see "RFC2060, section 6.3.11"
	 */
	public void append(final String mbox, final Flags f, final Date d, final Literal data) throws ProtocolException {
		appenduid(mbox, f, d, data, false);
	}

	public AppendUID appenduid(final String mbox, final Flags f, final Date d, final Literal data) throws ProtocolException {
		return appenduid(mbox, f, d, data, true);
	}

	public AppendUID appenduid(String mbox, final Flags f, final Date d, final Literal data, boolean flag) throws ProtocolException {
		// RFC2060 ɏ] mbox GR[h܂
// Sugisawa added. 2004/7/1
if (charset == null)
		mbox = BASE64MailboxEncoder.encode(mbox);

// Sugisawa added. 2004/8/17
if (mbox.indexOf(' ') > -1) mbox = '"' + mbox + '"';

		Argument args = new Argument();
// Sugisawa changed. 2004/7/1	
//		args.writeString(mbox);
// Sugisawa added. 2004/7/1
if (charset == null)
	args.writeASCII(mbox);
else {
	try {
		args.writeRawString(mbox, charset);
	} catch (UnsupportedEncodingException e) {
		args.writeASCII(mbox);
	}
}

		if (f != null) // ǉbZ[W Flags ݒ肵܂
			// APPEND ł \Recent tOݒł܂
			f.remove(Flags.Flag.RECENT);
		/*
		 * HACK ALERT: We want the flag_list to be written out
		 * without any checking/processing of the bytes in it. If
		 * I use writeString(), the flag_list will end up being
		 * quoted since it contains "illegal" characters. So I
		 * am depending on implementation knowledge that writeAtom()
		 * does not do any checking/processing - it just writes out
		 * the bytes. What we really need is a writeFoo() that just
		 * dumps out its argument.
		 */
		args.writeAtom(createFlagList(f));

		if (d != null) // ǉbZ[W INTERNALDATE ݒ肵܂
			args.writeASCII(INTERNALDATE.format(d));

		args.writeBytes(data);

		Response r[] = command("APPEND", args);
		notifyResponseHandlers(r);
		handleResult(r[r.length - 1]);

		if (flag)
			return getAppendUID(r[r.length - 1]);
		else
			return null;
	}

	private AppendUID getAppendUID(final Response r) {
		if (!r.isOK())
			return null;

		byte b;
		while ((b = r.readByte()) > 0 && b != 91)
			;
		if (b == 0)
			return null;

		String s = r.readAtom();
		if (!s.equalsIgnoreCase("APPENDUID"))
			return null;

		long uidvalidity = r.readLong();
		long uid = r.readLong();
		return new AppendUID(uidvalidity, uid);
	}

	/**
	 * CHECK R}h
	 * 
	 * @see "RFC2060, section 6.4.1"
	 */
	public void check() throws ProtocolException {
		simpleCommand("CHECK", null);
	}

	/**
	 * CLOSE R}h
	 * 
	 * @see "RFC2060, section 6.4.2"
	 */
	public void close() throws ProtocolException {
		simpleCommand("CLOSE", null);
	}

	/**
	 * EXPUNGE R}h
	 * 
	 * @see "RFC2060, section 6.4.3"
	 */
	public void expunge() throws ProtocolException {
		simpleCommand("EXPUNGE", null);
	}

	public void uidexpunge(final UIDSet[] auidset) throws ProtocolException {
		if (!hasCapability("UIDPLUS"))
			throw new BadCommandException("UID EXPUNGE not supported");
		simpleCommand("UID EXPUNGE " + UIDSet.toString(auidset), null);
	}

	/**
	 * w肳ꂽbZ[W BODYSTRUCTURE tFb`܂B
	 */
	public BODYSTRUCTURE fetchBodyStructure(final int msgno) throws ProtocolException {
		Response[] r = fetch(msgno, "BODYSTRUCTURE");
		notifyResponseHandlers(r);

		Response response = r[r.length - 1];
		if (response.isOK())
			return (BODYSTRUCTURE) FetchResponse.getItem(r, msgno, BODYSTRUCTURE.class);
		else if (response.isNO())
			return null;
		else {
			handleResult(response);
			return null;
		}
	}

	/**
	 * SEEN ƂăbZ[W}[NȂŁAw肳ꂽ BODY tFb`܂B
	 */
	public BODY peekBody(final int msgno, final String section) throws ProtocolException {
		return fetchBody(msgno, section, true);
	}

	/**
	 * w肳ꂽ BODY tFb`܂B
	 */
	public BODY fetchBody(final int msgno, final String section) throws ProtocolException {
		return fetchBody(msgno, section, false);
	}

	private BODY fetchBody(final int msgno, final String section, final boolean peek)
		throws ProtocolException {

		Response[] r;

		if (peek)
			r = fetch(msgno, "BODY.PEEK[" + (section == null ? "]" : section + "]"));
		else
			r = fetch(msgno, "BODY[" + (section == null ? "]" : section + "]"));

		notifyResponseHandlers(r);

		Response response = r[r.length - 1];
		if (response.isOK())
			return (BODY) FetchResponse.getItem(r, msgno, BODY.class);
		else if (response.isNO())
			return null;
		else {
			handleResult(response);
			return null;
		}
	}

	public BODY peekBody(final int msgno, final String section, final int start, final int size) throws ProtocolException {
		return fetchBody(msgno, section, start, size, true);
	}

	/**
	 * w肳ꂽ BODY ̕I FETCH łB
	 */
	public BODY fetchBody(final int msgno, final String section, final int start, final int size) throws ProtocolException {
        return fetchBody(msgno, section, start, size, false);
	}

	private BODY fetchBody(final int msgno, final String section, final int start, final int size, final boolean peek) throws ProtocolException {
		Response[] r = fetch(
			msgno, 
			(peek ? "BODY.PEEK[" : "BODY[") + (section == null ? "]<" : section +"]<") +
			String.valueOf(start) + '.' + String.valueOf(size) + '>');

		notifyResponseHandlers(r);

		Response response = r[r.length - 1];
		if (response.isOK())
			return (BODY) FetchResponse.getItem(r, msgno, BODY.class);
		else if (response.isNO())
			return null;
		else {
			handleResult(response);
			return null;
		}
	}

	/**
	 * w肳ꂽ RFC822 f[^ڂtFb`܂B
	 * 'what' names the item to be fetched.
	 * 'what' can be <code>null</code> to fetch the whole message.
	 */
	public RFC822DATA fetchRFC822(final int msgno, final String what) throws ProtocolException {
		Response[] r = fetch(msgno, what == null ? "RFC822" : "RFC822." + what);

		// dispatch untagged responses
		notifyResponseHandlers(r);

		Response response = r[r.length - 1]; 
		if (response.isOK())
			return (RFC822DATA) FetchResponse.getItem(r, msgno, RFC822DATA.class);
		else if (response.isNO())
			return null;
		else {
			handleResult(response);
			return null;
		}
	}

	/**
	 * w肳ꂽbZ[W FLAGS tFb`܂B
	 */
	public Flags fetchFlags(final int msgno) throws ProtocolException {
		Flags flags = null;
		Response[] r = fetch(msgno, "FLAGS");

		// FLAGS X|X܂
		for (int i = 0, len = r.length; i < len; i++) {
			if (r[i] == null ||
				!(r[i] instanceof FetchResponse) ||
				((FetchResponse)r[i]).getNumber() != msgno)
				continue;

			FetchResponse fr = (FetchResponse) r[i];
			if ((flags = (Flags)fr.getItem(Flags.class)) == null)
				continue;
			r[i] = null; // ̃X|X폜܂
			break;
		}

		// dispatch untagged responses
		notifyResponseHandlers(r);
		handleResult(r[r.length - 1]);
		return flags;
	}

	/**
	 * w肳ꂽbZ[W IMAP UID tFb`܂B
	 */
	public UID fetchUID(final int msgno) throws ProtocolException {
		Response[] r = fetch(msgno, "UID");

		// dispatch untagged responses
		notifyResponseHandlers(r);

		Response response = r[r.length - 1]; 
		if (response.isOK())
			return (UID) FetchResponse.getItem(r, msgno, UID.class);
		else if (response.isNO()) // XXX: Issue NOOP ?
			return null;
		else {
			handleResult(response);
			return null; // NOTREACHED
		}
	}
		
	/**
	 * Get the sequence number for the given UID. A UID object
	 * containing the sequence number is returned. If the given UID
	 * is invalid, <code>null</code> is returned.
	 */
	public UID fetchSequenceNumber(final long uid) throws ProtocolException {
		UID u = null;
		Response[] r = fetch(String.valueOf(uid), "UID", true);	

		for (int i = 0, len = r.length; i < len; i++) {
			if (r[i] == null || !(r[i] instanceof FetchResponse))
				continue;

			FetchResponse fr = (FetchResponse) r[i];
			if ((u = (UID)fr.getItem(UID.class)) != null) {
				if (u.uid == uid) // this is the one we want
					break;
				u = null;
		    }
		}

		notifyResponseHandlers(r);
		handleResult(r[r.length - 1]);
		return u;
	}

	/**
	 * Get the sequence numbers for UIDs ranging from start till end.
	 * UID objects that contain the sequence numbers are returned.
	 * If no UIDs in the given range are found, an empty array is returned.
	 */
	public UID[] fetchSequenceNumbers(final long start, final long end) throws ProtocolException {
		Response[] r = fetch(String.valueOf(start) + ':' + (end == UIDFolder.LASTUID ? "*" : String.valueOf(end)), "UID", true);

		UID u;
		Vector v = new Vector();
		for (int i = 0, len = r.length; i < len; i++) {
			if (r[i] == null || !(r[i] instanceof FetchResponse))
				continue;

			FetchResponse fr = (FetchResponse) r[i];
			if ((u = (UID)fr.getItem(UID.class)) != null)
				v.addElement(u);
		}

		notifyResponseHandlers(r);
		handleResult(r[r.length - 1]);

		UID[] ua = new UID[v.size()];
		v.copyInto(ua);
		return ua;
	}

	/**
	 * Get the sequence numbers for UIDs ranging from start till end.
	 * UID objects that contain the sequence numbers are returned.
	 * If no UIDs in the given range are found, an empty array is returned.
	 */
	public UID[] fetchSequenceNumbers(final long[] uids) throws ProtocolException {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < uids.length; i++) {
			if (i > 0)
				sb.append(',');
			sb.append(String.valueOf(uids[i]));
		}

		Response[] r = fetch(sb.toString(), "UID", true);	

		UID u;
		Vector v = new Vector();
		for (int i = 0, len = r.length; i < len; i++) {
			if (r[i] == null || !(r[i] instanceof FetchResponse))
				continue;

			FetchResponse fr = (FetchResponse) r[i];
			if ((u = (UID)fr.getItem(UID.class)) != null)
				v.addElement(u);
		}

		notifyResponseHandlers(r);
		handleResult(r[r.length - 1]);

		UID[] ua = new UID[v.size()];
		v.copyInto(ua);
		return ua;
	}

	public Response[] fetch(final MessageSet[] msgsets, final String what) throws ProtocolException {
		return fetch(MessageSet.toString(msgsets), what, false);
	}

	public Response[] fetch(final int start, final int end, final String what) throws ProtocolException {
		return fetch(String.valueOf(start) + ':' + String.valueOf(end), what, false);
	}

	public Response[] fetch(final int msg, final String what) throws ProtocolException {
		return fetch(String.valueOf(msg), what, false);
	}

	private Response[] fetch(final String msgSequence, final String what, final boolean uid) throws ProtocolException {
		if (uid)
			return command("UID FETCH " + msgSequence +" (" + what + ')', null);
		return command("FETCH " + msgSequence + " (" + what + ')', null);
	}

	/**
	 * COPY R}h
	 */
	public void copy(final MessageSet[] msgsets, final String mbox) throws ProtocolException {
		copy(MessageSet.toString(msgsets), mbox);
	}

	public void copy(final int start, final int end, final String mbox) throws ProtocolException {
		copy(String.valueOf(start) + ':' + String.valueOf(end), mbox);
	}

	private void copy(final String msgSequence, String mbox) throws ProtocolException {
// Sugisawa added. 2004/11/17
if (charset == null)
		// RFC2060 ɏ] mbox GR[h܂
		mbox = BASE64MailboxEncoder.encode(mbox);

// Sugisawa added. 2004/12/21
if (mbox.indexOf(' ') > -1) mbox = '"' + mbox + '"';

		Argument args = new Argument();	
		args.writeAtom(msgSequence);
// Sugisawa added. 2004/11/17
if (charset == null)
		args.writeASCII(mbox);
// Sugisawa added. 2004/11/17
else {
	try {
		args.writeRawString(mbox, charset);
	} catch (UnsupportedEncodingException e) {
		args.writeASCII(mbox);
	}
}

		simpleCommand("COPY", args);
	}

	public void storeFlags(final MessageSet[] msgsets, final Flags flags, final boolean set) throws ProtocolException {
		storeFlags(MessageSet.toString(msgsets), flags, set);
	}

	public void storeFlags(final int start, final int end, final Flags flags, final boolean set) throws ProtocolException {
		storeFlags(String.valueOf(start) + ':' + String.valueOf(end), flags, set);
	}

	/**
	 * ̃bZ[WɎw肳ꂽtOݒ肵܂B<p>
	 */
	public void storeFlags(final int msg, final Flags flags, final boolean set) throws ProtocolException { 
		storeFlags(String.valueOf(msg), flags, set);
	}

	private void storeFlags(final String msgset, final Flags flags, final boolean set) throws ProtocolException {
		Response[] r;
		if (set)
			r = command("STORE " + msgset + " +FLAGS " + createFlagList(flags), null);
		else
			r = command("STORE " + msgset + " -FLAGS " + createFlagList(flags), null);

		// Dispatch untagged responses
		notifyResponseHandlers(r);
		handleResult(r[r.length - 1]);
	}

	/**
	 * w肳ꂽ Flags IuWFNg IMAP tOXg쐬܂B
	 */
	private String createFlagList(final Flags flags) {
		StringBuffer sb = new StringBuffer();
		sb.append('('); // tOXg̊Jn

		Flags.Flag[] sf = flags.getSystemFlags(); // VXetO擾
		boolean first = true;
		for (int i = 0; i < sf.length; i++) {
			String s;
			Flags.Flag f = sf[i];
			if (f == Flags.Flag.ANSWERED)
				s = "\\Answered";
			else if (f == Flags.Flag.DELETED)
				s = "\\Deleted";
			else if (f == Flags.Flag.DRAFT)
				s = "\\Draft";
			else if (f == Flags.Flag.FLAGGED)
				s = "\\Flagged";
			else if (f == Flags.Flag.RECENT) {
				s = "\\Recent";				
			} else {
				if (f != Flags.Flag.SEEN)
					continue;	// XLbv܂
				s = "\\Seen";
			}
			if (first)
				first = false;
			else
				sb.append(' ');
			sb.append(s);
		}

		String[] uf = flags.getUserFlags(); // [U`tO̕擾
		for (int i = 0; i < uf.length; i++) {
			if (first)
				first = false;
			else
				sb.append(' ');
			sb.append(uf[i]);
		}

		sb.append(')'); // tOXg̏I
		return sb.toString();
	}

	/**
	 * Issue the given search criterion on the specified message sets.
	 * Returns array of matching sequence numbers. An empty array
	 * is returned if no matches are found.
	 * 
	 * @param msgsets MessageSet ̔z
	 * @param term SearchTerm
	 * @return vV[PXԍ̔z
	 */
	public int[] search(final MessageSet[] msgsets, final SearchTerm term) throws ProtocolException, SearchException {
		return search(MessageSet.toString(msgsets), term);
	}

	/**
	 * Issue the given search criterion on all messages in this folder.
	 * Returns array of matching sequence numbers. An empty array
	 * is returned if no matches are found.
	 * 
	 * @param term SearchTerm
	 * @return vV[PXԍ̔z
	 */
	public int[] search(final SearchTerm term) throws ProtocolException, SearchException {
		return search("ALL", term);
	}

	/* Apply the given SearchTerm on the specified sequence.
	 * Returns array of matching sequence numbers. Note that an empty
	 * array is returned for no matches.
	 */
	private int[] search(final String msgSequence, final SearchTerm term) throws ProtocolException, SearchException {
		// Check if the search "text" terms contain only ASCII chars
		if (SearchSequence.isAscii(term))
			try {
				return issueSearch(msgSequence, term, null);
			} catch (IOException ioex) { /* will not happen */ }

		/* The search "text" terms do contain non-ASCII chars. We need to
		 * use SEARCH CHARSET <charset> ...
		 *	The charsets we try to use are UTF-8 and the locale's
		 * default charset. If the server supports UTF-8, great, 
		 * always use it. Else we try to use the default charset.
		 */

		// Cycle thru the list of charsets
		for (int i = 0; i < searchCharsets.length; i++)
			if (searchCharsets[i] != null)
				try {
					return issueSearch(msgSequence, term, searchCharsets[i]);
				} catch (CommandFailedException cfx) {
					/* Server returned NO. For now, I'll just assume that 
					 * this indicates that this charset is unsupported.
					 * We can check the BADCHARSET response code once
					 * that's spec'd into the IMAP RFC ..
					 */
					searchCharsets[i] = null;
				} catch (IOException ioex) {
					/* Charset conversion failed. Try the next one */
				} catch (ProtocolException pex) {
					throw pex;
				} catch (SearchException sex) {
					throw sex;
				}

		// No luck.
		throw new SearchException("Search failed");
	}

	/* Apply the given SearchTerm on the specified sequence, using the
	 * given charset. <p>
	 * Returns array of matching sequence numbers. Note that an empty
	 * array is returned for no matches.
	 */
	private int[] issueSearch(final String msgSequence, final SearchTerm term, final String charset)
		throws ProtocolException, SearchException, IOException {

		// Generate a search-sequence with the given charset
		Argument args = SearchSequence.generateSequence(term, charset == null ? null : MimeUtility.javaCharset(charset));
		args.writeAtom(msgSequence);

		Response[] r;

		if (charset == null) // text is all US-ASCII
			r = command("SEARCH", args);
		else
			r = command("SEARCH CHARSET " + charset, args);

		Response response = r[r.length - 1];
		int[] matches = null;

		// SĂ SEARCH X|X݂͂܂
		if (response.isOK()) { // command succesful
			Vector v = new Vector();
			for (int i = 0, len = r.length; i < len; i++)
				if (r[i] instanceof IMAPResponse) {
					IMAPResponse ir = (IMAPResponse) r[i];
					// There *will* be one SEARCH response.
					if (ir.keyEquals("SEARCH")) {
						int num;
						while ((num = ir.readNumber()) != -1)
							v.addElement(new Integer(num));
						r[i] = null;
					}
				}

			// Copy the vector into 'matches'
			int vsize = v.size();
			matches = new int[vsize];
			for (int i = 0; i < vsize; i++)
				matches[i] = ((Integer)v.elementAt(i)).intValue();
		}

		// dispatch remaining untagged responses
		notifyResponseHandlers(r);
		handleResult(response);
		return matches;
	}

	/**
	 * NAMESPACE R}h
	 * 
	 * @see "RFC2342"
	 */
	public Namespaces namespace() throws ProtocolException {
		if (!hasCapability("NAMESPACE")) 
			throw new BadCommandException("NAMESPACE not supported");

		Response[] r = command("NAMESPACE", null);

		Namespaces namespace = null;
		Response response = r[r.length - 1];

		// NAMESPACE X|X݂͂܂
		if (response.isOK()) { // command succesful 
			for (int i = 0, len = r.length; i < len; i++) {
				if (!(r[i] instanceof IMAPResponse))
					continue;

				IMAPResponse ir = (IMAPResponse) r[i];
				if (ir.keyEquals("NAMESPACE")) {
					if (namespace == null)
						namespace = new Namespaces(ir);
					r[i] = null;
				}
			}
		}

		// dispatch remaining untagged responses
		notifyResponseHandlers(r);
		handleResult(response);
		return namespace;
	}

	/**
	 * GETQUOTAROOT R}h
	 * 
	 * Returns an array of Quota objects, representing the quotas
	 * for this mailbox and, indirectly, the quotaroots for this
	 * mailbox.
	 * 
	 * @see "RFC2087"
	 */
	public Quota[] getQuotaRoot(String mbox) throws ProtocolException {
		if (!hasCapability("QUOTA")) 
			throw new BadCommandException("GETQUOTAROOT not supported");

		// RFC2060 ɏ] mbox GR[h܂
		mbox = BASE64MailboxEncoder.encode(mbox);

		Argument args = new Argument();	
		args.writeASCII(mbox);

		Response[] r = command("GETQUOTAROOT", args);

		Response response = r[r.length - 1];

		Hashtable tab = new Hashtable();

		// SĂ QUOTAROOT y QUOTA X|X݂͂܂
		if (response.isOK()) { // command succesful 
			for (int i = 0, len = r.length; i < len; i++) {
				if (!(r[i] instanceof IMAPResponse))
					continue;

				IMAPResponse ir = (IMAPResponse) r[i];
				if (ir.keyEquals("QUOTAROOT")) {
					// quotaroot_response ::= "QUOTAROOT" SP astring *(SP astring)

					// read name of mailbox and throw away
					ir.readAtomString();
					// for each quotaroot add a placeholder quota
					String root = null;
					while ((root = ir.readAtomString()) != null)
						tab.put(root, new Quota(root));

					r[i] = null;
				} else if (ir.keyEquals("QUOTA")) {
					Quota quota = parseQuota(ir);
					Quota q = (Quota) tab.get(quota.quotaRoot);
					if (q != null && q.resources != null) {
						// XXX - should merge resources
					}
					tab.put(quota.quotaRoot, quota);
					r[i] = null;
				}
			}
		}

		// dispatch remaining untagged responses
		notifyResponseHandlers(r);
		handleResult(response);

		Quota[] qa = new Quota[tab.size()];
		Enumeration e = tab.elements();
		for (int i = 0; e.hasMoreElements(); i++)
			qa[i] = (Quota) e.nextElement();

		return qa;
	}

	/**
	 * GETQUOTA R}h
	 * 
	 * Returns an array of Quota objects, representing the quotas for this quotaroot.
	 * 
	 * @see "RFC2087"
	 */
	public Quota[] getQuota(final String root) throws ProtocolException {
		if (!hasCapability("QUOTA")) 
		    throw new BadCommandException("QUOTA not supported");

		Argument args = new Argument();	
		args.writeASCII(root);

		Response[] r = command("GETQUOTA", args);

		Quota quota = null;
		Vector v = new Vector();
		Response response = r[r.length - 1];

		// SĂ QUOTA X|X݂͂܂
		if (response.isOK()) { // command succesful 
			for (int i = 0, len = r.length; i < len; i++) {
				if (!(r[i] instanceof IMAPResponse))
					continue;

				IMAPResponse ir = (IMAPResponse) r[i];
				if (ir.keyEquals("QUOTA")) {
					quota = parseQuota(ir);
					v.addElement(quota);
					r[i] = null;
				}
			}
		}

		// dispatch remaining untagged responses
		notifyResponseHandlers(r);
		handleResult(response);
		Quota[] qa = new Quota[v.size()];
		v.copyInto(qa);
		return qa;
	}

	/**
	 * SETQUOTA R}h
	 * 
	 * Set the indicated quota on the corresponding quotaroot.
	 * 
	 * @see "RFC2087"
	 */
	public void setQuota(final Quota quota) throws ProtocolException {
		if (!hasCapability("QUOTA")) 
			throw new BadCommandException("QUOTA not supported");

		Argument args = new Argument();	
		args.writeASCII(quota.quotaRoot);
		Argument qargs = new Argument();	
		if (quota.resources != null) {
			for (int i = 0; i < quota.resources.length; i++) {
				qargs.writeAtom(quota.resources[i].name);
				qargs.writeNumber(quota.resources[i].limit);
			}
		}
		args.writeArgument(qargs);

		Response[] r = command("SETQUOTA", args);
		Response response = r[r.length - 1];

		// XXX - It's not clear from the RFC whether the SETQUOTA command
		// will provoke untagged QUOTA responses.  If it does, perhaps
		// we should grab them here and return them?

/*
		Quota quota = null;
		Vector v = new Vector();

		// SĂ QUOTA X|X݂͂܂
		if (response.isOK()) { // command succesful 
			for (int i = 0, len = r.length; i < len; i++) {
				if (!(r[i] instanceof IMAPResponse))
					continue;

				IMAPResponse ir = (IMAPResponse) r[i];
				if (ir.keyEquals("QUOTA")) {
					quota = parseQuota(ir);
					v.addElement(quota);
					r[i] = null;
				}
			}
		}
*/

		// dispatch remaining untagged responses
		notifyResponseHandlers(r);
		handleResult(response);
/*
		Quota[] qa = new Quota[v.size()];
		v.copyInto(qa);
		return qa;
*/
	}

	/**
	 * QUOTA X|X͂܂B
	 */
	private Quota parseQuota(final Response r) throws ParsingException {
		// quota_response ::= "QUOTA" SP astring SP quota_list
		String quotaRoot = r.readAtomString();	// quotaroot ::= astring
		Quota q = new Quota(quotaRoot);
		r.skipSpaces();
		// quota_list ::= "(" #quota_resource ")"
		if (r.readByte() != '(')
			throw new ParsingException("parse error in QUOTA");

		Vector v = new Vector();
		while (r.peekByte() != ')') {
			// quota_resource ::= atom SP number SP number
			String name = r.readAtom();
			if (name != null) {
				long usage = r.readLong();
				long limit = r.readLong();
				Quota.Resource res = new Quota.Resource(name, usage, limit);
				v.addElement(res);
			}
		}
		r.readByte();
		q.resources = new Quota.Resource[v.size()];
		v.copyInto(q.resources);
		return q;
	}

	/**
	 * SETACL R}h
	 * 
	 * @see "RFC2086"
	 */
	public void setACL(String mbox, final char modifier, final ACL acl) throws ProtocolException {
		if (!hasCapability("ACL")) 
			throw new BadCommandException("ACL not supported");

		// RFC2060 ɏ] mbox GR[h܂
		mbox = BASE64MailboxEncoder.encode(mbox);

		Argument args = new Argument();	
		args.writeASCII(mbox);
		args.writeASCII(acl.getName());
		String rights = acl.getRights().toString();

		if (modifier == '+' || modifier == '-')
			rights = modifier + rights;
		args.writeASCII(rights);

		Response[] r = command("SETACL", args);
		Response response = r[r.length - 1];

		// dispatch untagged responses
		notifyResponseHandlers(r);
		handleResult(response);
	}

	/**
	 * DELETEACL R}h
	 * 
	 * @see "RFC2086"
	 */
	public void deleteACL(String mbox, final String user) throws ProtocolException {
		if (!hasCapability("ACL")) 
			throw new BadCommandException("ACL not supported");

		// RFC2060 ɏ] mbox GR[h܂
		mbox = BASE64MailboxEncoder.encode(mbox);

		Argument args = new Argument();	
		args.writeASCII(mbox);
		args.writeASCII(user);

		Response[] r = command("DELETEACL", args);
		Response response = r[r.length - 1];

		// dispatch untagged responses
		notifyResponseHandlers(r);
		handleResult(response);
	}

	/**
	 * GETACL R}h
	 * 
	 * @see "RFC2086"
	 */
	public ACL[] getACL(String mbox) throws ProtocolException {
		if (!hasCapability("ACL")) 
			throw new BadCommandException("ACL not supported");

		// RFC2060 ɏ] mbox GR[h܂
		mbox = BASE64MailboxEncoder.encode(mbox);

		Argument args = new Argument();	
		args.writeASCII(mbox);

		Response[] r = command("GETACL", args);
		Response response = r[r.length - 1];

		// SĂ ACL X|X݂͂܂
		Vector v = new Vector();
		if (response.isOK()) { // command succesful 
			for (int i = 0, len = r.length; i < len; i++) {
				if (!(r[i] instanceof IMAPResponse))
					continue;

				IMAPResponse ir = (IMAPResponse) r[i];
				if (ir.keyEquals("ACL")) {
					// acl_data ::= "ACL" SPACE mailbox
					//		*(SPACE identifier SPACE rights)
					// read name of mailbox and throw away
					ir.readAtomString();
					String name = null;
					while ((name = ir.readAtomString()) != null) {
						String rights = ir.readAtomString();
						if (rights == null)
							break;
						ACL acl = new ACL(name, new Rights(rights));
						v.addElement(acl);
					}
					r[i] = null;
				}
			}
		}

		// dispatch remaining untagged responses
		notifyResponseHandlers(r);
		handleResult(response);
		ACL[] aa = new ACL[v.size()];
		v.copyInto(aa);
		return aa;
	}

	/**
	 * LISTRIGHTS R}h
	 * 
	 * @see "RFC2086"
	 */
	public Rights[] listRights(String mbox, final String user) throws ProtocolException {
		if (!hasCapability("ACL")) 
			throw new BadCommandException("ACL not supported");

		// RFC2060 ɏ] mbox GR[h܂
		mbox = BASE64MailboxEncoder.encode(mbox);

		Argument args = new Argument();	
		args.writeASCII(mbox);
		args.writeASCII(user);

		Response[] r = command("LISTRIGHTS", args);
		Response response = r[r.length - 1];

		// LISTRIGHTS X|X݂͂܂
		Vector v = new Vector();
		if (response.isOK()) { // command succesful 
			for (int i = 0, len = r.length; i < len; i++) {
				if (!(r[i] instanceof IMAPResponse))
					continue;

				IMAPResponse ir = (IMAPResponse) r[i];
				if (ir.keyEquals("LISTRIGHTS")) {
					// listrights_data ::= "LISTRIGHTS" SPACE mailbox
					//		SPACE identifier SPACE rights *(SPACE rights)
					// read name of mailbox and throw away
					ir.readAtomString();
					// read identifier and throw away
					ir.readAtomString();
					String rights;
					while ((rights = ir.readAtomString()) != null)
						v.addElement(new Rights(rights));
					r[i] = null;
				}
			}
		}

		// dispatch remaining untagged responses
		notifyResponseHandlers(r);
		handleResult(response);
		Rights[] ra = new Rights[v.size()];
		v.copyInto(ra);
		return ra;
	}

	/**
	 * MYRIGHTS R}h
	 * 
	 * @see "RFC2086"
	 */
	public Rights myRights(String mbox) throws ProtocolException {
		if (!hasCapability("ACL")) 
			throw new BadCommandException("ACL not supported");

		// RFC2060 ɏ] mbox GR[h܂
		mbox = BASE64MailboxEncoder.encode(mbox);

		Argument args = new Argument();	
		args.writeASCII(mbox);

		Response[] r = command("MYRIGHTS", args);
		Response response = r[r.length - 1];

		// MYRIGHTS X|X݂͂܂
		Rights rights = null;
		if (response.isOK()) { // command succesful
			for (int i = 0, len = r.length; i < len; i++) {
				if (!(r[i] instanceof IMAPResponse))
					continue;

				IMAPResponse ir = (IMAPResponse) r[i];
				if (ir.keyEquals("MYRIGHTS")) {
					// myrights_data ::= "MYRIGHTS" SPACE mailbox SPACE rights
					// read name of mailbox and throw away
					ir.readAtomString();
					String rs = ir.readAtomString();
					if (rights == null)
						rights = new Rights(rs);
					r[i] = null;
				}
			}
		}

		// dispatch remaining untagged responses
		notifyResponseHandlers(r);
		handleResult(response);
		return rights;
	}

}
