/*
 * Copyright 1997-2004 Sun Microsystems, Inc. All Rights Reserved.
 */

package com.sun.mail.smtp;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.StringReader;
import java.net.InetAddress;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.Hashtable;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.mail.Address;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.SendFailedException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.URLName;
import javax.mail.event.TransportEvent;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimePart;
import javax.mail.internet.ParseException;

import com.sun.mail.util.ASCIIUtility;
import com.sun.mail.util.BASE64EncoderStream;
import com.sun.mail.util.LineInputStream;
import com.sun.mail.util.SocketFetcher;
import com.sun.mail.util.TraceInputStream;
import com.sun.mail.util.TraceOutputStream;

/**
 * ̃NX́AbZ[WTu~bVƓ]ׂ̈ SMTP gp Transport ۃNX܂B
 * 
 * SMTP vgRvoC_̏ڍׂɊւĂ <a href="package-summary.html">jp.sourceforge.livez.mail.smtp</a> pbP[WhLe[VQƂĉB<p>
 * 
 * @see javax.mail.event.ConnectionEvent
 * @see javax.mail.event.TransportEvent
 */
public class SMTPTransport extends Transport {

	private String name = "smtp";		// ̃vgR̖O
	private int defaultPort = 25;		// ftHg SMTP |[g
	private boolean isSSL = false;	// SSL gp邩ǂ

	// Following fields valid only during the sendMessage method.
	private MimeMessage message;	// MׂbZ[W
	private Address[] addresses;	// msg 𑗐MAhX
	// Valid sent, valid unsent and invalid addresses
	private Address[] validSentAddr, validUnsentAddr, invalidAddr;
	// ̃AhXłAbZ[W𑗂܂?
	boolean sendPartiallyFailed = false;
	// Ƃ΁AɁA邽߂ɕKvƂO܂B
	MessagingException exception;

	// T[õT|[g SMTP T[rXg(EHLO gp)̃}bv
	private Hashtable extMap;

	private boolean quitWait = false;	// true if we should wait
	private String saslRealm = UNKNOWN;

	private boolean reportSuccess;	// throw an exception even on success
	private boolean useStartTLS;		// STARTTLS R}hgp邩ǂ
	private boolean useRset;

	private PrintStream out;			// fobOpo̓Xg[
	private String localHostName;		// zXg
	private String lastServerResponse;	// Ō SMTP X|X
	private int lastReturnCode;		// Ō SMTP ^[R[h

	/**
	 * MɊ܂܂ׂłȂwb_łB
	 */
	private static final String[] ignoreList = { "Bcc", "Content-Length" };
	private static final byte[] CRLF = { (byte)'\r', (byte)'\n' };
	private static final String UNKNOWN = "UNKNOWN";	// v[XtH_

	/**
	 * RXgN^łB
	 */
	public SMTPTransport(final Session session, final URLName urlname)    {
		this(session, urlname, "smtp", 25, false);
	}

	/**
	 * ̃NX SMTPSSLTransport TuNXɂĎgpRXgN^łB
	 */
	public SMTPTransport(final Session session, final URLName urlname, String name, int defaultPort, boolean isSSL) {
		super(session, urlname);

		if (urlname != null)
			name = urlname.getProtocol();

		this.name = name;
		this.defaultPort = defaultPort;
		this.isSSL = isSSL;
		out = session.getDebugOut();

		// mail.smtp.quitwait  true ɐݒ肵ꍇAQUIT R}h牞҂܂B
		String s = session.getProperty("mail." + name + ".quitwait");
		quitWait = s == null || s.equalsIgnoreCase("true");

		// mail.smtp.reportsuccess ͐ɗO𓊂܂B
		s = session.getProperty("mail." + s + ".reportsuccess");
		reportSuccess = s != null && s.equalsIgnoreCase("true");

		// mail.smtp.starttls.enable  STARTTLS R}h̎gp\ɂ܂B
		s = session.getProperty("mail." + s + ".starttls.enable");
		useStartTLS = s != null && s.equalsIgnoreCase("true");

		s = session.getProperty("mail." + s + ".userset");
		useRset = s != null && s.equalsIgnoreCase("true");
	}

	/**
	 * EHLO  HELO R}hgp鎞̃[JzXg̖O擾܂B
	 * mail.smtp.localhost vpeB InetAddress I[o[Ch܂B
	 */
	private String getLocalHost() {
		try {
			// zXg擾LbV܂
			if (localHostName == null || localHostName.length() <= 0)
				localHostName = session.getProperty("mail." + name + ".localhost");
			if (localHostName == null || localHostName.length() <= 0)
				localHostName = session.getProperty("mail." + name + ".localaddress");
			if (localHostName == null || localHostName.length() <= 0) {
				InetAddress inetaddress = InetAddress.getLocalHost();
				localHostName = inetaddress.getHostName();
				if (localHostName == null)
					localHostName = "[" + inetaddress.getHostAddress() + "]";
			}
		} catch (UnknownHostException uhex) {}
		return localHostName;
	}

	/**
	 * EHLO  HELO R}hgp鎞̃[JzXg̖Oݒ肵܂B
	 * 
	 * @since JavaMail 1.3.1
	 */
	public final void setLocalHost(final String localHostName) {
		this.localHostName = localHostName;
	}

	/**
	 * Start the SMTP protocol on the given socket, which was already connected by the caller.
	 * Useful for implementing the SMTP ATRN command (RFC 2645) where an existing connection is used when the server reverses roles and becomes the client.<p>
	 * 
	 * @since JavaMail 1.3.3
	 */
	public synchronized final void connect(final Socket socket) throws MessagingException {
		serverSocket = socket;
		super.connect();
	}

	/**
	 * DIGEST-MD5 F؂Ɏgp SASL 擾܂B
	 * 
	 * @return	SASL F؂Ɏgp郌̖O
	 * @since JavaMail 1.3.1
	 */
	public final String getSASLRealm() {
		if (saslRealm == "UNKNOWN") {
			saslRealm = session.getProperty("mail." + name + ".sasl.realm");
			if (saslRealm == null)
				saslRealm = session.getProperty("mail." + name + ".saslrealm");
		}
		return saslRealm;
	}

	/**
	 * SASL  DIGEST-MD5 F؂gplɐݒ肵܂B
	 * 
	 * @param saslRealm SASL F؂Ɏgp郌̖O
	 * @since JavMail 1.3.1
	 */
	public final void setSASLRealm(final String saslRealm) {
		this.saslRealm = saslRealm;
	}

	/**
	 * Should we report even successful sends by throwing an exception?
	 * If so, a <code>SendFailedException</code> will always be thrown and
	 * an {@link com.sun.mail.smtp.SMTPAddressSucceededException
	 * SMTPAddressSucceededException} will be included in the exception
	 * chain for each successful address, along with the usual
	 * {@link com.sun.mail.smtp.SMTPAddressFailedException
	 * SMTPAddressFailedException} for each unsuccessful address.
	 * 
	 * @return	true if an exception will be thrown on successful sends.
	 * @since JavaMail 1.3.2
	 */
	public final boolean getReportSuccess() {
		return reportSuccess;
	}

	/**
	 * Set whether successful sends should be reported by throwing an exception.
	 * 
	 * @param reportSuccess should we throw an exception on success?
	 * @since JavaMail 1.3.2
	 */
	public final void setReportSuccess(final boolean reportSuccess) {
		this.reportSuccess = reportSuccess;
	}

	/**
	 * Should we use the STARTTLS command to secure the connection if the server supports it?
	 * 
	 * @return true if the STARTTLS command will be used
	 * @since JavaMail 1.3.2
	 */
	public final boolean getStartTLS() {
		return useStartTLS;
	}

	/**
	 * Set whether the STARTTLS command should be used.
	 * 
	 * @param useStartTLS should we use the STARTTLS command?
	 * @since JavaMail 1.3.2
	 */
	public final void setStartTLS(final boolean useStartTLS) {
		this.useStartTLS = useStartTLS;
	}

	/**
	 * Should we use the RSET command instead of the NOOP command in the @{link #isConnected isConnected} method?
	 * 
	 * @return true if RSET will be used
	 * @since JavaMail 1.4
	 */
	public boolean getUseRset() {
		return useRset;
	}

	/**
	 * Set whether the RSET command should be used instead of the NOOP command in the @{link #isConnected isConnected} method.
	 * 
	 * @param useRset should we use the RSET command?
	 * @since JavaMail 1.4
	 */
	public void setUseRset(final boolean useRset) {
		this.useRset = useRset;
    }

	/**
	 * Return the last response we got from the server.
	 * A failed send is often followed by an RSET command, but the response from the RSET command is not saved.
	 * Instead, this returns the response from the command before the RSET command.
	 * 
	 * @return last response from server
	 * @since JavaMail 1.3.2
	 */
	public final String getLastServerResponse() {
		return lastServerResponse;
	}

	private DigestMD5 md5support;

	private synchronized DigestMD5 getMD5() {
		if (md5support == null)
			md5support = new DigestMD5(debug ? out : null);
		return md5support;
	}

	/**
	 * ۂ̃vgRL̐ڑ݂s܂B
	 * zXg null ̏ꍇA"localhost" ɐڑ݂̂܂B<p>
	 * 
	 * mail.smtp.ehlo  false ɐݒ肳ȂꍇA
	 * ESMTP R}h EHLO gpĎg肵悤Ƃ܂B
	 * 
	 * If mail.smtp.auth is set to true, we insist on having a username
	 * and password, and will try to authenticate ourselves if the server
	 * supports the AUTH extension (RFC 2554).
	 * 
	 * @param	host the name of the host to connect to
	 * @param	port the port to use (-1 means use default port)
	 * @param	user the name of the user to login as
	 * @param	passwd the user's password
	 * @return	true if connection successful, false if authentication failed
	 * @throws MessagingException	for non-authentication failures
	 */
	protected final boolean protocolConnect(
		String host,
		int port,
		final String user,
		final String passwd)
		throws MessagingException {

		// mail.smtp.ehlo  false ݒ肵ꍇAEHLO gp鎎݂𖳌ɂ܂B
		String ehloStr = session.getProperty("mail." + name + ".ehlo");
		boolean useEhlo = ehloStr == null || !ehloStr.equalsIgnoreCase("false");
		// mail.smtp.auth  true ݒ肵ꍇAAUTH gp鎎݂\ɂ܂B
		String authStr = session.getProperty("mail." + name + ".auth");
		boolean useAuth = authStr != null && authStr.equalsIgnoreCase("true");
		DigestMD5 md5;
		if (debug)
			out.println("DEBUG SMTP: useEhlo " + useEhlo + ", useAuth " + useAuth);

		/*
		 * If mail.smtp.auth is set, make sure we have a valid username
		 * and password, even if we might not end up using it (e.g.,
		 * because the server doesn't support ESMTP or doesn't support
		 * the AUTH extension).
		 */
		if (useAuth && (user == null || passwd == null))
			return false;

		/*
		 * If port is not specified, set it to value of mail.smtp.port
		 * property if it exists, otherwise default to 25.
		 */
		if (port == -1) {
			String portstring = session.getProperty("mail." + name + ".port");
			if (portstring != null)
				port = Integer.parseInt(portstring);
			else
				port = defaultPort;
		}

		if (host == null || host.length() == 0)
			host = "localhost";

		boolean succeed = false;

		if (serverSocket != null)
			openServer();
		else
			openServer(host, port);

		if (useEhlo)
			succeed = ehlo(getLocalHost());
		if (!succeed)
			helo(getLocalHost());

		if (useStartTLS && supportsExtension("STARTTLS")) {
			startTLS();
			/*
			 * Have to issue another EHLO to update list of extensions
			 * supported, especially authentication mechanisms.
			 * Don't know if this could ever fail, but we ignore failure.
			 */
			ehlo(getLocalHost());
		}

		if (useAuth && (supportsExtension("AUTH") || supportsExtension("AUTH=LOGIN"))) {
			if (debug) {
				out.println("DEBUG SMTP: Attempt to authenticate");
			if (!supportsAuthentication("LOGIN") &&
				supportsExtension("AUTH=LOGIN"))
				out.println("DEBUG SMTP: use AUTH=LOGIN hack");
			}
			// if authentication fails, close connection and return false
			if (supportsAuthentication("LOGIN") || supportsExtension("AUTH=LOGIN")) {
				// XXX - could use "initial response" capability
				int resp = simpleCommand("AUTH LOGIN");

				/*
				 * A 530 response indicates that the server wants us to
				 * issue a STARTTLS command first.  Do that and try again.
				 */
				if (resp == 530) {
					startTLS();
					resp = simpleCommand("AUTH LOGIN");
				}

				/*
				 * Wrap a BASE64Encoder around a ByteArrayOutputstream
				 * to craft b64 encoded username and password strings.
				 *
				 * Also note that unlike the B64 definition in MIME, CRLFs 
				 * should *not* be inserted during the encoding process.
				 * So I use Integer.MAX_VALUE (0x7fffffff (> 1G)) as the
				 * bytesPerLine, which should be sufficiently large!
				 */
				try {
					ByteArrayOutputStream bos = new ByteArrayOutputStream();
					OutputStream b64os = new BASE64EncoderStream(bos, Integer.MAX_VALUE);

					if (resp == 334) {
						// obtain b64 encoded bytes
						b64os.write(ASCIIUtility.getBytes(user));
						b64os.flush(); 	// GR[h

						// [UM
						resp = simpleCommand(bos.toByteArray());
						bos.reset(); 	// Zbgobt@
					}
					if (resp == 334) {
						// obtain b64 encoded bytes
						b64os.write(ASCIIUtility.getBytes(passwd));
						b64os.flush(); 	// GR[h

						// pX[hM
						resp = simpleCommand(bos.toByteArray());
						bos.reset(); 	// Zbgobt@
					}
				} catch (IOException ex) {	// ċNׂłȂ̂ŖB
				} finally {
					if (resp != 235) {
						closeConnection();
						return false;
					}
				}
			} else if (supportsAuthentication("PLAIN")) {
				// XXX - could use "initial response" capability
				int resp = simpleCommand("AUTH PLAIN");
				try {
					ByteArrayOutputStream bos = new ByteArrayOutputStream();
					OutputStream b64os = new BASE64EncoderStream(bos, Integer.MAX_VALUE);
					if (resp == 334) {
						// send "<NUL>user<NUL>passwd"
						// XXX - we don't send an authorization identity
						b64os.write(0);
						b64os.write(ASCIIUtility.getBytes(user));
						b64os.write(0);
						b64os.write(ASCIIUtility.getBytes(passwd));
						b64os.flush(); 	// GR[h

						// [UM
						resp = simpleCommand(bos.toByteArray());
					}
				} catch (IOException ex) {	// ċNׂłȂ̂ŖB
				} finally {
					if (resp != 235) {
						closeConnection();
						return false;
					}
				}
			} else if (supportsAuthentication("DIGEST-MD5") && (md5 = getMD5()) != null) {
				int resp = simpleCommand("AUTH DIGEST-MD5");
				try {
					if (resp == 334) {
						byte[] b = md5.authClient(host, user, passwd, getSASLRealm(), lastServerResponse);
						resp = simpleCommand(b);
						if (resp == 334) { // client authenticated by server
							if (!md5.authServer(lastServerResponse))
								// server NOT authenticated by client !!!
								resp = -1;
							else
								// send null response
								resp = simpleCommand(new byte[0]);
						}
					}
				} catch (Exception ex) {	// should never happen, ignore
					if (debug)
						out.println("DEBUG SMTP: DIGEST-MD5: " + ex);
				} finally {
					if (resp != 235) {
						closeConnection();
						return false;
					}
				}
			}
		}

		// we connected correctly
		return true;
	}

	/**
	 * Message w肳ꂽXg̃AhXɑM܂B<p>
	 * 
	 * If all the <code>addresses</code> succeed the SMTP check
	 * using the <code>RCPT TO:</code> command, we attempt to send the message.
	 * A TransportEvent of type MESSAGE_DELIVERED is fired indicating the
	 * successful submission of a message to the SMTP host.<p>
	 * 
	 * If some of the <code>addresses</code> fail the SMTP check,
	 * and the <code>mail.stmp.sendpartial</code> property is not set,
	 * sending is aborted. The TransportEvent of type MESSAGE_NOT_DELIVERED
	 * is fired containing the valid and invalid addresses. The
	 * SendFailedException is also thrown. <p>
	 * 
	 * If some of the <code>addresses</code> fail the SMTP check,
	 * and the <code>mail.stmp.sendpartial</code> property is set to true,
	 * the message is sent. The TransportEvent of type
	 * MESSAGE_PARTIALLY_DELIVERED
	 * is fired containing the valid and invalid addresses. The
	 * SMTPSendFailedException is also thrown. <p>
	 * 
	 * MessagingException is thrown if the message can't write out
	 * an RFC822-compliant stream using its <code>writeTo</code> method. <p>
	 * 
	 * @param message M MimeMessage
	 * @param addresses ̃bZ[W̑MAhX̃Xg
	 * @throws SMTPSendFailedException if the send failed because of an SMTP command error
	 * @throws SendFailedException ȃAhẌׂɑMsꍇ
	 * @throws MessagingException ڑIĂꍇA͐ڑԂɂȂꍇA̓bZ[W MimeMessage ł͂Ȃꍇ
	 * @see javax.mail.event.TransportEvent
	 */
	public synchronized final void sendMessage(
		final Message message,
		final Address[] addresses)
		throws MessagingException, SendFailedException {

		checkConnected();

		// check if the message is a valid MIME/RFC822 message and that
		// it has all valid InternetAddresses; fail if not
		if (!(message instanceof MimeMessage)) {
			if (debug)
				out.println("DEBUG SMTP: Can only send RFC822 msgs");
			throw new MessagingException("SMTP can only send RFC822 messages");
		}
		for (int i = 0; i < addresses.length; i++) {
			if (!(addresses[i] instanceof InternetAddress))
				throw new MessagingException(addresses[i] + " is not an InternetAddress");
		}

		this.message = (MimeMessage) message;
		this.addresses = addresses;
		validUnsentAddr = addresses;	// until we know better
		expandGroups();

		boolean use8bit = false;
		if (message instanceof SMTPMessage)
			use8bit = ((SMTPMessage)message).getAllow8bitMIME();
		if (!use8bit) {
			String ebStr = session.getProperty("mail." + name + ".allow8bitmime");
			use8bit = ebStr != null && ebStr.equalsIgnoreCase("true");
		}
		if (debug)
			out.println("DEBUG SMTP: use8bit " + use8bit);
		if (use8bit && supportsExtension("8BITMIME"))
			convertTo8Bit(this.message);

		try {
			mailFrom();
			rcptTo();
			this.message.writeTo(data(), ignoreList);
			finishData();
			if (sendPartiallyFailed) {
				// throw the exception,
				// fire TransportEvent.MESSAGE_PARTIALLY_DELIVERED event
				if (debug)
					out.println("DEBUG SMTP: Sending partially failed because of invalid destination addresses");
				notifyTransportListeners(
					TransportEvent.MESSAGE_PARTIALLY_DELIVERED,
					validSentAddr, validUnsentAddr, invalidAddr,
					this.message);

				throw new SMTPSendFailedException(".", lastReturnCode,
						lastServerResponse, exception,
						validSentAddr, validUnsentAddr, invalidAddr);
			}
			notifyTransportListeners(TransportEvent.MESSAGE_DELIVERED, validSentAddr, validUnsentAddr, invalidAddr, this.message);
		} catch (MessagingException mex) {
			if (debug)
				mex.printStackTrace(out);
			notifyTransportListeners(TransportEvent.MESSAGE_NOT_DELIVERED, validSentAddr, validUnsentAddr, invalidAddr, this.message);
			throw mex;
		} catch (IOException ex) {
			if (debug)
				ex.printStackTrace(out);
			// if we catch an IOException, it means that we want
			// to drop the connection so that the message isn't sent
			try {
				closeConnection();
			} catch (MessagingException mex) { /*  */ }
			notifyTransportListeners(TransportEvent.MESSAGE_NOT_DELIVERED, validSentAddr, validUnsentAddr, invalidAddr, this.message);

			throw new MessagingException("IOException while sending message", ex);
		} finally {
			// no reason to keep this data around
			validSentAddr = validUnsentAddr = invalidAddr = null;
			this.addresses = null;
			this.message = null;
			this.exception = null;
			sendPartiallyFailed = false;
		}
	}

	/**
	 * T[oAڑI܂B
	 */
	public synchronized final void close() throws MessagingException {
		if (!super.isConnected()) // Already closed.
			return;
		try {
			if (serverSocket != null) {
				sendCommand("QUIT");
				if (quitWait) {
					int resp = readServerResponse();
					if (resp != 221 && resp != -1)
						out.println("DEBUG SMTP: QUIT failed with " + resp);
				}
			}
		} finally {
			closeConnection();
		}
	}

	private void closeConnection() throws MessagingException {
		try {
			if (serverSocket != null)
				serverSocket.close();
		} catch (IOException ioex) {	    // shouldn't happen
			throw new MessagingException("Server Close Failed", ioex);
		} finally {
			serverSocket = null;
			serverOutput = null;
			serverInput = null;
			lineInputStream = null;
			if (super.isConnected())	// only notify if already connected
				super.close();
		}
	}

	/**
	 * Check whether the transport is connected. Override superclass
	 * method, to actually ping our server connection.
	 */
	public synchronized final boolean isConnected() {
		if (!super.isConnected())
			// if we haven't been connected at all, don't bother with NOOP
			return false;

		try {
			if (useRset)
				sendCommand("RSET");
			else
				sendCommand("NOOP");
			int resp = readServerResponse();

			// NOOP should return 250 on success, however, SIMS 3.2 returns
			// 200, so we work around it.
			//
			// Hotmail doesn't implement the NOOP command at all so assume
			// any kind of response means we're still connected.
			// That is, any response except 421, which means the server
			// is shutting down the connection.
			//
			if (resp >= 0 && resp != 421)
				return true;
			try {
				closeConnection();
			} catch (MessagingException mex) {}	// 
			return false;
		} catch (Exception ex) {
			try {
				closeConnection();
			} catch (MessagingException mex) {}	// 
			return false;
		}
	}

	/**
	 * Expand any group addresses.
	 */
	private void expandGroups() {
		Vector groups = null;
		for (int i = 0; i < addresses.length; i++) {
			InternetAddress a = (InternetAddress) addresses[i];
			if (a.isGroup()) {
				if (groups == null) {
					// first group, catch up with where we are
					groups = new Vector();
					for (int k = 0; k < i; k++)
						groups.addElement(addresses[k]);
				}
				// parse it and add each individual address
				try {
					InternetAddress[] ia = a.getGroup(true);
					if (ia != null) {
						for (int j = 0; j < ia.length; j++)
							groups.addElement(ia[j]);
					} else
						groups.addElement(a);
				} catch (ParseException pex) {
					// parse failed, add the whole thing
					groups.addElement(a);
				}
			} else {
				// if we've started accumulating a list, add this to it
				if (groups != null)
					groups.addElement(a);
			}
		}

		// if we have a new list, convert it back to an array
		if (groups != null) {
			InternetAddress[] newa = new InternetAddress[groups.size()];
			groups.copyInto(newa);
			addresses = newa;
		}
	}

	/**
	 * If the Part is a text part and has a Content-Transfer-Encoding
	 * of "quoted-printable" or "base64", and it obeys the rules for
	 * "8bit" encoding, change the encoding to "8bit".  If the part is
	 * a multipart, recursively process all its parts.
	 * 
	 * XXX - This is really quite a hack.
	 */
	private void convertTo8Bit(final MimePart part) {
		try {
			if (part.isMimeType("text/*")) {
				String enc = part.getEncoding();
				if (enc.equalsIgnoreCase("quoted-printable") ||
					enc.equalsIgnoreCase("base64")) {
					InputStream is = part.getInputStream();
					if (is8Bit(is))
						part.setHeader("Content-Transfer-Encoding", "8bit");
				}
			} else if (part.isMimeType("multipart/*")) {
				MimeMultipart mp = (MimeMultipart)part.getContent();
				int count = mp.getCount();
				for (int i = 0; i < count; i++)
					convertTo8Bit((MimePart)mp.getBodyPart(i));
			}
		} catch (IOException ioex) {
			// any exception causes us to give up
		} catch (MessagingException mex) {
			// any exception causes us to give up
		}
	}

	/**
	 * Check whether the data in the given InputStream follows the
	 * rules for 8bit text.  Lines have to be 998 characters or less
	 * and no NULs are allowed.  CR and LF must occur in pairs but we
	 * don't check that because we assume this is text and we convert
	 * all CR/LF combinations into canonical CRLF later.
	 */
	private boolean is8Bit(final InputStream is) {
		int b;
		int linelen = 0;
		boolean need8bit = false;
		try {
			while ((b = is.read()) >= 0) {
			b &= 0xff;
			if (b == '\r' || b == '\n')
				linelen = 0;
			else {
				if (b == 0)
					return false;
				if (++linelen > 998)	// 1000 - CRLF
					return false;
			}
			if (b > 0x7f)
				need8bit = true;
			}
		} catch (IOException ex) {
			return false;
		}
		if (debug && need8bit)
			out.println("DEBUG SMTP: found an 8bit part");
		return need8bit;
	}

	protected void finalize() throws Throwable {
		super.finalize();
		try {
			closeConnection();
		} catch (MessagingException mex) {}	// 
	}

	///////////////////// smtp stuff ///////////////////////

	private BufferedInputStream  serverInput;
	private LineInputStream      lineInputStream;
	private OutputStream         serverOutput;
	private Socket               serverSocket;

	/////// SMTP vgR //////

	private void helo(final String domain) throws MessagingException {
		if (domain != null)
			issueCommand("HELO " + domain, 250);
		else
			issueCommand("HELO", 250);
	}

	private boolean ehlo(final String domain) throws MessagingException {
		String cmd;
		if (domain != null)
			cmd = "EHLO " + domain;
		else
			cmd = "EHLO";
		sendCommand(cmd);
		int resp = readServerResponse();
		if (resp == 250) {
			// extract the supported service extensions
			BufferedReader rd = new BufferedReader(new StringReader(lastServerResponse));
			extMap = new Hashtable();
			try {
				boolean first = true;
				String line;
				while ((line = rd.readLine()) != null) {
					if (first) {	// skip first line which is the greeting
						first = false;
					} else if (line.length() >= 5) {	// shouldn't happen
						line = line.substring(4);	// skip response code
						int i = line.indexOf(' ');
						String arg = "";
						if (i > 0) {
							arg = line.substring(i + 1);
							line = line.substring(0, i);
						}
						if (debug)
							out.println("DEBUG SMTP: Found extension \"" + line + "\", arg \"" + arg + "\"");
						extMap.put(line.toUpperCase(), arg);
					}
				}
			} catch (IOException ex) {}	// can't happen
		}
		return resp == 250;
	}

	/*
	 * Gets the sender's address in the following order:
	 * 0. SMTPMessage.getEnvelopeFrom()
	 * 1. mail.smtp.from property
	 * 2. From: header in the message
	 * 3. system username using the InternetAddress.getLocalAddress() method
	 */
	private void mailFrom() throws MessagingException {
		String from = null;
		if (message instanceof SMTPMessage)
			from = ((SMTPMessage)message).getEnvelopeFrom();
		if (from == null || from.length() <= 0)
			from = session.getProperty("mail." + name + ".from");
		if (from == null || from.length() <= 0) {
			Address[] fa;
			Address me;
			if (message != null && (fa = message.getFrom()) != null && fa.length > 0)
				me = fa[0];
			else
				me = InternetAddress.getLocalAddress(session);

			if (me != null)
				from = ((InternetAddress)me).getAddress();
			else
				throw new MessagingException("can't determine local email address");
		}

		String cmd = "MAIL FROM:" + normalizeAddress(from);

		// request delivery status notification?
		if (supportsExtension("DSN")) {
			String ret = null;
			if (message instanceof SMTPMessage)
				ret = ((SMTPMessage)message).getDSNRet();
			if (ret == null)
				ret = session.getProperty("mail." + name + ".dsn.ret");
			// XXX - check for legal syntax?
			if (ret != null)
				cmd += " RET=" + ret;
		}

		/*
		 * If an RFC 2554 submitter has been specified, and the server
		 * supports the AUTH extension, include the AUTH= element on
		 * the MAIL FROM command.
		 */
		if (supportsExtension("AUTH")) {
			String ret = null;
			if (message instanceof SMTPMessage)
				ret = ((SMTPMessage)message).getSubmitter();
			if (ret == null)
				ret = super.session.getProperty("mail." + name + ".submitter");
			if (ret != null)
				try {
					String s = xtext(ret);
					cmd = cmd + " AUTH=" + s;
				} catch (IllegalArgumentException illegalargumentexception) {
					if(debug)
						out.println("DEBUG SMTP: ignoring invalid submitter: " + ret + ", Exception: " + illegalargumentexception);
				}
		}

		/*
		 * Have any extensions to the MAIL command been specified?
		 */
		String ext = null;
		if (message instanceof SMTPMessage)
			ext = ((SMTPMessage)message).getMailExtension();
		if (ext == null)
			ext = super.session.getProperty("mail." + name + ".mailextension");
		if (ext != null && ext.length() > 0)
			cmd = cmd + ' ' + ext;

		issueSendCommand(cmd, 250);
	}

	/**
	 * Sends each address to the SMTP host and copies it either into
	 * the validSentAddr or invalidAddr arrays.
	 * Sets the <code>sendFailed</code>
	 * flag to true if any addresses failed.
	 */
	/*
	 * success/failure/error possibilities from the RCPT command
	 * from rfc821, section 4.3
	 * S: 250, 251
	 * F: 550, 551, 552, 553, 450, 451, 452
	 * E: 500, 501, 503, 421
	 *
	 * and how we map the above error/failure conditions to valid/invalid
	 * address vectors that are reported in the thrown exception:
	 * invalid addr: 550, 501, 503, 551, 553
	 * valid addr: 552 (quota), 450, 451, 452 (quota), 421 (srvr abort)
	 */
	private void rcptTo() throws MessagingException {
		Vector valid = new Vector();
		Vector validUnsent = new Vector();
		Vector invalid = new Vector();
		int retCode = -1;
		MessagingException mex = null;
		boolean sendFailed = false;
		MessagingException sfex = null;
		validSentAddr = validUnsentAddr = invalidAddr = null;
		boolean sendPartial = false;
		if (message instanceof SMTPMessage)
			sendPartial = ((SMTPMessage)message).getSendPartial();
		if (!sendPartial) {
			String sp = session.getProperty("mail." + name + ".sendpartial");
			sendPartial = sp != null && sp.equalsIgnoreCase("true");
		}

		boolean dsn = false;
		String notify = null;
		if (supportsExtension("DSN")) {
			if (message instanceof SMTPMessage)
				notify = ((SMTPMessage)message).getDSNNotify();
			if (notify == null)
				notify = session.getProperty("mail." + name + ".dsn.notify");
			// XXX - check for legal syntax?
			if (notify != null)
				dsn = true;
		}

		// try the addresses one at a time
		for (int i = 0; i < addresses.length; i++) {
			sfex = null;
			InternetAddress ia = (InternetAddress) addresses[i];
			String cmd = "RCPT TO:" + normalizeAddress(ia.getAddress());
			if (dsn)
				cmd += " NOTIFY=" + notify;
			// send the addresses to the SMTP server
			sendCommand(cmd);
			// check the server's response for address validity
			retCode = readServerResponse();
			switch (retCode) {
				case 250: case 251:
					valid.addElement(addresses[i]);
					if (!reportSuccess)
						break;

					// user wants exception even when successful, including
					// details of the return code

					// create and chain the exception
					sfex = new SMTPAddressSucceededException(ia, cmd, retCode, lastServerResponse);
					if (mex == null)
						mex = sfex;
					else
						mex.setNextException(sfex);
					break;

				case 550: case 553: case 503: case 551: case 501:
					// given address is invalid
					if (!sendPartial)
						sendFailed = true;
					invalid.addElement(ia);
					// create and chain the exception
					sfex = new SMTPAddressFailedException(ia, cmd, retCode, lastServerResponse);
					if (mex == null)
						mex = sfex;
					else
						mex.setNextException(sfex);
					break;

				case 552: case 450: case 451: case 452:
					// given address is valid
					if (!sendPartial)
						sendFailed = true;
					validUnsent.addElement(ia);
					// create and chain the exception
					sfex = new SMTPAddressFailedException(ia, cmd, retCode, lastServerResponse);
					if (mex == null)
						mex = sfex;
					else
						mex.setNextException(sfex);
					break;

		    	default:
					// handle remaining 4xy & 5xy codes
					if (retCode >= 400 && retCode <= 499) {
						// assume address is valid, although we don't really know
						validUnsent.addElement(ia);
					} else if (retCode >= 500 && retCode <= 599) {
						// assume address is invalid, although we don't really know
						invalid.addElement(ia);
					} else {
						// completely unexpected response, just give up
						if (debug)
							out.println("DEBUG SMTP: got response code " + retCode + ", with response: " + lastServerResponse);
						String _lsr = lastServerResponse; // else rset will nuke it
						int _lrc = lastReturnCode;
						if (serverSocket != null)	// hasn't already been closed
							issueCommand("RSET", 250);
						lastServerResponse = _lsr;	// restore, for get
						lastReturnCode = _lrc;
						throw new SMTPAddressFailedException(ia, cmd, retCode, _lsr);
					}
					if (!sendPartial)
						sendFailed = true;
					// create and chain the exception
					sfex = new SMTPAddressFailedException(ia, cmd, retCode, lastServerResponse);
					if (mex == null)
						mex = sfex;
					else
						mex.setNextException(sfex);
					break;
			}
		}

		// if we're willing to send to a partial list, and we found no
		// valid addresses, that's complete failure
		if (sendPartial && valid.size() == 0)
			sendFailed = true;

		// copy the vectors into appropriate arrays
		if (sendFailed) {
			// copy invalid addrs
			invalidAddr = new Address[invalid.size()];
			invalid.copyInto(invalidAddr);

			// copy all valid addresses to validUnsent, since something failed
			validUnsentAddr = new Address[valid.size() + validUnsent.size()];
			int i = 0;
			for (int j = 0; j < valid.size(); j++)
				validUnsentAddr[i++] = (Address) valid.elementAt(j);
			for (int j = 0; j < validUnsent.size(); j++)
				validUnsentAddr[i++] = (Address) validUnsent.elementAt(j);
		} else if (reportSuccess || (sendPartial && (invalid.size() > 0 || validUnsent.size() > 0))) {
			// we'll go on to send the message, but after sending we'll
			// throw an exception with this exception nested
			sendPartiallyFailed = true;
			exception = mex;

			// copy invalid addrs
			invalidAddr = new Address[invalid.size()];
			invalid.copyInto(invalidAddr);

			// copy valid unsent addresses to validUnsent
			validUnsentAddr = new Address[validUnsent.size()];
			validUnsent.copyInto(validUnsentAddr);

			// copy valid addresses to validSent
			validSentAddr = new Address[valid.size()];
			valid.copyInto(validSentAddr);
		} else        // all addresses pass
			validSentAddr = addresses;

		// print out the debug info
		if (debug) {
			if (validSentAddr != null && validSentAddr.length > 0) {
				out.println("DEBUG SMTP: Verified Addresses");
				for (int l = 0; l < validSentAddr.length; l++) {
					out.println("DEBUG SMTP:   " + validSentAddr[l]);
				}
			}
			if (validUnsentAddr != null && validUnsentAddr.length > 0) {
				out.println("DEBUG SMTP: Valid Unsent Addresses");
				for (int j = 0; j < validUnsentAddr.length; j++) {
					out.println("DEBUG SMTP:   " + validUnsentAddr[j]);
				}
			}
			if (invalidAddr != null && invalidAddr.length > 0) {
				out.println("DEBUG SMTP: Invalid Addresses");
				for (int k = 0; k < invalidAddr.length; k++) {
					out.println("DEBUG SMTP:   " + invalidAddr[k]);
				}
			}
		}

		// throw the exception, fire TransportEvent.MESSAGE_NOT_DELIVERED event
		if (sendFailed) {
			if (debug)
				out.println("DEBUG SMTP: Sending failed because of invalid destination addresses");
			notifyTransportListeners(TransportEvent.MESSAGE_NOT_DELIVERED, validSentAddr, validUnsentAddr, invalidAddr, this.message);

			// reset the connection so more sends are allowed
			String lsr = lastServerResponse;	// save, for get
			int lrc = lastReturnCode;
			try {
				if (serverSocket != null)
					issueCommand("RSET", 250);
			} catch (MessagingException ex) {
				// if can't reset, best to close the connection
				try {
					close();
				} catch (MessagingException ex2) {
					// thrown by close()--ignore, will close() later anyway
					if (debug)
						ex2.printStackTrace(out);
				}
			} finally {
				lastServerResponse = lsr;	// XgA
				lastReturnCode = lrc;
			}

			throw new SendFailedException("Invalid Addresses", mex, validSentAddr, validUnsentAddr, invalidAddr);
		}
	}

	/**
	 * Send the "data" command to smtp host and return an OutputStream
	 * to which the data is to be written.
	 */
	private OutputStream data() throws MessagingException {
		issueSendCommand("DATA", 354);
		return new SMTPOutputStream(serverOutput);
	}

	private void finishData() throws MessagingException {
		issueSendCommand("\r\n.", 250);	// XXX - adds blank line to message?
	}

	private void startTLS() throws MessagingException {
		issueCommand("STARTTLS", 220);
		// it worked, now switch the socket into TLS mode
		try {
			serverSocket = SocketFetcher.startTLS(serverSocket);
			initStreams();
		} catch (IOException ioex) {
			closeConnection();
			throw new MessagingException("Could not convert socket to TLS", ioex);
		}
	}

	/////// primitives ///////

	private void openServer(final String server, int port) throws MessagingException {
		if (debug)
			out.println("DEBUG SMTP: trying to connect to host \"" + server + "\", port " + port + ", isSSL " + isSSL);

		try {
			Properties props = session.getProperties();

			serverSocket = SocketFetcher.getSocket(server, port, props, "mail." + name, isSSL);

			// socket factory may've chosen a different port,
			// update it for the debug messages that follow
			port = serverSocket.getPort();

			initStreams();

			int r = -1;
			if ((r = readServerResponse()) != 220) {
				serverSocket.close();
				serverSocket = null;
				serverOutput = null;
				serverInput = null;
				lineInputStream = null;
				if (debug)
					out.println("DEBUG SMTP: could not connect to host \"" + server + "\", port: " + port + ", response: " + r + "\n");
				throw new MessagingException("Could not connect to SMTP host: " + server + ", port: " + port + ", response: " + r);
			}
			if (debug)
				out.println("DEBUG SMTP: connected to host \"" + server + "\", port: " + port + "\n");
		} catch (UnknownHostException uhex) {
			throw new MessagingException("Unknown SMTP host: " + server, uhex);
		} catch (IOException ioe) {
			throw new MessagingException("Could not connect to SMTP host: " + server + ", port: " + port, ioe);
		}
	}

	private void openServer() throws MessagingException {
		int i = -1;
		String s = "UNKNOWN";
		try {
			i = serverSocket.getPort();
			s = serverSocket.getInetAddress().getHostName();
			if (super.debug)
				out.println("DEBUG SMTP: starting protocol to host \"" + s + "\", port " + i);
			initStreams();
			int j = -1;
			if ((j = readServerResponse()) != 220) {
				serverSocket.close();
				serverSocket = null;
				serverOutput = null;
				serverInput = null;
				lineInputStream = null;
				if (super.debug)
					out.println("DEBUG SMTP: got bad greeting from host \"" + s + "\", port: " + i + ", response: " + j + "\n");
				throw new MessagingException("Got bad greeting from SMTP host: " + s + ", port: " + i + ", response: " + j);
			}
			if (super.debug) {
				out.println("DEBUG SMTP: protocol started to host \"" + s + "\", port: " + i + "\n");
				return;
			}
		} catch (IOException ioexception) {
			throw new MessagingException("Could not start protocol to SMTP host: " + s + ", port: " + i, ioexception);
		}
	}

	private void initStreams() throws IOException {
		Properties props = session.getProperties();
		PrintStream out = session.getDebugOut();
		boolean debug = session.getDebug();

		String s = props.getProperty("mail.debug.quote");
		boolean quote = s != null && s.equalsIgnoreCase("true");

		TraceInputStream traceInput = new TraceInputStream(serverSocket.getInputStream(), out);
		traceInput.setTrace(debug);
		traceInput.setQuote(quote);

		TraceOutputStream traceOutput = new TraceOutputStream(serverSocket.getOutputStream(), out);
		traceOutput.setTrace(debug);
		traceOutput.setQuote(quote);

		serverOutput = new BufferedOutputStream(traceOutput);
		serverInput = new BufferedInputStream(traceInput);
		lineInputStream = new LineInputStream(serverInput);
	}

	private void issueCommand(final String cmd, final int expect) throws MessagingException {
		sendCommand(cmd);

		// if server responded with an unexpected return code,
		// throw the exception, notifying the client of the response
		if (readServerResponse() != expect)
		    throw new MessagingException(lastServerResponse);
	}

	private void issueSendCommand(final String cmd, final int expect) throws MessagingException {
		sendCommand(cmd);

		// if server responded with an unexpected return code,
		// throw the exception, notifying the client of the response
		int ret;
		if ((ret = readServerResponse()) != expect) {
			// assume message was not sent to anyone,
			// combine valid sent & unsent addresses
			int vsl = validSentAddr == null ? 0 : validSentAddr.length;
			int vul = validUnsentAddr == null ? 0 : validUnsentAddr.length;
			Address[] valid = new Address[vsl + vul];
			if (vsl > 0)
				System.arraycopy(validSentAddr, 0, valid, 0, vsl);
			if (vul > 0)
				System.arraycopy(validUnsentAddr, 0, valid, vsl, vul);
			validSentAddr = null;
			validUnsentAddr = valid;
			throw new SMTPSendFailedException(cmd, ret, lastServerResponse, exception, validSentAddr, validUnsentAddr, invalidAddr);
		}
		return;
	}

	private int simpleCommand(final String cmd) throws MessagingException {
		sendCommand(cmd);
		return readServerResponse();
	}

	private int simpleCommand(final byte[] cmd) throws MessagingException {
		sendCommand(cmd);
		return readServerResponse();
	}

	/**
	 * Sends command <code>cmd</code> to the server terminating
	 * it with <code>CRLF</code>.
	 */
	private void sendCommand(final String cmd) throws MessagingException {
		sendCommand(ASCIIUtility.getBytes(cmd));
	}

	private void sendCommand(final byte[] cmdBytes) throws MessagingException {
		//if (debug)
			//out.println("DEBUG SMTP SENT: " + new String(cmdBytes, 0));

		try {
			serverOutput.write(cmdBytes);
			serverOutput.write(CRLF);
			serverOutput.flush();
		} catch (IOException ex) {
			throw new MessagingException("Can't send command to SMTP host", ex);
		}
	}

	/**
	 * Reads server reponse returning the <code>returnCode</code>
	 * as the number.  Returns -1 on failure. Sets
	 * <code>lastServerResponse</code> and <code>lastReturnCode</code>.
	 */
	private int readServerResponse() throws MessagingException {
		String serverResponse = "";
		int returnCode = 0;
		StringBuffer buf = new StringBuffer(100);

		// read the server response line(s) and add them to the buffer
		// that stores the response
		try {
			String line = null;

			do {
				line = lineInputStream.readLine();
				if (line == null) {
					serverResponse = buf.toString();
					if (serverResponse.length() == 0)
						serverResponse = "[EOF]";
					lastServerResponse = serverResponse;
					lastReturnCode = -1;
					if (debug)
						out.println("DEBUG SMTP: EOF: " + serverResponse);
					return -1;
				}
				buf.append(line);
				buf.append("\n");
			} while (isNotLastLine(line));

			serverResponse = buf.toString();
		} catch (IOException ioex) {
			if (debug)
				out.println("DEBUG SMTP: exception reading response: " + ioex);
			//ioex.printStackTrace(out);
			lastServerResponse = "";
			lastReturnCode = 0;
			throw new MessagingException("Exception reading response", ioex);
			//returnCode = -1;
		}

		// fobOo͂܂
		//if (debug)
			//out.println("DEBUG SMTP RCVD: " + serverResponse);

		// parse out the return code
		if (serverResponse != null && serverResponse.length() >= 3) {
			try {
				returnCode = Integer.parseInt(serverResponse.substring(0, 3));
			} catch (NumberFormatException nfe) {
				try {
					close();
				} catch (MessagingException mex) {
					// thrown by close()--ignore, will close() later anyway
					if (debug)
						mex.printStackTrace(out);
				}
				returnCode = -1;
			} catch (StringIndexOutOfBoundsException ex) {
				//if (debug) ex.printStackTrace(out);
				try {
					close();
				} catch (MessagingException mex) {
					// thrown by close()--ignore, will close() later anyway
					if (debug)
						mex.printStackTrace(out);
				}
				returnCode = -1;
			}
		} else {
			returnCode = -1;
		}
		if (returnCode == -1 && debug)
			out.println("DEBUG SMTP: bad server response: " + serverResponse);

		lastServerResponse = serverResponse;
		lastReturnCode = returnCode;
		return returnCode;
	}

	/**
	 * Check if we're in the connected state.  Don't bother checking
	 * whether the server is still alive, that will be detected later.
	 */
	private void checkConnected() {
		if (!super.isConnected())
			throw new IllegalStateException("Not connected");
	}

	// tests if the <code>line</code> is an intermediate line according to SMTP
	private boolean isNotLastLine(final String line) {
		return line != null && line.length() >= 4 && line.charAt(3) == '-';
	}

	// wraps an address in "<>"'s if necessary
	private String normalizeAddress(final String addr) {
		if ((!addr.startsWith("<")) && (!addr.endsWith(">")))
			return '<' + addr + '>';
		return addr;
	}

	/**
	 * Return true if the SMTP server supports the specified service extension.
	 * Extensions are reported as results of the EHLO command when connecting to the server.
	 * See <a href="http://www.ietf.org/rfc/rfc1869.txt">RFC 1869</a> and other RFCs that define specific extensions.
	 * 
	 * @param ext the service extension name
	 * @return true if the extension is supported
	 * @since JavaMail 1.3.2
	 */
	public boolean supportsExtension(final String ext) {
		return extMap != null && extMap.get(ext.toUpperCase()) != null;
	}

	/**
	 * Return the parameter the server provided for the specified service extension, or null if the extension isn't supported.
	 * 
	 * @param ext the service extension name
	 * @return the extension parameter
	 * @since JavaMail 1.3.2
	 */
	public final String getExtensionParameter(final String ext) {
		if (extMap == null)
			return null;
		return (String) extMap.get(ext.toUpperCase());
	}

	private boolean supportsAuthentication(final String auth) {
		if (extMap == null)
			return false;
		String a = (String) extMap.get("AUTH");
		if (a == null)
			return false;
		for (StringTokenizer st = new StringTokenizer(a); st.hasMoreTokens();) {
			String tok = st.nextToken();
			if (tok.equalsIgnoreCase(auth))
				return true;
		}
		return false;
	}

	private static final char[] hexchar = {
		'0', '1', '2', '3', '4', '5', '6', '7',
		'8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
	};

	/**
	 * Convert a string to RFC 1891 xtext format.
	 *
	 * <p><pre>
	 *     xtext = *( xchar / hexchar )
	 *
	 *     xchar = any ASCII CHAR between "!" (33) and "~" (126) inclusive,
	 *          except for "+" and "=".
	 *
	 * ; "hexchar"s are intended to encode octets that cannot appear
	 * ; as ASCII characters within an esmtp-value.
	 *
	 *     hexchar = ASCII "+" immediately followed by two upper case
	 *          hexadecimal digits
	 * </pre></p>
	 */
	private String xtext(final String s) {
		StringBuffer sb = null;
		for (int i = 0; i < s.length(); i++) {
			char c = s.charAt(i);
			if (c >= 128)	// not ASCII
				throw new IllegalArgumentException("Non-ASCII character in SMTP submitter: " + s);
			if (c < '!' || c > '~' || c == '+' || c == '=') {
				if (sb == null) {
					sb = new StringBuffer(s.length() + 4);
					sb.append(s.substring(0, i));
				}
				sb.append('+');
				sb.append(hexchar[(((int)c)& 0xf0) >> 4]);
				sb.append(hexchar[((int)c)& 0x0f]);
			} else {
				if (sb != null)
					sb.append(c);
			}
		}

		return sb != null ? sb.toString() : s;
	}

}
