/*
 * Copyright 1997-2004 Sun Microsystems, Inc. All Rights Reserved.
 */

package com.sun.mail.util;

import java.io.EOFException;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * ̃NX BASE64 fR[_܂B
 * It is implemented as
 * a FilterInputStream, so one can just wrap this class around
 * any input stream and read bytes from this filter. The decoding
 * is done as the bytes are read out.
 */
public final class BASE64DecoderStream extends FilterInputStream {

	private byte[] buffer; 	// fR[hoCg̃LbV
	private int bufsize = 0;	// LbVTCY
	private int index = 0;		// LbṼCfbNX

	private byte[] input_buffer = new byte[8190];
	private int input_pos;
	private int input_len;

	private boolean ignoreErrors = false;

	/** 
	 * w肳ꂽ̓Xg[fR[h BASE64 fR[_쐬܂B
	 * VXevpeB <code>mail.mime.base64.ignoreerrors</code> ́A
	 * fR[hꂽf[^ŃG[O𔭐邩邩ǂ䂵܂B
	 * ftHg false ł(G[̏ꍇ͗O𔭐܂)B
	 * 
	 * @param in ̓Xg[
	 */
	public BASE64DecoderStream(final InputStream in) {
		super(in);
		buffer = new byte[3];
		try {
			String s = System.getProperty("mail.mime.base64.ignoreerrors");
			// ftHg false
			ignoreErrors = s != null && !s.equalsIgnoreCase("false");
		} catch(SecurityException sex) {
			// 
		}
	}

	/** 
	 * w肳ꂽ̓Xg[fR[h BASE64 fR[_쐬܂B
	 * 
	 * @param in ̓Xg[
	 * @param ignoreErrors GR[hꂽf[^̃G[𖳎邩ǂ
	 */
	public BASE64DecoderStream(final InputStream in, boolean ignoreErrors) {
		super(in);
		buffer = new byte[3];
		this.ignoreErrors = ignoreErrors;
	}

	/**
	 * ̓Xg[玟̃oCgf[^ǂݍ݂܂B
	 * oCg̒l <code>0</code> ` <code>255</code> ͈̔͂̒lƂ <code>int</code> ƂĕԂ܂B
	 * Xg[̏IɒBēǂݍރf[^Ȃꍇ <code>-1</code> Ԃ܂B
	 * ̃\bh́A̓f[^ǂݍ߂lɂȂ邩AXg[̏I肪o邩A
	 * ͗OX[܂ŃubN܂B 
	 * 
	 * @return f[^̎̃oCgBXg[̏IɒBꍇ <code>-1</code>
	 * @throws IOException o͗Oꍇ
	 * @see java.io.FilterInputStream#in
	 */
	public int read() throws IOException {
		if (index >= bufsize) {
			// obt@𖄂߂܂
			bufsize = decode(buffer, 0, buffer.length);
			if (bufsize <= 0) // obt@̏ꍇ
				return -1;
			index = 0; // obt@̃CfbNXZbg܂
		}
		return buffer[index++] & 0xff; // Zero off the MSB
	}

	/**
	 * ̓Xg[ <code>len</code> oCg̃f[^oCgzɓǂݍ݂܂B
	 * ̃\bh́A̓f[^ǂݍ߂lɂȂ܂ŃubN܂B
	 * <p>
	 * 
	 * @param buf the buffer into which the data is read.
	 * @param off the start offset of the data.
	 * @param len the maximum number of bytes read.
	 * @return the total number of bytes read into the buffer, or
	 *             <code>-1</code> if there is no more data because the end of
	 *             the stream has been reached.
	 * @throws IOException o͗Oꍇ
	 */
	public int read(final byte[] buf, int off, int len) throws IOException {
		int i = off;
		for (; index < bufsize && len > 0; len--)
			buf[off++] = buffer[index++];

		if (index >= bufsize)
			bufsize = index = 0;
		int l = (len / 3) * 3;
		if (l > 0) {
			int i1 = decode(buf, off, l);
			off += i1;
			len -= i1;
			if (i1 != l) {
				if (off == i)
					return -1;
				return off - i;
			}
		}
		for (; len > 0; len--) {
			int c = read();
			if (c == -1)
				break;
			buf[off++] = (byte) c;
        }

		if (off == i)
			return -1;
		return off- i;
	}

	/**
	 * Tests if this input stream supports marks. Currently this class does not support marks
	 */
	public boolean markSupported() {
		return false; // Maybe later ..
	}

	/**
	 * Returns the number of bytes that can be read from this input
	 * stream without blocking. However, this figure is only
	 * a close approximation in case the original encoded stream
	 * contains embedded CRLFs; since the CRLFs are discarded, not decoded
	 */ 
	public int available() throws IOException {
		// This is only an estimate, since in.available()
		// might include CRLFs too ..
		return (in.available() * 3) / 4 + (bufsize - index);
	}

	/**
	 * This character array provides the character to value map
	 * based on RFC1521.
	 */  
	private final static char pem_array[] = {
		'A','B','C','D','E','F','G','H', // 0
		'I','J','K','L','M','N','O','P', // 1
		'Q','R','S','T','U','V','W','X', // 2
		'Y','Z','a','b','c','d','e','f', // 3
		'g','h','i','j','k','l','m','n', // 4
		'o','p','q','r','s','t','u','v', // 5
		'w','x','y','z','0','1','2','3', // 6
		'4','5','6','7','8','9','+','/'  // 7
	};

	private final static byte pem_convert_array[] = new byte[256];

	static {
		for (int i = 0; i < 255; i++)
			pem_convert_array[i] = -1;
		for (int i = 0; i < pem_array.length; i++)
			pem_convert_array[pem_array[i]] = (byte) i;
	}

	private int decode(final byte[] buf, int off, int len) throws IOException {
		int k = off;
		while (len >= 3) {
			int l = 0;
			int a;
			int i;
			for (a = 0; l < 4; a |= i) {
				i = getByte();
				if (i == -1 || i == -2) {
					boolean flag;
					if (i == -1) {
						if (l == 0)
							return off - k;
						if (!ignoreErrors)
							throw new IOException("Error in encoded stream: needed 4 valid base64 characters but only got " + l + " before EOF" + recentChars());
						flag = true;
					} else {
						if (l < 2 && !ignoreErrors)
							throw new IOException("Error in encoded stream: needed at least 2 valid base64 characters, but only got " + l + " before padding character (=)" + recentChars());
						if (l == 0)
							return off - k;
						flag = false;
					}
					int k1 = l - 1;
					if (k1 == 0)
						k1 = 1;
					l++;
					a <<= 6;
					for (; l < 4; l++) {
						if (!flag) {
							i = getByte();
							if (i == -1) {
								if (!ignoreErrors)
									throw new IOException("Error in encoded stream: hit EOF while looking for padding characters (=)" + recentChars());
							} else
								if (i != -2 && !ignoreErrors)
									throw new IOException("Error in encoded stream: found valid base64 character after a padding character (=)" + recentChars());
						}
						a <<= 6;
					}

					a >>= 8;
					if (k1 == 2)
						buf[off + 1] = (byte)(a & 0xff);
					a >>= 8;
					buf[off] = (byte)(a & 0xff);
					len -= k1;
					off += k1;
					return off - k;
				}
				a <<= 6;
				l++;
			}

			buf[off + 2] = (byte)(a & 0xff);
			a >>= 8;
			buf[off + 1] = (byte)(a & 0xff);
			a >>= 8;
			buf[off] = (byte)(a & 0xff);
			len -= 3;
			off += 3;
		}
		return off - k;
	}

	private int getByte() throws IOException {
		int i;
		do {
			if (input_pos >= input_len) {
				try {
					input_len = super.in.read(input_buffer);
				} catch (EOFException _ex) {
					return -1;
				}
				if (input_len <= 0)
					return -1;
				input_pos = 0;
			}
			i = input_buffer[input_pos++] & 0xff;
			if (i == '=')
				return -2;
			i = pem_convert_array[i];
		} while (i == -1);
		return i;
	}

	private String recentChars() {
		String s = "";
		int i = input_pos <= 10 ? input_pos : 10;
		if (i > 0) {
			s = s + ", the " + i + " most recent characters were: \"";
			for (int j = input_pos - i; j < input_pos; j++) {
				char c = (char)(input_buffer[j] & 0xff);
				switch (c) {
					case '\r':
						s = s + "\\r";
						break;

					case '\n':
						s = s + "\\n";
						break;

					case '\t':
						s = s + "\\t";
						break;

					case '\013':
					case '\f':
					default:
						if (c >= ' ' && c < '\177')
							s = s + c;
						else
							s = s + "\\" + (int)c;
						break;
				}
			}

			s = s + "\"";
		}
		return s;
	}

	/**
	 * Base64 decode a byte array.  No line breaks are allowed.
	 * This method is suitable for short strings, such as those
	 * in the IMAP AUTHENTICATE protocol, but not to decode the
	 * entire content of a MIME part.
	 * 
	 * NOTE: inbuf may only contain valid base64 characters.
	 *       Whitespace is not ignored.
	 */
	public static byte[] decode(final byte[] inbuf) {
		int size = (inbuf.length / 4) * 3;
		if (size == 0)
			return inbuf;

		if (inbuf[inbuf.length - 1] == '=') {
			size--;
			if (inbuf[inbuf.length - 2] == '=')
				size--;
		}
		byte[] outbuf = new byte[size];

		int inpos = 0, outpos = 0;
		for (size = inbuf.length; size > 0; size -= 4) {
			int a = 3;
			int b = pem_convert_array[inbuf[inpos++] & 0xff];

			b <<= 6;
			b |= pem_convert_array[inbuf[inpos++] & 0xff];
			b <<= 6;

			if (inbuf[inpos] != '=')
				b |= pem_convert_array[inbuf[inpos++] & 0xff];
			else
				a--;

			b <<= 6;
			if (inbuf[inpos] != '=')
				b |= pem_convert_array[inbuf[inpos++] & 0xff];
			else
				a--;

			b <<= 6;
			if (size > 2)
				inbuf[outpos + 2] = (byte)(b & 0xff);
			b >>= 8;
			if (size > 1)
				inbuf[outpos + 1] = (byte)(b & 0xff);
			b >>= 8;
			inbuf[outpos] = (byte)(b & 0xff);
			outpos += a;
		}
		return outbuf;
	}

}
