/*
 * Copyright 1998-2004 Sun Microsystems, Inc. All Rights Reserved.
 */

package com.sun.mail.util;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Properties;

import javax.net.SocketFactory;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;

/**
 * ̃NX́ASocket Q擾ׂɎgp܂B
 * Depending on the arguments passed
 * it will either return a plain java.net.Socket or dynamically load
 * the SocketFactory class specified in the classname param and return
 * a socket created by that SocketFactory.
 */
public final class SocketFetcher {

	/**
	 * pbP[WvCx[gRXgN^łB
	 */
	private SocketFetcher() {}

	/**
	 * This method returns a Socket.  Properties control the use of
	 * socket factories and other socket characteristics.  The properties
	 * used are: <p>
	 * <ul>
	 * <li> <i>prefix</i>.socketFactory.class
	 * <li> <i>prefix</i>.socketFactory.fallback
	 * <li> <i>prefix</i>.socketFactory.port
	 * <li> <i>prefix</i>.timeout
	 * <li> <i>prefix</i>.connectiontimeout
	 * <li> <i>prefix</i>.localaddress
	 * <li> <i>prefix</i>.localport
	 * </ul> <p>
	 * If the socketFactory.class property isn't set, the socket
	 * returned is an instance of java.net.Socket connected to the
	 * given host and port. If the socketFactory.class property is set,
	 * it is expected to contain a fully qualified classname of a
	 * javax.net.SocketFactory subclass.  In this case, the class is
	 * dynamically instantiated and a socket created by that
	 * SocketFactory is returned. <p>
	 * 
	 * If the socketFactory.fallback property is set to false, don't
	 * fall back to using regular sockets if the socket factory fails. <p>
	 * 
	 * The socketFactory.port specifies a port to use when connecting
	 * through the socket factory.  If unset, the port argument will be
	 * used.  <p>
	 * 
	 * If the connectiontimeout property is set, we use a separate thread
	 * to make the connection so that we can timeout that connection attempt.
	 * <p>
	 * 
	 * If the timeout property is set, it is used to set the socket timeout.
	 * <p>
	 * 
	 * If the localaddress property is set, it's used as the local address
	 * to bind to.  If the localport property is also set, it's used as the
	 * local port number to bind to.
	 * 
	 * @param host The host to connect to
	 * @param port The port to connect to at the host
	 * @param props Properties object containing socket properties
	 * @param prefix Property name prefix, e.g., "mail.imap"
	 * @param useSSL use the SSL socket factory as the default
	 */
	public static Socket getSocket(
		final String host,
		final int port,
		Properties props,
		String prefix,
		boolean useSSL)
		throws IOException {

		if (prefix == null)
			prefix = "socket";
		if (props == null)
			props = new Properties();	// 
		String s = props.getProperty(prefix + ".connectiontimeout", null);
		int cto = -1;
		if (s != null) {
			try {
				cto = Integer.parseInt(s);
			} catch (NumberFormatException nfex) {}
		}

		Socket socket = null;
		String sfClass = props.getProperty(prefix + ".socketFactory.class", null);
		String timeout = props.getProperty(prefix + ".timeout", null);
		String localaddrstr = props.getProperty(prefix + ".localaddress", null);
		InetAddress localaddr = null;
		if (localaddrstr != null)
			localaddr = InetAddress.getByName(localaddrstr);
		String localportstr = props.getProperty(prefix + ".localport", null);
		int localport = 0;
		if (localportstr != null)
			try {
				localport = Integer.parseInt(localportstr);
			} catch (NumberFormatException ex) {}

		if (sfClass != null && sfClass.length() > 0) {
			// dynamically load the class 
			int sfPort = -1;
			boolean fb = false;
			// dynamically load the class 
			String fallback = props.getProperty(prefix + ".socketFactory.fallback", null);
			fb = fallback == null || !fallback.equalsIgnoreCase("false");
			String sfPortStr = props.getProperty(prefix + ".socketFactory.port", null);
			if (sfPortStr != null)
				try {
					sfPort = Integer.parseInt(sfPortStr);
				} catch (NumberFormatException ex) {}

			try {
				ClassLoader cl = getContextClassLoader();
				Class clsSockFact = null;
				if (cl != null)
					try {
						clsSockFact = cl.loadClass(sfClass);
					} catch (ClassNotFoundException ex) {}

				if (clsSockFact == null)
					clsSockFact = Class.forName(sfClass);
				// get & invoke the getDefault() method
				Method mthGetDefault = clsSockFact.getMethod("getDefault", new Class[0]);
				SocketFactory socketfactory = (SocketFactory) mthGetDefault.invoke(new Object(), new Object[0]);
				if (sfPort == -1)
					sfPort = port;
				socket = createSocket(localaddr, localport, s, sfPort, port, socketfactory, useSSL);
			} catch (SocketTimeoutException stex) {
				throw stex;
			} catch (Exception ex) {
				if (!fb) {
					if (ex instanceof InvocationTargetException) {
						Throwable t = ((InvocationTargetException)ex).getTargetException();
						if (t instanceof Exception)
							ex = (Exception) t;
					}
					if (ex instanceof IOException) {
						throw (IOException) ex;
					} else {
						IOException ioex = new IOException("Couldn't connect using \"" + sfClass + "\" socket factory to host, port: " + s + ", " + sfPort + "; Exception: " + ex);
						ioex.initCause(ex);
						throw ioex;
					}
				}
			}
		}

		if (socket == null)
			socket = createSocket(localaddr, localport, s, port, cto, null, useSSL);

		int to = -1;
		if (timeout != null) {
			try {
				to = Integer.parseInt(timeout);
			} catch (NumberFormatException nfex) {}
		}
		if (to >= 0)
			socket.setSoTimeout(to);

		return socket;
	}

	public static Socket getSocket(final String host, final int port, final Properties props, final String prefix) throws IOException {
		return getSocket(host, port, props, prefix, false);
	}

	private static Socket createSocket(
		InetAddress addr,
		int port,
		String hostname,
		int j,
		int timeout,
		SocketFactory socketfactory,
		boolean useSSL)
		throws IOException {

		Socket socket;
		if (socketfactory != null)
			socket = socketfactory.createSocket();
		else if (useSSL)
			socket = SSLSocketFactory.getDefault().createSocket();
		else
			socket = new Socket();

		if (addr != null)
			socket.bind(new InetSocketAddress(addr, port));

		if (timeout >= 0)
			socket.connect(new InetSocketAddress(hostname, j), timeout);
		else
			socket.connect(new InetSocketAddress(hostname, j));
		return socket;
	}

	/**
	 * Start TLS on an existing socket.
	 * Supports the "STARTTLS" command in many protocols.
	 */
	public static Socket startTLS(Socket socket) throws IOException {
		InetAddress a = socket.getInetAddress();
		String host = a.getHostName();
		int port = socket.getPort();
		try {
			SSLSocketFactory sslsocketfactory = (SSLSocketFactory) SSLSocketFactory.getDefault();
			socket = sslsocketfactory.createSocket(socket, host, port, true);
			((SSLSocket)socket).setEnabledProtocols(new String[] { "TLSv1" });
		} catch (Exception ex) {
			if (ex instanceof InvocationTargetException) {
				Throwable t = ((InvocationTargetException)ex).getTargetException();
				if (t instanceof Exception)
					ex = (Exception) t;
			}
			if (ex instanceof IOException) {
				throw (IOException) ex;
			} else {
				IOException ioex = new IOException("Exception in startTLS: host " + host + ", port " + port + "; Exception: " + ex);
				ioex.initCause(ex);
				throw ioex;
			}
		}
		return socket;
	}

	private static ClassLoader getContextClassLoader() {
		return (ClassLoader) AccessController.doPrivileged(new PrivilegedAction() {
			public Object run() {
				ClassLoader classloader = null;
				try {
					classloader = Thread.currentThread().getContextClassLoader();
				} catch (SecurityException ex) {}
				return classloader;
			}
		});
	}

}
