/*
 * Copyright 1997-2003 Sun Microsystems, Inc. All Rights Reserved.
 */

package javax.mail;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.lang.reflect.Constructor;
import java.net.InetAddress;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

import com.sun.mail.util.LineInputStream;


/**
 * Session NX̓[ZbV\ATuNX܂B
 * Session NX́A[ API ɂgpvpeBƃftHgW܂B
 * 1 ̃ftHgZbVAfXNgbṽ̕AvP[Vɂ苤L\łB
 * LȂZbV쐬鎖ł܂B<p>
 * 
 * The Session class provides access to the protocol providers that
 * implement the <code>Store</code>, <code>Transport</code>, and related
 * classes.  The protocol providers are configured using the following files:
 * <ul>
 *  <li> <code>javamail.providers</code> and
 * 	<code>javamail.default.providers</code> </li>
 *  <li> <code>javamail.address.map</code> and
 * 	<code>javamail.default.address.map</code> </li>
 * </ul>
 * <p>
 * Each <code>javamail.</code><i>X</i> resource file is searched for using
 * three methods in the following order:
 * <ol>
 *  <li> <code>java.home/lib/javamail.</code><i>X</i> </li>
 *  <li> <code>META-INF/javamail.</code><i>X</i> </li>
 *  <li> <code>META-INF/javamail.default.</code><i>X</i> </li>
 * </ol>
 * <p>
 * The first method allows the user to include their own version of the
 * resource file by placing it in the <code>lib</code> directory where the
 * <code>java.home</code> property points.  The second method allows an
 * application that uses the JavaMail APIs to include their own resource
 * files in their application's or jar file's <code>META-INF</code>
 * directory.  The <code>javamail.default.</code><i>X</i> default files
 * are part of the JavaMail <code>mail.jar</code> file. <p>
 * 
 * File location depends upon how the <code>ClassLoader</code> method
 * <code>getResource</code> is implemented.  Usually, the
 * <code>getResource</code> method searches through CLASSPATH until it
 * finds the requested file and then stops.  JDK 1.1 has a limitation that
 * the number of files of each name that will be found in the CLASSPATH is
 * limited to one.  However, this only affects method two, above; method
 * one is loaded from a specific location (if allowed by the
 * SecurityManager) and method three uses a different name to ensure that
 * the default resource file is always loaded successfully.  J2SE 1.2 and
 * later are not limited to one file of a given name. <p>
 * 
 * The ordering of entries in the resource files matters.  If multiple
 * entries exist, the first entries take precedence over the later
 * entries.  For example, the first IMAP provider found will be set as the
 * default IMAP implementation until explicitly changed by the
 * application.  The user- or system-supplied resource files augment, they
 * do not override, the default files included with the JavaMail APIs.
 * This means that all entries in all files loaded will be available. <p>
 * 
 * <b><code>javamail.providers</code></b> and
 * <b><code>javamail.default.providers</code></b><p>
 * 
 * These resource files specify the stores and transports that are
 * available on the system, allowing an application to "discover" what
 * store and transport implementations are available.  The protocol
 * implementations are listed one per line.  The file format defines four
 * attributes that describe a protocol implementation.  Each attribute is
 * an "="-separated name-value pair with the name in lowercase. Each
 * name-value pair is semi-colon (";") separated.  The following names
 * are defined. <p>
 * 
 * <table border=1>
 * <caption>
 * Attribute Names in Providers Files
 * </caption>
 * <tr>
 * <th>Name</th><th>Description</th>
 * </tr>
 * <tr>
 * <td>protocol</td>
 * <td>Name assigned to protocol.
 * For example, <code>smtp</code> for Transport.</td>
 * </tr>
 * <tr>
 * <td>type</td>
 * <td>Valid entries are <code>store</code> and <code>transport</code>.</td>
 * </tr>
 * <tr>
 * <td>class</td>
 * <td>Class name that implements this protocol.</td>
 * </tr>
 * <tr>
 * <td>vendor</td>
 * <td>Optional string identifying the vendor.</td>
 * </tr>
 * <tr>
 * <td>version</td>
 * <td>Optional string identifying the version.</td>
 * </tr>
 * </table><p>
 * 
 * Here's an example of <code>META-INF/javamail.default.providers</code>
 * file contents:
 * <pre>
 * protocol=imap; type=store; class=com.sun.mail.imap.IMAPStore; vendor=Sun Microsystems, Inc.;
 * protocol=smtp; type=transport; class=com.sun.mail.smtp.SMTPTransport; vendor=Sun Microsystems, Inc.;
 * </pre><p>
 * 
 * <b><code>javamail.address.map</code></b> and
 * <b><code>javamail.default.address.map</code></b><p>
 * 
 * These resource files map transport address types to the transport
 * protocol.  The <code>getType</code> method of
 * </code>javax.mail.Address</code> returns the address type.  The
 * <code>javamail.address.map</code> file maps the transport type to the
 * protocol.  The file format is a series of name-value pairs.  Each key
 * name should correspond to an address type that is currently installed
 * on the system; there should also be an entry for each
 * <code>javax.mail.Address</code> implementation that is present if it is
 * to be used.  For example, the
 * <code>javax.mail.internet.InternetAddress</code> method
 * <code>getType</code> returns "rfc822". Each referenced protocol should
 * be installed on the system.  For the case of <code>news</code>, below,
 * the client should install a Transport provider supporting the nntp
 * protocol. <p>
 * 
 * Here are the typical contents of a <code>javamail.address.map</code> file:
 * <pre>
 * rfc822=smtp
 * news=nntp
 * </pre>
 */
public final class Session {

	private final Properties props;
	private final Authenticator authenticator;
	private final Hashtable authTable = new Hashtable();
	private boolean debug = false;
	private PrintStream out;			// fobOpo̓Xg[
	private final Vector providers = new Vector();
	private final Hashtable providersByProtocol = new Hashtable();
	private final Hashtable providersByClassName = new Hashtable();
	private final Properties addressMap = new Properties();
						// maps type to protocol
	// ftHgZbV
	private static Session defaultSession = null;

	// fobOo͂Ŏgp JavaMail ̃o[Wԍ
	private static final String version = "1.4ea";

	// Constructor is not public
	private Session(final Properties props, final Authenticator authenticator) {
		this.props = props;
		this.authenticator = authenticator;

		if (Boolean.valueOf(props.getProperty("mail.debug")).booleanValue())
			debug = true;

		if (debug)
			pr("DEBUG: JavaMail version " + version);

		// get the Class associated with the Authenticator
		Class cl;
		if (authenticator != null)
			cl = authenticator.getClass();
		else
			cl = this.getClass();

		// \[Xǂݍ݂܂
		loadProviders(cl);
		loadAddressMap(cl);
    }

	/**
	 * V Session IuWFNg擾܂B
	 * 
	 * @param props ֘AvpeBێ Properties IuWFNg<br>
	 * NCAǵAJavaMail dl̕t^ A ɋLڂvpeB ( mail.store.protocolA
	 * mail.transport.protocolAmail.hostAmail.userAy mail.from) ̒lȂ΂Ȃ܂B
	 * ́AftHgSĂ̏ꍇɋ@\Ƃ͌ȂׂłB
	 * @param authenticator [UƃpX[hKvȏꍇA
	 * AvP[VɃR[obNׂɎgp Authenticator IuWFNg
	 * 
	 * @return V Session IuWFNg
	 * @see javax.mail.Authenticator
	 */
	public static Session getInstance(final Properties props, final Authenticator authenticator) {
		return new Session(props, authenticator);
	}

	/**
	 * V Session IuWFNg擾܂B
	 * 
	 * @param props ֘AvpeBێ Properties IuWFNg<br>
	 * NCAǵAJavaMail dl̕t^ A ɋLڂvpeB ( mail.store.protocolA
	 * mail.transport.protocolAmail.hostAmail.userAy mail.from) ̒lȂ΂Ȃ܂B
	 * ́AftHgSĂ̏ꍇɋ@\Ƃ͌ȂׂłB
	 * 
	 * @return V Session IuWFNg
	 * @since JavaMail 1.2
	 */
	public static Session getInstance(final Properties props) {
		return new Session(props, null);
	}

	/**
	 * ftHg Session IuWFNg擾܂B
	 * ftHg܂ݒ肳ĂȂꍇAV Session IuWFNg쐬A
	 * ftHgƂăCXg[܂B<p>
	 * 
	 * ftHg̃ZbV͓ Java z}VŎsSẴR[hŎgpł\A
	 * ZbV̓[UpX[hƂZLeB@̏܂ގ̂ŁA
	 * ftHg̃ZbVւ̃ANZX͐܂B
	 * Ăяoɂ쐬Ȃ΂ȂȂ Authenticator IuWFNǵA
	 * ANZX`FbNׂɊԐړIɎgp܂B
	 * ZbV쐬ۂɓn Authenticator IuWFNǵA
	 * ftHg̃ZbV擾ׂ̌㑱vɓn Authenticator IuWFNgƔr܂B
	 * 2 ̃IuWFNgꍇA͓ ClassLoader ̏ꍇAv͋܂B
	 * łȂꍇ͋ۂ܂B<p>
	 * 
	 * ZbV쐬ׂɎgp Authenticator IuWFNg null ̏ꍇAnull 
	 * nɂNłftHg̃ZbV擾ł܂B<p>
	 * 
	 * Note also that the Properties object is used only the first time
	 * this method is called, when a new Session object is created.
	 * Subsequent calls return the Session object that was created by the
	 * first call, and ignore the passed Properties object.  Use the
	 * <code>getInstance</code> method to get a new Session object every
	 * time the method is called. <p>
	 * 
	 * JDK 1.2 ł́AftHg̃ZbVւ̃ANZX𐧌䂷ׂɁA
	 * ǉ̃ZLeB Permission IuWFNggpł܂B
	 * 
	 * @param props Properties IuWFNgBV Session IuWFNg쐬ꍇɂ̂ݎgp܂B<br>
	 * NCAǵAJavaMail dl̕t^ A ɋLڂvpeB ( mail.store.protocolA
	 * mail.transport.protocolAmail.hostAmail.userAy mail.from) ̒lȂ΂Ȃ܂B
	 * ́AftHgSĂ̏ꍇɋ@\Ƃ͌ȂׂłB
	 * @param authenticator Authenticator IuWFNgB
	 * V Session IuWFNg쐬ꍇɂ̂ݎgp܂B
	 * łȂꍇASession 쐬ׂɎgp Authenticator ɈvȂ΂Ȃ܂B
	 * @return ftHg Session IuWFNg
	 */
	public static synchronized Session getDefaultInstance(
		final Properties props,
		final Authenticator authenticator) {

		if (defaultSession == null)
			defaultSession = new Session(props, authenticator);
		else {
			// have to check whether caller is allowed to see default session
			if (defaultSession.authenticator == authenticator)
				;	// either same object or both null, either way OK
			else if (defaultSession.authenticator != null &&
				authenticator != null &&
				defaultSession.authenticator.getClass().getClassLoader() ==
				authenticator.getClass().getClassLoader())
				;	// both objects came from the same class loader, OK
			else
				// anything else is not allowed
				throw new SecurityException("Access to default session denied");
		}

		return defaultSession;
	}

	/**
	 * ftHg Session IuWFNg擾܂B
	 * ftHg܂ݒ肳ĂȂꍇAV Session IuWFNg쐬A
	 * ftHgƂăCXg[܂B<p>
	 * 
	 * Authenticator Ȃō쐬ꂽftHg̃ZbV́A
	 *  Java z}VŎs̑SẴR[hgp\łA
	 * ZbVɂ̓[UpX[hƂZLeB@̏񂪊܂܂ꍇ܂B
	 * 
	 * @param props Properties IuWFNgBV Session IuWFNg쐬ꍇɂ̂ݎgp܂B<br>
	 * NCAǵAJavaMail dl̕t^ A ɋLڂvpeB ( mail.store.protocolA
	 * mail.transport.protocolAmail.hostAmail.userAy mail.from) ̒lȂ΂Ȃ܂B
	 * ́AftHgׂĂ̏ꍇɋ@\Ƃ͌ȂׂłB
	 * @return ftHg Session IuWFNg
	 * @since JavaMail 1.2
	 */
	public static Session getDefaultInstance(final Properties props) {
		return getDefaultInstance(props, null);
	}

	/**
	 *  Session ̃fobOݒs܂B
	 * <p>
	 * Session 쐬ɂfobOݒLɂłȂׁA
	 * Session RXgN^̃fobMOLɂɂ́A
	 * RXgN^ɓn Properties IuWFNg
	 *  <code>mail.debug</code> vpeB true ɐݒ肵܂B
	 * <code>mail.debug</code> vpeB̒ĺASession Pʂ̃fobMOtOׂɎgp܂B
	 * <code>setDebug</code> \bhɑ΂㑱̌ĂяóASession Pʂ̃fobMOtO𑀍삵A
	 * <code>mail.debug</code> vpeBɉe^܂B
	 * 
	 * @param debug fobOݒ
	 */
	public synchronized void setDebug(final boolean debug) {
		this.debug = debug;
		if (debug)
			pr("DEBUG: setDebug: JavaMail version " + version);
    }

	/**
	 *  Session ̃fobOݒ擾܂B
	 * 
	 * @return ݂̃fobOݒ
	 */
	public synchronized boolean getDebug() {
		return debug;
	}

	/**
	 * Set the stream to be used for debugging output for this session.
	 * If <code>out</code> is null, <code>System.out</code> will be used.
	 * Note that debugging output that occurs before any session is created,
	 * as a result of setting the <code>mail.debug</code> system property,
	 * will always be sent to <code>System.out</code>.
	 * 
	 * @param out the PrintStream to use for debugging output
	 * @since JavaMail 1.3
	 */
	public synchronized void setDebugOut(PrintStream out) {
		this.out = out;
	}

	/**
	 * Returns the stream to be used for debugging output.  If no stream
	 * has been set, <code>System.out</code> is returned.
	 * 
	 * @return the PrintStream to use for debugging output
	 * @since JavaMail 1.3
	 */
	public synchronized PrintStream getDebugOut() {
		if (out == null)
			return System.out;
		return out;
	}

	/**
	 * ̃\bh́ÃAvP[VŎgp\ ClassLoader ɂ胍[h
	 * ł javamail.[default.]providers t@CoRŃCXg[ꂽSĂ̎̔zԂ܂B
	 * 
	 * @return ݒ肳ꂽvoC_̔z
	 */
	public synchronized Provider[] getProviders() {
		Provider[] _providers = new Provider[providers.size()];
		providers.copyInto(_providers);
		return _providers;
	}

	/**
	 * w肳ꂽvgR̃ftHg Provider Ԃ܂B
	 * ܂ mail.&lt;protocol&gt;.class vpeB`FbNA݂ꍇ́A
	 * ̎Ɋ֘Atꂽ Provider Ԃ܂B
	 * ݂Ȃꍇ́Aݒt@CɍŏɌ Provider Ԃ܂B
	 * vgR̎Ȃꍇ́ANoSuchProviderException X[܂B
	 * 
	 * @param protocol ݒ肳ꂽvgR (smtpAimap )
	 * @return w肳ꂽvgRɂČݐݒ肳Ă Provider
	 * @throws NoSuchProviderException w肳ꂽvgR̃voC_Ȃꍇ
	 */
	public synchronized Provider getProvider(final String protocol) throws NoSuchProviderException {
		if (protocol == null || protocol.length() <= 0)
			throw new NoSuchProviderException("Invalid protocol: null");

		Provider _provider = null;

		// check if the mail.<protocol>.class property exists
		String _className = props.getProperty("mail." + protocol + ".class");
		if (_className != null) {
			if (debug)
				pr("DEBUG: mail." + protocol + ".class property exists and points to " + _className);
			_provider = (Provider) providersByClassName.get(_className);
		}

		if (_provider != null)
			return _provider;

		// returning currently default protocol in providersByProtocol
		_provider = (Provider) providersByProtocol.get(protocol);

		if (_provider == null)
			throw new NoSuchProviderException("No provider for " + protocol);
		if (debug)
			pr("DEBUG: getProvider() returning " + _provider.toString());

		return _provider;
	}

	/**
	 * nꂽ Provider AȑȎSĂ̒lI[o[Ch邱ƂɂA
	 * Provider.protocol ̃vgRɑ΂ftHg̎ƂĐݒ肵܂B
	 * 
	 * @param provider vgR̃ftHgƂĐݒ肳錻ݐݒ肳Ă Provider 
	 * @throws NoSuchProviderException nꂽvoC_̏ꍇ
	 */
	public synchronized void setProvider(final Provider provider) throws NoSuchProviderException {
		if (provider == null)
			throw new NoSuchProviderException("Can't set null provider");
		providersByProtocol.put(provider.getProtocol(), provider);
		props.put("mail." + provider.getProtocol() + ".class", provider.getClassName());
	}

	/**
	 * ̃[U] Store vgR Store IuWFNg擾܂B
	 * <code>mail.store.protocol</code> vpeB͊]vgRw肵܂B
	 * K؂ Store IuWFNg擾ȂꍇANoSuchProviderException X[܂B
	 * 
	 * @return Store IuWFNg 
	 * @throws NoSuchProviderException w肳ꂽvgR̃voC_Ȃꍇ
	 */
	public Store getStore() throws NoSuchProviderException {
		return getStore(getProperty("mail.store.protocol"));
	}

	/**
	 * w肳ꂽvgR Store IuWFNg擾܂B
	 * K؂ Store IuWFNg擾łȂꍇANoSuchProviderException X[܂B
	 * 
	 * @param protocol
	 * @return Store IuWFNg 
	 * @throws NoSuchProviderException w肳ꂽvgR̃voC_Ȃꍇ
	 */
	public Store getStore(final String protocol) throws NoSuchProviderException {
		return getStore(new URLName(protocol, null, -1, null, null, null));
	}

	/**
	 * w肳ꂽ URLName  Store IuWFNg擾܂B
	 * vꂽ Store IuWFNg擾łȂꍇANoSuchProviderException X[܂B
	 * 
	 * Store vgR̓ɂ́AURL  "scheme"  (RFC 1738 Q) gp܂B<p>
	 * 
	 * @param url ] Store \ URLName
	 * @return  Store IuWFNg
	 * @throws NoSuchProviderException w肳ꂽ URLName ̃voC_Ȃꍇ
	 * @see #getFolder(URLName)
	 * @see javax.mail.URLName
	 */
	public Store getStore(final URLName url) throws NoSuchProviderException {
		String protocol = url.getProtocol();
		Provider p = getProvider(protocol);
		return getStore(p, url);
	}

	/**
	 * Provider ɂw肳ꂽXgÃCX^X擾܂B
	 * XgACX^XĕԂ܂B
	 * 
	 * @param provider CX^XXgA Provider
	 * @return CX^Xꂽ Store
	 * @throws NoSuchProviderException w肳ꂽ Provider ̃voC_Ȃꍇ
	 */
	public Store getStore(final Provider provider) throws NoSuchProviderException {
		return getStore(provider, null);
	}

	/**
	 * Get an instance of the store specified by Provider. If the URLName
	 * is not null, uses it, otherwise creates a new one. Instantiates
	 * the store and returns it. This is a private method used by
	 * getStore(Provider) and getStore(URLName)
	 * 
	 * @param provider Store Provider that will be instantiated
	 * @param url URLName used to instantiate the Store
	 * @return Instantiated Store
	 * @throws NoSuchProviderException If a provider for the given Provider/URLName is not found.
	 */
	private Store getStore(final Provider provider, final URLName url) throws NoSuchProviderException {
		// make sure we have the correct type of provider
		if (provider == null || provider.getType() != Provider.Type.STORE )
		    throw new NoSuchProviderException("invalid provider");

		try {
		    return (Store) getService(provider, url);
		} catch (ClassCastException cce) {
		    throw new NoSuchProviderException("incorrect class");
		}
    }

	/**
	 * w肳ꂽ URLName ̕ Folder IuWFNg擾܂B
	 * v Folder IuWFNg擾łȂꍇAnull Ԃ܂B<p>
	 * 
	 * Store vgR̓ɂ́AURL  "scheme"  (RFC 1738 Q) gp܂B
	 * URL ̎c (܂ARFC 1738 ŋK肳 "schemepart")  Store ɂA
	 * vgRɈˑ@ŁAK؂ Folder IuWFNg̓ƃCX^XɎgp܂B<p>
	 * 
	 * RFC 1738 ́AIP x[X̃vgR (IMAP4APOP3 )  "schemepart" Ɋւ\w肵Ă܂B
	 * IP x[X̃[ Store ̃voC_́AFolder QƂ邽߂̍\Ȃ΂Ȃ܂B<p>
	 * 
	 * @param url ] Folder \ URLName
	 * @return Folder
	 * @throws NoSuchProviderException w肳ꂽ URLName ̃voC_Ȃꍇ
	 * @throws MessagingException Folder 薔͍쐬łȂꍇ 
	 * @see #getStore(URLName)
	 * @see javax.mail.URLName
	 */
	public Folder getFolder(final URLName url) throws MessagingException {
		// First get the Store
		Store store = getStore(url);
		store.connect();
		return store.getFolder(url);
	}

	/**
	 * ̃[U] Transport vgR Transport IuWFNg擾܂B
	 * <code>mail.transport.protocol</code> vpeB͊]vgRw肵܂B
	 * K؂ Transport IuWFNg擾łȂꍇAMessagingException X[܂B
	 * 
	 * @return Transport IuWFNg 
	 * @throws NoSuchProviderException voC_Ȃꍇ
	 */
	public Transport getTransport() throws NoSuchProviderException {
		return getTransport(getProperty("mail.transport.protocol"));
	}

	/**
	 * w肳ꂽvgR Transport IuWFNg擾܂B
	 * K؂ Transport IuWFNg擾łȂꍇAnull Ԃ܂B
	 * 
	 * @return Transport IuWFNg 
	 * @throws NoSuchProviderException w肳ꂽvgR̃voC_Ȃꍇ
	 */
	public Transport getTransport(final String protocol) throws NoSuchProviderException {
		return getTransport(new URLName(protocol, null, -1, null, null, null));
	}

	/**
	 * w肳ꂽ URLName  Transport IuWFNg擾܂B
	 * vꂽ Transport IuWFNg擾łȂꍇANoSuchProviderException X[܂B
	 * 
	 * Transport vgR̓ɂ́AURL  "scheme"  (RFC 1738 Q) gp܂B<p>
	 * 
	 * @param url ] Transport \ URLName
	 * @return  Transport IuWFNg
	 * @throws NoSuchProviderException w肳ꂽ URLName ̃voC_Ȃꍇ
	 * @see javax.mail.URLName
	 */
	public Transport getTransport(final URLName url) throws NoSuchProviderException {
		String protocol = url.getProtocol();
		Provider p = getProvider(protocol);
		return getTransport(p, url);
	}

	/**
	 * Provider Ɏw肳ꂽgX|[g̃CX^X擾܂B
	 * gX|[gCX^XĕԂ܂B
	 * 
	 * @param provider CX^XgX|[g Provider
	 * @return CX^Xꂽ Transport
	 * @throws NoSuchProviderException w肳ꂽvoC_̃voC_Ȃꍇ
	 */
	public Transport getTransport(final Provider provider) throws NoSuchProviderException {
		return getTransport(provider, null);
	}

	/**
	 * Message w肳ꂽAhX^ɃgX|[g\ Transport IuWFNg擾܂B
	 * 
	 * @param address
	 * @return Transport IuWFNg
	 * @throws NoSuchProviderException Address ^̃voC_Ȃꍇ 
	 * @see javax.mail.Address
	 */
	public Transport getTransport(final Address address) throws NoSuchProviderException {
		String transportProtocol = (String) addressMap.get(address.getType());
		if (transportProtocol == null)
			throw new NoSuchProviderException("No provider for Address type: "+ address.getType());
		return getTransport(transportProtocol);
	}

	/**
	 * w肳ꂽvoC_ URLName gp Transport IuWFNg擾܂B
	 * 
	 * @param provider gpvoC_
	 * @param url gp URL  (null ̏ꍇ܂)
	 * @return Transport IuWFNg
	 * @throws NoSuchProviderException w肳ꂽvoC_̃voC_Ȃꍇ
	 */
	private Transport getTransport(
		final Provider provider,
		final URLName url)
		throws NoSuchProviderException {

		// make sure we have the correct type of provider
		if (provider == null || provider.getType() != Provider.Type.TRANSPORT)
			throw new NoSuchProviderException("invalid provider");

		try {
			return (Transport) getService(provider, url);
		} catch (ClassCastException cce) {
			throw new NoSuchProviderException("incorrect class");
		}
	}

	/**
	 * Get a Service object.  Needs a provider object, but will
	 * create a URLName if needed.  It attempts to instantiate
	 * the correct class.
	 * 
	 * @param provider which provider to use
	 * @param url which URLName to use (can be null)
	 * @throws NoSuchProviderException thrown when the class cannot be
	 *			found or when it does not have the correct constructor
	 *			(Session, URLName), or if it is not derived from
	 *			Service.
	 */
	private Object getService(
		final Provider provider, URLName url)
		throws NoSuchProviderException {

		// need a provider and url
		if (provider == null)
			throw new NoSuchProviderException("null");

		// create a url if needed
		if (url == null)
			url = new URLName(provider.getProtocol(), null, -1, null, null, null);

		Object service = null;

		// get the ClassLoader associated with the Authenticator
		ClassLoader cl;
		if (authenticator != null)
			cl = authenticator.getClass().getClassLoader();
		else
			cl = this.getClass().getClassLoader();

		// now load the class
		Class serviceClass = null;
		try {
			// First try the "application's" class loader.
			ClassLoader ccl = getContextClassLoader();
			if (ccl != null)
				try {
					serviceClass = ccl.loadClass(provider.getClassName());
				} catch (ClassNotFoundException ex) {}
			if (serviceClass == null)
				serviceClass = cl.loadClass(provider.getClassName());
		} catch (Exception ex1) {
			// That didn't work, now try the "system" class loader.
			// (Need both of these because JDK 1.1 class loaders
			// may not delegate to their parent class loader.)
			try {
				serviceClass = Class.forName(provider.getClassName());
			} catch (Exception ex) {
				// Nothing worked, give up.
				if (debug)
					ex.printStackTrace(getDebugOut());
				throw new NoSuchProviderException(provider.getProtocol());
			}
		}

		// construct an instance of the class
		try {
			Class[] c = {javax.mail.Session.class, javax.mail.URLName.class};
			Constructor cons = serviceClass.getConstructor(c);

			Object[] o = { this, url };
			service = cons.newInstance(o);
		} catch (Exception ex) {
			if (debug)
				ex.printStackTrace(getDebugOut());
			throw new NoSuchProviderException(provider.getProtocol());
		}

		return service;
    }

	/**
	 *  (XgA̓gX|[g) URLName  PasswordAuthentication ۑ܂B
	 * pw  null ̏ꍇAURLName ɑΉGg͍폜܂B
	 * <p>
	 * ͒ʏAXgA̓gX|[g̎ɂA
	 * F؏ 1 ̃ZbVŕgpł悤ɂׂɎgp܂B
	 */
	public void setPasswordAuthentication(final URLName url, final PasswordAuthentication pw) {
		if (pw == null)
			authTable.remove(url);
		else
			authTable.put(url, pw);
	}

	/**
	 *  (XgA̓gX|[g) URLName ̔Cӂ̕ۑꂽ PasswordAuthentication Ԃ܂B
	 * ʏAXgA̓gX|[g̎ɂĂ̂ݎgp܂B
	 * 
	 * @return URLName ɑΉ PasswordAuthentication
	 */
	public PasswordAuthentication getPasswordAuthentication(final URLName url) {
		return (PasswordAuthentication) authTable.get(url);
	}

	/**
	 * Kvȃ[UƃpX[h擾ׂɁAAvP[VɃR[obN܂B
	 * AvP[V́Aȉ̗lȃ_CAO\Ȃ΂Ȃ܂B
	 * <p> <pre>
	 * Connecting to &lt;protocol&gt; mail service on host &lt;addr&gt;, port &lt;port&gt;.
	 * &lt;prompt&gt;
	 * 
	 * [U: &lt;defaultUserName&gt;
	 * pX[h:
	 * </pre>
	 * 
	 * @param addr zXg InetAddressBnull ̏ꍇ܂B
	 * @param protocol vgŘn (imapApop3 )
	 * @param prompt vvg̈ꕔƂĕ\ׂ̔Cӂ̒ǉ StringBnull ̏ꍇ܂B
	 * @param defaultUserName ftHg̃[UBnull ̏ꍇ܂B
	 * @return F؎҂ɂWꂽF؁Bnull ̏ꍇ܂B 
	 */
	public PasswordAuthentication requestPasswordAuthentication(
		final InetAddress addr, final int port,
		final String protocol, final String prompt, final String defaultUserName) {

		if (authenticator != null)
			return authenticator.requestPasswordAuthentication(addr, port, protocol, prompt, defaultUserName);
		return null;
	}

	/**
	 *  Session Ɋ֘Atꂽ Properties IuWFNgԂ܂B
	 * 
	 * @return Properties IuWFNg
	 */
	public Properties getProperties() { 
		return props; 
	}

	/**
	 * w肳ꂽvpeB̒lԂ܂B
	 * ̃vpeB݂Ȃꍇ null Ԃ܂B
	 * 
	 * @return vpeBlƂȂ镶
	 */
	public String getProperty(final String name) { 
		return props.getProperty(name); 
	}

	/**
	 * vgRvoC_ݒt@Cǂݍ݂܂B
	 */
	private void loadProviders(final Class cl) {
		StreamLoader loader = new StreamLoader() {
			public void load(final InputStream is) throws IOException {
				loadProvidersFromStream(is);
			}
		};

		// load system-wide javamail.providers from the <java.home>/lib dir
		try {
			String res = System.getProperty("java.home") + File.separator + "lib" + File.separator + "javamail.providers";
			loadFile(res, loader);
		} catch (SecurityException sex) {
			if (debug)
				pr("DEBUG: can't get java.home: " + sex);
		}

		// load the META-INF/javamail.providers file supplied by an application
		loadAllResources("META-INF/javamail.providers", cl, loader);

		// load default META-INF/javamail.default.providers from mail.jar file
		loadResource("/META-INF/javamail.default.providers", cl, loader);

		if (providers.size() == 0) {
			if (debug)
				pr("DEBUG: failed to load any providers, using defaults");
			// failed to load any providers, initialize with our defaults
			addProvider(new Provider(Provider.Type.STORE, "imap", "com.sun.mail.imap.IMAPStore", "Sun Microsystems, Inc.", version));
			addProvider(new Provider(Provider.Type.STORE, "imaps", "com.sun.mail.imap.IMAPSSLStore", "Sun Microsystems, Inc.", version));
			addProvider(new Provider(Provider.Type.STORE, "pop3", "com.sun.mail.pop3.POP3Store", "Sun Microsystems, Inc.", version));
			addProvider(new Provider(Provider.Type.STORE, "pop3s", "com.sun.mail.pop3.POP3SSLStore", "Sun Microsystems, Inc.", version));
			addProvider(new Provider(Provider.Type.TRANSPORT, "smtp", "com.sun.mail.smtp.SMTPTransport", "Sun Microsystems, Inc.", version));
			addProvider(new Provider(Provider.Type.TRANSPORT, "smtps", "com.sun.mail.smtp.SMTPSSLTransport", "Sun Microsystems, Inc.", version));
		}

		if (debug) {
			// dump the output of the tables for debugging
			pr("DEBUG: Tables of loaded providers");
			pr("DEBUG: Providers Listed By Class Name: " + providersByClassName.toString());
			pr("DEBUG: Providers Listed By Protocol: " + providersByProtocol.toString());
		}
	}

    private void loadProvidersFromStream(final InputStream is) throws IOException {
		if (is != null) {
			LineInputStream lis = new LineInputStream(is);
			String currLine;

			// load and process one line at a time using LineInputStream
			while ((currLine = lis.readLine()) != null) {
				if (currLine.startsWith("#"))
					continue;
				Provider.Type type = null;
				String protocol = null, className = null;
				String vendor = null, version = null;

				// separate line into key-value tuples
				StringTokenizer tuples = new StringTokenizer(currLine,";");
				while (tuples.hasMoreTokens()) {
					String currTuple = tuples.nextToken().trim();

					// set the value of each attribute based on its key
					int sep = currTuple.indexOf("=");
					if (currTuple.startsWith("protocol="))
						protocol = currTuple.substring(sep+1);
					else if (currTuple.startsWith("type=")) {
						String strType = currTuple.substring(sep+1);
						if (strType.equalsIgnoreCase("store"))
							type = Provider.Type.STORE;
						else if (strType.equalsIgnoreCase("transport"))
							type = Provider.Type.TRANSPORT;
					} else if (currTuple.startsWith("class="))
						className = currTuple.substring(sep+1);
					else if (currTuple.startsWith("vendor="))
						vendor = currTuple.substring(sep+1);
					else if (currTuple.startsWith("version="))
						version = currTuple.substring(sep+1);
				}

				// check if a valid Provider; else, continue
				if (type == null || protocol == null || className == null || protocol.length() <= 0 || className.length() <= 0) {
					if (debug)
						pr("DEBUG: Bad provider entry: " + currLine);
				    continue;
				}
				Provider provider = new Provider(type, protocol, className, vendor, version);

				// add the newly-created Provider to the lookup tables
				addProvider(provider);
		    }
		}
	}

	/**
	 * Add a provider to the session.
	 * 
	 * @param provider the provider to add
	 * @since JavaMail 1.4
	 */
	public synchronized void addProvider(final Provider provider) {
		providers.addElement(provider);
		providersByClassName.put(provider.getClassName(), provider);
		if (!providersByProtocol.containsKey(provider.getProtocol()))
			providersByProtocol.put(provider.getProtocol(), provider);
	}

	// load maps in reverse order of preference so that the preferred
	// map is loaded last since its entries will override the previous ones
	private void loadAddressMap(final Class cl) {
		StreamLoader loader = new StreamLoader() {
			public void load(final InputStream is) throws IOException {
				addressMap.load(is);
		    }
		};

		// load default META-INF/javamail.default.address.map from mail.jar
		loadResource("/META-INF/javamail.default.address.map", cl, loader);

		// load the META-INF/javamail.address.map file supplied by an app
		loadAllResources("META-INF/javamail.address.map", cl, loader);

		// load system-wide javamail.address.map from the <java.home>/lib dir
		try {
			String res = System.getProperty("java.home") + File.separator + "lib" + File.separator + "javamail.address.map";
			loadFile(res, loader);
		} catch (SecurityException sex) {
			if (debug)
				pr("DEBUG: can't get java.home: " + sex);
		}

		if (addressMap.isEmpty()) {
			if (debug)
				pr("DEBUG: failed to load address map, using defaults");
			addressMap.put("rfc822", "smtp");
		}
	}

	/**
	 * Set the default transport protocol to use for addresses of the specified type.
	 * Normally the default is set by the javamail.default.address.map or javamail.address.map files or resources.
	 * 
	 * @param addresstype type of address
	 * @param protocol name of protocol
	 * @since JavaMail 1.4
	 * @see #getTransport(Address)
	 */
	public synchronized void setProtocolForAddress(final String addresstype, final String protocol) {
		if (protocol == null)
			addressMap.remove(addresstype);
		else
			addressMap.put(addresstype, protocol);
	}

	/**
	 * w肳ꂽt@Cǂݍ݂܂B
	 */
	private void loadFile(final String name, final StreamLoader loader) {
		BufferedInputStream clis = null;
		try {
			clis = new BufferedInputStream(new FileInputStream(name));
			if (clis != null) {
				loader.load(clis);
				if (debug)
					pr("DEBUG: successfully loaded file: " + name);
			} else
				if (debug)
					pr("DEBUG: not loading file: " + name);
		} catch (IOException e) {
			if (debug)
				pr("DEBUG: " + e);
		} catch (SecurityException sex) {
			if (debug)
				pr("DEBUG: " + sex);
		} finally {
			try {
				if (clis != null)
					clis.close();
			} catch (IOException ex) {}	// 
		}
    }

	/**
	 * w肳ꂽ\[Xǂݍ݂܂B
	 */
	private void loadResource(final String name, final Class cl, final StreamLoader loader) {
		InputStream clis = null;
		try {
			clis = getResourceAsStream(cl, name);
			if (clis != null) {
				loader.load(clis);
				if (debug)
					pr("DEBUG: successfully loaded resource: " + name);
			} else
				if (debug)
					pr("DEBUG: not loading resource: " + name);
		} catch (IOException e) {
			if (debug)
				pr("DEBUG: " + e);
		} catch (SecurityException sex) {
			if (debug)
				pr("DEBUG: " + sex);
		} finally {
			try {
				if (clis != null)
					clis.close();
			} catch (IOException ex) {}	// 
		}
    }

	/**
	 * w肳ꂽ\[XSēǂݍ݂܂B
	 */
	private void loadAllResources(final String name, final Class cl, final StreamLoader loader) {
		boolean anyLoaded = false;
		try {
			ClassLoader cld = null;
			// First try the "application's" class loader.
			cld = getContextClassLoader();
			if (cld == null)
				cld = cl.getClassLoader();
			URL[] urls;
			if (cld != null)
				urls = getResources(cld, name);
			else
				urls = getSystemResources(name);
			if (urls != null) {
				for (int i = 0; i < urls.length; i++) {
					URL url = urls[i];
					InputStream clis = null;
					if (debug)
						pr("DEBUG: URL " + url);
					try {
						clis = openStream(url);
						if (clis != null) {
							loader.load(clis);
							anyLoaded = true;
							if (debug)
								pr("DEBUG: successfully loaded resource: " + url);
						} else
							if (debug)
								pr("DEBUG: not loading resource: " + url);
					} catch (IOException ioex) {
						if (debug)
							pr("DEBUG: " + ioex);
					} catch (SecurityException sex) {
						if (debug)
							pr("DEBUG: " + sex);
					} finally {
						try {
							if (clis != null)
								clis.close();
						} catch (IOException cex) {}
					}
				}
			}
		} catch (Exception ex) {
			if (debug)
				pr("DEBUG: " + ex);
		}

		// if failed to load anything, fall back to old technique, just in case
		if (!anyLoaded) {
			if (debug)
				pr("DEBUG: !anyLoaded");
			loadResource('/' + name, cl, loader);
		}
	}

	private void pr(final String str) {
		getDebugOut().println(str);
	}

	private static ClassLoader getContextClassLoader() {
		return (ClassLoader) AccessController.doPrivileged(new PrivilegedAction() {
			public Object run() {
				ClassLoader cl = null;
				try {
					cl = Thread.currentThread().getContextClassLoader();
				} catch (SecurityException ex) {}
				return cl;
			}
		});
	}

	private static InputStream getResourceAsStream(final Class c, final String name) throws IOException {
		try {
			return (InputStream) AccessController.doPrivileged(new PrivilegedExceptionAction() {
				public Object run() throws IOException {
					return c.getResourceAsStream(name);
				}
			});
		} catch (PrivilegedActionException e) {
			throw (IOException)e.getException();
		}
	}

	private static URL[] getResources(final ClassLoader cl, final String name) {
		return (URL[]) AccessController.doPrivileged(new PrivilegedAction() {
			public Object run() {
				URL[] ret = null;
				try {
					Vector v = new Vector();
					Enumeration e = cl.getResources(name);
					while (e != null && e.hasMoreElements()) {
						URL url = (URL) e.nextElement();
						if (url != null)
							v.addElement(url);
					}
					if (v.size() > 0) {
						ret = new URL[v.size()];
						v.copyInto(ret);
					}
				} catch (IOException ioex) {
				} catch (SecurityException ex) {}
				return ret;
			}
		});
	}

	private static URL[] getSystemResources(final String name) {
		return (URL[]) AccessController.doPrivileged(new PrivilegedAction() {
			public Object run() {
				URL[] ret = null;
				try {
					Vector v = new Vector();
					Enumeration e = ClassLoader.getSystemResources(name);
					while (e != null && e.hasMoreElements()) {
						URL url = (URL) e.nextElement();
						if (url != null)
							v.addElement(url);
					}
					if (v.size() > 0) {
						ret = new URL[v.size()];
						v.copyInto(ret);
					}
				} catch (IOException ioex) {
				} catch (SecurityException ex) {}
				return ret;
			}
		});
	}

	private static InputStream openStream(final URL url) throws IOException {
		try {
			return (InputStream) AccessController.doPrivileged(new PrivilegedExceptionAction() {
				public Object run() throws IOException {
					return url.openStream();
				}
			});
		} catch (PrivilegedActionException e) {
			throw (IOException)e.getException();
		}
	}

}

/**
 * Support interface to generalize
 * code that loads resources from stream.
 */
interface StreamLoader {

	public void load(InputStream is) throws IOException;

}
