/*
 * Copyright 1997-2003 Sun Microsystems, Inc. All Rights Reserved.
 */

package javax.mail.internet;

import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.mail.Address;
import javax.mail.Session;

/**
 * ̃NX <a href="http://www.ietf.org/rfc/rfc822.txt">RFC822</a> AhXf܂B
 * Typical address syntax is of the form "user@host.domain" or
 * "Personal Name <user@host.domain>".
 */
public class InternetAddress extends Address implements Cloneable {

	private static final long serialVersionUID = -7507595530758302903L;

	protected String address;	// dq[AhX

	/**
	 * lłB
	 */
	protected String personal;

	/**
	 * l RFC 2047 GR[ho[WłB<p>
	 * 
	 * ̃tB[h <code>personal</code> tB[h݂͌ɒǐՂ܂B
	 * ]āATuNX̃tB[h 1 𒼐ڐݒ肷ꍇA
	 * K؂ȍČvZsȂlɑ <code>null</code> ɐݒ肵Ȃ΂Ȃ܂B
	 */
	protected String encodedPersonal;

	/**
	 * ftHgRXgN^łB
	 */
	public InternetAddress() {}

	/**
	 * RXgN^łB<p>
	 * 
	 * w肳ꂽ<code>\͂</code>AInternetAddress \z܂B
	 * 
	 * @param address RFC822 `̃AhX
	 * @throws AddressException \͂Ɏsꍇ
	 */
	public InternetAddress(final String address) throws AddressException {
		// use our address parsing utility routine to parse the string
		InternetAddress a[] = parse(address, true);
		// if we got back anything other than a single address, it's an error
		if (a.length != 1)
			throw new AddressException("Illegal address", address);

		/*
		 * Now copy the contents of the single address we parsed
		 * into the current object, which will be returned from the
		 * constructor.
		 * XXX - this sure is a round-about way of getting this done.
		 */
		this.address = a[0].address;
		this.personal = a[0].personal;
		this.encodedPersonal = a[0].encodedPersonal;
	}

	/**
	 * Parse the given string and create an InternetAddress.
	 * If <code>strict</code> is false, the detailed syntax of the
	 * address isn't checked.
	 * 
	 * @param address RFC822 `̃AhX
	 * @param strict enforce RFC822 syntax
	 * @throws AddressException \͂Ɏsꍇ
	 * @since JavaMail 1.3
	 */
	public InternetAddress(final String address, final boolean strict) throws AddressException {
		this(address);
		if (strict)
			checkAddress(this.address, true, true);
	}

	/**
	 * AhXƌlw肳ꂽ InternetAddress \z܂B
	 * ̃AhX͍\IɗL RFC822 AhXƌȂ܂B
	 * 
	 * @param address RFC822 `̃AhX
	 * @param personal l
	 */
	public InternetAddress(final String address, final String personal) throws UnsupportedEncodingException {
		this(address, personal, null);
	}

	/**
	 * AhXƌlw肳ꂽ InternetAddress \z܂B
	 * ̃AhX͍\IɗL RFC822 AhXƌȂ܂B
	 * 
	 * @param address RFC822 `̃AhX
	 * @param personal l
	 * @param charset	MIME Zbg̖O
	 */
	public InternetAddress(final String address, final String personal, final String charset) throws UnsupportedEncodingException {
		this.address = address;
		setPersonal(personal, charset);
	}

	/**
	 *  InternetAddress IuWFNg̃Rs[Ԃ܂B
	 * 
	 * @since JavaMail 1.2
	 */
	public final Object clone() {
		InternetAddress a = null;
		try {
			a = (InternetAddress) super.clone();
		} catch (CloneNotSupportedException e) {} // Won't happen
		return a;
	}

	/**
	 * ̃AhX̌^Ԃ܂BInternetAddress ̌^ "rfc822" łB
	 */
	public final String getType() {
		return "rfc822";
	}

	/**
	 * dq[AhXݒ肵܂B
	 * 
	 * @param address dq[AhX
	 */
	public final void setAddress(final String address) {
		this.address = address;
	}

	/**
	 * lݒ肵܂BO US-ASCII ȊO̕܂܂ꍇA
	 * O RFC 2047 ɏ]Aw肳ꂽZbggpăGR[h܂B
	 * O US-ASCII ܂܂ꍇAGR[fBO͍sꂸAO͂̂܂܎gp܂B<p>
	 * 
	 * @param name l
	 * @param charset RFC 2047 ɏ]ĖOGR[hׂɎgp MIME Zbg 
	 * @throws UnsupportedEncodingException Zbg̃GR[fBOsꍇ
	 * @see #setPersonal(String)
	 */
	public final void setPersonal(final String name, final String charset) throws UnsupportedEncodingException {
		personal = name;
		if (name != null)
			encodedPersonal = MimeUtility.encodeWord(name, charset, null);
		else
			encodedPersonal = null;
	}

	/**
	 * lݒ肵܂BO US-ASCII ȊO̕܂܂ꍇA
	 * O̓vbgtH[̃ftHgZbggpăGR[h܂B
	 * O US-ASCII ܂܂ꍇAGR[fBO͍sꂸAO͂̂܂܎gp܂B<p>
	 * 
	 * @param name l
	 * @throws UnsupportedEncodingException Zbg̃GR[fBOsꍇ
	 * @see #setPersonal(String name, String charset)
	 */
	public final void setPersonal(final String name) throws UnsupportedEncodingException {
		personal = name;
		if (name != null)
			encodedPersonal = MimeUtility.encodeWord(name);
		else
			encodedPersonal = null;
	}

	/**
	 * dq[AhX擾܂B
	 * 
	 * @return dq[AhX
	 */
	public final String getAddress() {
		return address;
	}

	/**
	 * l擾܂BO RFC 2047 ɏ]ăGR[fBOĂꍇA
	 * Unicode ɃfR[hyѕϊ܂BfR[fBO͕ϊsꍇA
	 * f[^̂܂ܕԂ܂B
	 * 
	 * @return l
	 */
	public final String getPersonal() {
		if (personal != null)
			return personal;

		if (encodedPersonal != null) {
			try {
				personal = MimeUtility.decodeText(encodedPersonal);
				return personal;
			} catch (Exception ex) {
				// 1. ParseException: either its an unencoded string or
				//	it can't be parsed
				// 2. UnsupportedEncodingException: can't decode it.
				return encodedPersonal;
			}
		}
		// No personal or encodedPersonal, return null
		return null;
	}

	/**
	 * ̃AhX RFC 822 / RFC 2047 GR[hAhXɕϊ܂B
	 * ϊ̕ɂ US-ASCII ܂܂ׁA[ňSɎgpł܂B
	 * 
	 * @return 炭GR[hꂽAhX
	 */
	public final String toString() {
		if (encodedPersonal == null && personal != null)
			try {
				encodedPersonal = MimeUtility.encodeWord(personal);
			} catch (UnsupportedEncodingException ex) {}

		if (encodedPersonal != null)
			return quotePhrase(encodedPersonal) + " <" + address + '>';
		else if (isGroup() || isSimple())
			return address;
		else
			return '<' + address + '>';
	}

	/**
	 * ` (RFC 822 \)  Unicode ̃AhXԂ܂B
	 * 
	 * @return Unicode AhX
	 * @since JavaMail 1.2
	 */  
	public final String toUnicodeString() {
		String p = getPersonal();
		if (p != null)
	    	return quotePhrase(p) + " <" + address + '>';
		else if (isGroup() || isSimple())
			return address;
		else
			return '<' + address + '>';
	}

	/*
	 * quotePhrase() quotes the words within a RFC822 phrase.
	 * 
	 * This is tricky, since a phrase is defined as 1 or more
	 * RFC822 words, separated by LWSP. Now, a word that contains
	 * LWSP is supposed to be quoted, and this is exactly what the 
	 * MimeUtility.quote() method does. However, when dealing with
	 * a phrase, any LWSP encountered can be construed to be the
	 * separator between words, and not part of the words themselves.
	 * To deal with this funkiness, we have the below variant of
	 * MimeUtility.quote(), which essentially ignores LWSP when
	 * deciding whether to quote a word.
	 *
	 * It aint pretty, but it gets the job done :)
	 */

	private static final String rfc822phrase = HeaderTokenizer.RFC822.replace(' ', '\0').replace('\t', '\0');

	private static String quotePhrase(final String phrase) {
		int len = phrase.length();
		boolean needQuoting = false;

		for (int i = 0; i < len; i++) {
			char c = phrase.charAt(i);
			if (c == '"' || c == '\\') { 
				// need to escape them and then quote the whole string
				StringBuffer sb = new StringBuffer(len + 3);
				sb.append('"');
				for (int j = 0; j < len; j++) {
					char cc = phrase.charAt(j);
					if (cc == '"' || cc == '\\')
						// Escape the character
						sb.append('\\');
					sb.append(cc);
				}
				sb.append('"');
				return sb.toString();
			}
			if ((c < 040 && c != '\r' && c != '\n' && c != '\t') || c >= 0177 || rfc822phrase.indexOf(c) >= 0)
				// These characters cause the string to be quoted
				needQuoting = true;
		}

		if (needQuoting) {
			StringBuffer sb = new StringBuffer(len + 2);
			sb.append('"').append(phrase).append('"');
			return sb.toString();
		}
		return phrase;
	}

	private static String unquote(String s) {
		if (s.startsWith("\"") && s.endsWith("\"")) {
			s = s.substring(1, s.length() - 1);
			// check for any escaped characters
			if (s.indexOf('\\') >= 0) {
				StringBuffer sb = new StringBuffer(s.length());	// approx
				for (int i = 0; i < s.length(); i++) {
					char c = s.charAt(i);
					if (c == '\\' && i < s.length() - 1)
						c = s.charAt(++i);
					sb.append(c);
				}
				s = sb.toString();
			}
		}
		return s;
	}

	/**
	 * ZqłB
	 */
	public final boolean equals(final Object a) {
		if (!(a instanceof InternetAddress))
			return false;

		String s = ((InternetAddress)a).getAddress();
		if (s == address)
			return true;

		return (address != null && address.equalsIgnoreCase(s));
	}

	/**
	 * AhX̃nbVR[hvZ܂B
	 */
	public final int hashCode() {
		if (address == null)
			return 0;
		return address.toLowerCase().hashCode();
	}

	/**
	 * w肳ꂽ InternetAddress IuWFNg̔zAhX̃R}؂V[PXɕϊ܂B
	 * ϊ̕ɂ US-ASCII ܂܂ׁA[ňSɎgpł܂B<p>
	 * 
	 * @param addresses InternetAddress IuWFNg̔z
	 * @return R}؂̃AhX
	 * @throws ClassCastException w肳ꂽz̉ꂩ̃AhXIuWFNg InternetAddress IuWFNgłȂꍇB
	 *  RuntimeException ł鎖ɒӂĉB 
	 */
	public static final String toString(final Address[] addresses) {
		return toString(addresses, 0);
	}

	/**
	 * w肳ꂽ InternetAddress IuWFNg̔zAhX̃R}؂V[PXɕϊ܂B
	 * ϊ̕ɂ US-ASCII ܂܂ׁA[ňSɎgpł܂B<p>
	 * 
	 * 'used' p[^́Aϊ̃AhXV[PX񂪑}
	 * tB[hɐL镶ʒu̐w肵܂B
	 * ͕ϊ̃AhXV[PX񒆂̉sʒuʂׂɎgp܂B
	 * 
	 * @param addresses InternetAddress IuWFNg̔z
	 * @param used AhX񂪑}tB[hŊɎgpĂ镶ʒu̐
	 * @return R}؂̃AhX
	 * @throws ClassCastException w肳ꂽz̉ꂩ̃AhXIuWFNg InternetAddress IuWFNgłȂꍇB
	 *  RuntimeException ł鎖ɒӂĉB 
	 */
	public static final String toString(final Address[] addresses, int used) {
		if (addresses == null || addresses.length == 0)
			return null;

		StringBuffer sb = new StringBuffer();

		for (int i = 0; i < addresses.length; i++) {
			if (i != 0) { // need to append comma
				sb.append(", ");
				used += 2;
		    }

			String s = addresses[i].toString();
			int len = lengthOfFirstSegment(s); // length till CRLF
			if (used + len > 76) { // overflows ...
				sb.append("\r\n\t"); // .. start new continuation line
				used = 8; // account for the starting <tab> char
			}
			sb.append(s);
			used = lengthOfLastSegment(s, used);
		}

		return sb.toString();
	}

	/* Return the length of the first segment within this string.
	 * If no segments exist, the length of the whole line is returned.
	 */
	private static int lengthOfFirstSegment(final String s) {
		int pos;
		if ((pos = s.indexOf("\r\n")) != -1)
			return pos;
		return s.length();
	}

	/*
	 * Return the length of the last segment within this string.
	 * If no segments exist, the length of the whole line plus
	 * <code>used</code> is returned.
	 */
	private static int lengthOfLastSegment(final String s, final int used) {
		int pos;
		if ((pos = s.lastIndexOf("\r\n")) != -1)
			return s.length() - pos - 2;
		return s.length() + used;
	}

	/**
	 * ݂̃[U\ InternetAddress IuWFNgԂ܂B
	 * dq[AhXŜ "mail.from" vpeBŎw肳ꍇ܂B
	 * ݒ肳ĂȂꍇ "mail.user" y "mail.host" vpeB܂B
	 * 炪ݒ肳ĂȂꍇA"user.name" vpeB <code>InetAddress.getLocalHost</code> \bh܂B
	 * ̏̃ANZXɃZLeBOꍇ͖܂B
	 * dq[AhXʂ鎖łȂꍇ null Ԃ܂B
	 * 
	 * @param session vpeB̌Ɏgp Session IuWFNg
	 * @return ݂̃[U̓dq[AhX
	 */
	public static final InternetAddress getLocalAddress(final Session session) {
		String user = null, host = null, address = null;
		try {
		    if (session == null) {
				user = System.getProperty("user.name");
				host = InetAddress.getLocalHost().getHostName();
		    } else {
				address = session.getProperty("mail.from");
				if (address == null) {
					user = session.getProperty("mail.user");
					if (user == null || user.length() == 0)
						user = session.getProperty("user.name");
					if (user == null || user.length() == 0)
						user = System.getProperty("user.name");
					host = session.getProperty("mail.host");
					if (host == null || host.length() == 0) {
						InetAddress me = InetAddress.getLocalHost();
						if (me != null)
							host = me.getHostName();
				    }
				}
			}

			if (address == null && user != null && user.length() != 0 && host != null && host.length() != 0)
				address = user + '@' + host;

			if (address != null)
				return new InternetAddress(address);
		} catch (SecurityException sex) {		// 
		} catch (AddressException ex) {			// 
		} catch (UnknownHostException ex) {}	// 
		return null;
	}

	/**
	 * w肳ꂽR}؂̃AhXV[PX InternetAddress IuWFNgɍ\͂܂B
	 * AhX RFC822 \ɏȂ΂Ȃ܂B
	 * 
	 * @param addresslist R}؂̃AhX
	 * @return InternetAddress IuWFNg̔z
	 * @throws AddressException \͂Ɏsꍇ
	 */
	public static final InternetAddress[] parse(final String addresslist) throws AddressException {
		return parse(addresslist, true);
	}

	/**
	 * w肳ꂽAhX̃V[PX InternetAddress IuWFNgɍ\͂܂B
	 * <code>strict</code>  false ̏ꍇAXy[Xŋ؂ꂽPȓdq[AhXł܂B
	 * <code>strict</code>  true ̏ꍇARFC822 \K̑ (SĂł͂Ȃ) Kp܂B
	 * ɁA<code>strict</code>  true łĂA
	 * PȖOō\ꂽAhX ("@domain" Ȃ) ͋܂B
	 *  "s" AhX͎ۂ̃bZ[Wł͒܂B<p>
	 * 
	 * iłȂ\͈͂ʂɎ蓮œ͂[AhX̃Xg\͂鎞Ɏgp܂B
	 * iȍ\͈͂ʂɃ[bZ[W̃AhXwb_\͂鎞Ɏgp܂B
	 * 
	 * @param addresslist R}؂̃AhX
	 * @param strict RFC822 \Kp邩ǂ
	 * @return InternetAddress IuWFNg̔z
	 * @throws AddressException \͂Ɏsꍇ
	 */
	public static final InternetAddress[] parse(final String addresslist, final boolean strict) throws AddressException {
		return parse(addresslist, strict, false);
	}

	/**
	 * Parse the given sequence of addresses into InternetAddress
	 * objects.  If <code>strict</code> is false, the full syntax rules for
	 * individual addresses are not enforced.  If <code>strict</code> is
	 * true, many (but not all) of the RFC822 syntax rules are enforced. <p>
	 * 
	 * To better support the range of "invalid" addresses seen in real
	 * messages, this method enforces fewer syntax rules than the
	 * <code>parse</code> method when the strict flag is false
	 * and enforces more rules when the strict flag is true.  If the
	 * strict flag is false and the parse is successful in separating out an
	 * email address or addresses, the syntax of the addresses themselves
	 * is not checked.
	 * 
	 * @param addresslist R}؂̃AhX
	 * @param strict RFC822 \Kp邩ǂ
	 * @return InternetAddress IuWFNg̔z
	 * @throws AddressException \͂Ɏsꍇ
	 * @since JavaMail 1.3
	 */
	public static final InternetAddress[] parseHeader(final String addresslist, final boolean strict) throws AddressException {
		return parse(addresslist, strict, true);
	}

	/*
	 * RFC822 Address parser.
	 * 
	 * XXX - This is complex enough that it ought to be a real parser,
	 *       not this ad-hoc mess, and because of that, this is not perfect.
	 * 
	 * XXX - Deal with encoded Headers too.
	 */
	private static InternetAddress[] parse(
		final String s,
		final boolean strict,
		final boolean parseHdr)
		throws AddressException {

		int start, end, index, nesting;
		int start_personal = -1, end_personal = -1;
		int length = s.length();
		boolean in_group = false;		// we're processing a group term
		boolean route_addr = false;	// address came from route-addr term
		boolean rfc822 = false;		// looks like an RFC822 address
		Vector v = new Vector();

		for (start = end = -1, index = 0; index < length; index++) {
			char c = s.charAt(index);

			switch (c) {
				case '(': // We are parsing a Comment. Ignore everything inside.
					// XXX - comment fields should be parsed as whitespace,
					//	 more than one allowed per address
					rfc822 = true;
					if (start >= 0 && end == -1)
						end = index;
					if (start_personal == -1)
						start_personal = index + 1;
					for (index++, nesting = 1; index < length && nesting > 0; index++) {
						c = s.charAt(index);
						switch (c) {
							case '\\':
								index++; // skip both '\' and the escaped char
								break;
							case '(':
								nesting++;
								break;
							case ')':
								nesting--;
								break;
							default:
								break;
						}
					}
					if (nesting > 0)
						throw new AddressException("Missing ')'", s, index);
					index--;	// point to closing paren
					if (end_personal == -1)
						end_personal = index;
					break;

				case ')':
					throw new AddressException("Missing '('", s, index);

				case '<':
					rfc822 = true;
					if (route_addr)
						throw new AddressException("Extra route-addr", s, index);
					if (!in_group) {
						start_personal = start;
						if (start_personal >= 0)
							end_personal = index;
						start = index + 1;
					}

					boolean inquote = false;
					outf:
					for (index++; index < length; index++) {
						c = s.charAt(index);
						switch (c) {
							case '\\':	// XXX - is this needed?
								index++; // skip both '\' and the escaped char
								break;
							case '"':
								inquote = !inquote;
								break;
							case '>':
								if (inquote)
									continue;
								break outf; // out of for loop
							default:
								break;
						}
					}
					if (index >= length) {
						if (inquote)
							throw new AddressException("Missing '\"'", s, index);
						throw new AddressException("Missing '>'", s, index);
					}
					route_addr = true;
					end = index;
					break;

				case '>':
					throw new AddressException("Missing '<'", s, index);

				case '"':	// parse quoted string
					rfc822 = true;
					if (start == -1)
						start = index;
					outq:
					for (index++; index < length; index++) {
						c = s.charAt(index);
						switch (c) {
							case '\\':
								index++; // skip both '\' and the escaped char
								break;
							case '"':
								break outq; // out of for loop
							default:
								break;
						}
					}
					if (index >= length)
						throw new AddressException("Missing '\"'", s, index);
					break;

				case '[':	// a domain-literal, probably
					rfc822 = true;
					outb:
					for (index++; index < length; index++) {
						c = s.charAt(index);
						switch (c) {
							case '\\':
								index++; // skip both '\' and the escaped char
								break;
							case ']':
								break outb; // out of for loop
							default:
								break;
						}
					}
					if (index >= length)
						throw new AddressException("Missing ']'", s, index);
					break;

				case ',':	// end of an address, probably
					if (start == -1) {
						route_addr = false;
						rfc822 = false;
						start = end = -1;
						break;	// nope, nothing there
					}
					if (in_group) {
						route_addr = false;
						break;
					}
					// got a token, add this to our InternetAddress vector
					if (end == -1)
						end = index;
					String addr = s.substring(start, end).trim();
					if (rfc822 || strict || parseHdr) {
						if (strict || !parseHdr)
							checkAddress(addr, route_addr, false);
						InternetAddress ma = new InternetAddress();
						ma.setAddress(addr);
						if (start_personal >= 0) {
							ma.encodedPersonal = unquote(s.substring(start_personal, end_personal).trim());
							start_personal = end_personal = -1;
						}
						v.addElement(ma);
					} else {
						// maybe we passed over more than one space-separated addr
						StringTokenizer st = new StringTokenizer(addr);
						while (st.hasMoreTokens()) {
							String a = st.nextToken();
							checkAddress(a, false, false);
							InternetAddress ma = new InternetAddress();
							ma.setAddress(a);
							v.addElement(ma);
						}
					}

					route_addr = false;
					rfc822 = false;
					start = end = -1;
					break;

				case ':':
					rfc822 = true;
					if (in_group)
						throw new AddressException("Nested group", s, index);
					in_group = true;
					if (start == -1)
						start = index;
					break;

				case ';':
					if (start == -1)
						start = index;
					if (!in_group)
						throw new AddressException("Illegal semicolon, not in group", s, index);
					in_group = false;
					if (start == -1)
						start = index;
					InternetAddress ma = new InternetAddress();
					end = index + 1;
					ma.setAddress(s.substring(start, end).trim());
					v.addElement(ma);

					route_addr = false;
					start = end = -1;
					break;

				// Ignore whitespace
				case ' ':
				case '\t':
				case '\r':
				case '\n':
					break;

				default:
					if (start == -1)
						start = index;
					break;
			}
		}

		if (start >= 0) {
			/*
			 * The last token, add this to our InternetAddress vector.
			 * Note that this block of code should be identical to the
			 * block above for "case ','".
			 */
			if (end == -1)
				end = index;
			String addr = s.substring(start, end).trim();
			if (rfc822 || strict || parseHdr) {
				if (strict || !parseHdr)
					checkAddress(addr, route_addr, false);
				InternetAddress ma = new InternetAddress();
				ma.setAddress(addr);
				if (start_personal >= 0)
					ma.encodedPersonal = unquote(s.substring(start_personal, end_personal).trim());
				v.addElement(ma);
			} else {
				// maybe we passed over more than one space-separated addr
				StringTokenizer st = new StringTokenizer(addr);
				while (st.hasMoreTokens()) {
					String a = st.nextToken();
					checkAddress(a, false, false);
					InternetAddress ma = new InternetAddress();
					ma.setAddress(a);
					v.addElement(ma);
				}
			}
		}

		InternetAddress[] a = new InternetAddress[v.size()];
		v.copyInto(a);
		return a;
	}

	/**
	 * Validate that this address conforms to the syntax rules of
	 * RFC 822.  The current implementation checks many, but not
	 * all, syntax rules.  Note that even though the syntax of
	 * the address may be correct, there's no guarantee that a
	 * mailbox of that name exists.
	 *
	 * @throws AddressException if the address isn't valid.
	 * @since JavaMail 1.3
	 */
	public void validate() throws AddressException {
		checkAddress(getAddress(), true, true);
	}

	private static final String specialsNoDotNoAt = "()<>,;:\\\"[]";
	private static final String specialsNoDot = specialsNoDotNoAt + "@";

	/**
	 * Check that the address is a valid "mailbox" per RFC822.
	 * (We also allow simple names.)
	 *
	 * XXX - much more to check
	 * XXX - doesn't handle domain-literals properly (but no one uses them)
	 */
	private static void checkAddress(
		final String addr,
		final boolean routeAddr,
		final boolean validate)
		throws AddressException {

		int start = 0;
		if (addr.indexOf('"') >= 0)
			return;			// quote in address, too hard to check
		if (routeAddr) {
			/*
			 * Check for a legal "route-addr":
			 *		[@domain[,@domain ...]:]local@domain
			 */
			int i;
			for (start = 0; (i = indexOfAny(addr, ",:", start)) >= 0; start = i+1) {
				if (addr.charAt(start) != '@')
					throw new AddressException("Illegal route-addr", addr);
				if (addr.charAt(i) == ':') {
					// end of route-addr
					start = i + 1;
					break;
				}
			}
		}
		/*
		 * The rest should be "local@domain", but we allow simply "local"
		 * unless called from validate.
		 */
		int i;
		String local;
		String domain;
		if ((i = addr.indexOf('@', start)) >= 0) {
			if (i == start)
				throw new AddressException("Missing local name", addr);
			if (i == addr.length() - 1)
				throw new AddressException("Missing domain", addr);
			local = addr.substring(start, i);
			domain = addr.substring(i + 1);
		} else {
			/*
			 * Note that the MimeMessage class doesn't remember addresses
			 * as separate objects; it writes them out as headers and then
			 * parses the headers when the addresses are requested.
			 * In order to support the case where a "simple" address is used,
			 * but the address also has a personal name and thus looks like
			 * it should be a valid RFC822 address when parsed, we only check
			 * this if we're explicitly called from the validate method.
			 */
			if (validate)
				throw new AddressException("Missing final '@domain'", addr);

			/*
			 * No '@' so it's not *really* an RFC822 address, but still
			 * we allow some simple forms.
			 */
			local = addr;
			domain = null;
		}
		// there better not be any whitespace in it
		if (indexOfAny(addr, " \t\n\r") >= 0)
			throw new AddressException("Illegal whitespace in address", addr);
		// local-part must follow RFC822, no specials except '.'
		if (indexOfAny(local, specialsNoDot) >= 0)
			throw new AddressException("Illegal character in local name", addr);
		// check for illegal chars in the domain, but ignore domain literals
		if (domain != null && domain.indexOf('[') < 0 && indexOfAny(domain, specialsNoDot) >= 0)
			throw new AddressException("Illegal character in domain", addr);
	}

	/**
	 * Is this a "simple" address?  Simple addresses don't contain quotes
	 * or any RFC822 special characters other than '@' and '.'.
	 */
	private boolean isSimple() {
		return address == null || indexOfAny(address, specialsNoDotNoAt) < 0;
	}

	/**
	 * Indicates whether this address is an RFC 822 group address.
	 * Note that a group address is different than the mailing
	 * list addresses supported by most mail servers.  Group addresses
	 * are rarely used; see RFC 822 for details.
	 * 
	 * @return true if this address represents a group
	 * @since JavaMail 1.3
	 */
	public boolean isGroup() {
		// quick and dirty check
		return address != null && address.endsWith(";") && address.indexOf(':') > 0;
	}

	/**
	 * Return the members of a group address.  A group may have zero,
	 * one, or more members.  If this address is not a group, null
	 * is returned.  The <code>strict</code> parameter controls whether
	 * the group list is parsed using strict RFC 822 rules or not.
	 * The parsing is done using the <code>parseHeader</code> method.
	 * 
	 * @return		array of InternetAddress objects, or null
	 * @throws AddressException if the group list can't be parsed
	 * @since JavaMail 1.3
	 */
	public InternetAddress[] getGroup(final boolean strict) throws AddressException {
//		Vector groups = null;
		String addr = getAddress();
		// groups are of the form "name:addr,addr,...;"
		if (!addr.endsWith(";"))
			return null;
		int ix = addr.indexOf(':');
		if (ix < 0)
			return null;
		// extract the list
		String list = addr.substring(ix + 1, addr.length() - 1);
		// parse it and return the individual addresses
		return InternetAddress.parseHeader(list, strict);
    }

	/**
	 * Return the first index of any of the characters in "any" in "s",
	 * or -1 if none are found.
	 * 
	 * This should be a method on String.
	 */
	private static int indexOfAny(final String s, final String any) {
		return indexOfAny(s, any, 0);
	}

    private static int indexOfAny(final String s, final String any, final int start) {
		try {
			int len = s.length();
			for (int i = start; i < len; i++)
				if (any.indexOf(s.charAt(i)) >= 0)
					return i;

			return -1;
		} catch (StringIndexOutOfBoundsException e) {
			return -1;
		}
	}

}
