/*
 * Copyright 1997-2004 Sun Microsystems, Inc. All Rights Reserved.
 */

package javax.mail.internet;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.Enumeration;
import java.util.Vector;

import javax.activation.DataHandler;
import javax.activation.FileDataSource;
import javax.mail.BodyPart;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;

import com.sun.mail.util.ASCIIUtility;
import com.sun.mail.util.LineOutputStream;

/**
 * ̃NX MIME {\܂B
 *  <code>BodyPart</code> ۃNX <code>MimePart</code> C^tF[X܂B
 * MimeBodyParts  MimeMultipart IuWFNgɊ܂܂܂B<p>
 * 
 * MimeBodyPart  <code>InternetHeaders</code> NXgpA
 * {̃wb_\͂Ai[܂B<p>
 * 
 * <hr><strong>RFC822 y MIME wb_Ɋւ郁</strong><p>
 * 
 * RFC822 wb_tB[h US-ASCII ܂܂Ȃ<strong>Ȃ܂</strong>B
 * MIME ł ASCII ȊO̕GR[fBO鎖ɂA
 * ̃wb_̓̕ɐݒ\łBRFC 2047 ͂̏̋Kw肵܂B
 * ̃pbP[WŒ񋟂 MimeUtility NXpāȀsƂł܂B
 * <code>setHeader</code>A<code>addHeader</code>Ay <code>addHeaderLine</code> \bh
 * Ăяo͎w肳ꂽwb_ MIME vɏ]ۏ؂Ȃ΂Ȃ܂B
 * Ãwb_tB[h]ɂs̒ (SMTP ̏ꍇ 1000oCg) 𒴂ꍇ́A
 * M̑Oɐ܂肽 (bv) Kv܂BMwb_͐܂肽܂Ă\܂B
 * AvP[V̓wb_K؂ɐ܂肽݁AWJӔC܂B<p>
 * 
 * @see javax.mail.Part
 * @see javax.mail.internet.MimePart
 * @see javax.mail.internet.MimeUtility
 */
public class MimeBodyPart extends BodyPart implements MimePart {

	// Paranoia:
	// allow this last minute change to be disabled if it causes problems
	private static boolean setDefaultTextCharset = true;

	private static boolean setContentTypeFileName = true;

	private static boolean encodeFileName = false;
	private static boolean decodeFileName = false;

	static boolean cacheMultipart = true;

	static {
		try {
			String s = System.getProperty("mail.mime.setdefaulttextcharset");
			// default to true
			setDefaultTextCharset = s == null || !s.equalsIgnoreCase("false");

			s = System.getProperty("mail.mime.setcontenttypefilename");
			setContentTypeFileName = s == null || !s.equalsIgnoreCase("false");

			s = System.getProperty("mail.mime.encodefilename");
			// default to false
			encodeFileName = s != null && !s.equalsIgnoreCase("false");

			s = System.getProperty("mail.mime.decodefilename");
			// default to false
			decodeFileName = s != null && !s.equalsIgnoreCase("false");

			s = System.getProperty("mail.mime.cachemultipart");
			cacheMultipart = s == null || !s.equalsIgnoreCase("false");
		} catch (SecurityException sex) {
			// 
		}
	}

	/**
	 *  Part ̓e\ DataHandler IuWFNgłB
	 */
	protected DataHandler dh;

	/**
	 *  Part ̓ẽoCgێoCgzłB
	 */
	protected byte[] content;

	/**
	 * ̖{̃f[^ SharedInputStream C^tF[X InputStream ɂ苟ꂽꍇA
	 * <code>contentStream</code> ̖{̓e\ 1 ̃Xg[łB
	 * ̏ꍇA<code>content</code>  null łB
	 * 
	 * @since JavaMail 1.2
	 */
	protected InputStream contentStream;

	/**
	 * ̖{̑SẴwb_i[ InternetHeaders IuWFNgłB
	 */
	protected InternetHeaders headers;

	/**
	 *  MimeBodyPart IuWFNg쐬܂B
	 * ̖{ multipart bZ[W\zNCAgɂLꍇ܂B
	 */
	public MimeBodyPart() {
		super();
		headers = new InternetHeaders();
	}

	/**
	 * w肳ꂽ̓Xg[̃f[^ǂݎA
	 * \͂邱ƂɂAMimeBodyPart \z܂B
	 * p[T́Aw肳ꂽ̓Xg[̏I܂Ńf[^܂B
	 * ̓Xg[͗L MIME {̐擪n܂A
	 * ̖{̏IŏIȂ΂Ȃ܂B<p>
	 * 
	 * {؂ "E" ͓̓Xg[Ɋ܂܂ĂĂ<strong>Ȃ܂</strong>B
	 * MimeMultipart p[T maultipart Xg[e{̃oCg𒊏oA
	 * ؂蕶Ȃł̃RXgN^ɗ^܂B
	 * 
	 * @param is {̓̓Xg[
	 */
	public MimeBodyPart(InputStream is) throws MessagingException {
		if (!(is instanceof ByteArrayInputStream) && !(is instanceof BufferedInputStream) && !(is instanceof SharedInputStream))
			is = new BufferedInputStream(is);

		headers = new InternetHeaders(is);

		if (is instanceof SharedInputStream) {
			SharedInputStream sis = (SharedInputStream) is;
			contentStream = sis.newStream(sis.getPosition(), -1);
		} else {
			try {
				content = ASCIIUtility.getBytes(is);
			} catch (IOException ioex) {
				throw new MessagingException("Error reading input stream", ioex);
			}
		}
	}

	/**
	 * w肳ꂽwb_ƓeoCggp MimeBodyPart \z܂B<p>
	 * 
	 * voC_gp܂B
	 * 
	 * @param headers ̃̕wb_
	 * @param content ̖̕{\oCg
	 */
	public MimeBodyPart(final InternetHeaders headers, final byte[] content) /* throws MessagingException */ {
		super();
		this.headers = headers;
		this.content = content;
	}

	/**
	 * ̖{̓eTCYoCgPʂŕԂ܂B
	 * TCYʂłȂꍇ -1 Ԃ܂B<p>
	 * 
	 * ̐l͓eTCY̐mȑlłȂ\A
	 * e̓]GR[fBOłȂ_ɒӂĉB<p>
	 * 
	 * ̎́A<code>content</code> z ̃TCY (null łȂꍇ)A
	 *  <code>contentStream</code>  null łȂA
	 * <code>available</code> \bh̐ԂꍇA
	 * ̐TCYƂĕԂ܂BȊȌꍇ -1 Ԃ܂B
	 * 
	 * @return oCgPʂ̃TCYA͕s̏ꍇ -1
	 */
	public int getSize() throws MessagingException {
		if (content != null)
			return content.length;
		if (contentStream != null) {
		    try {
				int size = contentStream.available();
				// only believe the size if it's greate than zero, since zero
				// is the default returned by the InputStream class itself
				if (size > 0)
					return size;
		    } catch (IOException ex) {}	// 
		}
		return -1;
	}

	/**
	 *  Part ̓e̍sԂ܂B
	 * ̐lʂłȂꍇ -1 Ԃ܂B<p>
	 * 
	 * ̐l͓e̒̐mȑlłȂ\A
	 * e̓]GR[fBOłȂ_ɒӂĉB<p>
	 * 
	 * ̎ -1 Ԃ܂B
	 * 
	 * @return sA͕s̏ꍇ -1
	 */  
	 public int getLineCount() throws MessagingException {
	 	return -1;
	 }

	/**
	 * RFC 822 "Content-Type" wb_tB[h̒lԂ܂B
	 * ́A̖{̓ẽRec`\܂B
	 * ̒l null łĂ͂Ȃ܂B
	 * ̃tB[hgp\łȂꍇA"text/plain" Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader(name)</code> gpA
	 * Kvȃwb_tB[h擾܂B
	 * 
	 * @return ̖{ Content-Type
	 */
	public String getContentType() throws MessagingException {
		String s = getHeader("Content-Type", null);
		if (s == null)
			s = "text/plain";

// Sugisawa added. 2005/03/04
if (s.equalsIgnoreCase("text"))
	s = "text/plain";

		return s;
	}

	/**
	 * w肳ꂽ MIME ^Cv Part ǂʂ܂B̃\bh́A
	 * <strong><code>primaryType</code>  <code>subType</code> ݂̂r܂</strong>B
	 * Rec`̃p[^͖܂B<p>
	 * 
	 * Ⴆ΁ARec` <strong>"text/plain"</strong>  <strong>"text/plain; charset=Shift_JIS"</strong>  Part rƁA
	 * ̃\bh <code>true</code> Ԃ܂B<p>
	 * 
	 * <code>mimeType</code>  <code>subType</code> ꕶ '*' ̏ꍇA
	 * rɃTu^Cv͖܂B
	 */
	public final boolean isMimeType(final String mimeType) throws MessagingException {
		return isMimeType(this, mimeType);
	}

	/**
	 * "Content-Disposition" wb_tB[h̒lԂ܂B͂̏̕u\܂B
	 * ̕ǂ̗lɃ[Uɒ񎦂邩܂B<p>
	 * 
	 * Content-Disposition tB[hgpłȂꍇ null Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader(name)</code> gpAKvȃwb_tB[h擾܂B
	 * 
	 * @see #headers
	 */
	public String getDisposition() throws MessagingException {
		return getDisposition(this);
	}

	/**
	 * ̖{ "Content-Disposition" wb_tB[hݒ肵܂B
	 * u null ̏ꍇ́ȂSĂ "Content-Disposition" wb_tB[h폜܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̖{ READ_ONLY tH_擾ꂽꍇ
	 */
	public void setDisposition(final String disposition) throws MessagingException {
		setDisposition(this, disposition);
	}

	/**
	 * "Content-Transfer-Encoding" wb_tB[h̓e]GR[fBOԂ܂B
	 * wb_gpłȂꍇA͂̒l݂Ȃꍇ <code>null</code> Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader(name)</code> gpAKvȃwb_tB[h擾܂B
	 * 
	 * @see #headers
	 */
	public String getEncoding() throws MessagingException {
		return getEncoding(this);
	}

	/**
	 * "Content-ID" wb_tB[h̒lԂ܂BtB[hgpłȂꍇA
	 * ͂̒l݂Ȃꍇ <code>null</code> Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader(name)</code> gpAKvȃwb_tB[h擾܂B
	 */
	public String getContentID() throws MessagingException {
		return getHeader("Content-Id", null);
	}

	/**
	 * Set the "Content-ID" header field of this body part.
	 * If the <code>cid</code> parameter is null, any existing 
	 * "Content-ID" is removed.
	 *
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̖{ READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 * @since JavaMail 1.3
	 */
	public final void setContentID(final String cid) throws MessagingException {
		if (cid == null)
			removeHeader("Content-ID");
		else
			setHeader("Content-ID", cid);
	}

	/**
	 * "Content-MD5" wb_tB[h̒lԂ܂BtB[hgpłȂꍇA
	 * ͂̒l݂Ȃꍇ <code>null</code> Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader(name)</code> gpAKvȃwb_tB[h擾܂B
	 */
	public String getContentMD5() throws MessagingException {
		return getHeader("Content-MD5", null);
	}

	/**
	 * ̖{ "Content-MD5" wb_tB[hݒ肵܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̖{ READ_ONLY tH_擾ꂽꍇ
	 */
	public void setContentMD5(final String md5) throws MessagingException {
		setHeader("Content-MD5", md5);
	}

	/**
	 *  MimePart  Content-Language wb_Ŏw肳錾擾܂B
	 * Content-Language wb_ RFC 1766 ɂ`܂B
	 * wb_gpłȂꍇA͂̒l݂Ȃꍇ <code>null</code> Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader(name)</code> gpAKvȃwb_tB[h擾܂B
	 */
	public final String[] getContentLanguage() throws MessagingException {
		return getContentLanguage(this);
	}

	/**
	 *  MimePart  Content-Language wb_ݒ肵܂B
	 * Content-Language wb_ RFC 1766 ɂ`܂B
	 * 
	 * @param languages ^O̔z
	 */
	public final void setContentLanguage(final String[] languages) throws MessagingException {
		setContentLanguage(this, languages);
	}

	/**
	 * ̖{ "Content-Description" wb_tB[hԂ܂B
	 * ͒ʏꕔ̏̕Ɋ֘At܂B̃tB[hgpłȂꍇA
	 * ͂̒l݂Ȃꍇ <code>null</code> Ԃ܂B<p>
	 * 
	 * Content-Description tB[h RFC 2047 ɏ]ăGR[fBOĂꍇA
	 * fR[hAUnicode ɕϊ܂BfR[fBO܂͕ϊsꍇA
	 * f[^̂܂ܕԂ܂B<p>
	 * 
	 * ̎ <code>getHeader(name)</code> gpAKvȃwb_tB[h擾܂B
	 * 
	 * @return e̐
	 */
	public String getDescription() throws MessagingException {
		return getDescription(this);
	}

	/**
	 * ̖{ "Content-Description" wb_tB[hݒ肵܂B
	 * description p[^ <code>null</code> ̏ꍇ́A
	 * ̑SĂ "Content-Description" tB[h폜܂B<p>
	 * 
	 * description  US-ASCII ȊO̕܂܂ꍇ́A
	 * vbgtH[̃ftHgZbggpăGR[h܂B
	 * description  US-ASCII ܂܂ꍇ́A
	 * GR[fBO͍sꂸÂ܂܎gp܂B<p>
	 * 
	 * ZbgGR[fBOsꍇAMessagingException X[A
	 * UnsupportedEncodingException  MessagingException 
	 * qɂꂽÕ`F[Ɋ܂܂鎖ɒӂĉB
	 * 
	 * @param description e̐
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̖{ READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException ZbgϊsƁA
	 * UnsupportedEncodingException O`F[Ɋ܂܂ꍇ܂B
	 */
	public final void setDescription(final String description) throws MessagingException {
		setDescription(description, null);
	}

	/**
	 * ̖{ "Content-Description" wb_ݒ肵܂B
	 * description p[^ <code>null</code> ̏ꍇ́A
	 * ̑SĂ "Content-Description" tB[h폜܂B<p>
	 * 
	 * description  US-ASCII ȊO̕܂܂ꍇ́A
	 * w肳ꂽZbggpăGR[h܂B
	 * description  US-ASCII ܂܂ꍇ́A
	 * GR[fBO͍sꂸÂ܂܎gp܂B<p>
	 * 
	 * ZbgGR[fBOsꍇAMessagingException X[A
	 * UnsupportedEncodingException  MessagingException ̓qɂꂽ
	 * Õ`F[Ɋ܂܂鎖ɒӂĉB
	 * 
	 * @param description 
	 * @param charset GR[fBÔ߂̕Zbg
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̖{ READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException ZbgϊsƁA
	 * UnsupportedEncodingException O`F[Ɋ܂܂ꍇ܂B
	 */
	public void setDescription(final String description, final String charset) throws MessagingException {
		setDescription(this, description, charset);
	}

	/**
	 * ̖{Ɋ֘Atꂽt@C擾܂B<p>
	 * 
	 * ̖{ "Content-Disposition" wb_tB[hɂ "filename" p[^̒lԂ܂B
	 * ̃p[^gpłȂꍇ́A
	 * { "Content-Type" wb_tB[hɂ "name" p[^̒lԂ܂B
	 * Ƃ݂Ȃꍇ <code>null</code> Ԃ܂B
	 * 
	 * @return t@C
	 */
	/*
	 * XXX - for JavaMail 1.4
	 * If the System property <code>mail.mime.decodefilename</code>
	 * is set to <code>true</code>, the <code>MimeUtility</code>
	 * method <code>decodeText</code> will be used to decode any
	 * non-ASCII characters in the filename.  Note that this decoding
	 * violates the MIME specification, but is useful for interoperating
	 * with some mail clients that use this convention.
	 */
	public String getFileName() throws MessagingException {
		return getFileName(this);
	}

	/**
	 * \ł΁A̖{Ɋ֘Atꂽt@Cݒ肵܂B<p>
	 * 
	 * ̖{ "Content-Disposition" wb_tB[hɂ "filename" p[^ݒ肵܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̖{ READ_ONLY tH_擾ꂽꍇ
	 */
	/*
	 * XXX - for JavaMail 1.4
	 * If the System property <code>mail.mime.encodefilename</code>
	 * is set to <code>true</code>, the <code>MimeUtility</code>
	 * method <code>encodeText</code> will be used to encode any
	 * non-ASCII characters in the filename.  Note that this encoding
	 * violates the MIME specification, but is useful for interoperating
	 * with some mail clients that use this convention.
	 */
	public void setFileName(final String filename) throws MessagingException {
		setFileName(this, filename);
	}

	/**
	 * ̖{ "content" ɑ΂fR[hꂽ̓Xg[Ԃ܂B<p>
	 * 
	 * ̎ DataHandler ̓Xg[擾܂B
	 * ܂AgetDataHandler().getInputStream(); Ăяo܂B
	 * 
	 * @return InputStream
	 * @throws MessagingException
	 * @throws IOException ͒ʏ DataHandler ɂX[܂B
	 * ڍׂɂĂ javax.activation.DataHandler ̃}jAQƂĉB
	 * @see #getContentStream
	 * @see javax.activation.DataHandler#getInputStream
	 */
	public final InputStream getInputStream() throws IOException, MessagingException {
		return getDataHandler().getInputStream();
	}

	/**
	 * e̐oCg𐶐܂B
	 * ̃\bh͓e DataHandler IuWFNg쐬鎞Ɏgp܂B
	 * Part e̕ʌ̓̓Xg[񋟉\
	 * TuNX͂̃\bhI[o[Chꍇ܂B<p>
	 * 
	 * @see #content
	 * @see MimeMessage#getContentStream
	 */
	protected InputStream getContentStream() throws MessagingException {
		if (contentStream != null)
			return ((SharedInputStream)contentStream).newStream(0, -1);
		if (content != null)
			return new ByteArrayInputStream(content);

		throw new MessagingException("No content");
	}

	/**
	 * Cӂ Content-Transfer-Encoding ̂܂܂ɂāAf[^ւ InputStream Ԃ܂B
	 * ̃\bh "Content-Transfer-Encoding" wb_ȂĂ肷ꍇɖ𗧂܂B
	 * ̏ꍇA<code>getInputStream</code> \bh <code>getContent</code> \bh́A
	 * f[^ԂƂł܂B
	 * AvP[V͂̃\bhgpAf[^̂̂̃fR[h݂鎖ł܂B<p>
	 * 
	 * ̎͒P <code>getContentStream</code> \bhĂяo܂B
	 * 
	 * @since JavaMail 1.2
	 * @see #getInputStream
	 * @see #getContentStream
	 */
	public final InputStream getRawInputStream() throws MessagingException {
		return getContentStream();
	}

	/**
	 * ̖{̓e DataHandler Ԃ܂B<p>
	 * 
	 * Œ񋟂́Ax MimeMessage ̗̎lɋ@\܂B
	 * 
	 * @see MimeMessage#getDataHandler
	 */  
	public DataHandler getDataHandler() throws MessagingException {
		if (dh == null)
			dh = new DataHandler(new MimePartDataSource(this));
		return dh;
	}

	/**
	 * e java IuWFNgƂĕԂ܂B
	 * ԂIuWFNg̃^Cv͂e̕Ɉˑ܂B
	 * Ⴆ΁Atext/plain RečŗL`͕ String IuWFNgłB
	 * "multipart" RečŗL`͏ Multipart TuNXłB
	 * DataHandler VXeɕsȃRec^̏ꍇA
	 * ̓Xg[RecƂĕԂ܂B<p>
	 * 
	 * ̎ DataHandler Rec擾܂B
	 * ܂AgetDataHandler().getContent(); Ăяo܂B
	 * 
	 * @return IuWFNg
	 * @throws MessagingException
	 * @throws IOException ͒ʏ DataHandler ɂX[܂B
	 * ڍׂɂĂ javax.activation.DataHandler ̃}jAQƂĉB
	 */
	public final Object getContent() throws IOException, MessagingException {
		Object obj = getDataHandler().getContent();
		if (cacheMultipart && ((obj instanceof Multipart) || (obj instanceof Message)) && !(dh instanceof CachedDataHandler) && (content != null || contentStream != null))
			dh = createCachedDataHandler(obj, getContentType());
		return obj;
	}

	/**
	 * ̃\bh́A̖{̓eݒ肷ׂ̎dg݂񋟂܂B
	 * w肳ꂽ DataHandler IuWFNǵAۂ̓ebvȂ΂Ȃ܂B
	 * 
	 * @param dh Rec DataHandler
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̖{ READ_ONLY tH_擾ꍇ
	 */                 
	public void setDataHandler(final DataHandler dh) throws MessagingException {
		this.dh = dh;
		invalidateContentHeaders(this);
	}

	/**
	 * ̖{̓eݒ肷ׂ̊ȈՃ\bhłB<p>
	 * 
	 * Rec DataHandler IuWFNgɃbv܂Bꂪ@\ׂɁA
	 * w肳ꂽ^Cv DataContentHandler NX JavaMail ɑ΂Ďgp\łȂ΂ȂȂɒӂĉB
	 * ܂A<code>setContent(foobar, "application/x-foobar")</code> sׂɁA
	 * "application/x-foobar"  DataContentHandler CXg[ĂKv܂B
	 * ڍׂɂĂ Java Activation Framework QƂĉB
	 * 
	 * @param o RecIuWFNg
	 * @param type IuWFNg MIME ^Cv
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̖{ READ_ONLY tH_擾ꂽꍇ
	 */
	public void setContent(final Object o, final String type) throws MessagingException {
		if (o instanceof Multipart)
			setContent((Multipart)o);
		else
			setDataHandler(new DataHandler(o, type));
	}

	/**
	 * w肳ꂽ String ̃̕RecƂāA
	 * "text/plain"  MIME ^CvŐݒ肷ȈՃ\bhłB
	 *  US-ASCII ȊO̕܂܂ꍇA
	 * vbgtH[̃ftHg̕ZbggpăGR[h܂B
	 * ̕Zbg "charset" p[^ݒ肷ׂɂgp܂B<p>
	 * 
	 * ̃\bh͎gp镶Zbg߂ׂɑSĂ̕𑖍Ȃ΂ȂȂꍇ̂ŁA
	 * <code>text</code> 傫ꍇ́A\ቺ\鎖ɒӂĉB<p>
	 * 
	 * Zbgm̏ꍇAZbgp[^󂯎 setText() o[WgpĉB
	 * 
	 * @see #setText(String text, String charset)
	 */
	public final void setText(final String text) throws MessagingException {
		setText(text, null);
	}

	/**
	 * "text/plain"  MIME ^Cvyюw肳ꂽZbggpA
	 * w肳ꂽ String ̃̕RecƂĐݒ肷ȈՃ\bhłB
	 * w肳ꂽ Unicode ́Aw肳ꂽZbggpĕZbgGR[h܂B
	 * ̕Zbg "charset" p[^ݒ肷ׂɂgp܂B
	 */
	public final void setText(final String text, final String charset) throws MessagingException {
		setText(this, text, charset, "plain");
	}

	/**
	 * "text"  MIME ^Cvyюw肳ꂽ MIME Tu^CvgpA
	 * w肳ꂽ String ̃̕RecƂĐݒ肷ȈՃ\bhłB
	 * w肳ꂽ Unicode ́Aw肳ꂽZbggpĕZbgGR[h܂B
	 * ̕Zbg "charset" p[^ݒ肷ׂɂgp܂B
	 */
	public void setText(final String text, final String charset, final String subtype) throws MessagingException {
		setText(this, text, charset, subtype);
	}

	/**
	 * ̃\bh͖{̓e Multipart IuWFNgɐݒ肵܂B
	 * 
	 * @param mp Message ̃Recł multipart IuWFNg
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̖{ READ_ONLY tH_擾ꂽꍇ
	 */
	public void setContent(final Multipart mp) throws MessagingException {
		setDataHandler(new DataHandler(mp, mp.getContentType()));
		mp.setParent(this);
	}

	/**
	 * Use the specified file to provide the data for this part.
	 * The simple file name is used as the file name for this part and the data in the file is used as the data for this part.
	 * The encoding will be chosen appropriately for the file data.
	 * 
	 * @param file the File object to attach
	 * @throws IOException errors related to accessing the file
	 * @throws MessagingException message related errors
	 * @since JavaMail 1.4
	 */
	public final void attachFile(final File file) throws IOException, MessagingException {
		FileDataSource fds = new FileDataSource(file);
		setDataHandler(new DataHandler(fds));
		setFileName(fds.getName());
	}

	/**
	 * Use the specified file to provide the data for this part.
	 * The simple file name is used as the file name for this part and the data in the file is used as the data for this part.
	 * The encoding will be chosen appropriately for the file data.
	 * 
	 * @param file the name of the file to attach
	 * @throws IOException errors related to accessing the file
	 * @throws MessagingException message related errors
	 * @since JavaMail 1.4
	 */
	public final void attachFile(final String file) throws IOException, MessagingException {
		File f = new File(file);
		attachFile(f);
	}

	/**
	 * Save the contents of this part in the specified file.
	 * The content is decoded and saved, without any of the MIME headers.
	 * 
	 * @param file the File object to write to
	 * @throws IOException errors related to accessing the file
	 * @throws MessagingException message related errors
	 * @since JavaMail 1.4
	 */
	public final void saveFile(final File file) throws IOException, MessagingException {
		BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(file));
		InputStream is = getInputStream();
		try {
			byte[] bytes = new byte[8192];
			int i;
			while ((i = is.read(bytes)) > 0) 
				bos.write(bytes, 0, i);
		} finally {
			try {
				if (is != null)
					is.close();
			} catch (IOException ioex) {}
			try {
				if (bos != null)
					bos.close();
			} catch (IOException ioex) {}
		}
	}

	/**
	 * Save the contents of this part in the specified file.
	 * The content is decoded and saved, without any of the MIME headers.
	 * 
	 * @param file the name of the file to write to
	 * @throws IOException errors related to accessing the file
	 * @throws MessagingException message related errors
	 * @since JavaMail 1.4
	 */
	public final void saveFile(final String file) throws IOException, MessagingException {
		File f = new File(file);
		saveFile(f);
	}

	/**
	 * { RFC 822 `̃Xg[Ƃďo͂܂B
	 * 
	 * @throws MessagingException
	 * @throws IOException Xg[ւ̏ݒɃG[ꍇA
	 *  javax.activation CɂG[ꍇ
	 * @see javax.activation.DataHandler#writeTo
	 */
	public void writeTo(final OutputStream os) throws IOException, MessagingException {
		writeTo(this, os, null);
	}

	/**
	 *  header_name ̑SẴwb_擾܂B
	 * wb_ US-ASCII ȊO̕܂ޏꍇARFC 2047 ɊÂGR[h鎖A
	 * fR[hKvƂȂ鎖ɒӂĉB
	 * 
	 * @param name wb_̖O
	 * @return wb_̔z
	 * @see javax.mail.internet.MimeUtility
	 */  
	public String[] getHeader(final String name) throws MessagingException {
		return headers.getHeader(name);
	}

	/**
	 * ̃wb_̑SẴwb_擾A؂蕶ŋ؂ꂽwb_P String ƂĕԂ܂A
	 * ؂蕶 <code>null</code> ̏ꍇ́Aŏ̃wb_Ԃ܂B
	 * 
	 * @param name ̃wb_̖O
	 * @param delimiter delimiter between fields in returned string
	 * @return ̖OSẴwb_ɑ΂ltB[h 
	 * @throws MessagingException
	 */
	public final String getHeader(final String name, final String delimiter) throws MessagingException {
		return headers.getHeader(name, delimiter);
	}

	/**
	 *  header_name ̒lݒ肵܂B̑SẴwb_l̐VlŒu܂B
	 * RFC 822 wb_ US-ASCII ݂̂܂܂Ȃ΂ȂȂׁA
	 * US-ASCII ȊO̕܂ރwb_ RFC 2047 ̋Kɏ]ăGR[hȂ΂ȂȂɒӂĉB
	 * 
	 * @param name wb_
	 * @param value wb_l
	 * @see javax.mail.internet.MimeUtility
	 */
	public void setHeader(final String name, final String value) throws MessagingException {
		headers.setHeader(name, value);
	}
 
	/**
	 * ̒l header_name ̊lɒǉ܂B
	 * RFC 822 wb_ US-ASCII ݂̂܂܂Ȃ΂ȂȂׁA
	 * US-ASCII ȊO̕܂ރwb_ RFC 2047 ̋Kɏ]ăGR[hȂ΂ȂȂɒӂĉB
	 * 
	 * @param name wb_
	 * @param value wb_l
	 * @see javax.mail.internet.MimeUtility
	 */
	public void addHeader(final String name, final String value) throws MessagingException {
		headers.addHeader(name, value);    
	}

	/**
	 * ̖OSẴwb_폜܂B
	 */
	public void removeHeader(final String name) throws MessagingException {
		headers.removeHeader(name);
	}
 
	/**
	 *  Message ̑SẴwb_ Header IuWFNg̗񋓌^ƂĕԂ܂B
	 */
	public Enumeration getAllHeaders() throws MessagingException {
		return headers.getAllHeaders();
	}

	/**
	 *  Message ̈vwb_ Header IuWFNg̗񋓌^ƂĕԂ܂B<p>
	 */
	public Enumeration getMatchingHeaders(final String[] names) throws MessagingException {
		return headers.getMatchingHeaders(names);
	}

	/**
	 *  Message ̈vȂwb_ Header IuWFNg̗񋓌^ƂĕԂ܂B
	 */
	public Enumeration getNonMatchingHeaders(final String[] names) throws MessagingException {
		return headers.getNonMatchingHeaders(names);
	}

	/**
	 * wb_s̖{ɒǉ܂B
	 */
	public void addHeaderLine(final String line) throws MessagingException {
		headers.addHeaderLine(line);
	}

	/**
	 * SẴwb_s String ̗񋓌^ƂĎ擾܂B
	 * Header s͐ RFC 822 wb_słA"name" y "value" ̗tB[h܂݂܂B
	 */
	public Enumeration getAllHeaderLines() throws MessagingException {
		return headers.getAllHeaderLines();
	}

	/**
	 * vwb_s String ̗񋓌^ƂĎ擾܂B
	 * Header s͐ RFC 822 wb_słA"name" y "value" ̗tB[h܂݂܂B
	 */
	public Enumeration getMatchingHeaderLines(final String[] names) throws MessagingException {
		return headers.getMatchingHeaderLines(names);
	}

	/**
	 * vȂwb_s String ̗񋓌^ƂĎ擾܂B
	 * Header s͐ RFC 822 wb_słA"name" y "value" ̗tB[h܂݂܂B
	 */
	public Enumeration getNonMatchingHeaderLines(final String[] names) throws MessagingException {
		return headers.getNonMatchingHeaderLines(names);
	}

	/**
	 * ̖{̓eAK؂ MIME wb_XV܂B
	 * Őݒ肳ʓIȃwb_ <code>Content-Type</code>  <code>Content-Transfer-Encoding</code> łB
	 * wb_̍XV͎ 2 ̏ꍇɕKvɂȂ܂B
	 * 
	 * <br>
	 * - [AvP[Vɂ쐬郁bZ[ẂAwb_𖞂ׂɁA
	 * 鎞_ł̃\bhANeBuɂKv܂B
	 * 
	 * <br>
	 * - Store ǂݍ܂郁bZ[W͑SẴwb_i[ꏊ擾̂ŁA͕Kvł͂܂B
	 * AAbZ[WҏW\œe̓bZ[W\ɕҏWsꂽꍇA
	 * wb_̍ēKvƂȂꍇ܂B
	 * 
	 * <br>
	 * ǂ̏ꍇÃ\bh͒ʏ <code>Message.saveChanges</code> \bhɂĂяo܂B
	 */
	protected void updateHeaders() throws MessagingException {
		updateHeaders(this);
	}

	/////////////////////////////////////////////////////////////
	// Package private convenience methods to share code among //
	// MimeMessage and MimeBodyPart                            //
	/////////////////////////////////////////////////////////////

	static final boolean isMimeType(final MimePart part, final String mimeType) throws MessagingException {
		// XXX - lots of room for optimization here!
		try {
			ContentType ct = new ContentType(part.getContentType());
			return ct.match(mimeType);
		} catch (ParseException ex) {
			return part.getContentType().equalsIgnoreCase(mimeType);
		}
	}

	static final void setText(final MimePart part, final String text, String charset, final String subtype) throws MessagingException {
		if (charset == null) {
			if (MimeUtility.checkAscii(text) != MimeUtility.ALL_ASCII)
				charset = MimeUtility.getDefaultMIMECharset();
			else
				charset = "us-ascii";
		}
		part.setContent(text, "text/" + subtype + "; charset=" + MimeUtility.quote(charset, HeaderTokenizer.MIME));
	}

	static final String getDisposition(final MimePart part) throws MessagingException {
		String s = part.getHeader("Content-Disposition", null);

		if (s == null)
			return null;
	
		ContentDisposition cd = new ContentDisposition(s);
		return cd.getDisposition();
	}

	static final void setDisposition(final MimePart part, String disposition) throws MessagingException {
		if (disposition == null)
			part.removeHeader("Content-Disposition");
		else {
			String s = part.getHeader("Content-Disposition", null);
			if (s != null) { 
				/* A Content-Disposition header already exists ..
				 *
				 * Override disposition, but attempt to retain 
				 * existing disposition parameters
				 */
				ContentDisposition cd = new ContentDisposition(s);
				cd.setDisposition(disposition);
				disposition = cd.toString();
			}
			part.setHeader("Content-Disposition", disposition);
		}
	}

	static final String getDescription(final MimePart part) throws MessagingException {
		String rawvalue = part.getHeader("Content-Description", null);
		if (rawvalue == null)
			return null;

		try {
			return MimeUtility.decodeText(MimeUtility.unfold(rawvalue));
		} catch (UnsupportedEncodingException ex) {
			return rawvalue;
		}
	}

	static final void setDescription(final MimePart part, final String description, final String charset) throws MessagingException {
		if (description == null) {
			part.removeHeader("Content-Description");
			return;
		}

		try {
			part.setHeader("Content-Description", MimeUtility.fold(21, MimeUtility.encodeText(description, charset, null)));
		} catch (UnsupportedEncodingException uex) {
			throw new MessagingException("Encoding error", uex);
		}
	}

	static final String getFileName(final MimePart part) throws MessagingException {
		String filename = null;
		String s = part.getHeader("Content-Disposition", null);

		if (s != null) {
			// Parse the header ..
			ContentDisposition cd = new ContentDisposition(s);
			filename = cd.getParameter("filename");
		}

		if (filename == null) {
			// Still no filename ? Try the "name" ContentType parameter
			String s2 = part.getHeader("Content-Type", null);
			if (s2 != null) {
				try {
					ContentType ct = new ContentType(s2);
					filename = ct.getParameter("name");
				} catch (ParseException pex) { }	// 
			}
		}

		if (decodeFileName && filename != null)
			try {
				filename = MimeUtility.decodeText(s);
			} catch (UnsupportedEncodingException e) {
				throw new MessagingException("Can't decode filename", e);
			}

		return filename;
	}

	static final void setFileName(final MimePart part, String name) throws MessagingException {
		if (encodeFileName && name != null) {
			try {
				name = MimeUtility.encodeText(name);
			} catch (UnsupportedEncodingException ex) {
				throw new MessagingException("Can't encode filename", ex);
			}
		}

		// Set the Content-Disposition "filename" parameter
		String s = part.getHeader("Content-Disposition", null);
		ContentDisposition cd = new ContentDisposition(s == null ? Part.ATTACHMENT : s);

		cd.setParameter("filename", name);
		part.setHeader("Content-Disposition", cd.toString());

		/* Also attempt to set the Content-Type "name" parameter,
		 * to satisfy ancient MUAs. XXX: This is not RFC compliant.
		 */
		if (setContentTypeFileName) {
			s = part.getHeader("Content-Type", null);
			if (s != null) {
				try {
					ContentType cType = new ContentType(s);
					cType.setParameter("name", name);
					part.setHeader("Content-Type", cType.toString());
				} catch (ParseException pex) {}	// 
			}
		}
	}

	static final String[] getContentLanguage(final MimePart part) throws MessagingException {
		String s = part.getHeader("Content-Language", null);

		if (s == null)
		    return null;

		// Tokenize the header to obtain the Language-tags (skip comments)
		HeaderTokenizer h = new HeaderTokenizer(s, HeaderTokenizer.MIME);
		Vector v = new Vector();

		while (true) {
			HeaderTokenizer.Token tk = h.next(); // get a language-tag
			int tkType = tk.getType();
			if (tkType == HeaderTokenizer.Token.EOF)
				break; // done
			else if (tkType == HeaderTokenizer.Token.ATOM)
				v.addElement(tk.getValue());
			else // invalid token, skip it.
				continue;
		}

		if (v.size() == 0)
		    return null;

		String[] language = new String[v.size()];
		v.copyInto(language);
		return language;	
	}

	static final void setContentLanguage(final MimePart part, final String[] languages) throws MessagingException {
		StringBuffer sb = new StringBuffer(languages[0]);
		for (int i = 1; i < languages.length; i++)
			sb.append(',').append(languages[i]);
		part.setHeader("Content-Language", sb.toString());
	}

	static final String getEncoding(final MimePart part) throws MessagingException {
		String s = part.getHeader("Content-Transfer-Encoding", null);

		if (s == null)
			return null;

		s = s.trim();	// get rid of trailing spaces
		// quick check for known values to avoid unnecessary use
		// of tokenizer.
		if (s.equalsIgnoreCase("7bit") || s.equalsIgnoreCase("8bit") || s.equalsIgnoreCase("quoted-printable") || s.equalsIgnoreCase("binary") || s.equalsIgnoreCase("base64"))
			return s;

		// Tokenize the header to obtain the encoding (skip comments)
		HeaderTokenizer h = new HeaderTokenizer(s, HeaderTokenizer.MIME);

		for (;;) {
			HeaderTokenizer.Token tk = h.next(); // get a token
			int tkType = tk.getType();
			if (tkType == HeaderTokenizer.Token.EOF)
				break; // done
			else if (tkType == HeaderTokenizer.Token.ATOM)
				return tk.getValue();
			else // invalid token, skip it.
				continue;
		}
		return s;
	}

	static final void setEncoding(final MimePart part, final String encoding) throws MessagingException {
		part.setHeader("Content-Transfer-Encoding", encoding);
	}

	static final DataHandler createCachedDataHandler(final Object obj, final String mimeType) {
		return new CachedDataHandler(obj, mimeType);
	}

	static final void updateHeaders(final MimePart part) throws MessagingException {
		DataHandler dh = part.getDataHandler();
		if (dh == null) // Huh ?
			return;

		try {
			String type = dh.getContentType();
			boolean composite = false;
			boolean needCTHeader = part.getHeader("Content-Type") == null;

			ContentType cType = new ContentType(type);
			if (cType.match("multipart/*")) {
				// If multipart, recurse
				composite = true;
				Object o = dh.getContent();
				if (o instanceof MimeMultipart)
					((MimeMultipart)o).updateHeaders();
				else
					throw new MessagingException("MIME part of type \"" + type + "\" contains object of type " + o.getClass().getName() + " instead of MimeMultipart");
		    } else if (cType.match("message/rfc822"))
				composite = true;

			// Content-Transfer-Encoding, but only if we don't
			// already have one
			if (!composite) {	// not allowed on composite parts
				if (part.getHeader("Content-Transfer-Encoding") == null)
					setEncoding(part, MimeUtility.getEncoding(dh));

				if (needCTHeader && setDefaultTextCharset && cType.match("text/*") && cType.getParameter("charset") == null) {
					/*
					 * Set a default charset for text parts.
					 * We really should examine the data to determine
					 * whether or not it's all ASCII, but that's too
					 * expensive so we make an assumption:  If we
					 * chose 7bit encoding for this data, it's probably
					 * ASCII.  (MimeUtility.getEncoding will choose
					 * 7bit only in this case, but someone might've
					 * set the Content-Transfer-Encoding header manually.)
					 */
					String charset;
					String enc = part.getEncoding();
					if (enc != null && enc.equalsIgnoreCase("7bit"))
						charset = "us-ascii";
					else
						charset = MimeUtility.getDefaultMIMECharset();
					cType.setParameter("charset", charset);
					type = cType.toString();
				}
			}

			// Now, let's update our own headers ...

			// Content-type, but only if we don't already have one
			if (needCTHeader) {
				/*
				 * Pull out "filename" from Content-Disposition, and
				 * use that to set the "name" parameter. This is to
				 * satisfy older MUAs (DtMail, Roam and probably
				 * a bunch of others).
				 */
				String s = part.getHeader("Content-Disposition", null);
				if (s != null) {
					// Parse the header ..
					ContentDisposition cd = new ContentDisposition(s);
					String filename = cd.getParameter("filename");
					if (filename != null) {
						cType.setParameter("name", filename);
						type = cType.toString();
					}
				}

				part.setHeader("Content-Type", type);
			}
		} catch (IOException ex) {
			throw new MessagingException("IOException updating headers", ex);
		}
	}

	static final void invalidateContentHeaders(final MimePart part) throws MessagingException {
		part.removeHeader("Content-Type");
		part.removeHeader("Content-Transfer-Encoding");
	}

	static final void writeTo(
		final MimePart part,
		OutputStream os,
		final String[] ignoreList)
    	throws IOException, MessagingException {

		// see if we already have a LOS
		LineOutputStream los = null;
		if (os instanceof LineOutputStream)
			los = (LineOutputStream) os;
		else
			los = new LineOutputStream(os);

		// First, write out the header
		Enumeration hdrLines = part.getNonMatchingHeaderLines(ignoreList);
		while (hdrLines.hasMoreElements())
			los.writeln((String)hdrLines.nextElement());

		// The CRLF separator between header and content
		los.writeln();

		// Finally, the content. Encode if required.
		// XXX: May need to account for ESMTP ?
		os = MimeUtility.encode(os, part.getEncoding());
		part.getDataHandler().writeTo(os);
		os.flush(); // Needed to complete encoding
    }

}
