/*
 * Copyright 1997-2004 Sun Microsystems, Inc. All Rights Reserved.
 */

package javax.mail.internet;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectStreamException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.text.ParseException;
import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;

import javax.activation.DataHandler;
import javax.mail.Address;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.util.SharedByteArrayInputStream;

import com.sun.mail.util.ASCIIUtility;
import com.sun.mail.util.LineOutputStream;

/**
 * ̃NX MIME `̓dq[bZ[W\܂B
 *  <code>Message</code> ۃNX <code>MimePart</code> C^tF[X܂B<p>
 *
 * V MIME `bZ[W쐬悤ƂNCAǵA
 *  MimeMessage IuWFNgCX^gAK؂ȑƓeŖ܂B<p> 
 * 
 * MIME ̃obNGhXgAT[rXvoC_́A
 * MimeMessage TuNXA郁\bhI[o[ChāA̎񋟂ł܂B
 * ƂPȃP[X́A炭voC_ MIME `̓̓Xg[𐶐A
 * Xg[̍\͂̃NXɔC鎖łB<p>
 *
 * MimeMessage  <code>InternetHeaders</code> NXgpāA
 * bZ[W̃gbvx RFC 822 wb_\͋yъi[܂B<p>
 *
 * <hr><strong>RFC822 y MIME wb_Ɋւ郁</strong><p>
 *
 * RFC822 wb_tB[h US-ASCII <strong></strong>܂܂Ȃ΂Ȃ܂B
 * MIME ł ASCII ȊO̕GR[fBO邱ƂɂA
 * ̃wb_̓̕ɐݒ\łB
 * RFC 2047 ͂̏̋Kw肵܂B
 * ̃pbP[WŒ񋟂 MimeUtility NXpĂ̏sł܂B
 * <code>setHeader</code>A<code>addHeader</code>A
 * y <code>addHeaderLine</code> \bȟĂяóA
 * w肳ꂽwb_ MIME vɏ]Ƃۏ؂Kv܂B
 * Ãwb_tB[h]ɂs̒ (SMTP ̏ꍇ 1000oCg) 𒴂ꍇ́A
 * M̑Oɐ܂肽 (bv) Kv܂B
 * Mwb_͐܂肽܂Ă\܂B
 * AvP[V̓wb_K؂ɐ܂肽݁AWJӔC܂B
 * 
 * @see javax.mail.internet.MimeUtility
 * @see javax.mail.Part
 * @see javax.mail.Message
 * @see javax.mail.internet.MimePart
 * @see javax.mail.internet.InternetAddress
 */
public class MimeMessage extends Message implements MimePart {

	/**
	 *  Message ̓e\ DataHandler IuWFNgłB
	 */
	protected DataHandler dh;

	/**
	 *  Message ̓ẽoCgێoCgzłB
	 */
	protected byte[] content;

	/**
	 * ̃bZ[W̃f[^ SharedInputStream C^tF[X InputStream ɂ苟ꂽꍇA
	 * <code>contentStream</code> ̃bZ[W̓e\ 1 ̃Xg[łB
	 * ̏ꍇA<code>content</code>  null łB
	 * 
	 * @since JavaMail 1.2
	 */
	protected InputStream contentStream;

	/**
	 * ̃bZ[W̃wb_i[ InternetHeaders IuWFNgłB
	 */
	protected InternetHeaders headers;

	/**
	 * ̃bZ[W Flags łB
	 */
	protected Flags flags;

	/**
	 * bZ[WύXꂽǂtOłB
	 * bZ[WύXĂȂꍇA<code>content</code> z̔Cӂ̃f[^͗LłƂ݂ȂA
	 * <code>writeTo</code> \bhŒڎgp܂B
	 * ̃tÓÃbZ[W쐬鎞A
	 *  <code>saveChanges</code> \bhĂяo鎞 true ɐݒ肳܂B
	 * 
	 * @since JavaMail 1.2
	 */
	protected boolean modified = false;

	/**
	 * ̃bZ[W <code>saveChanges</code> \bhĂяoKvꍇ̃tOłB
	 * ̃tÓApublic RXgN^ɂ false ɐݒ肳A
	 * <code>saveChanges</code> \bhɂ true ɐݒ肳܂B
	 * <code>writeTo</code> \bh͂̃tO`FbNA
	 * Kvɉ <code>saveChanges</code> \bhĂяo܂B
	 * ɂAV\zꂽbZ[WŁA<code>saveChanges</code> \bhĂяoYƂʓIȌ邱Ƃł܂B
	 * 
	 * @since JavaMail 1.2
	 */
	protected boolean saved = false;

	// Used to parse dates
	private static MailDateFormat mailDateFormat = new MailDateFormat();

	// Should addresses in headers be parsed in "strict" mode?
	private boolean strict = true;

	/**
	 * ftHg̃RXgN^łB̃bZ[WIuWFNg쐬܂B
	 * <code>headers</code> tB[h͋ InternetHeaders IuWFNgɐݒ肳܂B
	 * <code>flags</code> tB[h͋ Flags IuWFNgɐݒ肳܂B
	 * <code>modified</code> tO true ɐݒ肳܂B
	 */
	public MimeMessage(final Session session) {
		super(session);
		modified = true;
		headers = new InternetHeaders();
		flags = new Flags();	//  Flags IuWFNg
		initStrict();
	}

	/**
	 * w肳ꂽ MIME InputStream ̃f[^ǂݎA
	 * \͂邱ƂŁAMimeMessage \z܂B
	 * InputStream ̓bZ[Wf[^̍ŌɈʒut܂B<p>
	 * 
	 * ̓Xg[̍\͂́ÃRXgN^̂̓ōs邱ƂɒӂĂB
	 * 
	 * @param session ̃bZ[W Session IuWFNg
	 * @param is bZ[W̓Xg[
	 * @throws MessagingException
	 */
	public MimeMessage(final Session session, final InputStream is) throws MessagingException {
		super(session);
		flags = new Flags(); //  Flags IuWFNg
		initStrict();
		parse(is);
		saved = true;
	}

	/**
	 * <code>source</code> MimeMessage 珉ꂽeV MimeMessage \z܂B
	 * VbZ[W͌̃bZ[WƓƗĂ܂B<p>
	 * 
	 * : ݂̎͂ǂ炩ƂƔłBKvȏɃf[^̃Rs[s܂B
	 * 
	 * @param source ẽRs[ƂȂ郁bZ[W
	 * @throws MessagingException
	 * @since JavaMail 1.2
	 */
	public MimeMessage(final MimeMessage source) throws MessagingException {
		super(source.session);
		flags = source.getFlags();
		ByteArrayOutputStream bos;
		int size = source.getSize();

		if (size > 0)
			bos = new ByteArrayOutputStream(size);
		else
			bos = new ByteArrayOutputStream();

		try {
			strict = source.strict;
			source.writeTo(bos);
			bos.close();
			SharedByteArrayInputStream bis = new SharedByteArrayInputStream(bos.toByteArray());
			parse(bis);
			bis.close();
			saved = true;
		} catch (IOException ex) {
			// should never happen, but just in case...
			throw new MessagingException("IOException while copying message", ex);
		}
	}

	/**
	 * w肳ꂽ Folder ƃbZ[Wԍ MimeMessage IuWFNg\z܂B<p>
	 * 
	 * ̃\bh <code>MimeMessage</code> TuNXvoC_łB
	 */
	protected MimeMessage(final Folder folder, final int msgnum) {
		super(folder, msgnum);
		flags = new Flags();  //  Flags IuWFNg
		saved = true;
		initStrict();
	}

	/**
	 * w肳ꂽ MIME InputStream ̃f[^ǂݎA
	 * \͂邱ƂŁAMimeMessage \z܂B
	 * InputStream ̓bZ[Wf[^̍ŌɈʒut܂B
	 * ̓Xg[̍\͂́ÃRXgN^̂̓ōs邱ƂɒӂĂB<p>
	 * 
	 * ̃\bh <code>MimeMessage</code> TuNXvoC_łB
	 * 
	 * @param folder ܂tH_
	 * @param is bZ[W̓Xg[
	 * @param msgnum tH_̂̃bZ[W̃bZ[Wԍ
	 * @throws MessagingException
	 */
	protected MimeMessage(final Folder folder, final InputStream is, final int msgnum) throws MessagingException {
		this(folder, msgnum);
		initStrict();
		parse(is);
	}

	/**
	 * w肳ꂽ InternetHeaders IuWFNgѓe MimeMessage \z܂B
	 * 
	 * ̃\bh <code>MimeMessage</code> TuNXvoC_łB
	 * 
	 * @param folder ܂tH_
	 * @param headers bZ[W̓Xg[
	 * @param content The message content
	 * @param msgnum tH_̂̃bZ[W̃bZ[Wԍ
	 * @throws MessagingException
	 */
	protected MimeMessage(
		final Folder folder,
		final InternetHeaders headers,
		final byte[] content,
		final int msgnum) /* throws MessagingException */ {

		this(folder, msgnum);
		this.headers = headers;
		this.content = content;
		initStrict();
	}

	/**
	 * Set the strict flag based on property.
	 */
	private void initStrict() {
		if (session != null) {
			String s = session.getProperty("mail.mime.address.strict");
			strict = s == null || !s.equalsIgnoreCase("false");
		}
	}

	/**
	 * <code>headers</code> y <code>content</code> tB[hK؂ɐݒ肷 InputStream \͂܂B
	 * A<code>modified</code> tOZbg܂B<p>
	 * 
	 * ̃\bh́AInputStream ̍\͎ɐ䂷KvTuNXɂgp܂B
	 * 
	 * @param is bZ[W̓Xg[
	 * @throws MessagingException
	 */
	protected final void parse(InputStream is) throws MessagingException {
		if (!(is instanceof ByteArrayInputStream) &&
			!(is instanceof BufferedInputStream) &&
			!(is instanceof SharedInputStream))
				is = new BufferedInputStream(is);

		headers = createInternetHeaders(is);

		if (is instanceof SharedInputStream) {
			SharedInputStream sis = (SharedInputStream)is;
			contentStream = sis.newStream(sis.getPosition(), -1);
		} else {
			try {
				content = ASCIIUtility.getBytes(is);
			} catch (IOException ioex) {
				throw new MessagingException("IOException", ioex);
			}
		}

		modified = false;
    }

	/**
	 * RFC 822 "From" wb_tB[h̒lԂ܂B
	 * ̃wb_tB[h݂Ȃꍇ́A"Sender" wb_tB[hgp܂B
	 * "Sender" wb_tB[h݂Ȃꍇ́A<code>null</code> Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader</code> \bhgpAKvȃwb_tB[h擾܂B
	 * 
	 * @return Address IuWFNg
	 * @throws MessagingException
	 * @see #headers
	 */
	public Address[] getFrom() throws MessagingException {
		Address[] a = getAddressHeader("From");
		if (a == null)
			a = getAddressHeader("Sender");
		return a;
	}

	/**
	 * RFC 822 "From" wb_tB[hݒ肵܂B
	 * SĂ̊̒l͎w肳ꂽAhXŒu܂B
	 * address  <code>null</code> ̏ꍇ́Ãwb_폜܂B
	 * 
	 * @param address ̃bZ[W̑M
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void setFrom(final Address address) throws MessagingException {
		if (address == null)
			removeHeader("From");
		else
			setHeader("From", address.toString());
	}

	/**
	 * <code>InternetAddress.getLocalAddress</code> \bh̒lgpāA
	 * RFC 822 "Form" wb_tB[hݒ肵܂B
	 * 
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public final void setFrom() throws MessagingException {
		InternetAddress me = InternetAddress.getLocalAddress(session);
		if (me != null)
			setFrom(me);
		else
			throw new MessagingException("No From address");
	}

	/**
	 * w肳ꂽAhX "From" tB[hɒǉ܂B
	 * "From" tB[h݂Ȃꍇ͍쐬܂B
	 * 
	 * @param addresses ̃bZ[W̑M
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void addFrom(final Address[] addresses) throws MessagingException {
		addAddressHeader("From", addresses);
	}

	/** 
	 * Returns the value of the RFC 822 "Sender" header field.
	 * If the "Sender" header field is absent, <code>null</code>
	 * is returned.<p>
	 * 
	 * This implementation uses the <code>getHeader</code> method
	 * to obtain the requisite header field.
	 * 
	 * @return Address object
	 * @throws MessagingException
	 * @since JavaMail 1.3
	 * @see #headers
	 */
	public Address getSender() throws MessagingException {
		Address[] a = getAddressHeader("Sender");
		if (a == null)
			return null;
		return a[0];	// there can be only one
	}

	/**
	 * Set the RFC 822 "Sender" header field. Any existing values are 
	 * replaced with the given address. If address is <code>null</code>,
	 * this header is removed.
	 * 
	 * @param address the sender of this message
	 * @throws IllegalWriteException if the underlying
	 *			implementation does not support modification
	 *			of existing values
	 * @throws IllegalStateException if this message is obtained from a READ_ONLY folder.
	 * @throws MessagingException
	 * @since JavaMail 1.3
	 */
	public void setSender(final Address address) throws MessagingException {
		if (address == null)
			removeHeader("Sender");
		else
			setHeader("Sender", address.toString());
	}

	/**
	 * ̓NX javax.mail.Message.RecipientType NXgāA
	 * ⑫I RecipientType ǉ܂B
	 * ݂Œ`Ă 1 ̕⑫I RecipientType  NEWSGROUPS łB
	 * 
	 * @see javax.mail.Message.RecipientType
	 */
	public static final class RecipientType extends Message.RecipientType {

		private static final long serialVersionUID = 1L;

		/**
		 * "Newsgroup" (Usenet j[X) M҂łB
		 */
		public static final RecipientType NEWSGROUPS = new RecipientType("Newsgroups");

		protected RecipientType(final String type) {
			super(type);
		}

		protected Object readResolve() throws ObjectStreamException {
			if (type.equals("Newsgroups"))
				return NEWSGROUPS;
			return super.readResolve();
		}

	}

	/**
	 * ^ɂw肳ꂽM҂Ԃ܂B
	 * ^ƑΉ RFC 822 wb_Ƃ̊Ԃ̃}bsO͎̒ʂłB
	 * <pre>
	 *		Message.RecipientType.TO		"To"
	 *		Message.RecipientType.CC		"Cc"
	 *		Message.RecipientType.BCC		"Bcc"
	 *		MimeMessage.RecipientType.NEWSGROUPS	"Newsgroups"
	 * </pre><br>
	 * 
	 * ^ɂw肳wb_ȂA܂͂̒l̏ꍇ́Anull Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader</code> \bhgpAKvȃwb_tB[h擾܂B
	 * 
	 * @param type M҂̌^
	 * @return Address IuWFNg̔z
	 * @throws MessagingException wb_擾łȂꍇ
	 * @throws AddressException wb_̌`Ɍ肪ꍇ
	 * @see #headers
	 * @see javax.mail.Message.RecipientType#TO
	 * @see javax.mail.Message.RecipientType#CC
	 * @see javax.mail.Message.RecipientType#BCC
	 * @see javax.mail.internet.MimeMessage.RecipientType#NEWSGROUPS
	 */
	public Address[] getRecipients(final Message.RecipientType type) throws MessagingException {
		if (type == RecipientType.NEWSGROUPS) {
			String s = getHeader("Newsgroups", ",");
			return (s == null) ? null : NewsAddress.parse(s);
		}
		return getAddressHeader(getHeaderName(type));
	}

	/**
	 * bZ[W̑SĂ̎M҃AhX擾܂B
	 * TOACCABCCA NEWSGROUPS ̎M҂𒊏o܂B
	 * 
	 * @return Address IuWFNg̔z
	 * @throws MessagingException
	 * @see javax.mail.Message.RecipientType#TO
	 * @see javax.mail.Message.RecipientType#CC
	 * @see javax.mail.Message.RecipientType#BCC
	 * @see javax.mail.internet.MimeMessage.RecipientType#NEWSGROUPS
	 */
	public final Address[] getAllRecipients() throws MessagingException {
		Address[] all = super.getAllRecipients();
		Address[] ng = getRecipients(RecipientType.NEWSGROUPS);

		if (ng == null)
			return all;		// the common case

		int numRecip = (all != null ? all.length : 0) + (ng != null ? ng.length : 0);
		Address[] addresses = new Address[numRecip];
		int pos = 0;

		if (all != null) {
			System.arraycopy(all, 0, addresses, pos, all.length);
			pos += all.length;
		}
		if (ng != null) {
			System.arraycopy(ng, 0, addresses, pos, ng.length);
			pos += ng.length;
		}

		return addresses;
    }

	/**
	 * w肳ꂽM҂̃^Cvw肳ꂽAhXɐݒ肵܂B
	 * AhXp[^ <code>null</code> ̏ꍇ́AΉM҃tB[h폜܂B
	 * 
	 * @param type M҃^Cv
	 * @param addresses AhX
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 * @see #getRecipients
	 */
	public void setRecipients(final Message.RecipientType type, final Address[] addresses) throws MessagingException {
		if (type == RecipientType.NEWSGROUPS) {
			if (addresses == null || addresses.length == 0)
				removeHeader("Newsgroups");
			else
				setHeader("Newsgroups", NewsAddress.toString(addresses));
		} else
			setAddressHeader(getHeaderName(type), addresses);
	}

	/**
	 * w肳ꂽM҂̃^Cvw肳ꂽAhXɐݒ肵܂B
	 * AhXp[^ <code>null</code> ̏ꍇ́AΉM҃tB[h폜܂B
	 * 
	 * @param type M҃^Cv
	 * @param addresses AhX
	 * @throws AddressException AhX̍\͂Ɏsꍇ
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 * @since JavaMail 1.2
	 * @see #getRecipients
	 */
	public final void setRecipients(final Message.RecipientType type, final String addresses) throws MessagingException {
		if (type == RecipientType.NEWSGROUPS) {
			if (addresses == null || addresses.length() == 0)
				removeHeader("Newsgroups");
			else
				setHeader("Newsgroups", addresses);
		} else
			setAddressHeader(getHeaderName(type), InternetAddress.parse(addresses));
	}

	/**
	 * w肳ꂽAhXw肳ꂽM҃^Cvɒǉ܂B
	 * 
	 * @param type M҃^Cv
	 * @param addresses AhX
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void addRecipients(final Message.RecipientType type, final Address[] addresses) throws MessagingException {
		if (type == RecipientType.NEWSGROUPS) {
			String s = NewsAddress.toString(addresses);
			if (s != null)
				addHeader("Newsgroups", s);
		} else
			addAddressHeader(getHeaderName(type), addresses);
	}

	/**
	 * w肳ꂽAhXw肳ꂽM҃^Cvɒǉ܂B
	 * 
	 * @param type M҃^Cv
	 * @param addresses AhX
	 * @throws AddressException AhX̍\͂Ɏsꍇ
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 * @since JavaMail 1.2
	 */
	public final void addRecipients(final Message.RecipientType type, final String addresses) throws MessagingException {
		if (type == RecipientType.NEWSGROUPS) {
			if (addresses != null && addresses.length() != 0)
				addHeader("Newsgroups", addresses);
		} else
			addAddressHeader(getHeaderName(type), InternetAddress.parse(addresses));
	}

	/**
	 * RFC 822 "Reply-To" wb_tB[h̒lԂ܂B
	 * ̃wb_gpłȂA͂̒l݂Ȃꍇ́A
	 * <code>getFrom</code> \bhĂяoA̒lԂ܂B
	 * 
	 * ̎ <code>getHeader</code> \bhgpAKvȃwb_tB[h擾܂B
	 * 
	 * @throws MessagingException
	 * @see #headers
	 */
	public Address[] getReplyTo() throws MessagingException {
		Address[] a = getAddressHeader("Reply-To");
		if (a == null)
			a = getFrom();
		return a;
	}

	/**
	 * RFC 822 "Reply-To" wb_tB[hݒ肵܂B
	 * addresses p[^ <code>null</code> ̏ꍇ́Ãwb_폜܂B
	 * 
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void setReplyTo(final Address[] addresses) throws MessagingException {
		setAddressHeader("Reply-To", addresses);
	}

	// Convenience method to get addresses
	private Address[] getAddressHeader(final String name) throws MessagingException {
		String s = getHeader(name, ",");
		return (s == null) ? null : InternetAddress.parseHeader(s, strict);
	}

	// Convenience method to set addresses
	private void setAddressHeader(final String name, final Address[] addresses) throws MessagingException {
		String s = InternetAddress.toString(addresses);
		if (s == null)
			removeHeader(name);
		else
			setHeader(name, s);
	}

	private void addAddressHeader(final String name, final Address[] addresses) throws MessagingException {
		String s = InternetAddress.toString(addresses);
		if (s == null)
			return;
		addHeader(name, s);
	}

	/**
	 * "Subject" wb_tB[h̒lԂ܂B
	 * tB[hgpłȂꍇA͂̒l݂Ȃꍇ null Ԃ܂B<p>
	 * 
	 *  RFC 2047 ɏ]ăGR[hĂꍇAfR[hAUnicode ɕϊ܂B
	 * fR[fBO܂͕ϊsꍇAf[^̂܂ܕԂ܂B<p>
	 * 
	 * ̎ <code>getHeader</code> \bhgpAKvȃwb_tB[h擾܂B
	 * 
	 * @return 
	 * @throws MessagingException
	 * @see #headers
	 */
	public String getSubject() throws MessagingException {
		String rawvalue = getHeader("Subject", null);
		if (rawvalue == null)
			return null;

		try {
			return MimeUtility.decodeText(MimeUtility.unfold(rawvalue));
		} catch (UnsupportedEncodingException ex) {
			return rawvalue;
		}
	}

	/**
	 * "Subject" wb_tB[hݒ肵܂B
	 *  US-ASCII ȊO̕܂܂ꍇ́A
	 * vbgtH[̃ftHgZbggpăGR[h܂B
	 *  US-ASCII Zbg܂܂ꍇ́A
	 * GR[fBOsꂸÂ܂܎gp܂B
	 *  null ̏ꍇ́A "Subject" tB[h폜܂B<p>
	 * 
	 * ZbgGR[fBOsꍇAMessagingException X[A
	 * UnsupportedEncodingException  MessagingException ̓qɂꂽÕ`F[Ɋ܂܂邱ƂɒӂĉB
	 * 
	 * @param subject 
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException ZbgϊsƁAUnsupportedEncodingException O`F[Ɋ܂܂ꍇ܂
	 */
	public final void setSubject(final String subject) throws MessagingException {
		setSubject(subject, null);
	}

	/**
	 * "Subject" wb_tB[hݒ肵܂B
	 *  US-ASCII ȊO̕܂܂ꍇ́A
	 * w肳ꂽZbggpăGR[h܂B
	 *  US-ASCII Zbg܂܂ꍇ́A
	 * GR[fBOsꂸÂ܂܎gp܂B
	 *  null ̏ꍇ́A "Subject" tB[h폜܂B<p>
	 * 
	 * ZbgGR[fBOsꍇAMessagingException X[A
	 * UnsupportedEncodingException  MessagingException ̓qɂꂽÕ`F[Ɋ܂܂邱ƂɒӂĉB
	 * 
	 * @param subject 
	 * @param charset Zbg
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException ZbgϊsƁAUnsupportedEncodingException O`F[Ɋ܂܂ꍇ܂
	 */
	public void setSubject(final String subject, final String charset) throws MessagingException {
		if (subject == null)
			removeHeader("Subject");

		try {
			setHeader("Subject", MimeUtility.fold(9, MimeUtility.encodeText(subject, charset, null)));
		} catch (UnsupportedEncodingException uex) {
			throw new MessagingException("Encoding error", uex);
		}
    }

	/**
	 * RFC 822 "Date" tB[h̒lԂ܂B
	 * ͂̃bZ[WMꂽtłB
	 * ̃tB[hgpłȂA܂͂̒l݂Ȃꍇ null Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader</code> \bhgpAKvȃwb_tB[h擾܂B
	 * 
	 * @return M
	 * @throws MessagingException
	 */
	public Date getSentDate() throws MessagingException {
		String s = getHeader("Date", null);
		if (s != null) {
// Sugisawa added. 2004/03/09 JST ^C][w肵ĂAz MUA ւ̑ΉR[hǉ
// JST w肳Ă +̎w肪Ȃꍇ
int i = s.indexOf(" JST");
if (i > 0 && s.indexOf('+') == -1) {
	// +0900 
	s = s.substring(0, i) + " +0900";
}

			try {
				synchronized (mailDateFormat) {
					return mailDateFormat.parse(s);
				}
			} catch (ParseException pex) {
				return null;
			}
		}

		return null;
	}

	/**
	 * RFC 822 "Date" wb_tB[hݒ肵܂B
	 * ̓bZ[W̍쐬҂bZ[W̊ƔzMtłB
	 * ̓tp[^ <code>null</code> ̏ꍇ͊ "Date" tB[h폜܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void setSentDate(final Date d) throws MessagingException {
		if (d == null)
			removeHeader("Date");
		else {
			synchronized (mailDateFormat) {
				setHeader("Date", mailDateFormat.format(d));
			}
		}
	}

	/**
	 * ̃bZ[WMꂽԂ܂B
	 * ̓t擾łȂꍇ <code>null</code> Ԃ܂B<p>
	 * 
	 * RFC 822 ͎M̃tB[h`ȂƂɒӂĂB
	 * ׁ̈A̓t񋟉\Ȏ݂̂LlԂKv܂B<p>
	 * 
	 * ̎ <code>null</code> Ԃ܂B
	 * 
	 * @return ̃bZ[WMꂽ
	 * @throws MessagingException
	 */
	public Date getReceivedDate() throws MessagingException {
		return null;
	}

	/**
	 * ̃bZ[W̓eTCYoCgPʂŕԂ܂B
	 * TCYʂłȂꍇ -1 Ԃ܂B<p>
	 * 
	 * ̐l͓eTCY̐mȑlłȂ\A
	 * e̓]GR[fBOƂłȂƂ_ɒӂĂB<p>
	 * 
	 * ̎́A<code>content</code> z̃TCY (null łȂꍇ)A
	 *  <code>contentStream</code>  null łȂA
	 * <code>available</code> \bh̐ԂꍇA̐TCYƂĕԂ܂B
	 * łȂꍇ -1 Ԃ܂B
	 * 
	 * @return eoCg̃TCY
	 * @throws MessagingException
	 */
	public int getSize() throws MessagingException {
		if (content != null)
			return content.length;
		if (contentStream != null) {
			try {
				int size = contentStream.available();
				// only believe the size if it's greater than zero, since zero
				// is the default returned by the InputStream class itself
				if (size > 0)
					return size;
			} catch (IOException ex) {}	// 
		}
		return -1;
	}

	/**
	 * ̃bZ[W̓e̍sԂ܂B̐lʂłȂꍇ -1 Ԃ܂B<p>
	 * 
	 * ̐l͓e̒̐mȑlłȂ\A
	 * e̓]GR[fBOƂłȂƂ_ɒӂĉB<p>
	 * 
	 * ̎ -1 Ԃ܂B
	 * 
	 * @return Rec̍s
	 * @throws MessagingException
	 */  
	public int getLineCount() throws MessagingException {
		return -1;
	}

	/**
	 * RFC 822 "Content-Type" wb_tB[h̒lԂ܂B
	 * ́ÃbZ[W̓ẽRec`\܂B
	 * ̒l null łĂ͂Ȃ܂B
	 * ̃tB[hgp\łȂꍇA"text/plain" Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader</code> \bhgpAKvȃwb_tB[h擾܂B
	 * 
	 * @return ̃̕Rec`
	 * @throws MessagingException
	 * @see javax.activation.DataHandler
	 */
	public String getContentType() throws MessagingException {
		String s = getHeader("Content-Type", null);
		if (s == null)
			return "text/plain";

// Sugisawa added. 2005/03/04
if (s.equalsIgnoreCase("text"))
	s = "text/plain";

		return s;
	}

	/**
	 * w肳ꂽ MIME ^Cṽp[gǂʂ܂B
	 * ̃\bh́A<strong><code>primaryType</code>  <code>subType</code> ̂</strong>r܂B
	 * Rec`̃p[^͖܂B<p>
	 * 
	 * Ƃ΁ARec` <strong>"text/plain"</strong> ̃p[g <strong>"text/plain; charset=foobar"</strong> ƔrƁA
	 * ̃\bh <code>true</code> Ԃ܂B<p>
	 * 
	 * <code>mimeType</code>  <code>subType</code> ꕶ '*' ̏ꍇA
	 * rɃTu^Cv͖܂B
	 */
	public final boolean isMimeType(final String mimeType) throws MessagingException {
		return MimeBodyPart.isMimeType(this, mimeType);
	}

	/**
	 * "Content-Disposition" wb_tB[h̒lԂ܂B
	 * ͂̃p[g̃fBX|WV\܂B
	 * ̃p[gǂ̂悤Ƀ[Uɒ񎦂邩܂B<p>
	 * 
	 * Content-Disposition tB[hgpłȂꍇ <code>null</code> Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader</code> \bhgpAKvȃwb_tB[h擾܂B
	 * 
	 * @return ̃p[g̃fBX|WVAs̏ꍇ null
	 * @throws MessagingException
	 */
	public String getDisposition() throws MessagingException {
		return MimeBodyPart.getDisposition(this);
	}

	/**
	 * ̃bZ[W "Content-Disposition" wb_tB[hݒ肵܂B
	 * <code>disposition</code>  null ̏ꍇ́A
	 * ̑SĂ "Content-Disposition" wb_tB[h폜܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void setDisposition(final String disposition) throws MessagingException {
		MimeBodyPart.setDisposition(this, disposition);
	}

	/**
	 * "Content-Transfer-Encoding" wb_tB[h̓]eGR[fBOԂ܂B
	 * wb_gpłȂꍇA͂̒l݂Ȃꍇ <code>null</code> Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader</code> \bhgpAKvȃwb_tB[h擾܂B
	 * 
	 * @return ]eGR[fBO
	 * @throws MessagingException
	 */
	public String getEncoding() throws MessagingException {
		return MimeBodyPart.getEncoding(this);
	}

	/**
	 * "Content-ID" wb_tB[h̒lԂ܂B
	 * tB[hgpłȂꍇA͂͂̒l݂Ȃꍇ <code>null</code> Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader</code> \bhgpAKvȃwb_tB[h擾܂B
	 * 
	 * @return Rec ID
	 * @throws MessagingException
	 */
	public String getContentID() throws MessagingException {
		return getHeader("Content-Id", null);
	}

	/**
	 * ̃bZ[W "Content-ID" wb_tB[hݒ肵܂B
	 * <code>cid</code> p[^ null ̏ꍇ́A
	 * ̑SĂ "Content-ID" 폜܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void setContentID(final String cid) throws MessagingException {
		if (cid == null)
			removeHeader("Content-ID");
		else
			setHeader("Content-ID", cid);
	}

	/**
	 * "Content-MD5" wb_tB[h̒lԂ܂B
	 * tB[hgpłȂꍇA͂̒l݂Ȃꍇ <code>null</code> Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader</code> \bhgpAKvȃwb_tB[h擾܂B
	 * 
	 * @return Rec MD5
	 * @throws MessagingException
	 */
	public String getContentMD5() throws MessagingException {
		return getHeader("Content-MD5", null);
	}

	/**
	 * ̃bZ[W "Content-MD5" wb_tB[hݒ肵܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void setContentMD5(final String md5) throws MessagingException {
		setHeader("Content-MD5", md5);
	}

	/**
	 * ̃bZ[W "Content-Description" wb_tB[hԂ܂B
	 * ͒ʏAIȏ̃p[gɊ֘At܂B
	 * ̃tB[hgpłȂꍇA͂̒l݂Ȃꍇ null Ԃ܂B<p>
	 * 
	 * Content-Description tB[h RFC 2047 ɏ]ăGR[hĂꍇA
	 * fR[hAUnicode ɕϊ܂B
	 * fR[fBO͕ϊsꍇAf[^̂܂ܕԂ܂B<p>
	 * 
	 * ̎ <code>getHeader</code> \bhgpAKvȃwb_tB[h擾܂B
	 * 
	 * @return content-description
	 * @throws MessagingException
	 */
	public String getDescription() throws MessagingException {
		return MimeBodyPart.getDescription(this);
	}

	/**
	 * ̃bZ[W "Content-Description" wb_tB[hݒ肵܂B
	 * description p[^ <code>null</code> ̏ꍇ́A
	 * ̑SĂ "Content-Description" tB[h폜܂B<p>
	 * 
	 * description  US-ASCII ȊO̕܂܂ꍇ́A
	 * vbgtH[̃ftHgZbggpăGR[h܂B
	 * description  US-ASCII ܂܂ꍇ́A
	 * GR[fBO͍sꂸÂ܂܎gp܂B<p>
	 * 
	 * ZbgGR[fBOsꍇA
	 * MessagingException X[A
	 * UnsupportedEncodingException  MessagingException ̓qɂꂽÕ`F[Ɋ܂܂邱ƂɒӂĉB
	 * 
	 * @param description content-description
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException ZbgϊsƁAUnsupportedEncodingException O`F[Ɋ܂܂ꍇ܂
	 */
	public final void setDescription(final String description) throws MessagingException {
		setDescription(description, null);
	}

	/**
	 * ̃bZ[W "Content-Description" wb_tB[hݒ肵܂B
	 * description p[^ <code>null</code> ̏ꍇ́A
	 * ̑SĂ "Content-Description" tB[h폜܂B<p>
	 * 
	 * description  US-ASCII ȊO̕܂܂ꍇ́A
	 * w肳ꂽZbggpăGR[h܂B
	 * description  US-ASCII ܂܂ꍇ́A
	 * GR[fBO͍sꂸÂ܂܎gp܂B<p>
	 * 
	 * ZbgGR[fBOsꍇA
	 * MessagingException X[A
	 * UnsupportedEncodingException  MessagingException ̓qɂꂽÕ`F[Ɋ܂܂邱ƂɒӂĉB
	 * 
	 * @param description Description
	 * @param charset LN^GR[fBO
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException ZbgϊsƁAUnsupportedEncodingException O`F[Ɋ܂܂ꍇ܂
	 */
	public void setDescription(final String description, final String charset) throws MessagingException {
		MimeBodyPart.setDescription(this, description, charset);
	}

	/**
	 * ̃bZ[W "Content-Language" wb_tB[hŎw肳錾擾܂B
	 * Content-Language wb_ RFC 1766 ɂ`܂B
	 * ̃tB[hgpłȂꍇA͂̒l݂Ȃꍇ <code>null</code> Ԃ܂B<p>
	 * 
	 * ̎ <code>getHeader</code> \bhgpAKvȃwb_tB[h擾܂B
	 * 
	 * @return Content-Language wb_̒l
	 * @throws MessagingException
	 */
	public final String[] getContentLanguage() throws MessagingException {
		return MimeBodyPart.getContentLanguage(this);
	}

	/**
	 *  MimePart  "Content-Language" wb_ݒ肵܂B
	 * Content-Language wb_ RFC 1766 ɂ`܂B
	 * 
	 * @param languages ^O̔z
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void setContentLanguage(final String[] languages) throws MessagingException {
		MimeBodyPart.setContentLanguage(this, languages);
	}

	/**
	 * "Message-ID" wb_tB[h̒lԂ܂B
	 * ̃tB[hgpłȂꍇA͂̒l݂Ȃꍇ null Ԃ܂B<p>
	 * 
	 * Œ񋟂ftHg̎ <code>getHeader</code> \bhgpA
	 * "Message-ID" tB[h̒lԂ܂B
	 * 
	 * @return Message-ID wb_̒l
	 * @throws MessagingException ̃tB[h̎擾ɂOꍇ
	 * @since JavaMail 1.1
	 * @see javax.mail.search.MessageIDTerm
	 */
	public String getMessageID() throws MessagingException {
		return getHeader("Message-ID", null);
	}

	/**
	 * ̃bZ[WɊ֘Atꂽt@C擾܂B<p>
	 * 
	 * ̃bZ[W "Content-Disposition" wb_tB[hɂ "filename" p[^̒lԂ܂B
	 * ̃p[^gpłȂꍇ́A BodyPart  "Content-Type" wb_tB[hɂ "name" p[^̒lԂ܂B
	 * Ƃ݂Ȃꍇ <code>null</code> Ԃ܂B
	 * 
	 * @return t@C
	 * @throws MessagingException ̃tB[h̎擾ɂOꍇ
	 */
	public String getFileName() throws MessagingException {
		return MimeBodyPart.getFileName(this);
	}

	/**
	 * \ł΁A̕Ɋ֘Atꂽt@Cݒ肵܂B<p>
	 * 
	 * ̃bZ[W "Content-Disposition" wb_tB[hɂ "filename" p[^ݒ肵܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void setFileName(final String filename) throws MessagingException {
		MimeBodyPart.setFileName(this, filename);	
	}

	private String getHeaderName(final Message.RecipientType type) throws MessagingException {
		String headerName;

		if (type == Message.RecipientType.TO)
			headerName = "To";
		else if (type == Message.RecipientType.CC)
			headerName = "Cc";
		else if (type == Message.RecipientType.BCC)
			headerName = "Bcc";
		else if (type == MimeMessage.RecipientType.NEWSGROUPS)
			headerName = "Newsgroups";
		else
			throw new MessagingException("Invalid Recipient Type");

		return headerName;
	}

	/**
	 * ̃bZ[W "content" ɑ΂fR[hꂽ̓Xg[Ԃ܂B<p>
	 * 
	 * ̎ DataHandler ̓Xg[擾܂B
	 * ܂A<code>getDataHandler().getInputStream()</code> Ăяo܂B
	 * 
	 * @return ̓Xg[
	 * @throws MessagingException
	 * @throws IOException ͒ʏ DataHandler ɂX[B
	 * 			ڍׂɂĂ javax.activation.DataHandler ̃}jAQƂĉB
	 * @see #getContentStream
	 * @see javax.activation.DataHandler#getInputStream
	 */
	public final InputStream getInputStream() throws IOException, MessagingException {
		return getDataHandler().getInputStream();
	}

	/**
	 * e̐oCg𐶐܂B
	 * ̃\bh́A\͎Ae DataHandler IuWFNg쐬ׂɎgp܂B
	 * bZ[W̓e̕ʌ̓̓Xg[񋟉\ȃTuNX́A
	 * ̃\bhI[o[Chꍇ܂B<p>
	 * 
	 * ̎́A<code>content</code> oCgz񂩂\z ByteArrayInputStream ԂłB
	 * 
	 * @see #content
	 */
	protected InputStream getContentStream() throws MessagingException {
		if (contentStream != null)
			return ((SharedInputStream)contentStream).newStream(0, -1);
		if (content != null)
			return new SharedByteArrayInputStream(content);

		throw new MessagingException("No content");
	}

	/**
	 * Cӂ Content-Transfer-Encoding ̂܂܂ɂāA
	 * f[^ւ InputStream Ԃ܂B
	 * ̃\bh "Content-Transfer-Encoding" wb_ȂĂ肷ꍇɖ𗧂܂B
	 * ̏ꍇA<code>getInputStream</code> \bh܂ <code>getContent</code> \bh͐f[^Ԃo܂B
	 * AvP[V͂̃\bhgpAf[^̃̕fR[h݂鎖o܂B<p>
	 * 
	 * ̎͒P <code>getContentStream</code> \bhĂяo܂B
	 * 
	 * @since JavaMail 1.2
	 * @see #getInputStream
	 * @see #getContentStream
	 */
	public final InputStream getRawInputStream() throws MessagingException {
		return getContentStream();
	}

	/**
	 * ̃bZ[W̓e DataHandler Ԃ܂B<p>
	 * 
	 * Œ񋟂͎̂悤ɋ@\܂B
	 * <code>getContentStream</code> \bhgpēẽoCgXg[𐶐鎖ɒӂĉB
	 * ASĂ̓]fR[fBÕ\bh̓ŎIɎs鎖ɂӂĉB<p>
	 * 
	 * <blockquote><pre>
	 *  getDataHandler() {
	 *      if (dh == null) {
	 *          dh = new DataHandler(new MimePartDataSource(this));
	 *      }
	 *      return dh;
	 *  }
	 *  <p>
	 *  class MimePartDataSource implements DataSource {
	 *      public getInputStream() {
	 *          return MimeUtility.decode(
	 *		     getContentStream(), getEncoding());
	 *      }
	 *	
	 *		.... <̑ DataSource \bh>
	 *  }
	 * </pre></blockquote><p>
	 * 
	 * @throws MessagingException
	 */
	public synchronized DataHandler getDataHandler() throws MessagingException {
		if (dh == null)
			dh = new DataHandler(new MimePartDataSource(this));
		return dh;
	}

	/**
	 * e Java IuWFNgƂĕԂ܂B
	 * ̃IuWFNg̃^Cv͓e̕Ɉˑ܂B
	 * Ⴆ΁A"text/plain" RečŗL`͕ String IuWFNgłB
	 * "multipart" bZ[W̌ŗL`͏ Multipart TuNXłB
	 * sȃRec`̏ꍇ DataHandler VXeɁA
	 * ̓Xg[RecƂĕԂ܂B<p>
	 * 
	 * ̎ DataHandler Rec擾܂B
	 * ܂A<code>getDataHandler().getContent()</code> Ăяo܂B
	 * 
	 * @return IuWFNg
	 * @throws MessagingException
	 * @throws IOException ͒ʏ DataHandler ɂX[܂B
	 * 			ڍׂɂĂ javax.activation.DataHandler ̃}jAQƂĉB
	 * @see javax.mail.Part
	 * @see javax.activation.DataHandler#getContent
	 */
	public final Object getContent() throws IOException, MessagingException {
		Object obj = getDataHandler().getContent();
		if (MimeBodyPart.cacheMultipart && ((obj instanceof Multipart) || (obj instanceof Message)) && !(dh instanceof CachedDataHandler) && (content != null || contentStream != null))
			dh = MimeBodyPart.createCachedDataHandler(obj, getContentType());
		return obj;
	}

	/**
	 * ̃\bh͂̃p[g̓eݒ肷邽߂̎dg݂񋟂܂B
	 * w肳ꂽ DataHandler IuWFNg͎ۂ̓ebvȂ΂Ȃ܂B
	 * 
	 * @param dh Rec DataHandler
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public synchronized void setDataHandler(final DataHandler dh) throws MessagingException {
		this.dh = dh;
		MimeBodyPart.invalidateContentHeaders(this);
	}

	/**
	 * ̃bZ[W̃Recݒ肷邽߂̊ȈՃ\bhłB<p>
	 * 
	 * Rec DataHandler IuWFNgɃbv܂B
	 * ꂪ@\ׂɁA
	 * w肳ꂽ^Cv DataContentHandler NX JavaMail ɑ΂Ďgp\łȂ΂ȂȂɒӂĉB
	 * ܂A<code>setContent(foobar, "application/x-foobar")</code> sׂɁA
	 * "application/x-foobar"  DataContentHandler CXg[ĂKv܂B
	 * ڍׂɂĂ Java Activation Framework QƂĉB
	 * 
	 * @param o RecIuWFNg
	 * @param type IuWFNg MIME ^Cv
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public final void setContent(final Object o, final String type) throws MessagingException {
		setDataHandler(new DataHandler(o, type));
	}

	/**
	 * w肳ꂽ̃p[g̃RecƂāA
	 * "text/plain"  MIME ^CvŐݒ肷ȈՃ\bhłB
	 *  US-ASCII ȊO̕܂܂ꍇA
	 * vbgtH[̃ftHg̕ZbggpăGR[h܂B
	 * ̕Zbg "charset" p[^ݒ肷ׂɂgp܂B<p>
	 * 
	 * ̃\bh͎gp镶Zbg߂ׂɑSĂ̕𑖍Ȃ΂ȂȂꍇ̂ŁA
	 * <code>text</code> 傫ꍇ͐\ቺ\鎖ɒӂĉB<p>
	 * 
	 * Zbgm̏ꍇAZbgp[^󂯎 <code>setText()</code> gpĉB
	 * 
	 * @see #setText(String text, String charset)
	 */
	public final void setText(final String text) throws MessagingException {
		setText(text, null);
	}

	/**
	 * "text/plain"  MIME ^Cvyюw肳ꂽZbggpA
	 * w肳ꂽ̃p[g̃RecƂĐݒ肷ȈՃ\bhłB
	 * w肳ꂽ Unicode ́Aw肳ꂽZbggpĕZbgGR[h܂B
	 * ̕Zbg "charset" p[^ݒ肷ׂɂgp܂B
	 */
	public final void setText(final String text, final String charset) throws MessagingException {
// Sugisawa chenged.
//		MimeBodyPart.setText(this, text, charset, "plain");

// Sugisawa added.
// eLXg̏ꍇsȂꍇɉst܂B
if (text == null)
	MimeBodyPart.setText(this, "\r\n", charset, "plain");
else if (!text.endsWith("\r\n"))
	MimeBodyPart.setText(this, text + "\r\n", charset, "plain");
else
	MimeBodyPart.setText(this, text, charset, "plain");

	}

	/**
	 * "text"  MIME ^Cvyюw肳ꂽ MIME Tu^CvgpA
	 * w肳ꂽ̃p[g̃RecƂĐݒ肷ȈՃ\bhłB
	 * w肳ꂽ Unicode ́Aw肳ꂽZbggpĕZbgGR[h܂B
	 * ̕Zbg "charset" p[^ݒ肷ׂɂgp܂B
	 */
	public void setText(String text, String charset, String subtype) throws MessagingException {
		MimeBodyPart.setText(this, text, charset, subtype);
	}

	/**
	 * ̃\bh̓bZ[W̃Rec Multipart IuWFNgɐݒ肵܂B
	 * 
	 * @param mp bZ[W̃Recł Multipart IuWFNg
	 * @throws IllegalWriteException l̕ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public final void setContent(final Multipart mp) throws MessagingException {
		setDataHandler(new DataHandler(mp, mp.getContentType()));
		mp.setParent(this);
	}

	/**
	 * ̃bZ[Wւ̕ԐMɓKVbZ[W擾܂B
	 * VbZ[W̑ƃwb_͓K؂ɐݒ肳܂B
	 * ̐VbZ[W͋AȂ킿 "content" <strong>Ȃ</strong>ɒӂĉB
	 * ́ANCAgK؂ɖ߂Kv܂B<p>
	 * 
	 * <code>replyToAll</code>  true ̏ꍇA
	 * VbZ[W͂̃bZ[W̑SĂ̎M҂ɈĂ܂B
	 * łȂꍇAԐM͂̃bZ[W̑M҂ɈĂ܂ (<code>getReplyTo</code> \bh̒lgp)B<p>
	 * 
	 * "Subject" tB[h́ǍɐړuRe:vtċLq܂ (ɐ擪uRe:vłȂꍇ)B
	 * ̃bZ[W "Message-Id" wb_ꍇ́A
	 * VbZ[W̒ "In-Reply-To" wb_ݒ肳܂B
	 * ̃bZ[W <code>ANSWERED</code> tO true ݒ肳܂B
	 * 
	 * @param replyToAll ԐM͂̃bZ[W̑SĂ̎M҂ɑMȂ΂ȂȂ
	 * @return ԐMpbZ[W
	 * @throws MessagingException
	 */
	public final Message reply(final boolean replyToAll) throws MessagingException {
		MimeMessage reply = createMimeMessage(session);
		/*
		 * Have to manipulate the raw Subject header so that we don't lose
		 * any encoding information.  This is safe because "Re:" isn't
		 * internationalized and (generally) isn't encoded.  If the entire
		 * Subject header is encoded, prefixing it with "Re: " still leaves
		 * a valid and correct encoded header.
		 */
		String subject = getHeader("Subject", null);
		if (subject != null) {
			if (!subject.regionMatches(true, 0, "Re: ", 0, 4))
				subject = "Re: " + subject;
			reply.setHeader("Subject", subject);
		}

		Address a[] = getReplyTo();
		reply.setRecipients(Message.RecipientType.TO, a);

		if (replyToAll) {
			Vector v = new Vector();
			// add my own address to list
			InternetAddress me = InternetAddress.getLocalAddress(session);
			if (me != null)
				v.addElement(me);
			// add any alternate names I'm known by
			String alternates = null;
			if (session != null)
				alternates = session.getProperty("mail.alternates");
			if (alternates != null)
				eliminateDuplicates(v, InternetAddress.parse(alternates, false));
			// should we Cc all other original recipients?
			String replyallccStr = null;
			if (session != null)
				replyallccStr = session.getProperty("mail.replyallcc");
			boolean replyallcc = replyallccStr != null && replyallccStr.equalsIgnoreCase("true");
			// add the recipients from the To field so far
			eliminateDuplicates(v, a);
			a = getRecipients(Message.RecipientType.TO);
			a = eliminateDuplicates(v, a);
			if (a != null && a.length > 0) {
				if (replyallcc)
					reply.addRecipients(Message.RecipientType.CC, a);
				else
					reply.addRecipients(Message.RecipientType.TO, a);
			}
			a = getRecipients(Message.RecipientType.CC);
			a = eliminateDuplicates(v, a);
			if (a != null && a.length > 0)
				reply.addRecipients(Message.RecipientType.CC, a);
			// don't eliminate duplicate newsgroups
			a = getRecipients(RecipientType.NEWSGROUPS);
			if (a != null && a.length > 0)
				reply.setRecipients(RecipientType.NEWSGROUPS, a);
		}

		String msgId = getHeader("Message-Id", null);
		if (msgId != null)
			reply.setHeader("In-Reply-To", msgId);
		try {
			setFlags(answeredFlag, true);
		} catch (MessagingException mex) {}	// 

		return reply;
	}

	// used above in reply()
	private static final Flags answeredFlag = new Flags(Flags.Flag.ANSWERED);

	/**
	 * Check addrs for any duplicates that may already be in v.
	 * Return a new array without the duplicates.  Add any new
	 * addresses to v.  Note that the input array may be modified.
	 */
	private Address[] eliminateDuplicates(final Vector v, Address[] addrs) {
		if (addrs == null)
			return null;
		int gone = 0;

		for (int i = 0; i < addrs.length; i++) {
			boolean found = false;
			// search the vector for this address
			for (int j = 0; j < v.size(); j++) {
				if (((InternetAddress)v.elementAt(j)).equals(addrs[i])) {
					// found it; count it and remove it from the input array
					found = true;
					gone++;
					addrs[i] = null;
					break;
				}
			}
			if (!found)
				v.addElement(addrs[i]);	// add new address to vector
		}

		// if we found any duplicates, squish the array
		if (gone != 0) {
			Address[] a;
			// new array should be same type as original array
			// XXX - there must be a better way, perhaps reflection?
			if (addrs instanceof InternetAddress[])
				a = new InternetAddress[addrs.length - gone];
			else
				a = new Address[addrs.length - gone];
			for (int i = 0, j = 0; i < addrs.length; i++)
				if (addrs[i] != null)
					a[j++] = addrs[i];
			addrs = a;
		}

		return addrs;
	}

	/**
	 * bZ[W RFC 822 `̃Xg[Ƃďo͂܂B<p>
	 * 
	 * bZ[W̍\z@ɂAlXȍsgpꍇ鎖ɒӂKvłB
	 * ʂɁAo͓͂K؂ FilterOutputStream gpđMȂ΂Ȃ܂B
	 * ɂAs̏I[́AMIME  Internet vgRŎgp\ CRLFA
	 * ̓[J̃eLXgt@CɊi[\ȃ[JvbgtH[̍sI[̂ǂ炩]`ɕϊ܂B<p>
	 * 
	 * ̎́Anull Xggp <code>writeTo(OutputStream, String[])</code> \bhĂяo܂B
	 * 
	 * @throws IOException Xg[ւ̏ݒɃG[ꍇA
	 * 					 javax.activation CɂG[ꍇ
	 * @throws MessagingException
	 * @see javax.activation.DataHandler#writeTo
	 */
	public void writeTo(final OutputStream os) throws IOException, MessagingException {
		writeTo(os, null);
	}

	/**
	 * bZ[W RFC 822 `Xg[ƂāAw肳ꂽwb_ŏo͂܂B
	 * <code>saved</code> tOݒ肳Ȃꍇ́A
	 * <code>saveChanges</code> \bhĂяo܂B
	 * <code>modified</code> tOݒ肳ꂸA
	 * <code>content</code> z null łȂꍇ́A
	 * K؂ȃbZ[Wwb_̏݌A<code>content</code> z񂪒ڏ܂܂B
	 * 
	 * @throws javax.mail.MessagingException
	 * @throws IOException Xg[ւ̏ݒɃG[ꍇA
	 * 					 javax.activation CɂG[ꍇ
	 * @see javax.activation.DataHandler#writeTo
	 */
	public final void writeTo(final OutputStream os, final String[] ignoreList) throws IOException, MessagingException {
		if (!saved)
			saveChanges();

		if (modified) {
			MimeBodyPart.writeTo(this, os, ignoreList);
			return;
		}

		// Else, the content is untouched, so we can just output it
		// First, write out the header
		Enumeration hdrLines = getNonMatchingHeaderLines(ignoreList);
		LineOutputStream los = new LineOutputStream(os);
		while (hdrLines.hasMoreElements())
			los.writeln((String)hdrLines.nextElement());

		// The CRLF separator between header and content
		los.writeln();

		// Finally, the content. 
		if (content == null) {
			// call getContentStream to give subclass a chance to
			// provide the data on demand
			InputStream is = getContentStream();
			// now copy the data to the output stream
			byte[] buf = new byte[8192];
			int len;
			while ((len = is.read(buf)) > 0)
				os.write(buf, 0, len);
			is.close();
			buf = null;
		} else
			os.write(content);
		os.flush();
	}

	/**
	 *  header_name ̑SẴwb_擾܂B
	 * wb_ US-ASCII ȊO̕܂ޏꍇARFC 2047 ɊÂGR[hĂ̂ŁA
	 * fR[hKvƂȂ鎖ɒӂĉB<p>
	 * 
	 * ̎ <code>headers</code> InternetHeaders IuWFNgwb_擾܂B
	 * 
	 * @param name wb_̖O
	 * @return wb_̔z
	 * @throws MessagingException
	 * @see javax.mail.internet.MimeUtility
	 */
	public String[] getHeader(final String name) throws MessagingException {
		return headers.getHeader(name);
	}

	/**
	 * ̃wb_̑SẴwb_擾A
	 * ؂蕶ŋ؂ꂽwb_P̕ƂĕԂ܂B
	 * delimiter  <code>null</code> ̏ꍇ́Aŏ̃wb_Ԃ܂B
	 * 
	 * @param name ̃wb_̖O
	 * @param delimiter ؂蕶
	 * @return ̖OSẴwb_ɑ΂ltB[h 
	 * @throws MessagingException
	 */
	public String getHeader(final String name, final String delimiter) throws MessagingException {
		return headers.getHeader(name, delimiter);
	}

	/**
	 *  header_name ̒lݒ肵܂B
	 * ̑SẴwb_l̐VlŒu܂B
	 * RFC 822 wb_ US-ASCII ݂̂܂܂Ȃ΂ȂȂׁA
	 * US-ASCII ȊO̕܂ރwb_́AĂяoɂ RFC 2047 ̋Kɏ]ăGR[hȂ΂ȂȂƂɒӂĉB
	 * 
	 * @param name wb_
	 * @param value wb_l
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 * @see javax.mail.internet.MimeUtility
	 */
	public void setHeader(final String name, final String value) throws MessagingException {
		headers.setHeader(name, value);	
	}

	/**
	 * ̒l header_name ̊lɒǉ܂B
	 * RFC 822 wb_ US-ASCII ݂̂܂܂Ȃ΂ȂȂׁA
	 * US-ASCII ȊO̕܂ރwb_ RFC 2047 ̋Kɏ]ăGR[hȂ΂ȂȂɒӂĉB
	 * 
	 * @param name wb_
	 * @param value wb_l
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 * @see javax.mail.internet.MimeUtility
	 */
	public void addHeader(final String name, final String value) throws MessagingException {
		headers.addHeader(name, value);
	}

	/**
	 * ̖OSẴwb_폜܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void removeHeader(final String name) throws MessagingException {
		headers.removeHeader(name);
	}

	/**
	 * ̃bZ[W̑SẴwb_ Header IuWFNg̗񋓂ƂĕԂ܂B<p>
	 * 
	 * wb_ US-ASCII ȊO̕܂ޏꍇA
	 * RFC 2047 ɊÂăGR[hĂ̂ŁA
	 * fR[hKvƂȂ鎖ɒӂĉB<p>
	 * 
	 * ̎ <code>headers</code> InternetHeaders IuWFNgwb_擾܂B
	 * 
	 * @return wb_IuWFNg̔z
	 * @throws MessagingException
	 * @see javax.mail.internet.MimeUtility
	 */
	public Enumeration getAllHeaders() throws MessagingException {
		return headers.getAllHeaders();
	}

	/**
	 * ̃bZ[Wvwb_ Header IuWFNg Enumeration ƂĕԂ܂B
	 * ̎ <code>headers</code> InternetHeaders IuWFNgwb_擾܂B
	 * 
	 * @throws MessagingException
	 */
	public Enumeration getMatchingHeaders(final String[] names) throws MessagingException {
		return headers.getMatchingHeaders(names);
	}

	/**
	 * ̃bZ[WvȂwb_ Header IuWFNg Enumeration ƂĕԂ܂B
	 * ̎ <code>headers</code> InternetHeaders IuWFNgwb_擾܂B
	 * 
	 * @throws MessagingException
	 */
	public Enumeration getNonMatchingHeaders(final String[] names) throws MessagingException {
		return headers.getNonMatchingHeaders(names);
	}

	/**
	 *  RFC 822 wb_sǉ܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void addHeaderLine(final String line) throws MessagingException {
		headers.addHeaderLine(line);
	}

	/**
	 * SẴwb_s String  Enumeration ƂĎ擾܂B
	 * Header s͐ RFC 822 wb_słA"name" y "value" ̗tB[h܂݂܂B
	 * 
	 * @throws MessagingException
	 */
	public Enumeration getAllHeaderLines() throws MessagingException {
		return headers.getAllHeaderLines();
	}

	/**
	 * vwb_s String  Enumeration ƂĎ擾܂B
	 * Header s͐ RFC 822 wb_słA"name" y "value" ̗tB[h܂݂܂B
	 * 
	 * @throws MessagingException
	 */
	public Enumeration getMatchingHeaderLines(final String[] names) throws MessagingException {
		return headers.getMatchingHeaderLines(names);
	}

	/**
	 * vȂwb_s String  Enumeration ƂĎ擾܂B
	 * Header s͐ RFC 822 wb_słA"name" y "value" ̗tB[h܂݂܂B
	 * 
	 * @throws MessagingException
	 */
	public Enumeration getNonMatchingHeaderLines(final String[] names) throws MessagingException {
		return headers.getNonMatchingHeaderLines(names);
	}

	/**
	 * ̃bZ[W̃tO܂ <code>Flags</code> IuWFNgԂ܂B<p>
	 * 
	 *  Flags IuWFNg̕Ԃ̂ŁA
	 * Ԃꂽ Flags IuWFNgύXẴbZ[W̃tOɉeȂɒӂĉB
	 * 
	 * @return Flags ̃bZ[W̃tO܂ Flags IuWFNg
	 * @throws MessagingException
	 * @see javax.mail.Flags
	 */
	public synchronized Flags getFlags() throws MessagingException {
		return (Flags) flags.clone();
	}

	/**
	 * <code>flag</code> Ɏw肳ꂽtÕbZ[Wɐݒ肳Ă邩ǂ`FbN܂B<p>
	 * 
	 * ̎́ÃbZ[W̓ <code>flags</code> IuWFNg`FbN܂B
	 * 
	 * @param flag	tO
	 * @return ̃bZ[W̎w肳ꂽtO̒l
	 * @see javax.mail.Flags.Flag
	 * @see javax.mail.Flags.Flag#ANSWERED
	 * @see javax.mail.Flags.Flag#DELETED
	 * @see javax.mail.Flags.Flag#DRAFT
	 * @see javax.mail.Flags.Flag#FLAGGED
	 * @see javax.mail.Flags.Flag#RECENT
	 * @see javax.mail.Flags.Flag#SEEN
	 * @throws MessagingException
	 */
	public synchronized boolean isSet(final Flags.Flag flag) throws MessagingException {
		return flags.contains(flag);
	}

	/**
	 * ̃bZ[W̃tOݒ肵܂B<p>
	 * 
	 * ̎ <code>flags</code> tB[hύX܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public synchronized void setFlags(final Flags flag, final boolean set) throws MessagingException {
		if (set)
			flags.add(flag);
		else
			flags.remove(flag);
	}

	/**
	 * ̃bZ[W̓K؂ȃwb_tB[hXVA
	 * bZ[W̓eƈѐlɂ܂B
	 * ̃bZ[W Folder Ɋ܂܂ꍇA
	 * ̃bZ[Wɑ΂SĂ̕ύX́A܂tH_ɃR~bg܂B<p>
	 * 
	 * bZ[W̃wb_͓e̔Cӂ̕ύXꂽꍇA
	 * <code>saveChanges</code> ĂяoāA̕ύXiIɂȂ΂Ȃ܂B
	 * ȂƁAύX́AtH_̎ɂۑꂽ肳Ȃ肵܂B<p>
	 * 
	 * READ_ONLY ŊJĂtH_擾ꂽbZ[WύXĂ͂܂B
	 * AbZ[W saveChanges ĂяoĂ͂܂B<p>
	 * 
	 * ̃\bh <code>modified</code> tO true ɂA
	 * <code>save</code> tO true ɂāA
	 * <code>updateHeaders</code> \bhĂяo܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	public void saveChanges() throws MessagingException {
		modified = true;
		saved = true;
		updateHeaders();
	}

	/**
	 * Update the Message-ID header.
	 * This method is called by the updateHeaders and allows a subclass to override only the algorithm for choosing a Message-ID.
	 * 
	 * @throws MessagingException
	 * @since JavaMail 1.4
	 */
	protected void updateMessageID() throws MessagingException {
		setHeader("Message-ID", "<" + UniqueValue.getUniqueMessageIDValue(session) + ">");
	}

	/**
	 * <code>saveChanges</code> \bhɂĂяoA
	 * ۂ MIME wb_XV܂B
	 * ̎́A<code>Content-Transfer-Encoding</code> wb_ (KvȏꍇŊɐݒ肳ĂȂꍇ)A
	 * <code>Mime-Version</code> wb_Ay <code>Message-ID</code> wb_ݒ肵܂B
	 * ÃbZ[W̓e <code>MimeMultipart</code> ̏ꍇA
	 * <code>updateHeaders</code> \bhĂяo܂B
	 * 
	 * @throws IllegalWriteException ύXT|[gȂꍇ
	 * @throws IllegalStateException ̃bZ[W READ_ONLY tH_擾ꂽꍇ
	 * @throws MessagingException
	 */
	protected final void updateHeaders() throws MessagingException {
		MimeBodyPart.updateHeaders(this);	
		setHeader("Mime-Version", "1.0");
		updateMessageID();
	}

	/**
	 * w肳ꂽ InputStream wb_[h InternetHeaders IuWFNg쐬AԂ܂B
	 * Kvł΁ATuNX͂̃\bhI[o[ChA
	 * InternetHeaders ̃TuNXԂo܂B
	 * ̎͒P InternetHeaders IuWFNg\zAԂ܂B
	 * 
	 * @param is wb_ǂݍ݌ InputStream
	 * @throws MessagingException
	 * @since JavaMail 1.2
	 */
	protected final InternetHeaders createInternetHeaders(final InputStream is) throws MessagingException {
		return new InternetHeaders(is);
	}

	/**
	 * Create and return a MimeMessage object.
	 * The reply method uses this method to create the MimeMessage object that it will return.
	 * Subclasses can override this method to return a subclass of MimeMessage. This implementation simply constructs and returns a MimeMessage object using the supplied Session.
	 * 
	 * @param session the Session to use for the new message
	 * @return the new MimeMessage object
	 * @throws MessagingException
	 * @since JavaMail 1.4
	 */
	protected MimeMessage createMimeMessage(final Session session) throws MessagingException {
		return new MimeMessage(session);
	}

}
