/*
 * Copyright 1997-2003 Sun Microsystems, Inc. All Rights Reserved.
 */

package javax.mail.internet;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.Hashtable;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.mail.MessagingException;

import com.sun.mail.util.ASCIIUtility;
import com.sun.mail.util.BASE64DecoderStream;
import com.sun.mail.util.BASE64EncoderStream;
import com.sun.mail.util.BEncoderStream;
import com.sun.mail.util.LineInputStream;
import com.sun.mail.util.QDecoderStream;
import com.sun.mail.util.QEncoderStream;
import com.sun.mail.util.QPDecoderStream;
import com.sun.mail.util.QPEncoderStream;
import com.sun.mail.util.UUDecoderStream;
import com.sun.mail.util.UUEncoderStream;

import jp.sourceforge.livez.mail.util.MailUtils;
import jp.sourceforge.livez.mail.util.WindowsEUCJP;
import jp.sourceforge.livez.mail.util.WindowsJIS;

/**
 * lX MIME ֘A@\񋟂郆[eBeBNXłB<p>
 * 
 * MIME wb_ RFC 2047 ɏ]ăGR[hyуfR[h郁\bhꎮ܂B
 * wb_̏ɊւȒPȐȉɎ܂B<p>
 * 
 * RFC 822 [wb_ US-ASCII <strong></strong>܂܂Ȃ΂Ȃ܂B
 * US-ASCII ȊO̕܂ރwb_ US-ASCII ݂̂܂ޗlɃGR[hKv܂B
 * {IɁȀ BASE64 ܂ QP gpē̕GR[h鎖܂݂܂B
 * ڍׂ RFC 2047 ɐ܂B<p>
 * 
 * Java ɂāAString  (16 rbg) Unicode ܂݂܂B
 * ASCII  Unicode ̃TuZbgł (0 - 127 ͈̔͂߂܂)B
 * ASCII ݂̂܂ String ͊Ƀ[ňSɎgpo܂B
 * String  US-ASCII ȊO̕܂܂ꍇ̓GR[hKvłB
 * ̒iK蕡GɂĂ̂́AUnicode ܂LgpĂ镶ZbgłȂׁA
 * ܂ String ʂ̕ZbgɕZbgGR[hA
 * ̌ɓ]GR[fBOKv鎖łB<p>
 * 
 * [ňS String (Ⴆ΁ASMTP ̑MȂ) ̎ۂ̃oCg擾ɂ͎̑삪KvłB
 * <p><blockquote><pre>
 * 
 *	byte[] bytes = string.getBytes("iso-8859-1");	
 * 
 * </pre></blockquote><p>
 * 
 * MimeMessage y MimeBodyPart  <code>setHeader</code> y <code>addHeader</code> \bh́A
 * w肳ꂽwb_l US-ASCII ݂̂܂ Unicode łƉ肵܂B
 * ]āǍĂяo͓nl US-ASCII ȊO̕܂܂ȂmFȂ΂Ȃ܂B
 * ̃NX̃\bh́A̎sɖ𗧂܂B<p>
 * 
 * MimeMessage y MimeBodyPart  <code>getHeader</code> \bht@~[́A
 * wb_lԂ܂B RFC 2047 ɏ]ăGR[h\܂B
 * ̏ꍇAUnicode ɃfR[hKv܂B
 * ̃NX̃\bh́A̎sɖ𗧂܂B<p>
 * 
 * Several System properties control strict conformance to the MIME
 * spec.  Note that these are not session properties but must be set
 * globally as System properties. <p>
 * 
 * The <code>mail.mime.decodetext.strict</code> property controls
 * decoding of MIME encoded words.  The MIME spec requires that encoded
 * words start at the beginning of a whitespace separated word.  Some
 * mailers incorrectly include encoded words in the middle of a word.
 * If the <code>mail.mime.decodetext.strict</code> System property is
 * set to <code>"false"</code>, an attempt will be made to decode these
 * illegal encoded words. The default is true. <p>
 * 
 * The <code>mail.mime.encodeeol.strict</code> property controls the
 * choice of Content-Transfer-Encoding for MIME parts that are not of
 * type "text".  Often such parts will contain textual data for which
 * an encoding that allows normal end of line conventions is appropriate.
 * In rare cases, such a part will appear to contain entirely textual
 * data, but will require an encoding that preserves CR and LF characters
 * without change.  If the <code>mail.mime.encodeeol.strict</code>
 * System property is set to <code>"true"</code>, such an encoding will
 * be used when necessary.  The default is false. <p>
 * 
 * In addition, the <code>mail.mime.charset</code> System property can
 * be used to specify the default MIME charset to use for encoded words
 * and text parts that don't otherwise specify a charset.  Normally, the
 * default MIME charset is derived from the default Java charset, as
 * specified in the <code>file.encoding</code> System property.  Most
 * applications will have no need to explicitly set the default MIME
 * charset.  In cases where the default MIME charset to be used for
 * mail messages is different than the charset used for files stored on
 * the system, this property should be set.
 */
public final class MimeUtility {

	// This class cannot be instantiated
	private MimeUtility() {}

	public static final int ALL = -1;

	private static boolean decodeStrict = true;
	private static boolean encodeEolStrict = false;
	private static boolean foldEncodedWords = false;
	private static boolean foldText = true;

	static {
		try {
			String s = System.getProperty("mail.mime.decodetext.strict");
			// default to true
			decodeStrict = s == null || !s.equalsIgnoreCase("false");
			s = System.getProperty("mail.mime.encodeeol.strict");
			// default to false
			encodeEolStrict = s != null && s.equalsIgnoreCase("true");
			s = System.getProperty("mail.mime.foldencodedwords");
			// default to false
			foldEncodedWords = s != null && s.equalsIgnoreCase("true");
			s = System.getProperty("mail.mime.foldtext");
			// default to true
			foldText = s == null || !s.equalsIgnoreCase("false");
		} catch (SecurityException sex) {
			// 
		}
	}

	/**
	 * [ňSɎgpolɂׁA
	 * ̃f[^\[X̓̓Xg[ɓKpȂ΂ȂȂ content-transfer-encoding 擾܂B<p>
	 * 
	 * ŎgpASY͎̒ʂłB<br>
	 * <ul>
	 * <li>
	 * ̃f[^\[X̃vC}^Cv "text" ŁA
	 * ̓Xg[̑SẴoCg US-ASCII ̏ꍇA
	 * GR[fBO "7bit" łB
	 * 𒴂oCg US-ASCII ȊȌꍇAGR[fBO "base64" łB
	 * ̃oCg US-ASCII ȊȌꍇAGR[fBO "quoted-printable" łB
	 * <li>
	 * ̃f[^\[X̃vC}^Cv "text" łȂA
	 * ̓Xg[̑SẴoCg US-ASCII ̏ꍇAGR[fBO "7bit" łB
	 * US-ASCII ȊO̕ 1 łꍇAGR[fBO "base64" łB
	 * </ul>
	 *
	 * @param ds DataSource
	 * @return GR[fBOB"7bit"A"quoted-printable"A "base64"
	 */ 
	public static String getEncoding(final DataSource ds) {
		ContentType cType = null;
		InputStream is = null;
		String encoding = null;

		try {
			cType = new ContentType(ds.getContentType());
			is = ds.getInputStream();
		} catch (Exception ex) {
			return "base64"; // what else ?!
		}

		boolean isText = cType.match("text/*");
		// if not text, stop processing when we see non-ASCII
		int i = checkAscii(is, ALL, !isText);
		switch (i) {
			case ALL_ASCII:
				encoding = "7bit"; // all ascii
				break;
			case MOSTLY_ASCII:
				encoding = "quoted-printable"; // mostly ascii
				break;
			default:
				encoding = "base64"; // mostly binary
				break;
		}

		// ̓Xg[܂
		try {
			is.close();
		} catch (IOException ioex) {}

		return encoding;
	}

	/**
	 * <code>getEncoding(DataSource)</code> ƓłB
	 * AA<code>InputStream</code> f[^ǂݍނ̂ł͂ȂA
	 * <code>writeTo</code> \bhgpăf[^܂B
	 * ́A<code>DataHandler</code> IuWFNg MIME ^Cv (Ⴆ΁A"text/plain" String) ō쐬ʓIȃP[XɂāAIłB
	 * ȂȂASĂ I/O ̃XbhŎs邩łB
	 * <code>InputStream</code> KvȏꍇA<code>DataHandler</code> ̓XbhA
	 * pCvXg[̑΁Ay <code>writeTo</code> \bhgpăf[^𐶐܂B<p>
	 * 
	 * @since	JavaMail 1.2
	 */
	public static String getEncoding(final DataHandler dh) {
		ContentType cType = null;
		String encoding = null;

		/*
		 * Try to pick the most efficient means of determining the
		 * encoding.  If this DataHandler was created using a DataSource,
		 * the getEncoding(DataSource) method is typically faster.  If
		 * the DataHandler was created with an object, this method is
		 * much faster.  To distinguish the two cases, we use a heuristic.
		 * A DataHandler created with an object will always have a null name.
		 * A DataHandler created with a DataSource will usually have a
		 * non-null name.
		 *
		 * XXX - This is actually quite a disgusting hack, but it makes
		 *	 a common case run over twice as fast.
		 */
		if (dh.getName() != null)
			return getEncoding(dh.getDataSource());

		try {
			cType = new ContentType(dh.getContentType());
		} catch (Exception ex) {
			return "base64"; // what else ?!
		}

		if (cType.match("text/*")) {
			// Check all of the available bytes
			AsciiOutputStream aos = new AsciiOutputStream(false, false);
			try {
				dh.writeTo(aos);
			} catch (IOException ex) {}	// 

			switch (aos.getAscii()) {
				case ALL_ASCII:
					encoding = "7bit"; // all ascii
					break;
				case MOSTLY_ASCII:
					encoding = "quoted-printable"; // mostly ascii
					break;
				default:
					encoding = "base64"; // mostly binary
					break;
			}
		} else { // "text" łȂꍇ
			// Check all of available bytes, break out if we find
			// at least one non-US-ASCII character
			AsciiOutputStream aos = new AsciiOutputStream(true, encodeEolStrict);
			try {
				dh.writeTo(aos);
			} catch (IOException ex) {}	// 
			if (aos.getAscii() == ALL_ASCII) // all ascii
				encoding = "7bit";
			else // found atleast one non-ascii character, use b64 
				encoding = "base64";
		}

		return encoding;
	}

	/**
	 * w肳ꂽ̓Xg[fR[h܂B
	 * Ԃ̓Xg[̓fR[hꂽ̓Xg[łB
	 * RFC 2045 Œ`ĂSẴGR[fBO͂ŃT|[g܂B
	 * ɂ "base64"A"quoted-printable"A"7bit"A"8bit"Ay "binary" ܂܂܂B
	 * ɁA"uuencode" T|[g܂B
	 *
	 * @param is ̓Xg[
	 * @param encoding Xg[̃GR[fBO
	 * @return fR[hꂽ̓Xg[
	 */
	public static InputStream decode(final InputStream is, final String encoding)
		throws MessagingException {

		if (encoding.equalsIgnoreCase("base64"))
			return new BASE64DecoderStream(is);
		else if (encoding.equalsIgnoreCase("quoted-printable"))
			return new QPDecoderStream(is);
		else if (encoding.equalsIgnoreCase("uuencode") ||
			 encoding.equalsIgnoreCase("x-uuencode") ||
			 encoding.equalsIgnoreCase("x-uue"))

			return new UUDecoderStream(is);
		else if (encoding.equalsIgnoreCase("binary") ||
			 encoding.equalsIgnoreCase("7bit") ||
			 encoding.equalsIgnoreCase("8bit"))

			return is;
		else
			throw new MessagingException("Unknown encoding: " + encoding);
	}

	/**
	 * w肳ꂽo̓Xg[ɃGR[_bv܂B
	 * RFC 2045 Œ`ĂSẴGR[fBO͂ŃT|[g܂B
	 * ɂ "base64"A"quoted-printable"A"7bit"A"8bit"Ay "binary" ܂܂܂B
	 * ɁA"uuencode" T|[g܂B
	 * 
	 * @param os o̓Xg[
	 * @param encoding Xg[̃GR[fBO
	 * @return w肳ꂽGR[fBOKpo̓Xg[
	 */
	public static OutputStream encode(
		final OutputStream os,
		final String encoding)
		throws MessagingException {

		if (encoding == null)
			return os;
		else if (encoding.equalsIgnoreCase("base64"))
			return new BASE64EncoderStream(os);
		else if (encoding.equalsIgnoreCase("quoted-printable"))
			return new QPEncoderStream(os);
		else if (encoding.equalsIgnoreCase("uuencode") ||
			 encoding.equalsIgnoreCase("x-uuencode") ||
			 encoding.equalsIgnoreCase("x-uue"))

			return new UUEncoderStream(os);
		else if (encoding.equalsIgnoreCase("binary") ||
			 encoding.equalsIgnoreCase("7bit") ||
			 encoding.equalsIgnoreCase("8bit"))

			return os;
		else
			throw new MessagingException("Unknown encoding: " +encoding);
	}

	/**
	 * w肳ꂽo̓Xg[ɃGR[_bv܂B
	 * RFC 2045 Œ`ĂSẴGR[fBO͂ŃT|[g܂B
	 * ɂ "base64"A"quoted-printable"A"7bit"A"8bit"Ay "binary" ܂܂܂B
	 * ɁA"uuencode" T|[g܂B
	 * <code>filename</code> p[^ "uuencode" GR[fBOƋɎgpA
	 * GR[hꂽo͂Ɋ܂܂܂B
	 * 
	 * @param os o̓Xg[
	 * @param encoding Xg[̃GR[fBO
	 * @param filename GR[ht@C̖O (uuencode Ƌɂ̂ݎgp܂)
	 * @return w肳ꂽGR[fBOKpo̓Xg[
	 * @since JavaMail 1.2
	 */
	public static OutputStream encode(
		final OutputStream os,
		final String encoding,
		final String filename)
		throws MessagingException {

		if (encoding == null)
			return os;
		else if (encoding.equalsIgnoreCase("base64"))
			return new BASE64EncoderStream(os);
		else if (encoding.equalsIgnoreCase("quoted-printable"))
			return new QPEncoderStream(os);
		else if (encoding.equalsIgnoreCase("uuencode") ||
			encoding.equalsIgnoreCase("x-uuencode") ||
			encoding.equalsIgnoreCase("x-uue"))

			return new UUEncoderStream(os, filename);
		else if (encoding.equalsIgnoreCase("binary") ||
			encoding.equalsIgnoreCase("7bit") ||
			encoding.equalsIgnoreCase("8bit"))

			return os;
		else
			throw new MessagingException("Unknown encoding: " +encoding);
	}

	/**
	 * RFC 822 "text" g[N RFC 2047 ɏ][ɈSȌ`ɃGR[h܂B<p>
	 * 
	 * w肳ꂽ Unicode  US-ASCII ȊO̕ɂČ܂B
	 *  US-ASCII ܂܂ꍇ͂̂܂ܕԂ܂B
	 *  US-ASCII ȊO̕܂܂ꍇA
	 * ܂vbgtH[̃ftHg̕ZbggpĕGR[hA
	 *  B ܂ Q GR[fBOgpē]GR[h܂B
	 * ʂ̃oCg ASCII ݂̂܂ Unicode ƂĕԂ܂B<p>
	 * 
	 * ̃\bh "\Ȃ" RFC 822 wb_GR[hׂɂgpȂ΂Ȃ܂B<p>
	 * 
	 * gp:
	 * <p><blockquote><pre>
	 * 
	 *  MimePart part = ...
	 *  String rawvalue = "FooBar Mailer, Japanese version 1.1"
	 *  try {
	 *    // If we know for sure that rawvalue contains only US-ASCII 
	 *    // characters, we can skip the encoding part
	 *    part.setHeader("X-mailer", MimeUtility.encodeText(rawvalue));
	 *  } catch (UnsupportedEncodingException e) {
	 *    // encoding failure
	 *  } catch (MessagingException me) {
	 *   // setHeader() failure
	 *  }
	 * 
	 * </pre></blockquote><p>
	 * 
	 * @param text Unicode 
	 * @return US-ASCII ݂̂܂ Unicode 
	 * @throws UnsupportedEncodingException GR[fBOɎsꍇ
	 */
	public static String encodeText(final String text) throws UnsupportedEncodingException {
		return encodeText(text, null, null);
	}

	/**
	 * RFC 822 "text" g[N RFC 2047 ɏ][ɈSȌ`ɃGR[h܂B<p>
	 * 
	 * w肳ꂽ Unicode  US-ASCII ȊO̕ɂČ܂B
	 *  US-ASCII ܂܂ꍇ͂̂܂ܕԂ܂B
	 *  US-ASCII ȊO̕܂܂ꍇA
	 * ܂w肳ꂽZbggpĕGR[hA
	 *  B ܂ Q GR[fBOgpē]GR[h܂B
	 * ʂ̃oCg ASCII ݂̂܂ Unicode ƂĕԂ܂B<p>
	 * 
	 * ̃\bh "\Ȃ" RFC 822 wb_GR[hׂɂgpȂ΂Ȃ܂B
	 * 
	 * @param text wb_l
	 * @param charset LN^ZbgB̃p[^ null ̏ꍇAvbgtH[̃ftHg̃LN^Zbggp܂
	 * @param encoding gpGR[fBOB
	 * 			݃T|[gĂl "B" y "Q" łB
	 * 			̃p[^ null ̎A
	 * 			GR[h镶̖wǂ ASCII Zbgɂꍇ "Q" GR[fBOgpA
	 * 			łȂꍇ "B" GR[fBOgp܂
	 * @return US-ASCII ݂̂܂ Unicode 
	 */
	public static String encodeText(
		final String text,
		final String charset,
		final String encoding)
		throws UnsupportedEncodingException {

		return encodeWord(text, charset, encoding, false);
	}

	/**
	 * "\Ȃ" wb_A܂ARFC 822  '*text' ƒ`Ăwb_fR[h܂B<p>
	 * 
	 *  RFC 2047, Section 6.1.1 ŋK肳ĂASYgpăfR[h܂B
	 * Cӂ̃V[PXŕϊɎsƁAUnsupportedEncodingException X[܂B
	 * String  RFC 2047 `ŃGR[hꂽwb_łȂꍇ͂̂܂ܕԂ܂B<p>
	 * 
	 * gp:
	 * <p><blockquote><pre>
	 * 
	 *  MimePart part = ...
	 *  String rawvalue = null;
	 *  String  value = null;
	 *  try {
	 *    if ((rawvalue = part.getHeader("X-mailer")[0]) != null)
	 *      value = MimeUtility.decodeText(rawvalue);
	 *  } catch (UnsupportedEncodingException e) {
	 *      // Don't care
	 *      value = rawvalue;
	 *  } catch (MessagingException me) { }
	 * 
	 *  return value;
	 * 
	 * </pre></blockquote><p>
	 * 
	 * @param etext GR[hĂ\̂l
	 * @throws UnsupportedEncodingException LN^Zbg̕ϊsꍇ
	 */
	public static String decodeText(final String etext) throws UnsupportedEncodingException {
// Sugisawa added. 2004/03/10
// JavaMail W̃fR[hARFC ɏĂȂĂfR[h\ MailUtils ̂炬tfR[hs܂B
return MailUtils.decodeText(etext);

// Sugisawa changed. 2004/03/10 ȉ̃ubNSăRg

		/*
		 * We look for sequences separated by "linear-white-space".
		 * (as per RFC 2047, Section 6.1.1)
		 * RFC 822 defines "linear-white-space" as SPACE | HT | CR | NL.
		 */
/*
		String lwsp = " \t\n\r";
		StringTokenizer st;
*/

		/*
		 * First, lets do a quick run thru the string and check
		 * whether the sequence "=?"  exists at all. If none exists,
		 * we know there are no encoded-words in here and we can just
		 * return the string as-is, without suffering thru the later 
		 * decoding logic. 
		 * This handles the most common case of unencoded headers 
		 * efficiently.
		 */
/*
		if (etext.indexOf("=?") == -1) return etext;

		// Encoded words found. fR[hJn܂B

		st = new StringTokenizer(etext, lwsp, true);
		StringBuffer sb = new StringBuffer();  // fR[h obt@
		StringBuffer wsb = new StringBuffer(); // zCg Xy[X obt@
		boolean prevWasEncoded = false;

		while (st.hasMoreTokens()) {
		    char c;
		    String s = st.nextToken();
		    // If whitespace, append it to the whitespace buffer
		    if (((c = s.charAt(0)) == ' ') || (c == '\t') || (c == '\r') || (c == '\n'))
				wsb.append(c);
		    else {
				// Check if token is an 'encoded-word' ..
				String word;
	
				try {
				    word = decodeWord(s);
				    // Yes, this IS an 'encoded-word'.
				    if (!prevWasEncoded && wsb.length() > 0) {
						// if the previous word was also encoded, we
						// should ignore the collected whitespace. Else
						// we include the whitespace as well.
						sb.append(wsb);
				    }
				    prevWasEncoded = true;
				} catch (ParseException pex) {
				    // This is NOT an 'encoded-word'.
				    word = s;
				    // possibly decode inner encoded words
				    if (!decodeStrict) word = decodeInnerWords(word);
				    // include colleced whitespace ..
				    if (wsb.length() > 0) sb.append(wsb);
				    prevWasEncoded = false;
				}
				sb.append(word); // append the actual word
				wsb.setLength(0); // reset wsb for reuse
		    }
		}

		return sb.toString();
*/
	}

	/**
	 * RFC 822 "word" g[N RFC 2047 ɏ][ɈSȌ`ɃGR[h܂B<p>
	 * 
	 * w肳ꂽ Unicode  US-ASCII ȊO̕ɂČ܂B
	 *  US-ASCII ܂܂ꍇ͂̂܂ܕԂ܂B
	 *  US-ASCII ȊO̕܂܂ꍇA
	 * ܂vbgtH[̃ftHg̕ZbggpĕGR[hA
	 *  B  Q GR[fBOgpē]GR[h܂B
	 * ʂ̃oCg ASCII ݂̂܂ Unicode ƂĕԂ܂B<p>
	 * 
	 * ̃\bh RFC 822 "phrases" 쐬鎞Ɏgp鎖Ӑ}Ă܂B
	 * Ⴆ΁AInternetAddress NX͂gp 'phrase' R|[lgGR[h܂B 
	 * 
	 * @param word	Unicode 
	 * @return US-ASCII ݂̂܂ Unicode ̔z 
	 * @throws UnsupportedEncodingException GR[fBOɎsꍇ
	 */
	public static String encodeWord(final String word) throws UnsupportedEncodingException {
		return encodeWord(word, null, null);
	}

	/**
	 * RFC 822 "word" g[N RFC 2047 ɏ][ɈSȌ`ɃGR[h܂B<p>
	 * 
	 * w肳ꂽ Unicode  US-ASCII ȊO̕ɂČ܂B
	 *  US-ASCII ܂܂ꍇ͂̂܂ܕԂ܂B
	 *  US-ASCII ȊO̕܂܂ꍇA
	 * ܂w肳ꂽZbggpĕGR[hA
	 *  B  Q GR[fBOgpē]GR[h܂B
	 * ʂ̃oCg ASCII ݂̂܂ Unicode ƂĕԂ܂B<p>
	 * 
	 * @param word Unicode 
	 * @param charset MIME LN^Zbg
	 * @param encoding gpGR[fBOB
	 * 		݃T|[gĂl "B" y "Q" łB
	 * 		̃p[^ null ̎A
	 * 		GR[h镶̖wǂ ASCII Zbgɂꍇ "Q" GR[fBOgpA
	 * 		łȂꍇ "B" GR[fBOgp܂B
	 * @return US-ASCII ݂̂܂ Unicode 
	 * @throws UnsupportedEncodingException GR[fBOɎsꍇ
	 */
	public static String encodeWord(
		final String word,
		final String charset,
		final String encoding)
		throws UnsupportedEncodingException {

		return encodeWord(word, charset, encoding, true);
	}

	/*
	 * Encode the given string. The parameter 'encodingWord' should
	 * be true if a RFC 822 "word" token is being encoded and false if a
	 * RFC 822 "text" token is being encoded. This is because the 
	 * "Q" encoding defined in RFC 2047 has more restrictions when
	 * encoding "word" tokens. (Sigh)
	 */ 
	private static String encodeWord(
		final String string,
		String charset,
		String encoding,
		final boolean encodingWord)
		throws UnsupportedEncodingException {

		// If 'string' contains only US-ASCII characters, just
		// return it.
		int ascii = checkAscii(string);
		if (ascii == ALL_ASCII)
			return string;
	
		// Else, apply the specified charset conversion.
		String jcharset;
		if (charset == null) { // use default charset
			jcharset = getDefaultJavaCharset(); // the java charset
			charset = getDefaultMIMECharset(); // the MIME equivalent
		} else // MIME charset -> java charset
			jcharset = javaCharset(charset);

		// If no transfer-encoding is specified, figure one out.
		if (encoding == null) {
			if (ascii != MOSTLY_NONASCII)
				encoding = "Q";
			else
				encoding = "B";
		}
	
		boolean b64;
		if (encoding.equalsIgnoreCase("B")) 
			b64 = true;
		else if (encoding.equalsIgnoreCase("Q"))
			b64 = false;
		else
			throw new UnsupportedEncodingException("Unknown transfer encoding: " + encoding);

		StringBuffer outb = new StringBuffer(); // the output buffer
		doEncode(string, b64, jcharset, 
			 // As per RFC 2047, size of an encoded string should not
			 // exceed 75 bytes.
			 // 7 = size of "=?", '?', 'B'/'Q', '?', "?="
			 75 - 7 - charset.length(), // the available space
			 "=?" + charset + '?' + encoding + '?', // prefix
			 true, encodingWord, outb);

		return outb.toString();
	}

	private static void doEncode(
		final String string,
		final boolean b64, 
		final String jcharset,
		final int avail,
		final String prefix, 
		final boolean first,
		final boolean encodingWord,
		final StringBuffer buf) 
		throws UnsupportedEncodingException {

	// First find out what the length of the encoded version of
	// 'string' would be.
// Sugisawa changed. 2004/03/08
//	byte[] bytes = string.getBytes(jcharset);

// Sugisawa added. 2004/03/08
byte[] bytes;
if (jcharset.equalsIgnoreCase("ISO-2022-JP"))
	bytes = WindowsJIS.encode(string);
else if (jcharset.equalsIgnoreCase("Shift_JIS"))
	bytes = string.getBytes("Windows-31J");
else if (jcharset.equalsIgnoreCase("euc-jp"))
	bytes = WindowsEUCJP.encode(string);
else
	bytes = string.getBytes(jcharset);

		int len;
		if (b64)	// "B" GR[fBO
			len = BEncoderStream.encodedLength(bytes);
		else		// "Q" GR[fBO
			len = QEncoderStream.encodedLength(bytes, encodingWord);

		int size;
		if ((len > avail) && ((size = string.length()) > 1)) { 
			// If the length is greater than 'avail', split 'string'
			// into two and recurse.
			doEncode(string.substring(0, size / 2), b64, jcharset, avail, prefix, first, encodingWord, buf);
			doEncode(string.substring(size / 2, size), b64, jcharset, avail, prefix, false, encodingWord, buf);
		} else {
			// length <= than 'avail'. Encode the given string
			ByteArrayOutputStream os = new ByteArrayOutputStream();
			OutputStream eos; // GR[_[
			if (b64)	// "B" GR[fBO
				eos = new BEncoderStream(os);
			else		// "Q" GR[fBO
				eos = new QEncoderStream(os, encodingWord);

			try { // GR[fBO܂
				eos.write(bytes);
				eos.close();
			} catch (IOException ioex) {}

			byte[] encodedBytes = os.toByteArray(); // the encoded stuff
			// Now write out the encoded (all ASCII) bytes into our StringBuffer
			if (!first) // not the first line of this sequence
				if (foldEncodedWords)
					buf.append("\r\n "); // start a continuation line
				else
					buf.append(' '); // line will be folded later

			buf.append(prefix);
			for (int i = 0; i < encodedBytes.length; i++)
				buf.append((char)encodedBytes[i]);
			buf.append("?="); // terminate the current sequence
		}
	}

	/**
	 * ́ARFC 2047  "encoded-word" \͋Kgpč\͂܂B
	 * \͂ɎsꍇAParseException X[܂B
	 * łȂꍇ́A]fR[hꂽAUnicode ɕϊ܂B
	 * ϊɎsƁAUnsupportedEncodingException X[܂B<p>
	 * 
	 * @param eword GR[hĂ\̂l
	 * @throws ParseException  RFC 2047 ŋK肳Ă encoded-word łȂꍇ
	 * @throws UnsupportedEncodingException LN^Zbgϊsꍇ
	 */
	public static String decodeWord(final String eword) throws ParseException, UnsupportedEncodingException {
		if (!eword.startsWith("=?")) // encoded word Ȃꍇ
		    throw new ParseException();

		// LN^Zbg擾
		int start = 2; int pos; 
		if ((pos = eword.indexOf('?', start)) == -1)
			throw new ParseException();
		String charset = javaCharset(eword.substring(start, pos));

		// GR[fBO擾
		start = pos + 1;
		if ((pos = eword.indexOf('?', start)) == -1)
			throw new ParseException();
		String encoding = eword.substring(start, pos);

		// encoded-sequence 擾
		start = pos + 1;
		if ((pos = eword.indexOf("?=", start)) == -1)
			throw new ParseException();
		String word = eword.substring(start, pos);

		try {
			// Extract the bytes from word
			ByteArrayInputStream bis = new ByteArrayInputStream(ASCIIUtility.getBytes(word));

			// Get the appropriate decoder
			InputStream is;
			if (encoding.equalsIgnoreCase("B")) 
				is = new BASE64DecoderStream(bis);
			else if (encoding.equalsIgnoreCase("Q"))
				is = new QDecoderStream(bis);
			else
				throw new UnsupportedEncodingException("unknown encoding: " + encoding);

			// For b64 & q, size of decoded word <= size of word. So
			// the decoded bytes must fit into the 'bytes' array. This
			// is certainly more efficient than writing bytes into a
			// ByteArrayOutputStream and then pulling out the byte[]
			// from it.
			int count = bis.available();
			byte[] bytes = new byte[count];
			// count is set to the actual number of decoded bytes 
			count = is.read(bytes, 0, count);

			// Finally, convert the decoded bytes into a String using
			// the specified charset
// Sugisawa changed. 2004/03/08
//			String s = new String(bytes, 0, count, charset);

// Sugisawa added. 2004/03/08
String s;
if ("ISO-2022-JP".equalsIgnoreCase(charset))
	s = WindowsJIS.decode(bytes, count);
else if ("Shift_JIS".equalsIgnoreCase(charset))
	s = new String(bytes, 0, count, "Windows-31J");
else if ("euc-jp".equalsIgnoreCase(charset))
	s = WindowsEUCJP.decode(bytes, count);
else
	s = new String(bytes, 0, count, charset);

			if (pos + 2 < eword.length()) {
				// there's still more text in the string
				String rest = eword.substring(pos + 2);
				if (!decodeStrict)
					rest = decodeInnerWords(rest);
				s += rest;
			}
			return s;
		} catch (UnsupportedEncodingException uex) {
			// explicitly catch and rethrow this exception, otherwise
			// the below IOException catch will swallow this up!
			throw uex;
		} catch (IOException ioex) {
			// Shouldn't happen.
			throw new ParseException();
		} catch (IllegalArgumentException iex) {
			/* An unknown charset of the form ISO-XXX-XXX, will cause
			 * the JDK to throw an IllegalArgumentException ... Since the
			 * JDK will attempt to create a classname using this string,
			 * but valid classnames must not contain the character '-',
			 * and this results in an IllegalArgumentException, rather than
			 * the expected UnsupportedEncodingException. Yikes
			 */
			throw new UnsupportedEncodingException();
		}
	}

	/**
	 * Look for encoded words within a word.  The MIME spec doesn't
	 * allow this, but many broken mailers, especially Japanese mailers,
	 * produce such incorrect encodings.
	 */
	private static String decodeInnerWords(final String word) throws UnsupportedEncodingException {
		int start = 0, i;
		StringBuffer buf = new StringBuffer();

		while ((i = word.indexOf("=?", start)) >= 0) {
			buf.append(word.substring(start, i));
			int end = word.indexOf("?=", i);
			if (end < 0)
				break;
			String s = word.substring(i, end + 2);

			try {
				s = decodeWord(s);
			} catch (ParseException pex) {
				// ignore it, just use the original string
			}

			buf.append(s);
			start = end + 2;
		}

		if (start == 0)
			return word;
		if (start < word.length())
			buf.append(word.substring(start));

		return buf.toString();
	}

	/**
	 * w肳ꂽꕶꗗ̔Cӂ̕PɊ܂܂ĂꍇɁA
	 * Pp郆[eBeB\bhłB<p>
	 * 
	 * <code>HeaderTokenizer</code> NX́A
	 * MIME  RFC 822 Ƃ 2 g̓؂蕶Zbg`܂B<p>
	 * 
	 * ʂɂ̃\bh́ARFC 822 y MIME wb_tB[h̐Ɏgp܂B
	 * 
	 * @param word pP
	 * @param specials ꕶ̃Zbg
	 * @return 炭pP
	 * @see javax.mail.internet.HeaderTokenizer#MIME
	 * @see javax.mail.internet.HeaderTokenizer#RFC822
	 */
	public static String quote(final String word, final String specials) {
		int len = word.length();

		/*
		 * Look for any "bad" characters, Escape and
		 *  quote the entire string if necessary.
		 */
		boolean needQuoting = false;

		for (int i = 0; i < len; i++) {
			char c = word.charAt(i);
			if (c == '"' || c == '\\' || c == '\r' || c == '\n') {
				// need to escape them and then quote the whole string
				StringBuffer sb = new StringBuffer(len + 3);
				sb.append('"');
				sb.append(word.substring(0, i));
				int lastc = 0;
	
				for (int j = i; j < len; j++) {
					char cc = word.charAt(j);
					if ((cc == '"') || (cc == '\\') || (cc == '\r') || (cc == '\n'))
						if (cc == '\n' && lastc == '\r')
							;	// do nothing, CR was already escaped
						else
							sb.append('\\');	// Escape the character

					sb.append(cc);
					lastc = cc;
				}
				sb.append('"');
				return sb.toString();
			} else if (c < 040 || c >= 0177 || specials.indexOf(c) >= 0)
				// These characters cause the string to be quoted
				needQuoting = true;
		}

		if (needQuoting) {
			StringBuffer sb = new StringBuffer(len + 2);
			sb.append('"').append(word).append('"');
			return sb.toString();
		} 
		return word;
	}

	/**
	 * Fold a string at linear whitespace so that each line is no longer
	 * than 76 characters, if possible.  If there are more than 76
	 * non-whitespace characters consecutively, the string is folded at
	 * the first whitespace after that sequence.  The parameter
	 * <code>used</code> indicates how many characters have been used in
	 * the current line; it is usually the length of the header name. <p>
	 * 
	 * Note that line breaks in the string aren't escaped; they probably
	 * should be.
	 * 
	 * @param	used	characters used in line so far
	 * @param	s	the string to fold
	 * @return		the folded string
	 * @since JavaMail 1.4
	 */
	public static String fold(int used, String s) {
		if (!foldText)
			return s;

		int end;

		// Strip trailing spaces and newlines
		for (end = s.length() - 1; end >= 0; end--) {
			char c = s.charAt(end);
			if (c != ' ' && c != '\t' && c != '\r' && c != '\n')
				break;
		}
		if (end != s.length() - 1)
			s = s.substring(0, end + 1);

		// if the string fits now, just return it
		if (used + s.length() <= 76)
			return s;

		// have to actually fold the string
		StringBuffer sb = new StringBuffer(s.length() + 4);
		char lastc = 0;

		while (used + s.length() > 76) {
			int lastspace = -1;
			for (int i = 0; i < s.length(); i++) {
				if (lastspace != -1 && used + i > 76)
					break;
				char c = s.charAt(i);
				if (c == ' ' || c == '\t')
					if (!(lastc == ' ' || lastc == '\t'))
						lastspace = i;
				lastc = c;
			}

			if (lastspace == -1) {
				// no space, use the whole thing
				sb.append(s);
				s = "";
				used = 0;
				break;
			}
			sb.append(s.substring(0, lastspace));
			sb.append("\r\n");
			lastc = s.charAt(lastspace);
			sb.append(lastc);
			s = s.substring(lastspace + 1);
			used = 1;
		}

		sb.append(s);
		return sb.toString();
	}

	/**
	 * Unfold a folded header.  Any line breaks that aren't escaped and
	 * are followed by whitespace are removed.
	 * 
	 * @param s the string to unfold
	 * @return the unfolded string
	 * @since JavaMail 1.4
	 */
	public static String unfold(String s) {
		if (!foldText)
			return s;

		StringBuffer sb = null;
		int i;

		while ((i = indexOfAny(s, "\r\n")) >= 0) {
			int start = i;
			int l = s.length();
			i++;		// skip CR or NL
			if (i < l && s.charAt(i - 1) == '\r' && s.charAt(i) == '\n')
				i++;	// skip LF

			if (start == 0 || s.charAt(start - 1) != '\\') {
				char c;
				// if next line starts with whitespace, skip all of it
				// XXX - always has to be true?
				if (i < l && ((c = s.charAt(i)) == ' ' || c == '\t')) {
					i++;	// skip whitespace
					while (i < l && ((c = s.charAt(i)) == ' ' || c == '\t'))
						i++;
					if (sb == null)
						sb = new StringBuffer(s.length());
					if (start != 0) {
						sb.append(s.substring(0, start));
						sb.append(' ');
					}
					s = s.substring(i);
					continue;
				}
				// it's not a continuation line, just leave it in
				if (sb == null)
					sb = new StringBuffer(s.length());
				sb.append(s.substring(0, i));
				s = s.substring(i);
			} else {
				// there's a backslash at "start - 1"
				// strip it out, but leave in the line break
				if (sb == null)
					sb = new StringBuffer(s.length());
				sb.append(s.substring(0, start - 1));
				sb.append(s.substring(start, i));
				s = s.substring(i);
			}
		}

		if (sb != null) {
			sb.append(s);
			return sb.toString();
		}
		return s;
	}

	/**
	 * Return the first index of any of the characters in "any" in "s",
	 * or -1 if none are found.
	 * 
	 * This should be a method on String.
	 */
	private static int indexOfAny(final String s, final String any) {
		return indexOfAny(s, any, 0);
	}

	private static int indexOfAny(final String s, final String any, final int start) {
		try {
			int len = s.length();
			for (int i = start; i < len; i++)
				if (any.indexOf(s.charAt(i)) >= 0)
					return i;

			return -1;
		} catch (StringIndexOutOfBoundsException e) {
			return -1;
		}
	}

	/**
	 * MIME LN^ZbgL Java LN^Zbg֕ϊ܂B<p>
	 * 
	 * @param charset MIME LN^Zbg
	 * @return Java LN^ZbgB
	 * 		K؂ȃ}bsOgpłȂꍇAnLN^ZbĝԂ܂B
	 */
	public static String javaCharset(final String charset) {
		if (mime2java == null || charset == null)
			// no mapping table, or charset parameter is null
			return charset;
	
		String alias = (String) mime2java.get(charset.toLowerCase());
		return alias == null ? charset : alias;
	}

	/**
	 * Java LN^Zbg MIME LN^Zbg֕ϊ܂B<p>
	 * 
	 * JDK ̏̃o[W (1.2 ̌) ͂̋@\񋟂\鎖ɒӂĉB
	 * ̏ꍇÃ\bhȂȂ邱Ƃl܂B
	 * 
	 * @param charset JDK ̃LN^Zbg
	 * @return MIME/IANA B
	 * 			}bsOgpłȂꍇAnLN^ZbĝԂ܂B
	 * @since JavaMail 1.1
	 */
	public static String mimeCharset(final String charset) {
		if (java2mime == null || charset == null) 
			// no mapping table or charset param is null
			return charset;

		String alias = (String) java2mime.get(charset.toLowerCase());
		return alias == null ? charset : alias;
	}

	private static String defaultJavaCharset;
	private static String defaultMIMECharset;

	/**
	 * VXě݂̃ftHgP[ɑΉftHg̕Zbg擾܂B<p>
	 * 
	 * @return VXẽftHgP[̃ftHg Java LN^Zbg (MIME LN^Zbgł͂Ȃ)B 
	 * @since JavaMail 1.1
	 */
	public static String getDefaultJavaCharset() {
		if (defaultJavaCharset == null) {
			/*
			 * If mail.mime.charset is set, it controls the default
			 * Java charset as well.
			 */
			String mimecs = null;
			try {
				mimecs = System.getProperty("mail.mime.charset");
			} catch (SecurityException ex) {}	// 

			if (mimecs != null && mimecs.length() > 0) {
				defaultJavaCharset = javaCharset(mimecs);
				return defaultJavaCharset;
	    	}

			try {
				defaultJavaCharset = System.getProperty("file.encoding", "8859_1");
			} catch (SecurityException sex) {
				final class NullInputStream extends InputStream {
					public int read() {
						return 0;
					}
				}

				InputStreamReader reader = new InputStreamReader(new NullInputStream());
				defaultJavaCharset = reader.getEncoding();
				if (defaultJavaCharset == null)
					defaultJavaCharset = "8859_1";
		    }
		}

		return defaultJavaCharset;
	}

	/*
	 * Get the default MIME charset for this locale.
	 */
	static String getDefaultMIMECharset() {
		if (defaultMIMECharset == null) {
			try {
				defaultMIMECharset = System.getProperty("mail.mime.charset");
			} catch (SecurityException ex) {}	// 
		}

		if (defaultMIMECharset == null)
			defaultMIMECharset = mimeCharset(getDefaultJavaCharset());

		return defaultMIMECharset;
	}

	// Tables to map MIME charset names to Java names and vice versa.
	// XXX - Should eventually use J2SE 1.4 java.nio.charset.Charset
	private static Hashtable mime2java;
	private static Hashtable java2mime;

	static {
		java2mime = new Hashtable(40);
		mime2java = new Hashtable(10);

		try {
			// Use this class's classloader to load the mapping file
			// XXX - we should use SecuritySupport, but it's in another package
			InputStream is = javax.mail.internet.MimeUtility.class.getResourceAsStream("/META-INF/javamail.charset.map");

			if (is != null) {
				try {
					is = new LineInputStream(is);
					// Load the JDK-to-MIME charset mapping table
					loadMappings((LineInputStream)is, java2mime);
					// Load the MIME-to-JDK charset mapping table
					loadMappings((LineInputStream)is, mime2java);
				} finally {
					try {
						is.close();
					} catch (Exception cex) {
						// 
					}
				}
			}
		} catch (Exception ex) {}

		// If we didn't load the tables, e.g., because we didn't have
		// permission, load them manually.  The entries here should be
		// the same as the default javamail.charset.map.
		if (java2mime.isEmpty()) {
			java2mime.put("8859_1", "ISO-8859-1");
			java2mime.put("iso8859_1", "ISO-8859-1");
			java2mime.put("iso8859-1", "ISO-8859-1");

			java2mime.put("8859_2", "ISO-8859-2");
			java2mime.put("iso8859_2", "ISO-8859-2");
			java2mime.put("iso8859-2", "ISO-8859-2");

			java2mime.put("8859_3", "ISO-8859-3");
			java2mime.put("iso8859_3", "ISO-8859-3");
			java2mime.put("iso8859-3", "ISO-8859-3");

			java2mime.put("8859_4", "ISO-8859-4");
			java2mime.put("iso8859_4", "ISO-8859-4");
			java2mime.put("iso8859-4", "ISO-8859-4");

			java2mime.put("8859_5", "ISO-8859-5");
			java2mime.put("iso8859_5", "ISO-8859-5");
			java2mime.put("iso8859-5", "ISO-8859-5");

			java2mime.put("8859_6", "ISO-8859-6");
			java2mime.put("iso8859_6", "ISO-8859-6");
			java2mime.put("iso8859-6", "ISO-8859-6");

			java2mime.put("8859_7", "ISO-8859-7");
			java2mime.put("iso8859_7", "ISO-8859-7");
			java2mime.put("iso8859-7", "ISO-8859-7");

			java2mime.put("8859_8", "ISO-8859-8");
			java2mime.put("iso8859_8", "ISO-8859-8");
			java2mime.put("iso8859-8", "ISO-8859-8");

			java2mime.put("8859_9", "ISO-8859-9");
			java2mime.put("iso8859_9", "ISO-8859-9");
			java2mime.put("iso8859-9", "ISO-8859-9");

			java2mime.put("sjis", "Shift_JIS");
			java2mime.put("jis", "ISO-2022-JP");
			java2mime.put("iso2022jp", "ISO-2022-JP");
			java2mime.put("euc_jp", "euc-jp");
			java2mime.put("koi8_r", "koi8-r");
			java2mime.put("euc_cn", "euc-cn");
			java2mime.put("euc_tw", "euc-tw");
			java2mime.put("euc_kr", "euc-kr");
		}

		if (mime2java.isEmpty()) {
			mime2java.put("iso-2022-cn", "ISO2022CN");
			mime2java.put("iso-2022-kr", "ISO2022KR");
			mime2java.put("utf-8", "UTF8");
			mime2java.put("utf8", "UTF8");
			mime2java.put("ja_jp.iso2022-7", "ISO2022JP");
			mime2java.put("ja_jp.eucjp", "EUCJIS");
			mime2java.put("euc-kr", "KSC5601");
			mime2java.put("euckr", "KSC5601");
			mime2java.put("us-ascii", "ISO-8859-1");
			mime2java.put("x-us-ascii", "ISO-8859-1");
// Sugisawa added. 2005/03/04
mime2java.put("utf-7", "UTF7");
mime2java.put("utf7", "UTF7");
		}
	}

	private static void loadMappings(final LineInputStream is, final Hashtable table) {
		String currLine;

		do {
			try {
				currLine = is.readLine();
			} catch (IOException ioex) {
				break; // error in reading, stop
			}

			if (currLine == null) // end of file, stop
				break;
			if (currLine.startsWith("--") && currLine.endsWith("--"))
				// end of this table
				break;	

			// ignore empty lines and comments
			if (currLine.trim().length() == 0 || currLine.startsWith("#"))
				continue;

			// A valid entry is of the form <key><separator><value>
			// where, <separator> := SPACE | HT. Parse this
			StringTokenizer tk = new StringTokenizer(currLine, " \t");
			try {
				String key = tk.nextToken();
				String value = tk.nextToken();
				table.put(key.toLowerCase(), value);
		    } catch (NoSuchElementException nex) {}
		} while (true);
	}

	static final int ALL_ASCII 		= 1;
	static final int MOSTLY_ASCII 	= 2;
	static final int MOSTLY_NONASCII 	= 3;

	/** 
	 * Check if the given string contains non US-ASCII characters.
	 * @param	s	string
	 * @return		ALL_ASCII if all characters in the string 
	 *			belong to the US-ASCII charset. MOSTLY_ASCII
	 *			if more than half of the available characters
	 *			are US-ASCII characters. Else MOSTLY_NONASCII.
	 */
	static int checkAscii(final String s) {
		int ascii = 0, non_ascii = 0;
		int l = s.length();

		for (int i = 0; i < l; i++) {
			if (nonascii(s.charAt(i))) // non-ascii
				non_ascii++;
			else
				ascii++;
		}

		if (non_ascii == 0)
			return ALL_ASCII;
		if (ascii > non_ascii)
			return MOSTLY_ASCII;

		return MOSTLY_NONASCII;
	}

	/** 
	 * Check if the given byte array contains non US-ASCII characters.
	 * 
	 * @param b oCgz
	 * @return ALL_ASCII if all characters in the string 
	 *			belong to the US-ASCII charset. MOSTLY_ASCII
	 *			if more than half of the available characters
	 *			are US-ASCII characters. Else MOSTLY_NONASCII.
	 * 
	 * XXX - this method is no longer used
	 */
	static int checkAscii(final byte[] b) {
		int ascii = 0, non_ascii = 0;

		for (int i=0; i < b.length; i++) {
			// The '&' operator automatically causes b[i] to be promoted
			// to an int, and we mask out the higher bytes in the int 
			// so that the resulting value is not a negative integer.
			if (nonascii(b[i] & 0xff)) // non-ascii
				non_ascii++;
			else
				ascii++;
		}

		if (non_ascii == 0)
			return ALL_ASCII;
		if (ascii > non_ascii)
			return MOSTLY_ASCII;

		return MOSTLY_NONASCII;
	}

	/** 
	 * Check if the given input stream contains non US-ASCII characters.
	 * Upto <code>max</code> bytes are checked. If <code>max</code> is
	 * set to <code>ALL</code>, then all the bytes available in this
	 * input stream are checked. If <code>breakOnNonAscii</code> is true
	 * the check terminates when the first non-US-ASCII character is
	 * found and MOSTLY_NONASCII is returned. Else, the check continues
	 * till <code>max</code> bytes or till the end of stream.
	 * 
	 * @param is ̓Xg[
	 * @param max maximum bytes to check for. The special value
	 *			ALL indicates that all the bytes in this input
	 *			stream must be checked.
	 * @param breakOnNonAscii if <code>true</code>, then terminate the
	 *			the check when the first non-US-ASCII character
	 *			is found.
	 * @return ALL_ASCII if all characters in the string 
	 *			belong to the US-ASCII charset. MOSTLY_ASCII
	 *			if more than half of the available characters
	 *			are US-ASCII characters. Else MOSTLY_NONASCII.
	 */
	static int checkAscii(final InputStream is, int max, final boolean breakOnNonAscii) {
		int ascii = 0, non_ascii = 0;
		int len;
		int block = 4096;
		int linelen = 0;
		boolean longLine = false, badEOL = false;
		boolean checkEOL = encodeEolStrict && breakOnNonAscii;
		byte buf[] = null;

		if (max != 0) {
			block = (max == ALL) ? 4096 : Math.min(max, 4096);
			buf = new byte[block]; 
		}

		while (max != 0) {
			try {
				if ((len = is.read(buf, 0, block)) == -1)
					break;
				int lastb = 0;
				for (int i = 0; i < len; i++) {
					// The '&' operator automatically causes b[i] to 
					// be promoted to an int, and we mask out the higher
					// bytes in the int so that the resulting value is 
					// not a negative integer.
					int b = buf[i] & 0xff;
					if (checkEOL && ((lastb == '\r' && b != '\n') || (lastb != '\r' && b == '\n')))
						badEOL = true;
					if (b == '\r' || b == '\n')
						linelen = 0;
					else {
						linelen++;
						if (linelen > 998)	// 1000 - CRLF
							longLine = true;
				    }

					if (nonascii(b)) {	// non-ascii
						if (breakOnNonAscii) // we are done
							return MOSTLY_NONASCII;
						non_ascii++;
					} else
						ascii++;

					lastb = b;
				}
			} catch (IOException ioex) {
				break;
			}
			if (max != ALL)
				max -= len;
		}

		if (max == 0 && breakOnNonAscii)
			// We have been told to break on the first non-ascii character.
			// We haven't got any non-ascii character yet, but then we
			// have not checked all of the available bytes either. So we
			// cannot say for sure that this input stream is ALL_ASCII,
			// and hence we must play safe and return MOSTLY_NONASCII
			return MOSTLY_NONASCII;

		if (non_ascii == 0) { // no non-us-ascii characters so far
			// If we're looking at non-text data, and we saw CR without LF
			// or vice versa, consider this mostly non-ASCII so that it
			// will be base64 encoded (since the quoted-printable encoder
			// doesn't encode this case properly).
			if (badEOL)
				return MOSTLY_NONASCII;
			// if we've seen a long line, we degrade to mostly ascii
			else if (longLine)
				return MOSTLY_ASCII;
			else
				return ALL_ASCII;
		}

		if (ascii > non_ascii) // mostly ascii
			return MOSTLY_ASCII;

		return MOSTLY_NONASCII;
	}

	static final boolean nonascii(final int b) {
		return b >= 0177 || (b < 040 && b != '\r' && b != '\n' && b != '\t');
	}

	static final Class _mthclass$(final String s) {
		try {
			return Class.forName(s);
		} catch (ClassNotFoundException e) {
			throw new NoClassDefFoundError(e.getMessage());
		}
	}

}

/**
 * An OutputStream that determines whether the data written to
 * it is all ASCII, mostly ASCII, or mostly non-ASCII.
 */
final class AsciiOutputStream extends OutputStream {

	private boolean breakOnNonAscii;
	private int ascii = 0, non_ascii = 0;
	private int linelen = 0;
	private boolean longLine = false;
	private boolean badEOL = false;
	private boolean checkEOL = false;
	private int lastb = 0;
	private int ret = 0;

	public AsciiOutputStream(final boolean breakOnNonAscii, final boolean encodeEolStrict) {
		this.breakOnNonAscii = breakOnNonAscii;
		checkEOL = encodeEolStrict && breakOnNonAscii;
	}

	public void write(final int b) throws IOException {
		check(b);
	}

	public void write(final byte b[]) throws IOException {
		write(b, 0, b.length);
	}

	public void write(final byte b[], final int off, int len) throws IOException {
		len += off;
		for (int i = off; i < len ; i++)
			check(b[i]);
	}

	private final void check(int b) throws IOException {
		b &= 0xff;
		if (checkEOL && ((lastb == '\r' && b != '\n') || (lastb != '\r' && b == '\n')))
			badEOL = true;
		if (b == '\r' || b == '\n')
			linelen = 0;
		else {
			linelen++;
			if (linelen > 998)	// 1000 - CRLF
				longLine = true;
		}

		if (MimeUtility.nonascii(b)) { // non-ascii
			non_ascii++;
			if (breakOnNonAscii) {	// we are done
				ret = MimeUtility.MOSTLY_NONASCII;
				throw new EOFException();
			}
		} else
			ascii++;

		lastb = b;
	}

	/**
	 * Return ASCII-ness of data stream.
	 */
	public int getAscii() {
		if (ret != 0)
			return ret;
		// If we're looking at non-text data, and we saw CR without LF
		// or vice versa, consider this mostly non-ASCII so that it
		// will be base64 encoded (since the quoted-printable encoder
		// doesn't encode this case properly).
		if (badEOL)
			return MimeUtility.MOSTLY_NONASCII;
		if (non_ascii == 0)	// no non-us-ascii characters so far
			// if we've seen a long line, we degrade to mostly ascii
			return longLine ? MimeUtility.MOSTLY_ASCII : MimeUtility.ALL_ASCII;

		if (ascii > non_ascii) // mostly ascii
			return MimeUtility.MOSTLY_ASCII;

		return MimeUtility.MOSTLY_NONASCII;
	}

}
