/*
 * Copyright 1997-2002 Sun Microsystems, Inc. All Rights Reserved.
 */

package javax.mail.internet;

import java.io.UnsupportedEncodingException;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * ̃NX MIME p[^ (ƒl̑) ێ܂B
 */
public final class ParameterList {

	private static class ParamEnum implements Enumeration {

		private Iterator it;

		ParamEnum(Iterator iterator) {
			it = iterator;
		}

		public boolean hasMoreElements() {
			return it.hasNext();
		}

		public Object nextElement() {
			return it.next();
		}

	}

	private static class Value {

		String value;
		String encodedValue;

		private Value() {}

	}

	private Map list = new LinkedHashMap(); // internal map

	private static boolean encodeParameters = false;
	private static boolean decodeParameters = false;
	private static boolean decodeParametersStrict = false;
	private static final char hex[] = {
		'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
	};

	static {
		try {
			String s = System.getProperty("mail.mime.encodeparameters");
			encodeParameters = s != null && s.equalsIgnoreCase("true");
			s = System.getProperty("mail.mime.decodeparameters");
			decodeParameters = s != null && s.equalsIgnoreCase("true");
			s = System.getProperty("mail.mime.decodeparameters.strict");
			decodeParametersStrict = s != null && s.equalsIgnoreCase("true");
		} catch (SecurityException sex) {}
	}

	/**
	 * Ȃ̃RXgN^łB
	 */
	public ParameterList() {}

	/**
	 * p[^XggpRXgN^łB
	 * String ̍\͂ƃp[^̎Wyъi[͓Iɍs܂B
	 * \͂Ɏs ParseException X[܂B
	 * ̃p[^Xg͗LłA ParameterList ɍ\͂鎖ɒӂĉB
	 * 
	 * @param s p[^Xg
	 * @throws ParseException \͂Ɏsꍇ
	 */
	public ParameterList(final String s) throws ParseException {
		HeaderTokenizer h = new HeaderTokenizer(s, HeaderTokenizer.MIME);

		do {
			HeaderTokenizer.Token tk = h.next();
			int type = tk.getType();

			if (type == HeaderTokenizer.Token.EOF) // done
				return;

			if ((char)type == ';') {
				// expect parameter name
				tk = h.next();
				// tolerate trailing semicolon, even though it violates the spec
				if (tk.getType() == HeaderTokenizer.Token.EOF)
					return;
				// parameter name must be a MIME Atom
				if (tk.getType() != HeaderTokenizer.Token.ATOM)
					throw new ParseException("Expected parameter name, got \"" + tk.getValue() + "\"");
				String name = tk.getValue().toLowerCase();

				// expect '='
				tk = h.next();
				if ((char)tk.getType() != '=')
					throw new ParseException("Expected '=', got \"" + tk.getValue() + "\"");

				// expect parameter value
				tk = h.next();
				int type2 = tk.getType();
				// parameter value must be a MIME Atom or Quoted String
				if (type2 != HeaderTokenizer.Token.ATOM && type2 != HeaderTokenizer.Token.QUOTEDSTRING)
					throw new ParseException("Expected parameter value, got \"" + tk.getValue() + "\"");

				String value = tk.getValue();
				if (decodeParameters && name.endsWith("*")) {
					name = name.substring(0, name.length() - 1);
					list.put(name, decodeValue(value));
				} else
					list.put(name, value);
			} else
				throw new ParseException("Expected ';', got \"" + tk.getValue() + "\"");
		} while (true);
	}

	/**
	 * ̃Xgɂp[^̐Ԃ܂B
	 * 
	 * @return p[^̐
	 */
	public int size() {
		return list.size();
	}

	/**
	 * w肳ꂽp[^̒lԂ܂B
	 * p[^͑啶ƏʂȂɒӂĉB
	 * 
	 * @param name p[^
	 * @return p[^̒lBp[^݂Ȃꍇ <code>null</code> Ԃ܂B 
	 */
	public String get(final String name) {
		Object obj = list.get(name.trim().toLowerCase());
		if (obj instanceof Value)
			return ((Value)obj).value;
		return (String) obj;
	}

	/**
	 * p[^ݒ肵܂B̃p[^ɑ݂ꍇA̐VlŒu܂B
	 * 
	 * @param name p[^̖O
	 * @param value p[^̒l
	 */
	public void set(final String name, final String value) {
		list.put(name.trim().toLowerCase(), value);
	}

	/**
	 * p[^ݒ肵܂B̃p[^ɑ݂ꍇA̐VlŒu܂B
	 * If the mail.mime.encodeparameters System property is true, and the parameter value is non-ASCII, it will be encoded with the specified charset, as specified by RFC 2231.
	 * 
	 * @param name p[^̖O
	 * @param value p[^̒l
	 * @param charset charset of the parameter value.
	 * @since JavaMail 1.4
	 */
	public void set(final String name, final String value, final String charset) {
		if (encodeParameters) {
			Value v = encodeValue(value, charset);
			if (v != null)
				list.put(name.trim().toLowerCase(), v);
			else
				set(name, value);
		} else
			set(name, value);
	}

	/**
	 * w肳ꂽp[^ ParameterList 폜܂B
	 * p[^݂ȂꍇÃ\bh͉s܂B
	 * 
	 * @param name p[^̖O
	 */
	public void remove(final String name) {
		list.remove(name.trim().toLowerCase());
	}

	/**
	 * ̃Xg̑Sp[^̖O̗񋓂Ԃ܂B
	 * 
	 * @return ̃Xg̑Sp[^̗
	 */
	public Enumeration getNames() {
		return new ParamEnum(list.keySet().iterator());
	}

	/**
	 *  ParameterList  MIME String ɕϊ܂B
	 * ꂪ̃Xg̏ꍇA̕񂪕Ԃ܂B
	 * 
	 * @return String
	 */
	public String toString() {
		return toString(0);
	}

	/**
	 *  ParameterList  MIME String ɕϊ܂B
	 * ꂪ̃Xg̏ꍇA̕񂪕Ԃ܂B
	 * 
	 * 'used' p[^́Aʂ̃p[^Xg}tB[hɂāA
	 * ɐLĂ镶ʒu̐w肵܂B
	 * ʂ̃p[^Xg}ꏊʂׂɎgp܂B
	 * 
	 * @param used p[^Xg}tB[hŊɎgpĂ镶ʒu̐
	 * @return String
	 */  
	public String toString(int used) {
		StringBuffer sb = new StringBuffer();
		Iterator iterator = list.keySet().iterator();

		while (iterator.hasNext()) {
			String name = (String) iterator.next();
			Object obj = list.get(name);
			String value;
			if (obj instanceof Value) {
				value = ((Value)obj).encodedValue;
				name = name + '*';
			} else {
				value = (String) obj;
			}
			value = quote(value);
			sb.append("; ");
			used += 2;
			int len = name.length() + value.length() + 1;
			if (used + len > 76) { // overflows ...
				sb.append("\r\n\t"); // .. start new continuation line
				used = 8; // account for the starting <tab> char
			}
			sb.append(name).append('=');
			used += name.length() + 1;
			if (used + value.length() > 76) { // still overflows ...
				// have to fold value
				String s = MimeUtility.fold(used, value);
				sb.append(s);
				int lastlf = s.lastIndexOf('\n');
				if (lastlf >= 0)	// always true
					used += s.length() - lastlf - 1;
				else
					used += s.length();
			} else {
				sb.append(value);
				used += value.length();
			}
		}
		return sb.toString();
	}

	// Quote a parameter value token if required.
	private String quote(final String value) {
		return MimeUtility.quote(value, HeaderTokenizer.MIME);
	}

	private Value encodeValue(final String s, final String s1) {
		if (MimeUtility.checkAscii(s) == 1)
			return null;

		byte[] bytes;
		try {
			bytes = s.getBytes(MimeUtility.javaCharset(s1));
		} catch (UnsupportedEncodingException ex) {
			return null;
		}

		StringBuffer sb = new StringBuffer(bytes.length + s1.length() + 2);
		sb.append(s1).append("''");
		for (int i = 0; i < bytes.length; i++) {
			char c = (char) (bytes[i] & 0xff);
			if (c <= ' ' || c >= '\177' || c == '*' || c == '\'' || c == '%' || "()<>@,;:\\\"\t []/?=".indexOf(c) >= 0)
				sb.append('%').append(hex[c >> 4]).append(hex[c & 15]);
			else
				sb.append(c);
		}

		Value value = new Value();
		value.value = s;
		value.encodedValue = sb.toString();
		return value;
	}

	private Value decodeValue(String s) throws ParseException {
		Value value = new Value();
		value.encodedValue = s;
		value.value = s;

		try {
			int i = s.indexOf('\'');
			if (i <= 0)
				if (decodeParametersStrict)
					throw new ParseException("Missing charset in encoded value: " + s);
				else
					return value;
			String s1 = s.substring(0, i);
			int j = s.indexOf('\'', i + 1);
			if (j < 0)
				if (decodeParametersStrict)
					throw new ParseException("Missing language in encoded value: " + s);
				else
					return value;
			/* String s2 = */ s.substring(i + 1, j);
			s = s.substring(j + 1);
			byte[] bytes = new byte[s.length()];
			i = 0;
			int k = 0;
			for (; i < s.length(); i++) {
				char c = s.charAt(i);
				if (c == '%') {
					String s3 = s.substring(i + 1, i + 3);
					c = (char) Integer.parseInt(s3, 16);
					i += 2;
				}
				bytes[k++] = (byte) c;
			}

			value.value = new String(bytes, 0, k, MimeUtility.javaCharset(s1));
		} catch (NumberFormatException nfex) {
			if (decodeParametersStrict)
				throw new ParseException(nfex.toString());
		} catch (UnsupportedEncodingException ueex) {
			if (decodeParametersStrict)
				throw new ParseException(ueex.toString());
		} catch (StringIndexOutOfBoundsException sioobex) {
			if (decodeParametersStrict)
				throw new ParseException(sioobex.toString());
		}
		return value;
	}

}
