package javax.mail.internet;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Enumeration;

import javax.mail.MessagingException;

import com.sun.mail.util.LineOutputStream;

/**
 * A MimeBodyPart that handles data that has already been encoded.
 * This class is useful when constructing a message and attaching data that has already been encoded (for example, using base64 encoding).
 * The data may have been encoded by the application, or may have been stored in a file or database in encoded form.
 * The encoding is supplied when this object is created.
 * The data is attached to this object in the usual fashion, by using the setText, setContent, or setDataHandler methods.
 * 
 * @since JavaMail 1.4
 */
public final class PreencodedMimeBodyPart extends MimeBodyPart {

	private String encoding;

	/**
	 * Create a PreencodedMimeBodyPart that assumes the data is encoded using the specified encoding.
	 * The encoding must be a MIME supported Content-Transfer-Encoding.
	 */
	public PreencodedMimeBodyPart(final String encoding) {
		this.encoding = encoding;
	}

	/**
	 * Returns the content transfer encoding specified when this object was created.
	 */
	public String getEncoding() throws MessagingException {
		return encoding;
	}

	/**
	 * Output the body part as an RFC 822 format stream.
	 */
	public void writeTo(final OutputStream os) throws IOException, MessagingException {
		LineOutputStream los = null;

		if (os instanceof LineOutputStream)
			los = (LineOutputStream) os;
		else
			los = new LineOutputStream(os);

		Enumeration e = getAllHeaderLines();
		while (e.hasMoreElements())
			los.writeln((String)e.nextElement());
		los.writeln();
		getDataHandler().writeTo(os);
		os.flush();
	}

	/**
	 * Force the Content-Transfer-Encoding header to use the encoding that was specified when this object was created.
	 */
	protected void updateHeaders() throws MessagingException {
		super.updateHeaders();
		MimeBodyPart.setEncoding(this, encoding);
	}

}
