/*
 * Copyright 1997-2000 Sun Microsystems, Inc. All Rights Reserved.
 */

package javax.mail.internet;

import javax.mail.Session;

/**
 * This is a utility class that generates unique values. The generated
 * String contains only US-ASCII characters and hence is safe for use
 * in RFC822 headers. <p>
 *
 * This is a package private class.
 */
final class UniqueValue {

	/**
	 * A global part number.  Access is not synchronized because the
	 * value is only one part of the unique value and so doesn't need
	 * to be accurate.
	 */
	private static int part = 0;

	private static int id = 0;

	UniqueValue() {}

	/**
	 * Get a unique value for use in a multipart boundary string.
	 *
	 * This implementation generates it by concatenating a global
	 * part number, a newly created object's <code>hashCode()</code>,
	 * and the current time (in milliseconds).
	 */
	public static String getUniqueBoundaryValue() {
		StringBuffer s = new StringBuffer();

		// Unique string is ----=_Part_<part>_<hashcode>.<currentTime>
		s.append("----=_Part_").append(part++).append('_').append(s.hashCode()).append('.').append(System.currentTimeMillis());
		return s.toString();
	}

	/**
	 * Get a unique value for use in a Message-ID.
	 *
	 * This implementation generates it by concatenating a newly
	 * created object's <code>hashCode()</code>, the current
	 * time (in milliseconds), the string
	 * this user's local address generated by 
	 * <code>InternetAddress.getLocalAddress()</code>.
	 * (The address defaults to "user@localhost" if
	 * <code>getLocalAddress()</code> returns null.)
	 *
	 * @param ssn Session object used to get the local address
	 * @see javax.mail.internet.InternetAddress
	 */
	public static String getUniqueMessageIDValue(final Session ssn) {
		String suffix = null;

		InternetAddress addr = InternetAddress.getLocalAddress(ssn);
		if (addr != null)
			suffix = addr.getAddress();
		else
// Sugisawa changed. 2004/08/06
//			suffix = "javamailuser@localhost"; // worst-case default
// Sugisawa added. 2004/08/06
			suffix = "user@localhost"; // worst-case default

		StringBuffer s = new StringBuffer();

		// Unique string is <hashcode>.<currentTime>.JavaMail.<suffix>
		s.append(s.hashCode()).append('.').append(id++).append(System.currentTimeMillis()).append('.').
// Sugisawa changed. 2004/08/06
//			append("JavaMail.").
			append(suffix);

		return s.toString();
	}

}
