package javax.mail.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.activation.DataSource;
import javax.mail.internet.ContentType;
import javax.mail.internet.MimeUtility;
import javax.mail.internet.ParseException;

/**
 * A DataSource backed by a byte array.
 * The byte array may be passed in directly, or may be initialized from an InputStream or a String.
 * 
 * @since JavaMail 1.4
 */
public final class ByteArrayDataSource implements DataSource {

	private byte[] data;

	/**
	 * MIME type
	 */
	private String type;
	private String name = "";

	/**
	 * Create a ByteArrayDataSource with data from the specified InputStream and with the specified MIME type.
	 * The InputStream is read completely and the data is stored in a byte array.
	 * 
	 * @param is the InputStream
	 * @param type the MIME type
	 * @throws IOException errors reading the stream
	 */
	public ByteArrayDataSource(InputStream is, String type) throws IOException {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		byte[] data = new byte[8192];
		int i;
		while((i = is.read(data)) > 0) 
			baos.write(data, 0, i);
		this.data = baos.toByteArray();
		this.type = type;
	}

	/**
	 * Create a ByteArrayDataSource with data from the specified byte array and with the specified MIME type.
	 * 
	 * @param data the data
	 * @param type the MIME type
	 */
	public ByteArrayDataSource(final byte[] data, final String type) {
		this.data = data;
		this.type = type;
	}

	/**
	 * Create a ByteArrayDataSource with data from the specified String and with the specified MIME type.
	 * The MIME type should include a charset parameter specifying the charset to be used for the string.
	 * If the parameter is not included, the default charset is used.
	 * 
	 * @param data the String
	 * @param type the MIME type
	 * @throws IOException errors reading the String
	 */
	public ByteArrayDataSource(final String data, final String type) throws IOException {
		String s = null;

		try {
			ContentType contentType = new ContentType(type);
			s = contentType.getParameter("charset");
		} catch(ParseException parseexception) {}

		if (s == null)
			s = MimeUtility.getDefaultJavaCharset();

		this.data = data.getBytes(s);
		this.type = type;
	}

	/**
	 * Return an InputStream for the data.
	 * Note that a new stream is returned each time this method is called.
	 */
	public InputStream getInputStream() throws IOException {
		if (data == null)
			throw new IOException("no data");
		return new ByteArrayInputStream(data);
	}

	/**
	 * Return an OutputStream for the data.
	 * Writing the data is not supported; an IOException is always thrown.
	 */
	public OutputStream getOutputStream() throws IOException {
		throw new IOException("cannot do this");
	}

	/**
	 * Get the MIME content type of the data.
	 */
	public String getContentType() {
		return type;
	}

	/**
	 * Get the name of the data.
	 * By default, an empty string ("") is returned.
	 */
	public String getName() {
		return name;
	}

	/**
	 * Set the name of the data.
	 * @param name
	 */
	public void setName(final String name) {
		this.name = name;
	}

}
