/*
 * Copyright 1994-2004 Sun Microsystems, Inc. All Rights Reserved.
 */

package javax.mail.util;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;

import javax.mail.internet.SharedInputStream;

/**
 * <code>SharedFileInputStream</code>  <code>mark</code>  <code>reset</code> \bhT|[gA
 * t@Cf[^obt@O <code>BufferedInputStream</code> łB
 * A<code>newStream</code> t@C̕W\̃Xg[쐬郁\bhT|[g܂B
 * <code>RandomAccessFile</code> IuWFNǵAt@Cf[^ɃANZXׂɎgp܂B
 * 
 * @since JavaMail 1.4
 */
public final class SharedFileInputStream extends BufferedInputStream implements SharedInputStream {

	private static int defaultBufferSize = 2048;

	/**
	 * f[^܂ރt@CB֘ASĂ SharedFileInputStreams ɋL܂B
	 */
	protected RandomAccessFile in;

	/**
	 * Ǎ݃obt@̕WTCY
	 */
	protected int bufsize;

	/**
	 * The file offset that corresponds to the first byte in the read buffer.
	 */
	protected long bufpos;

	/**
	 * The file offset of the start of data in this subset of the file.
	 */
	protected long start = 0;

	/**
	 * The amount of data in this subset of the file.
	 */
	protected long datalen;

	private boolean master = true;
	private SharedFile sf;

	/**
	 * A shared class that keeps track of the references
	 * to a particular file so it can be closed when the
	 * last reference is gone.
	 */
	class SharedFile {

        private int cnt;
        private RandomAccessFile in;

		SharedFile(final String name) throws IOException {
			super();
			in = new RandomAccessFile(name, "r");
		}

		SharedFile(final File file) throws IOException {
			super();
			in = new RandomAccessFile(file, "r");
		}

		public RandomAccessFile open() {
			cnt++;
			return in;
		}

		public synchronized void close() throws IOException {
			if (cnt > 0 && --cnt <= 0)
				in.close();
		}

		public synchronized void forceClose() throws IOException {
			if (cnt > 0) {
				cnt = 0;
				in.close();
			} else {
				try {
					in.close();
				} catch (IOException e) {}
			}
		}

		protected void finalize() throws Throwable {
			super.finalize();
			in.close();
		}

	}

	/**
	 * Check to make sure that this stream has not been closed
	 */
	private void ensureOpen() throws IOException {
		if (in == null)
			throw new IOException("Stream closed");
	}

	/**
	 * Creates a SharedFileInputStream for the file.
	 * 
	 * @param file t@C
	 * @throws IOException
	 */
	public SharedFileInputStream(final File file) throws IOException {
		this(file, defaultBufferSize);
	}

	/**
	 * w肳ꂽt@C <code>SharedFileInputStream</code> 쐬܂B
	 * 
	 * @param file t@C
	 * @throws IOException
	 */
	public SharedFileInputStream(final String file) throws IOException {
		this(file, defaultBufferSize);
	}

	/**
	 * obt@TCYw肵 <code>SharedFileInputStream</code> 쐬܂B
	 * 
	 * @param file t@C
	 * @param size obt@TCY
	 * @throws IllegalArgumentException <code>size</code>  <code>0</code> ȉ̏ꍇ
	 * @throws IOException
	 */
	public SharedFileInputStream(final File file, final int size) throws IOException {
		super(null);	// XXX - will it NPE?

		if (size <= 0)
			throw new IllegalArgumentException("Buffer size <= 0");

		init(new SharedFile(file), size);
	}

	/**
	 * obt@TCYw肵 <code>SharedFileInputStream</code> 쐬܂B
	 * 
	 * @param file t@C
	 * @param size obt@TCY
	 * @throws IllegalArgumentException <code>size</code>  <code>0</code> ȉ̏ꍇ
	 * @throws IOException
	 */
	public SharedFileInputStream(final String file, final int size) throws IOException {
		super(null);	// XXX - will it NPE?

		if (size <= 0)
			throw new IllegalArgumentException("Buffer size <= 0");

		init(new SharedFile(file), size);
	}

	private void init(SharedFile file, int size) throws IOException {
		this.sf = file;
		this.in = file.open();
		this.start = 0;
		this.datalen = in.length();	// XXX - file can't grow
		this.bufsize = size;
		this.buf = new byte[size];
	}

	/**
	 * <code>newStream</code> \bhŎgp܂B
	 */
	private SharedFileInputStream(final SharedFile sf, final long start, final long len, final int bufsize) {
		super(null);
		this.master = false;
		this.sf = sf;
		this.in = sf.open();
		this.start = start;
		this.bufpos = start;
		this.datalen = len;
		this.bufsize = bufsize;
		this.buf = new byte[bufsize];
	}

	/**
	 * Fills the buffer with more data, taking into account
	 * shuffling and other tricks for dealing with marks.
	 * Assumes that it is being called by a synchronized method.
	 * This method also assumes that all data has already been read in,
	 * hence pos > count.
	 */
	private void fill() throws IOException {
		if (markpos < 0) {
			pos = 0;		/* no mark: throw away the buffer */
			bufpos += count;
		} else if (pos >= buf.length)	/* no room left in buffer */
			if (markpos > 0) {	/* can throw away early part of the buffer */
				int sz = pos - markpos;
				System.arraycopy(buf, markpos, buf, 0, sz);
				pos = sz;
				bufpos += markpos;
				markpos = 0;
			} else if (buf.length >= marklimit) {
				markpos = -1;	/* buffer got too big, invalidate mark */
				pos = 0;	/* drop buffer contents */
				bufpos += count;
			} else {		/* grow buffer */
				int nsz = pos * 2;
				if (nsz > marklimit)
					nsz = marklimit;
				byte nbuf[] = new byte[nsz];
				System.arraycopy(buf, 0, nbuf, 0, pos);
				buf = nbuf;
			}
		count = pos;
		in.seek(bufpos + (long)pos);
		// limit to datalen
		int len = buf.length - pos;
		if (bufpos - start + pos + len > datalen)
			len = (int)(datalen - (bufpos - start + pos));
		int n = in.read(buf, pos, len);
		if (n > 0)
			count = n + pos;
	}

	/**
	 * See
	 * the general contract of the <code>read</code>
	 * method of <code>InputStream</code>.
	 * 
	 * @return the next byte of data, or <code>-1</code> if the end of the stream is reached.
	 * @throws IOException o͗Oꍇ
	 */
	public synchronized int read() throws IOException {
		ensureOpen();
		if (pos >= count) {
			fill();
			if (pos >= count)
				return -1;
		}
		return buf[pos++] & 0xff;
	}

	/**
	 * Read characters into a portion of an array, reading from the underlying
	 * stream at most once if necessary.
	 */
	private int read1(final byte[] b, final int off, final int len) throws IOException {
		int avail = count - pos;
		if (avail <= 0) {
			fill();
			avail = count - pos;
			if (avail <= 0)
				return -1;
		}
		int cnt = (avail < len) ? avail : len;
		System.arraycopy(buf, pos, b, off, cnt);
		pos += cnt;
		return cnt;
	}

	/**
	 * Reads bytes from this stream into the specified byte array,
	 * starting at the given offset.
	 * 
	 * <p> This method implements the general contract of the corresponding
	 * <code>{@link java.io.InputStream#read(byte[], int, int) read}</code>
	 * method of the <code>{@link java.io.InputStream}</code> class.
	 * 
	 * @param b destination buffer.
	 * @param off offset at which to start storing bytes.
	 * @param len maximum number of bytes to read.
	 * @return the number of bytes read, or <code>-1</code> if the end of
	 *             the stream has been reached.
	 * @throws IOException o͗Oꍇ
	 */
	public synchronized int read(byte[] b, int off, int len) throws IOException {
		ensureOpen();
		if ((off | len | off + len | b.length - (off + len)) < 0)
			throw new IndexOutOfBoundsException();
		if (len == 0)
			return 0;

		int n = read1(b, off, len);
		if (n <= 0)
			return n;
		while ((n < len) /* && (in.available() > 0) */) {
			int n1 = read1(b, off + n, len - n);
			if (n1 <= 0)
				break;
			n += n1;
		}
		return n;
	}

	/**
	 * See the general contract of the <code>skip</code> method of <code>InputStream</code>.
	 * 
	 * @param n the number of bytes to be skipped.
	 * @return the actual number of bytes skipped.
	 * @throws IOException o͗Oꍇ
	 */
	public synchronized long skip(final long n) throws IOException {
		ensureOpen();
		if (n <= 0)
			return 0;

		long avail = count - pos;
		if (avail <= 0) {
			// If no mark position set then don't keep in buffer
			/*
				if (markpos <0) 
					return in.skip(n);
			*/

			// Fill in buffer to save bytes for reset
			fill();
			avail = count - pos;
			if (avail <= 0)
				return 0;
		}

		long skipped = (avail < n) ? avail : n;
		pos += skipped;
		return skipped;
	}

	/**
	 * ̓̓Xg[ubLOȂœǂގłoCgԂ܂B
	 * 
	 * @return ̓̓Xg[ubLOȂœǂގłoCg
	 * @throws IOException o͗Oꍇ
	 */
	public synchronized int available() throws IOException {
		ensureOpen();
		return (count - pos) + in_available();
	}

	private int in_available() /* throws IOException */ {
		// XXX - I[ot[
		return (int)((start + datalen) - (bufpos + count));
	}

	/**
	 * See the general contract of the <code>mark</code>
	 * method of <code>InputStream</code>.
	 * 
	 * @param readlimit the maximum limit of bytes that can be read before
	 *                      the mark position becomes invalid.
	 * @see #reset()
	 */
	public synchronized void mark(final int readlimit) {
		marklimit = readlimit;
		markpos = pos;
	}

	/**
	 * See the general contract of the <code>reset</code>
	 * method of <code>InputStream</code>.
	 * <p>
	 * If <code>markpos</code> is <code>-1</code>
	 * (no mark has been set or the mark has been
	 * invalidated), an <code>IOException</code>
	 * is thrown. Otherwise, <code>pos</code> is
	 * set equal to <code>markpos</code>.
	 * 
	 * @throws IOException if this stream has not been marked or
	 *               if the mark has been invalidated.
	 * @see #mark(int)
	 */
	public synchronized void reset() throws IOException {
		ensureOpen();
		if (markpos < 0)
            throw new IOException("Resetting to invalid mark");
		pos = markpos;
	}

	/**
	 * Tests if this input stream supports the <code>mark</code> 
	 * and <code>reset</code> methods. The <code>markSupported</code> 
	 * method of <code>SharedFileInputStream</code> returns 
	 * <code>true</code>. 
	 * 
	 * @return a <code>boolean</code> indicating if this stream type supports
	 *          the <code>mark</code> and <code>reset</code> methods.
	 * @see java.io.InputStream#mark(int)
	 * @see java.io.InputStream#reset()
	 */
	public boolean markSupported() {
		return true;
	}

	/**
	 * Closes this input stream and releases any system resources 
	 * associated with the stream. 
	 *
	 * @throws IOException o͗Oꍇ
	 */
	public void close() throws IOException {
		if (in == null)
			return;
		try {
			if (master)
				sf.forceClose();
			else
				sf.close();
		} finally {
			sf = null;
			in = null;
			buf = null;
		}
	}

	/**
	 * Return the current position in the InputStream, as an offset from the beginning of the InputStream.
	 */
	public long getPosition() {
//		System.out.println("getPosition: start " + start + " pos " + pos + " bufpos " + bufpos + " = " + (bufpos + pos - start));
		if (in == null)
			throw new RuntimeException("Stream closed");
		return bufpos + pos - start;
	}

	/**
	 * Return a new InputStream representing a subset of the data from this InputStream, starting at start (inclusive) up to end (exclusive).
	 * start must be non-negative. If end is -1, the new stream ends at the same place as this stream.
	 * The returned InputStream will also implement the SharedInputStream interface.
	 */
	public InputStream newStream(final long start, long end) {
		if (in == null)
			throw new RuntimeException("Stream closed");
		if (start < 0)
			throw new IllegalArgumentException("start < 0");
		if (end == -1)
			end = datalen;
		return new SharedFileInputStream(sf, this.start + (int)start, (int)(end - start), bufsize);
	}

	/**
	 * Force this stream to close.
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		close();
	}

}
