package org.lixm.core.model;

import org.lixm.core.common.XMLType;


/**
 * <p>要素をモデル化します。</p>
 * @author tasogare
 * @version 1.1.2
 */
public class ElementModel extends AbstractModel {

    private String name;
    private String pref;
    private String id;

    public ElementModel(
        String name, String pref, String id)
    {
        super(XMLType.ELEMENT);

        this.name = name;
        this.pref = pref;
        this.id = id;
    }

    /**
     * <p>実装依存の方法で要素名文字列を取得します。<br>
     * 主にデバッグやエンドユーザーメッセージに利用されるユーティリティです。</p>
     * 
     * 要素のローカル名や完全修飾名を取得したい場合は{@link #toLocalName()}か
     * {@link #toQname()}を利用して下さい。
     * 
     * @see #toLocalName()
     * @see #toQname()
     * 
     * @return 実装依存な要素名文字列
     */
    public String getName() {
        return name;
    }

	/**
	 * <p>要素のローカル名を設定します。</p>
	 * 
	 * @param name 新たな要素のローカル名
	 */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * <p>名前空間プレフィックスを取得します。存在しない場合は<code>null</code>となります。</p>
     * @return 名前空間プレフィックス。存在しない場合は<code>null</code>
     */
    public String getPefix() {
        return pref;
    }

    /**
     * <p>名前空間プレフィックスを設定します。存在しない場合は<code>null</code>となります。</p>
     * 
     * @param pref 新たな名前空間プレフィックス
     */
    public void setPefix(String pref) {
        this.pref = pref;
    }

    /**
     * <p>名前空間URIを取得します。存在しない場合は<code>nullとなります。</code></p>
     * @return 名前空間URI。存在しない場合は<code>null</code>
     */
    public String getNamespace() {
        return id;
    }

    /**
     * <p>名前空間URIを設定します。存在しない場合は<code>nullとなります。</code></p>
     * 
     * @param namespace 新たな名前空間URI
     */
    public void setNamespace(String namespace) {
        this.id = namespace;
    }

    /**
     * <p>要素のローカル名を取得します。</p>
     * 
     * @return 要素のローカル名
     */
    public String toLocalName() {
        return getName( );
    }

    /**
     * <p>要素の完全修飾名を取得します。完全修飾名が存在しない場合必ず<code>null</code>
     * になります。</p>
     * 
     * @return 要素の完全修飾名。完全修飾名が存在しない場合は<code>null</code>。
     */
    public String toQname() {
        String QName;
        if (getPefix( ) != null  && !getPefix().equals("")) {
            QName = getPefix( ) + ":" + getName( );
        } else {
            QName = getName( );
        }
        return QName;
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        return toQname();
    }

    /**
     * <p>モデルの同一性を論理的に調べその結果を返します。</p>
     * @return 同一のモデルなら<code>true</code>、違うなら<code>false</code>。
     */
    public boolean equals(
        Object o)
    {
        if (!super.equals(o)) return false;
        return ( this.toString( ) == o.toString( ) );
    }
}
