/*
 LKST log gate analyzer

 Copyright (C) HITACHI,LTD. 2004-2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */
#ifndef __LKSTLA_H__
#define __LKSTLA_H__

#include <linux/lkst.h>
#include <linux/lkst_buffer.h>
#include <timespec.h>
#include <logfile.h>
#include <slot.h>

#define ULMASK (~0UL)

#define NORM_ARGIDX(i) (((i - 1) % 4) + 1)
#define ARG1UL(rec) ((unsigned long)rec->log_arg1 & ULMASK)
#define ARG2UL(rec) ((unsigned long)rec->log_arg2 & ULMASK)
#define ARG3UL(rec) ((unsigned long)rec->log_arg3 & ULMASK)
#define ARG4UL(rec) ((unsigned long)rec->log_arg4 & ULMASK)
#define ARG(rec, i) ( *(&(rec)->log_arg1 + (i) - 1) )

#ifndef MAX
#define MAX(x, y) (((x) > (y))? (x): (y))
#endif
#define XORL(x, y) ((unsigned long)(x) ^ (unsigned long)(y))

#define ADDR_LEN ((int)(sizeof(void *) << 1))

extern int quiet_mode; /* true: do not print header */

struct command_option {
	char * opt;
	char * format;
	char * description;
	int (* handler)(int , char *);
};

struct analysis_formatter;

#define GAF_TIME_EV 1
#define GAF_LONG_EV 0

typedef enum {
	GAT_IGNORE=0,
	GAT_EVENT=1,
	GAT_START=2,
	GAT_END=3,
	GAT_PAUSE=4,
	GAT_RESUME=5,
	GAT_REMOVE=6,
	GAT_EN_ST=7,
} ga_type_t;

#define GA_MAXOPTS 64
#define FMT_SIZE 1024

struct column {
	char *title;
	int width;
	char *format;
};

#define COLUMN_MAX 4
#define COL_KEY 0
#define COL_ALIAS 1
#define COL_TIME 2
#define COL_METRIC 3 

#define SET_COL(ga, col, _title, _width, _format) \
do {                                              \
        ga.cols[col].title = _title;              \
        ga.cols[col].width = _width;              \
        ga.cols[col].format = _format;            \
} while (0)

/*
 * struct : gate_analyzer
 * description : event gate analyzer 
 */
struct gate_analyzer {
	/* common interfaces & proparties */
	char * name;		/*analyzer name*/
	char * description;	/*analyzer description*/	
	char format_data[FMT_SIZE]; /* format string of each raw */

	/* column info (title, width and format) */
	struct column cols[COLUMN_MAX];

	/* initialization function */
	int (*init)(void);
	
	/* optionals */
	int nr_options;
	struct command_option * options[GA_MAXOPTS]; /*extra options*/

	/*main analyzer loop*/
	int (*analyze_logs)(struct gate_analyzer *ga, 
			    struct lkstloglist *llist,
			    struct analysis_formatter *fmt);

	/*for distribution formatter -- default values*/
	int flag;
	int ranks;
	double max;
	double min;

	/* get event type */
	ga_type_t (*get_type)(struct lkst_log_record *);

	/* make session key (use session analyzers only)*/
	slot_hkey (*session_key)(struct lkst_log_record *);

	/* get alias name of information key */
	char *    (*get_alias)(slot_hkey key);

	/* in event analyzer, pass same record to following functions*/
	/* make information key (for classification) */
	slot_hkey (*info_key)(struct lkst_log_record *,
			      struct lkst_log_record *);

	/* for pid filter */
	int (*get_pid)(struct lkst_log_record *,
		       struct lkst_log_record *);
	/* get event time & time filter */
	double (*get_time)(struct timespec *ts, 
			   struct lkst_log_record *,
			   struct lkst_log_record *);
	/* get metric value */
	double (*get_metric)(struct lkst_log_record *,
			     struct lkst_log_record *);
	/* get cpu id */
	int (*get_cpu)(struct lkst_log_record *,
		       struct lkst_log_record *);

	/*session correspondence*/
	int multi_session;
};

/*default special functions*/
extern int init_none(void);
extern slot_hkey session_key_none(struct lkst_log_record *rec);
extern int get_cpu_1st(struct lkst_log_record *,
		       struct lkst_log_record *);
extern int get_pid_logpid(struct lkst_log_record *,
			  struct lkst_log_record *);
extern double get_time_logtime(struct timespec *,
			       struct lkst_log_record *,
			       struct lkst_log_record *);
extern double get_metric_logtime(struct lkst_log_record *,
				struct lkst_log_record *);

extern int analyze_sessions(struct gate_analyzer *ga,
			    struct lkstloglist *llist,
			    struct analysis_formatter *fmt);
extern int analyze_events(struct gate_analyzer *ga,
			  struct lkstloglist *llist,
			  struct analysis_formatter *fmt);

#define GA_INIT_DEF_TIME \
	.flag = GAF_TIME_EV, \
	.ranks = 8, \
	.min = 0.0000001,\
	.max = 10,

#define GA_INIT_DEF_LONG \
	.flag = GAF_LONG_EV, \
	.ranks = 9, \
	.min = 0,\
	.max = 100, 

#define GA_INIT_RET_EVENT_TIME \
	.get_time = get_time_logtime,

#define GA_INIT_1TO1_SESSION \
	.multi_session = 0,

#define GA_INIT_1TON_SESSION \
	.multi_session = 1,

#define GA_INIT_RET_SESSION_TIME \
	.get_time = get_time_logtime, \
	.get_metric = get_metric_logtime, \
	GA_INIT_DEF_TIME

#define GA_INIT_RET_EVENT_PID \
	.get_pid = get_pid_logpid,

#define GA_INIT_NO_EXTRAOPTS \
	.nr_options = 0, \
	.options[0] = NULL,

#define GA_INIT_SESSION_ANALYZER \
	.analyze_logs = analyze_sessions, \
	.get_cpu = get_cpu_1st,

#define GA_INIT_EVENT_ANALYZER \
	.analyze_logs = analyze_events, \
	.session_key = session_key_none,\
	.get_cpu = get_cpu_1st,

struct analysis_formatter {
	struct command_option opt;
	int active;
	
	/* analyze handler */
	int (*pre_analyze)(struct gate_analyzer *ga);
	void (*callback)(struct gate_analyzer *ga, slot_hkey key, 
			 struct timespec *ts, double metric, int pid);
	void (*post_analyze)(struct gate_analyzer *ga);
};

#endif
