/*
 LKST log analyzer base functions

 Copyright (C) HITACHI,LTD. 2004-2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
                                                                                                                             
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <logfile.h>
#include <lkstla.h>
#include <timespec.h>
#include <limits.h>
#include <ctype.h>

#define LOG_HASH_BITS 6
static generic_slots_t *gsl_session = NULL;

#define MAXPARAMS 255
//C lang template macro ...

#define FILTER_DEF(tname, type, strto) \
	struct tname##_parameter { \
		int nr; \
		struct { \
			int inv; \
			type begin,end; \
		} list[MAXPARAMS]; \
	};\
	static int tname##_parser(int c, char *opt, struct tname##_parameter * param) {\
		type val1,val2; \
		int i; \
		char *ptr = opt-1; \
		i = param->nr; \
		do { \
			opt = ptr+1; \
			if (opt[0]=='!') { param->list[i].inv = 1; opt++; } \
			else { param->list[i].inv = 0;} \
			val2 = val1 = strto(opt,&ptr,0); \
			if ( ptr == opt ) return -1; \
			switch (*ptr) { \
			case ':': \
				val2=0; \
			case '+': \
				opt = ptr+1; \
				val2 += strto(opt,&ptr,0); \
				if ( ptr == opt ) { \
					if(*opt==','||*opt=='\0') { \
						val2=LONG_MAX; \
					} else {\
						return -1; \
					} \
				} \
				break; \
			case '\0': \
			case ',': \
				break;\
			default: \
				printf("Perse error: %c\n",*ptr);\
				return -1;\
			} \
			if (val1>val2) return -1;\
			param->list[i].begin = val1; \
			param->list[i].end = val2; \
			if ((++i)==MAXPARAMS)break; \
		} while (*ptr==','); \
		param->nr = i; \
		if (*ptr!='\0') return -1; \
		return 0; \
	} \
	static int tname##_filter(type val, struct tname##_parameter * param) { \
		int i; \
		for (i=0;i<param->nr;i++) { \
			if ( ! param->list[i].inv ) { \
				if (val >= param->list[i].begin &&  \
					    val <= param->list[i].end) break; \
			} else { \
				if (val < param->list[i].begin || \
					    val > param->list[i].end) break; \
			} \
		} \
		return i!=param->nr || param->nr==0; \
	}

#define strtod2(a,b,c) strtod(a,b);

FILTER_DEF(ulong, unsigned long, strtoul);
FILTER_DEF(double, double, strtod2);

#define ULONG_FILTER_DEF(c,name,desc) \
	static struct ulong_parameter name##param={0,}; \
	static int opt_handler_##name(int c, char *opt) { \
		return ulong_parser(c, opt, &(name##param)); \
	} \
	static inline int name##_filter(unsigned long val) { \
		return ulong_filter(val, &(name##param)); \
	} \
	struct command_option name##_option = { \
		.opt = #c ":", \
		.format = "-" #c "<[!]" #name "[+span|:" #name "][,...]]>", \
		.description = desc, \
		.handler = opt_handler_##name,};

ULONG_FILTER_DEF(k,key,"specify infokey filter");
ULONG_FILTER_DEF(p,pid,"specify PID filter");
ULONG_FILTER_DEF(c,cpu,"specify CPU filter");

#define DOUBLE_FILTER_DEF(c,name,desc) \
	static struct double_parameter name##param={0,}; \
	static int opt_handler_##name(int c, char *opt) { \
		return double_parser(c, opt, &(name##param)); \
	} \
	static inline int name##_filter(double val) { \
		return double_filter(val, &(name##param)); \
	} \
	struct command_option name##_option = { \
		.opt = #c ":", \
		.format = "-" #c "<[!]start[+span|:end][,...]]>", \
		.description = desc, \
		.handler = opt_handler_##name,};

DOUBLE_FILTER_DEF(m,metric,"specify metric filter.");
DOUBLE_FILTER_DEF(t,time,"specify time filter.");


/*default special functions*/
static struct timespec zero_ts={0,0};

int get_pid_logpid(struct lkst_log_record *start,
		   struct lkst_log_record *end)
{
	if(start) {
		return start->posix.log_pid;
	} else
		return -1;
}
double get_time_logtime(struct timespec *ts,
			struct lkst_log_record *start,
			struct lkst_log_record *end)
{
	if (start) {
		*ts = start->posix.log_time;
	} else {
		*ts = zero_ts;
	}
	return ts2d(*ts);
}
double get_metric_logtime(struct lkst_log_record *start,
			 struct lkst_log_record *end) 
{
	struct timespec ts;
	if (start && end) {
		ts = end->posix.log_time;
		timespec_dec(ts,start->posix.log_time);
		return ts2d(ts);
	} else {
		return (double)0;
	}
}

int init_none(void) 
{
	return 0;
}

slot_hkey session_key_none(struct lkst_log_record *rec)
{
	return HKEY_UNUSED;
}

int get_cpu_1st(struct lkst_log_record *_1st, struct lkst_log_record *_2nd) 
{
	return _1st->posix.log_processor;
}

/*
 * function: analyze_logs
 * desc: analyze logfile and match same context entries.
 * args: llist -- list of logfiles
 *       ga -- gate analyzer
 *       fn -- matched entry handler
 * retval: 0 -- success
 *        -1 -- failed to read header.
 */

struct session_record{
	struct lkst_log_record rec;
	struct timespec pause_total;
	struct timespec pause_ts;
};

int analyze_sessions(struct gate_analyzer *ga, struct lkstloglist *llist,
		     struct analysis_formatter *fmt) 
{
	struct lkst_log_record *rec, *stopper, *starter, *ender;
	struct timespec ts;
	slot_t *slot;
	slot_hkey key, ikey;
	double metric;
	int pid;
	ga_type_t type;
	
	/*init session slots*/
	gsl_session = new_generic_slots(LOG_HASH_BITS, sizeof(struct session_record));
	if (gsl_session == NULL) {
		printf("error: fail to memory allocation\n");
		return -ENOMEM;
	}
	rec = stopper = NULL;
	while ((rec = loglist_earliest_entry(llist, rec)) != NULL) {
		if (rec->posix.log_event_type == LKST_ETYPE_LKST_OVWRTN_REC) {
			lkstlogfile_t *lf;
			release_all_slots(gsl_session);//dispose all sessions
			lf = find_logfile_from_entry(llist,rec);
			if (!lf) goto end_analyze; //debug!
			do{//read next valid entry
				if(logfile_read_one(lf)<1) {
					goto end_analyze;
				}
			}while (lf->entry.posix.log_event_type == 
				LKST_ETYPE_LKST_OVWRTN_REC);
			stopper = &(lf->entry);
		}
		if (stopper) {
			if( stopper != rec)
				continue;
			else
				stopper = NULL;
		}
		
		type = ga->get_type(rec);
		switch (type) {
		case GAT_END:
		case GAT_EN_ST:
			key = ga->session_key(rec);
			for (slot = find_slot(gsl_session, key);
			     slot != NULL; 
			     slot = find_slot(gsl_session, key)) {
				struct session_record *srec = slot_data(slot);
				starter = &srec->rec;
				ender = rec;
				/*decrement pausing time*/
				timespec_dec(ender->posix.log_time,srec->pause_total);
				
				metric = ga->get_metric(starter, ender);
				ikey = ga->info_key(starter, ender);
				pid = ga->get_pid(starter, ender);
				/*filtering*/
				if (!key_filter(ikey) ||
				    !pid_filter(pid) ||
				    !time_filter(ga->get_time(&ts, starter,ender)) || 
				    !cpu_filter(ga->get_cpu(starter,ender)) ||
				    !metric_filter(metric))
					break;
				fmt->callback(ga, ikey, &ts,metric, pid);
				release_slot(gsl_session, key);
				if(!ga->multi_session) break;
			}
			if ( type == GAT_END)
				break;
		case GAT_START:
			key = ga->session_key(rec);
			slot = get_free_slot(gsl_session, key); /*reserve slot*/
			if (slot!=NULL) {
				struct session_record *srec = slot_data(slot);
				srec->rec = *rec;
				srec->pause_ts = srec->pause_total = zero_ts;
			}
			break;
		case GAT_PAUSE:
			key = ga->session_key(rec);
			for (slot = find_slot(gsl_session, key);
			     slot != NULL && slot->key == key; 
			     slot = slot->next) {
				struct session_record *srec = slot_data(slot);
				srec->pause_ts = rec->posix.log_time;
				if(!ga->multi_session) break;
			}
			break;
		case GAT_RESUME:
			key = ga->session_key(rec);
			for (slot = find_slot(gsl_session, key);
			     slot != NULL && slot->key == key; 
			     slot = slot->next) {
				struct session_record *srec = slot_data(slot);
				struct timespec ts;
				ts = rec->posix.log_time;
				if (! timespec_eq(srec->pause_ts, zero_ts)){
					timespec_dec(ts, srec->pause_ts);
					timespec_inc(srec->pause_total, ts);
					srec->pause_ts = zero_ts;
				}
				if(!ga->multi_session) break;
			}
			break;
		case GAT_REMOVE:
			key = ga->session_key(rec);
			for (slot = find_slot(gsl_session, key);
			     slot != NULL; 
			     slot = find_slot(gsl_session, key)) {
				release_slot(gsl_session, key);
				if(!ga->multi_session) break;
			}
		default:
			break;
		}
	}
end_analyze:
	free_generic_slots(gsl_session);
	gsl_session = NULL;
	return 0;
}

int analyze_events(struct gate_analyzer *ga, struct lkstloglist *llist,
		   struct analysis_formatter *fmt) 
{
	struct lkst_log_record *rec;
	struct timespec ts;
	int pid;
	slot_hkey ikey;
	double metric;
	
	rec = NULL;
	while ((rec = loglist_earliest_entry(llist, rec)) != NULL) {
		if (rec->posix.log_event_type == LKST_ETYPE_LKST_OVWRTN_REC)
			continue;
		if (ga->get_type(rec) != GAT_EVENT) continue;
		
		;
		/*filtering*/
		if (!key_filter(ikey = ga->info_key(rec, rec)) ||
		    !pid_filter(pid = ga->get_pid(rec, rec)) ||
		    !time_filter(ga->get_time(&ts, rec, rec)) ||
		    !cpu_filter(ga->get_cpu(rec,rec)) ||
		    !metric_filter(metric = ga->get_metric(rec, rec)))
			continue;
		fmt->callback(ga, ikey, &ts,
			      metric, pid);
	}
	return 0;
}

