/*
 LKST distribution formatter

 Copyright (C) HITACHI,LTD. 2004-2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <lkstla.h>
#include <math.h>

static int pre_analyze_dist(struct gate_analyzer *ga);
static void callback_dist(struct gate_analyzer *ga, slot_hkey key,
			  struct timespec *ets, double metric, int pid);
static void post_analyze_dist(struct gate_analyzer *ga);
static int opt_handler_dist(int c, char *opt);

struct analysis_formatter dist_formatter = {
        .opt = { 
		.opt = "d::",
		.format = "-d[[min]:[max]:[rank]:[L|I][C]]",
		.description = "show distributions. I:linearscale, L:logscale",
		.handler = opt_handler_dist,
	},
	.active = 0,
	.pre_analyze = pre_analyze_dist,
	.callback = callback_dist,
	.post_analyze = post_analyze_dist,
};

static int o_flag = 0;
#define FL_MIN 1
#define FL_MAX 2
#define FL_RNK 4
#define FL_LOG 8

static double opt_min = 0;
static double opt_max = 0;
static int opt_ranks = 0;
static int opt_log = 0;
static char csv_char = ' ';

#define SYS_HASH_BITS 6
static generic_slots_t *gsl_dist = NULL;
static double *bounds;
static int nr_bounds;
static int nr_entries;

static void make_bounds(double min, double max, int ranks, int logscale)
{
	double metric;
	int i;
	nr_bounds = ranks+1;
	nr_entries = nr_bounds+1;
	bounds = malloc(sizeof(double)*nr_bounds);
	if (logscale) {
		min = log10(min); max = log10(max);
	}
	metric = max - min ;
	for ( i = 0; i<nr_bounds; i++) {
		bounds[i] = metric*i/(nr_bounds-1) + min;
	}
	if (logscale) {
		for ( i = 0; i<nr_bounds; i++) {
			bounds[i] = pow(10,bounds[i]);
		}
	}
}

static int pre_analyze_dist(struct gate_analyzer *ga)
{
	int i;
	if (!(o_flag & FL_MIN)) opt_min = ga->min;
	if (!(o_flag & FL_MAX)) opt_max = ga->max;
	if (!(o_flag & FL_RNK)) opt_ranks = ga->ranks;
	if (!(o_flag & FL_LOG)) opt_log = ga->flag;
	make_bounds(opt_min, opt_max, opt_ranks, opt_log);
	gsl_dist = new_generic_slots_sorted(SYS_HASH_BITS, 
					    sizeof(int)*nr_entries);
	if (gsl_dist == NULL) {
		printf("error: fail to memory allocation\n");
		return -1;
	}

	if (quiet_mode)
		return 0;

	printf("%s\n",ga->description);
	printf("%*s%c%*s", ga->cols[COL_KEY].width, ga->cols[COL_KEY].title,
	       csv_char, ga->cols[COL_ALIAS].width, ga->cols[COL_ALIAS].title);
	printf("%c  less-than-min",csv_char);
	for (i=0; i<nr_bounds-1; i++) {
		printf("%c%15.9f",csv_char,bounds[i]);
	}
	printf("%c  more-than-max\n",csv_char);

	return 0;
}

static void callback_dist(struct gate_analyzer *ga, slot_hkey key,
			  struct timespec *ets, double metric, int pid)
{
	int *counter;
	slot_t *slot;
	int i;
	slot = find_slot(gsl_dist, key);
	if (slot == NULL){
		slot = get_free_slot(gsl_dist, key);
		if (slot) {
			counter = (int *)slot_data(slot);
			memset(counter, 0, sizeof(int)*nr_bounds);
		}
	}
	if (slot != NULL)  {
		counter = (int *)slot_data(slot);
		for (i=0;i<nr_bounds;i++) {
			if (metric<bounds[i]) {
				counter[i]++;
				break;
			}
		}
		if (i==nr_bounds) counter[i]++;
	}
}
static void post_analyze_dist(struct gate_analyzer *ga)
{
	slot_hkey key;
	slot_t *slot;
	int i, *counter;
	for_each_sorted_slot(gsl_dist, slot) {
		key = slot->key;
		counter = (int *)slot_data(slot);
		printf(ga->cols[COL_KEY].format,
		       ga->cols[COL_KEY].width, key);
		putchar(csv_char);
		printf(ga->cols[COL_ALIAS].format,
		       ga->cols[COL_ALIAS].width, ga->get_alias(key));
		for ( i=0;i<nr_entries;i++) {
			printf("%c%15d", csv_char, counter[i]);
		}
		printf("\n");
	}
	free_generic_slots(gsl_dist);
	return ;
}

#include <string.h>
#include <ctype.h>

static int opt_handler_dist(int c, char *opt)
{
	double val;
	long lval;
	char *ptr=opt;
	if (opt) {
		if (*ptr!=':') {
			val = strtod(ptr,&ptr);
			if (*ptr != ':') goto format_err;
			opt_min = val;
			o_flag |= FL_MIN;
		}
		ptr++;
		if (*ptr!=':') {
			val = strtod(ptr,&ptr);
			if (*ptr != ':') goto format_err;
			opt_max = val;
			o_flag |= FL_MAX;
		}
		ptr++;
		if (*ptr!=':') {
			lval = strtol(ptr,&ptr,10);
			if (*ptr != ':') goto format_err;
			opt_ranks = lval;
			o_flag |= FL_RNK;
		}
		ptr++;
		while (*ptr != '\0'){
			switch(*ptr) {
			case 'L':
				opt_log = 1;
				o_flag |= FL_LOG;
				break;
			case 'I':
				opt_log = 0;
				o_flag |= FL_LOG;
				break;
			case 'C':
				csv_char = ',';
				break;
			default:
				goto format_err;
			}
			ptr ++;
		}
		if (*ptr != '\0') goto format_err;
	}
	dist_formatter.active = 1;
	return 0;
format_err:
	fprintf(stderr, "error: option format error(%c %s)(%s)\n", c, opt,ptr);
	return -EINVAL;
}
