/*
 Process living time analyzer

 Copyright (C) HITACHI,LTD. 2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#include <errno.h>
#include <linux/lkst.h>
#include <linux/lkst_buffer.h>
#include <lkstla.h>
#include <string.h>

static slot_hkey info_key_proclive(struct lkst_log_record *rec_start, struct lkst_log_record *rec_end);
static slot_hkey session_key_proclive(struct lkst_log_record *rec);
static char * get_alias_proclive(slot_hkey key);
static ga_type_t get_type_proclive(struct lkst_log_record *rec);
static int get_pid_proclive(struct lkst_log_record *rec,
			    struct lkst_log_record *dummy);
static int init_proclive(void);

static int pid_show=0;
extern struct gate_analyzer proclive_analyzer;
static int extra_opt_handler_pid(int c, char*v) {
	pid_show = 1;
	SET_COL(proclive_analyzer, COL_ALIAS, "pid", 6, "%*s");

	return 0;
}

static struct command_option pid_option = {
	.opt = "P",
	.format = "-P",
	.description = "show PID instead of task_name",
	.handler = extra_opt_handler_pid,
};

struct gate_analyzer proclive_analyzer = {
	.name = "proclive",
	.description = "Process living time analyzer",
	.cols = {
		[COL_KEY] = { "pid", 6, "%*d" },
		[COL_ALIAS] = { "task_name", 16, "%*s" },
		[COL_METRIC] = { "living-time" },
	},

	GA_INIT_SESSION_ANALYZER
	GA_INIT_RET_SESSION_TIME
	GA_INIT_1TO1_SESSION

	.info_key = info_key_proclive,
	.session_key = session_key_proclive,
	.get_alias = get_alias_proclive,

	.get_type = get_type_proclive,
	.get_pid = get_pid_proclive,

	.init = init_proclive,

	.nr_options = 1,
	.options[0] = &pid_option,
};

static generic_slots_t *gsl_info=NULL;

static int init_proclive(void)
{
	gsl_info = new_generic_slots(5,20);
	if (gsl_info == NULL)return -ENOMEM;
	return 0;
}

static void register_alias(struct lkst_log_record *rec) 
{
	slot_hkey key = info_key_proclive(rec,rec);
	slot_t *s;
	if (pid_show) return ;
	s = find_slot(gsl_info, key);
	if (!s) {
		s = get_free_slot(gsl_info,key);
	}
	if (s) {
		memcpy( (char*)slot_data(s), &rec->log_arg3, 8);
		memcpy( ((char*)slot_data(s))+8, &rec->log_arg4, 8);
		((char*)slot_data(s))[16]='\0';
	}
}

static char str_pid[16];

static char * get_alias_proclive(slot_hkey key)
{
	if (pid_show) {
		snprintf(str_pid, 10, "%d", (int)key);
		return str_pid;
	}else {
		slot_t *s = find_slot(gsl_info, key);
		if (s) {
			return (char *)slot_data(s);
		}else
			return "(unknown)";
	}
}

static int get_pid_proclive(struct lkst_log_record *rec,
			    struct lkst_log_record *dummy)
{
	int pid = -1;
	switch (rec->posix.log_event_type) { 
	case LKST_ETYPE_PROCESS_SCHED_ENTER:
		pid = rec->posix.log_pid&0xffffffff;
		break;
	case LKST_ETYPE_PROCESS_FORK:
	case LKST_ETYPE_PROCESS_EXIT:
		pid = rec->log_arg1&0xffffffff;
		break;
	}
	return pid;
}

static ga_type_t get_type_proclive(struct lkst_log_record *rec)
{
	switch (rec->posix.log_event_type) {
	case LKST_ETYPE_PROCESS_FORK:
		return GAT_START;
	case LKST_ETYPE_PROCESS_EXIT:
		return GAT_END;
	case LKST_ETYPE_PROCESS_SCHED_ENTER:
		register_alias(rec);
	default:
		return GAT_IGNORE;
	}
}

/*return pid*/
static slot_hkey info_key_proclive(struct lkst_log_record *rec,
				   struct lkst_log_record *rec_end)
{
	return (slot_hkey)get_pid_proclive(rec,rec_end);
}

static slot_hkey session_key_proclive(struct lkst_log_record *rec)
{	
	return (slot_hkey)get_pid_proclive(rec,rec);
}

