/*
 Process status analyzer

 Copyright (C) HITACHI,LTD. 2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <linux/lkst.h>
#include <linux/lkst_buffer.h>
#include <lkstla.h>
#include <string.h>
#include <extra_etypes.h>

static slot_hkey info_key_procstat(struct lkst_log_record *rec_start, struct lkst_log_record *rec_end);
static char * get_alias_procstat(slot_hkey key);
static ga_type_t get_type_procstat(struct lkst_log_record *rec);
static int get_pid_procstat(struct lkst_log_record *rec,
			    struct lkst_log_record *dummy);
static int init_procstat(void);
static double get_metric_procstat(struct lkst_log_record *rec_start,
				 struct lkst_log_record *rec_end);

#define PS_TASK_DIE     0
static unsigned state_map[65] = {
	[0]=7,	/*TASK_RUNNING*/
	[1]=6,	/*TASK_INTERRUPTIBLE*/
	[2]=5,	/*TASK_UNINTERRUPTIBLE*/
	[4]=4,	/*TASK_STOPPED*/
	[8]=3,	/*TASK_TRACED*/
	[16]=2,	/*TASK_ZOMBIE*/
	[32]=1,	/*TASK_DEAD*/
	[64]=8  /*TASK_RUNNING(REAL)*/
};
#define PS_TASK(state) (state_map[(state)])

static int extra_opt_handler_states(int c, char*v) {
	printf("%3s %s\n","num","state");
	printf("%3d %s\n",0,"CALL_EXIT");
	printf("%3d %s\n",1,"TASK_DEAD");
	printf("%3d %s\n",2,"TASK_ZOMBIE");
	printf("%3d %s\n",3,"TASK_TRACED");
	printf("%3d %s\n",4,"TASK_STOPPED");
	printf("%3d %s\n",5,"TASK_UNINTERRUPTIBLE");
	printf("%3d %s\n",6,"TASK_INTERRUPTIBLE");
	printf("%3d %s\n",7,"TASK_RUNNING(ready)");
	printf("%3d %s\n",8,"TASK_RUNNING(run)");
	exit(0);
}

static int pid_show=0;
struct gate_analyzer procstat_analyzer;
static int extra_opt_handler_pid(int c, char*v) {
	pid_show = 1;
	SET_COL(procstat_analyzer, COL_ALIAS, "pid", 5, "%*s");
	
	return 0;
}

static struct command_option cmd_options[2] = {
	{
		.opt = "P",
		.format = "-P",
		.description = "show PID instead of task_name",
		.handler = extra_opt_handler_pid,
	},{
		.opt = "L",
		.format = "-L",
		.description = "list task states",
		.handler = extra_opt_handler_states,
	}
};

struct gate_analyzer procstat_analyzer = {
	.name = "procstat",
	.description = "Process status analyzer",
	.cols = {
		[COL_KEY] = { "pid", 6, "%*d" },
		[COL_ALIAS] = { "task_name", 16, "%*s" },
		[COL_METRIC] = { "process-stat" },
	},

	GA_INIT_EVENT_ANALYZER
	GA_INIT_RET_EVENT_TIME
	.max = 9,
	.min = 0,
	.flag = GAF_LONG_EV,
	.ranks = 9,

	.info_key = info_key_procstat,
	.get_alias = get_alias_procstat,

	.get_type = get_type_procstat,
	.get_metric = get_metric_procstat,
	.get_pid = get_pid_procstat,

	.init = init_procstat,

	.nr_options = 2,
	.options = { &cmd_options[0], &cmd_options[1]},
};

static generic_slots_t *gsl_info=NULL;

static int init_procstat(void)
{
	gsl_info = new_generic_slots(5,20);
	if (gsl_info == NULL)return -ENOMEM;
	return 0;
}

static void register_alias(struct lkst_log_record *rec) 
{
	slot_hkey key = info_key_procstat(rec,rec);
	slot_t *s;
	if (pid_show) return ;
	s = find_slot(gsl_info, key);
	if (!s) {
		s = get_free_slot(gsl_info,key);
	}
	if (s) {
		memcpy( (char*)slot_data(s), &rec->log_arg3, 8);
		memcpy( ((char*)slot_data(s))+8, &rec->log_arg4, 8);
		((char*)slot_data(s))[16]='\0';
	}
}

static char str_pid[16];

static char * get_alias_procstat(slot_hkey key)
{
	if (pid_show) {
		snprintf(str_pid, 10, "%ld", key);
		return str_pid;
	}else {
		slot_t *s = find_slot(gsl_info, key);
		if (s) {
			return (char *)slot_data(s);
		}else
			return "(unknown)";
	}
}

static int get_pid_procstat(struct lkst_log_record *rec,
			    struct lkst_log_record *dummy)
{
	int pid = -1;
	switch (rec->posix.log_event_type) { 
	case LKST_ETYPE_PROCESS_SCHED_ENTER:
	case LKST_ETYPE_PROCESS_CONTEXTSWITCH:
		pid = rec->posix.log_pid&0xffffffff;
		break;
	case LKST_ETYPE_PROCESS_FORK:
	case LKST_ETYPE_PROCESS_EXIT:
	case LKST_ETYPE_PROCESS_WAKEUP2:
	case LKST_ETYPE_PROCESS_CONTEXTSW2:
		pid = rec->log_arg1&0xffffffff;
		break;
	}
	return pid;
}

static ga_type_t get_type_procstat(struct lkst_log_record *rec)
{
	switch (rec->posix.log_event_type) {
	case LKST_ETYPE_PROCESS_FORK:
	case LKST_ETYPE_PROCESS_EXIT:
	case LKST_ETYPE_PROCESS_CONTEXTSWITCH:
	case LKST_ETYPE_PROCESS_CONTEXTSW2:
	case LKST_ETYPE_PROCESS_WAKEUP2:
		return GAT_EVENT;
		break;
	case LKST_ETYPE_PROCESS_SCHED_ENTER:
		register_alias(rec);
	default:
		return GAT_IGNORE;
	}
}

/*return pid*/
static slot_hkey info_key_procstat(struct lkst_log_record *rec,
				   struct lkst_log_record *rec_end)
{
	return (slot_hkey)get_pid_procstat(rec,rec_end);
}

static double get_metric_procstat(struct lkst_log_record *rec,
				 struct lkst_log_record *rec_end)
{
	unsigned ret=0;
	switch (rec->posix.log_event_type) { 
	case LKST_ETYPE_PROCESS_CONTEXTSWITCH:
		ret = PS_TASK(rec->log_arg3 & 63);
		break;
	case LKST_ETYPE_PROCESS_WAKEUP2:
		ret = PS_TASK(0);/*waken-up tasks are always in running state*/
		break;
	case LKST_ETYPE_PROCESS_CONTEXTSW2:
		ret = PS_TASK(64);
		break;
	case LKST_ETYPE_PROCESS_FORK:
		ret = PS_TASK(rec->log_arg4 & 63);
		break;
	case LKST_ETYPE_PROCESS_EXIT:
		ret = PS_TASK_DIE;
		break;
	}
	return (double)ret;
}
