 /*
 system information analyzer

 Copyright (C) HITACHI,LTD. 2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by H. Kawai <h-kawai@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <linux/lkst.h>
#include <linux/lkst_buffer.h>
#include <lkstla.h>
#include <eventdesc.h>
#include <logfile.h>
#include <extra_etypes.h>
#include <multientry.h>

static int init_sysinfo(void);
static slot_hkey info_key_sysinfo(struct lkst_log_record *rec_start,
				  struct lkst_log_record *rec_end);
static char * get_alias_sysinfo(slot_hkey key);
static double get_metric_sysinfo(struct lkst_log_record *start,
				 struct lkst_log_record *end);
static ga_type_t get_type_sysinfo(struct lkst_log_record *rec);

static char *arg_id = NULL; // index or name of ARG n
static int arg_idx = -1; // idx 5-8 correspond to ARG1-4 of SYSTEM_INFO2 event
static int sysinfo_etype;

static int extra_opt_handler_arg(int c, char *v) {
	if (c != 'A' || v == NULL || *v == '\0')
		return -1;

	if ((arg_id = strdup(v)) == NULL)
		return -ENOMEM;
	
	return 0;
}

static struct command_option arg_option = {
	.opt = "A:",
	.format = "-A {index|desc}",
	.description = "specify an argument of sysinfo record to display",
	.handler = extra_opt_handler_arg,
};

struct gate_analyzer sysinfo_analyzer = {
	.name = "sysinfo",
	.description = "system information analyzer",
	.cols = {
		[COL_KEY] = { "arg_idx", 7, "%*d" },
		[COL_ALIAS] = { "arg_desc", 16, "%*s" },
		[COL_METRIC] = { "value" },
	},
	
	.flag = GAF_LONG_EV,
	.ranks = 5,
	.min = 0,
	.max = 1000000,
	GA_INIT_EVENT_ANALYZER
	GA_INIT_RET_EVENT_PID
	GA_INIT_RET_EVENT_TIME
		
	.info_key = info_key_sysinfo,
	.get_alias = get_alias_sysinfo,
	.get_metric = get_metric_sysinfo,

	.get_type = get_type_sysinfo,

	.init = init_sysinfo,
	.nr_options = 2,
	.options[0] = &arg_option,
	.options[1] = &edesc_option,
};

static int init_sysinfo(void)
{
	int etype_list[] = {
		LKST_ETYPE_SYSTEM_INFO,
		LKST_ETYPE_SYSTEM_INFO2,
		LKST_ETYPE_MAX
	};

	init_eventdesc();

	arg_idx = get_argidx(arg_id, etype_list);
	if (arg_idx < 1) 
		return -1;
	sysinfo_etype = etype_list[(arg_idx - 1) / 4];

	return 0;
}

static slot_hkey info_key_sysinfo(struct lkst_log_record *rec_start,
				  struct lkst_log_record *rec)
{
	return (slot_hkey)arg_idx;
}

static char * get_alias_sysinfo(slot_hkey key)
{
	char *s;

	s = lkst_etype_desc[sysinfo_etype].args[NORM_ARGIDX(arg_idx) - 1];
	if (s == NULL || *s == '\0')
		return "(unknown)";
	else
		return s;
}

static double get_metric_sysinfo(struct lkst_log_record *start,
				 struct lkst_log_record *end)
{
	return (double)ARG(start, NORM_ARGIDX(arg_idx));
}

static ga_type_t get_type_sysinfo(struct lkst_log_record *rec)
{
	if (rec->posix.log_event_type == sysinfo_etype)
		return GAT_EVENT;

	return GAT_IGNORE;
}
