/*
 * Copyright 1999 Silicon Graphics, Inc. All rights reserved.
 */
#include <lcrash.h>

#define LOG_BUF_SIZE 16384

void
print_log_buf(FILE *ofp)    
{
	int i, j, spacer = 1;
	char *np, *lbuf;
	kaddr_t symaddr;
	syment_t *sp;

	if (!(lbuf = (char *)malloc(LOG_BUF_SIZE))) {
		return;
	}
	if (!(np = (char *)malloc(256))) {
		free(lbuf);
		return;
	}
	if ((sp = kl_lkup_symname("log_buf"))) {
		symaddr = sp->s_addr;
	} else {
		fprintf(stderr, "Could not find the symbol \"log_buf\"\n");
		free(np);
		free(lbuf);
		return;
	}

	GET_BLOCK(symaddr, LOG_BUF_SIZE, lbuf);
	i = 0;
	j = 0;
	while (i < LOG_BUF_SIZE) {
		if (lbuf[i] == '\n') {
			np[j] = '\0';
			if (spacer) {
				fprintf(ofp, "    ");
			} else {
				spacer = 1;
			}
			fprintf(ofp, "%s\n", np);
			j = 0;
		} else {
			np[j] = lbuf[i];
			if (j == 255) {
				np[j] = '\0';
				if (spacer) {
					fprintf(ofp, "    ");
					spacer = 0;
				} else {
					spacer = 1;
				}
				fprintf(ofp, "%s", np);
				j = 0;
			} else {
				j++;
			}
		}
		i++;
	}
	if (j) {
		np[j] = '\0';
		if (spacer) {
			fprintf(ofp, "    ");
			spacer = 0;
		} else {
			spacer = 1;
		}
		fprintf(ofp, "%s\n", np);
	}
	free(np);
	free(lbuf);
}

void
stat_cmd(command_t *cmd)
{
	char *utsname;

	if((utsname = get_utsname())){
		fprintf(cmd->ofp, "\n");
		fprintf(cmd->ofp, "   sysname : %s\n", (char*)
			K_PTR(utsname, NEW_UTSNAME, "sysname"));
		fprintf(cmd->ofp, "  nodename : %s\n", (char*)
			K_PTR(utsname, NEW_UTSNAME, "nodename"));
		fprintf(cmd->ofp, "   release : %s\n", (char*)
			K_PTR(utsname, NEW_UTSNAME, "release"));
		fprintf(cmd->ofp, "   version : %s\n", (char*)
			K_PTR(utsname, NEW_UTSNAME, "version"));
		fprintf(cmd->ofp, "   machine : %s\n", (char*)
			K_PTR(utsname, NEW_UTSNAME, "machine"));
		fprintf(cmd->ofp, "domainname : %s\n", (char*)
			K_PTR(utsname, NEW_UTSNAME, "domainname"));
		fprintf(cmd->ofp, "\nLOG_BUF:\n\n");
		kl_free_block(utsname);
	} else {
		fprintf(cmd->efp, "COULD NOT GET SYSINFO!\n\n");
	}	
	print_log_buf(cmd->ofp);
}

#define _STAT_USAGE "[-w outfile]"

/*
 * stat_usage() -- Print the usage string for the 'stat' command.
 */
void
stat_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _STAT_USAGE);
}

/*
 * stat_help() -- Print the help information for the 'stat' command.
 */
void
stat_help(command_t *cmd)
{
	CMD_HELP(cmd, _STAT_USAGE,
	"Display system statistics and the log_buf array, which contains "
	"the latest messages printed via the kernel printf/cmn_err "
	"routines.");
}

/*
 * stat_parse() -- Parse the command line arguments for 'stat'.
 */
int
stat_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_FALSE|C_WRITE), 0)) {
		return(1);
	}
	return(0);
}

/*
 * stat_complete() -- Complete arguments of 'stat' command.
 */
char *
stat_complete(command_t *cmd)
{
	char *ret;

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	fprintf(cmd->ofp, "\n");
	stat_usage(cmd);
	return(DRAW_NEW_ENTIRE_LINE);
}
