#
#       snmp_printer.py
#
#       2003.8.22
#
#   Copyright (C) Hidetoshi Nakano
#
#   Please use this program at your own risk.
#   Without any warranty.
# 
#
############################
import string

import snmp_check
import oid
import Lpy.tools.langcode
import Lpy.tools.countrycode
import Lpy.tools.charset

class NameSpace:
    pass

class SNMP_Printer(snmp_check.SNMP_Check):
    P_name = 0
    P_func = 1

    S_general    = "99" # "05"
    S_covers     = "06"
    S_locale     = "07"
    S_inputTray  = "08"
    S_outputTray = "09"
    S_Marker     = "10"
    S_MarkerSupplies = "11"
    S_MarkerColorant = "12"
    S_MediaPath      = "13"
    S_Channel        = "14"
    S_Interpreter    = "15"
    S_ConsoleDisplay = "16"
    S_ConsoleLight   = "17"
    S_Alert          = "18"
    S_findevice      = "30" # Finisher
    S_finSupply      = "31"
    S_finMediaInput  = "32"
    S_finAttribute   = "33"

    def __init__(self,host,community,port):
        snmp_check.SNMP_Check.__init__(self,host,community,port)
        self.dlist = NameSpace()

        self.snmp_func = {
            self.S_general:    ("Default",       self.get_printer_general),
            self.S_covers:     ("Covers",        self.get_printer_cover),
            self.S_locale:     ("Locales",       self.get_printer_localization),
            self.S_inputTray : ("InputTray",     self.get_printer_inputTray),
            self.S_outputTray: ("OutputTray",    self.get_printer_outputTray),
            self.S_Marker:     ("Marker",        self.get_printer_Marker),
            self.S_MarkerSupplies: ("MarkerSupplies",self.get_printer_MarkerSupplies),
            self.S_MarkerColorant: ("MarkerColorant",self.get_printer_MarkerColorant),
            self.S_MediaPath:      ("MediaPath",     self.get_printer_MediaPath),
            self.S_Channel:        ("Channel",       self.get_printer_Channel),
            self.S_Interpreter:    ("Interpreter",   self.get_printer_Interpreter),
            self.S_ConsoleDisplay: ("ConsoleDisplay",self.get_printer_ConsoleDisplay),
            self.S_ConsoleLight:   ("ConsoleLight",  self.get_printer_ConsoleLight),
            self.S_Alert:          ("Alert",         self.get_printer_Alert),
            self.S_findevice:      ("Findevice",     self.get_printer_findevice),  # Finisher
            self.S_finSupply:      ("FinSupply",     self.get_printer_finSupply),
            self.S_finMediaInput:  ("FinMediaInput", self.get_printer_finMediaInput),
            #self.S_finAttribute:  ("FinAttribute",  self.get_printer_finAttribute),
        }

    def get_printer_all_mib(self,printer_mib,printerIndex):
        if printer_mib is None:
            return None

        keys = self.snmp_func.keys()
        keys.sort()
        for key in keys:
            mib = apply(self.snmp_func[key][self.P_func],(printer_mib,printerIndex))
            if mib:
                setattr(self.dlist,key,mib)

    def get_mib_func(self,key):
        if hasattr(self.dlist,key):
            return getattr(self.dlist,key)
        return None

    def get_direct_mib(self,data,flag,name,num_index):
        key = oid.get_key_oid(name,flag)
        if data.has_key(key + num_index):
            return data[key + num_index]
        return None

## System mib
    def check_snmp_system(self,mib):
        tmp = {}
        data = self.get_direct_mib(mib,oid.OID_rfc1213,'sysDescr',".0")
        if data:
            tmp["1"] = ('sysDescr' ,data)
        data = self.get_direct_mib(mib,oid.OID_rfc1213,'sysObjectID',".0")
        if data:
            tmp["2"] = ('sysObjectID' ,data)
        data = self.get_direct_mib(mib,oid.OID_rfc1213,'sysUpTime',".0")
        if data:
            tmp["3"] = ('sysUpTime',data)
        data = self.get_direct_mib(mib,oid.OID_rfc1213,'sysContact',".0")
        if data:
            tmp["4"] = ('sysContact',data)
        data = self.get_direct_mib(mib,oid.OID_rfc1213,'sysName',".0")
        if data:
            tmp["5"] = ('sysName',data)
        data = self.get_direct_mib(mib,oid.OID_rfc1213,'sysLocation',".0")
        if data:
            tmp["6"] = ('sysLocation',data)
        data = self.get_direct_mib(mib,oid.OID_rfc1213,'sysServices',".0")
        if data:
            tmp["7"] = ('sysServices',data)
        return tmp

## Interface mib
    def check_snmp_interface(self,mib):
        key = oid.get_key_oid('ifEntry',oid.OID_rfc1213)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.' % key)
        dat = self.get_index_mib(mib,key,count,2,22)
        if dat is None:
            print "interface MIB is None."
            return None

        keys = dat.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            data = dat[key]
            tmp = {}
            tmp["02"] = ('ifDescr',data[0])
            tmp["03"] = ('ifType',data[1])
            tmp["04"] = ('ifMtu',data[2])
            tmp["05"] = ('ifSpeed',data[3])
            tmp["06"] = ('ifPhysAddress',data[4])
            tmp["07"] = ('ifAdminStatus',data[5])
            tmp["08"] = ('ifOperStatus',data[6])
            tmp["09"] = ('ifLastChange',data[7])
            tmp["10"] = ('ifInOctets' ,data[8])
            tmp["11"] = ('ifInUcastPkts',data[9])
            tmp["12"] = ('ifInNUcastPkts',data[10])
            tmp["13"] = ('ifInDiscards',data[11])
            tmp["14"] = ('ifInErrors',data[12])
            tmp["15"] = ('ifInUnknownProtos',data[13])
            tmp["16"] = ('ifOutOctets',data[14])
            tmp["17"] = ('ifOutUcastPkts',data[15])
            tmp["18"] = ('ifOutNUcastPkts',data[16])
            tmp["19"] = ('ifOutDiscards',data[17])
            tmp["20"] = ('ifOutErrors',data[18])
            tmp["21"] = ('ifOutQLen',data[19])
            tmp["22"] = ('ifSpecific',data[20])
            mibdic[key] = tmp
        return  mibdic

    ## hostMIB
    def get_host_memory(self,host_mib):
        return self.get_direct_mib(host_mib,oid.OID_host,'hrMemorySize','.0')
        # Kbytes

    def get_host_storage(self,host_mib,name):
        key = oid.get_key_oid('hrStorageEntry',oid.OID_host)
        if key is None:
            return None
        count = self.mib_index_count(host_mib,'%s.2.' % key)
        dat = self.get_index_mib(host_mib,key,count,2,7)
        if dat is None:
            print "get_host_storage MIB is None."
            return None
        keys = dat.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            data = dat[key]
            tmp = {}
            #tmp["2"] = ("StorageType",data[0])
            tmp["3"] = ("StorageDescr",data[1])
            tmp["4"] = ("StorageAllocationUnits",data[2])
            tmp["5"] = ("StorageSize",data[3])
            tmp["6"] = ("StorageUsed",data[4])
            tmp["7"] = ("StorageAllocationFailures",data[5])
            mibdic[key] = tmp
        return mibdic

    def get_host_disk(self,host_mib,printer):
        oid = oid.get_key_oid('hrDiskStorageEntry',oid.OID_host)
        data = self.get_index_data(host_mib,oid,1,4,printer)
        if not data:
            return None
        tmp = {}
        tmp["1"] = ("Access",data[0])
        tmp["2"] = ("Media" ,data[1])
        tmp["3"] = ("Removeble",data[2])
        tmp["4"] = ("Capacity" ,data[3])
        return tmp

    def get_host_device(self,host_mib):
        key = oid.get_key_oid('hrDeviceEntry',oid.OID_host)
        if key is None:
            return None
        count = self.mib_index_count(host_mib,'%s.2.' % key)
        dat = self.get_index_mib(host_mib,key,count,1,6)
        if dat is None:
            print "get_host_storage MIB is None."
            return None
        keys = dat.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            data = dat[key]
            tmp = {}
            #tmp["1"] = ("DeviceIndex",data[0])
            tmp["2"] = ("DeviceType",data[1])
            tmp["3"] = ("DeviceDescr",data[2])
            tmp["4"] = ("DeviceID",data[3])
            tmp["5"] = ("DeviceStatus",oid.get_tc(data[4],"DeviceStatus"))
            tmp["6"] = ("DeviceErrors",data[5])
            mibdic[key] = tmp
        return mibdic

    def get_printer_status(self,host_mib,printer):
        key = oid.get_key_oid('hrPrinterEntry',oid.OID_host)
        data = self.get_index_data(host_mib,key,1,2,str(printer))
        if not data:
            return None
        tmp = {}
        tmp["1"] = ("PrinterStatus", oid.get_tc(data[0],"PrinterStatus"))
        tmp["2"] = ("PrinterDetectedErrorState", oid.get_tc(data[1],"DetectedError"))
        tmp["3"] = ("PrinterDetectedErrorState", data[1])
        return tmp

## printerMIB
    # Alert
    def get_printer_Alert(self,mib,printer,flag = 1):
        if mib == None:
            return None
        key = oid.get_key_oid('prtAlertEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer)) 
        data = self.get_index_mib2(mib,key,printer,count,2,9)
        if not data:
            return None

        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            if flag == 2:
                tmp["2"] = ("AlertSeverityLevel",oid.get_tc(data[key][0],"AlertServerityLevel"))
                tmp["7"] = ("AlertCode"      ,oid.get_tc(data[key][5],"AlertCode"))
                tmp["8"] = ("AlertDescription",data[key][6])
            else:

                tmp["2"] = ("AlertSeverityLevel",oid.get_tc(data[key][0],"AlertServerityLevel"))
                tmp["3"] = ("AlertTrainingLevel",oid.get_tc(data[key][1],"AlertTrainingLevel"))
                tmp["4"] = ("AlertGroup"     ,oid.get_tc(data[key][2],"AlertGroup"))
                if data[key][3] != '-1':
                    tmp["5"] = ("AlertGroupIndex",data[key][3])  # index or -1
                if data[key][4] != '-2':
                    tmp["6"] = ("AlertLocation"  ,data[key][4])  # index or -2
                tmp["7"] = ("AlertCode"      ,oid.get_tc(data[key][5],"AlertCode"))
                tmp["8"] = ("AlertDescription",data[key][6])
                tmp["9"] = ("AlertTime"       ,data[key][7])
            mibdic[key] = tmp
        return mibdic

    # General
    def snmp_default_printer(self,key,dat,name,name2=None):
        if not self.snmp_func.has_key(key) or self.get_mib_func(key) is None:
            #print "snmp_default_printer is None",key
            return None
        else:
            data = self.get_mib_func(key)
            Index = self.get_string_index(str(dat))
            if data.has_key(Index):
                if data[Index].has_key(name):
                    ret = data[Index][name][1]
                    if name2 and data[Index].has_key(name2):
                        ret += ",%s" % data[Index][name2][1]
                    return ret
        return None

    def get_printer_general(self,mib,printer):
        key = oid.get_key_oid('prtGeneralEntry',oid.OID_printer)
        data = self.get_index_data(mib,key,1,19,printer)
        if not data:
            return None

        tmp = {}
        tmp["01"] = ("GeneralConfigChanges", data[0])
        dat = self.snmp_default_printer(self.S_locale,data[1],"2")
        if dat:
            tmp["02"] = ("GeneralCurrentLocalization",dat) # Locale index

        tmp["03"] = ('GeneralReset',oid.get_tc(data[2],"GeneralReset"))
        tmp["04"] = ('GeneralCurrentOperator',data[3]) # Operator:name,tel,mail-address
        tmp["05"] = ('GeneralServicePerson',  data[4]) # last service person:name,tel,mail-address

        dat = self.snmp_default_printer(self.S_inputTray,data[5],"18")
        if dat:
            tmp["06"] = ('InputDefaultIndex',  dat)    # inputTray index

        dat = self.snmp_default_printer(self.S_outputTray,data[6],"12")
        if dat:
            tmp["07"] = ('OutputDefaultIndex', dat)    # outputTray index

        dat = self.snmp_default_printer(self.S_Marker,data[7],"02")
        if dat:
            tmp["08"] = ('MarkerDefaultIndex', dat)    # marker index

        dat = self.snmp_default_printer(self.S_MediaPath,data[8],"10")
        if dat:
            tmp["09"] = ('MediaPathDefaultIndex',dat)  # mediapath index

        dat = self.snmp_default_printer(self.S_locale,data[9],"2")
        if dat:
            tmp["10"] = ('ConsoleLocalization', dat)   # Console Locale  index

        tmp["11"] = ('ConsoleNumberOfDisplayLines',data[10])
        tmp["12"] = ('ConsoleNumberOfDisplayChars',data[11])
        tmp["13"] = ('ConsoleDisable',oid.get_tc(data[12],'ConsoleDisable'))
        if data[13] != '' or data[15] != '':  # new
                tmp["14"] = ('AuxiliarySheetStartupPage',data[13])
                tmp["15"] = ('AuxiliarySheetBannerPage', data[14])
                tmp["16"] = ('GeneralPrinterName',  data[15])
                tmp["17"] = ('GeneralSerialNumber', data[16])
                tmp["18"] = ('AlertCriticalEvents', data[17])
                tmp["19"] = ('AlertAllEvents', data[18])
        return tmp

    # Cover : 6
    def get_printer_cover(self,mib,printer):
        key = oid.get_key_oid('prtCoverEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer))
        dat = self.get_index_mib2(mib,key,printer,count,2,3)
        if not dat:
            return None
        keys = dat.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["2"] = ("CoverDescription",dat[key][0])
            tmp["3"] = ("CoverStatus",oid.get_tc(dat[key][1],"CoverStatus"))
            mibdic[key] = tmp
        return mibdic

    # Localization : 7
    def get_printer_localization(self,mib,printer):
        key = oid.get_key_oid('prtLocalizationEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer))
        data = self.get_index_mib2(mib,key,printer,count,2,4)
        if not data:
            return None

        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["2"] = ('LocalizationLanguage',Lpy.tools.langcode.get_langcode(string.lower(data[key][0])))
            tmp["3"] = ('LocalizationCountry',Lpy.tools.countrycode.get_countrycode(string.upper(data[key][1])))
            tmp["4"] = ('LocalizationCharacterSet',Lpy.tools.charset.get_charset(str(data[key][2])))
            mibdic[key] = tmp
        return mibdic

    # Input : 8
    def get_printer_inputTray(self,mib,printer):
        key = oid.get_key_oid('prtInputEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer))
        data = self.get_index_mib2(mib,key,printer,count,2,25)
        if not data:
            return None
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["02"] = ('InputType',oid.get_tc(data[key][0],"InputType"))
            tmp["03"] = ('InputDimUnit',oid.get_tc(data[key][1],"MediaUnit"))
            tmp["04"] = ('MediaDimFeedDirDeclared' ,data[key][2])  # paper length
            tmp["05"] = ('MediaDimXFeedDirDeclared',data[key][3])  # paper width
            tmp["06"] = ('MediaDimFeedDirChosen'  ,data[key][4])
            tmp["07"] = ('MediaDimXFeedDirChosen' ,data[key][5])
            tmp["08"] = ('InputCapacityUnit',oid.get_tc(data[key][6],"CapacityUnit"))
            tmp["09"] = ('InputMaxCapacity',data[key][7])
            if data[key][8] == '0':
                tmp["10"] = ('InputCurrentLevel',"no paper")
            elif data[key][8] == '-3':
                tmp["10"] = ('InputCurrentLevel',"one unit remains")
            else:
                tmp["10"] = ('InputCurrentLevel',oid.get_tc(data[key][8],"CapacityError"))

            tmp["11"] = ('InputStatus',oid.get_tc(data[key][9],"SubUnitStatus")) # Error check
            tmp["12"] = ('InputMediaName'   ,data[key][10]) # PLAIN,PAPER
            tmp["13"] = ('InputName'        ,data[key][11]) # TRAY1 , MPTRAY
            tmp["14"] = ('InputVendorName'  ,data[key][12])
            tmp["15"] = ('InputModel'       ,data[key][13])
            tmp["16"] = ('InputVersion'     ,data[key][14])
            tmp["17"] = ('InputSerialNumber',data[key][15])
            tmp["18"] = ('InputDescription' ,data[key][16])
            tmp["19"] = ('InputSecurity',oid.get_tc(data[key][17],"PresentOnOff"))
            tmp["20"] = ('InputMediaWeight',data[key][18])
            tmp["21"] = ('InputMediaType'  ,data[key][19])
            tmp["22"] = ('InputMediaColor' ,data[key][20])
            tmp["23"] = ('InputMediaFormParts',data[key][21])
            if data[key][22] != '' :  # new
                tmp["24"] = ('InputMediaLoadTimeout',data[key][22])
                #tmp["25"] = ('InputNextIndex', data[key][23])
            mibdic[key] = tmp
        return mibdic

    # Output : 9
    def get_printer_outputTray(self,mib,printer):
        key = oid.get_key_oid('prtOutputEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer))
        data = self.get_index_mib2(mib,key,printer,count,2,24)
        if not data:
           return None
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["02"] = ("OutputType",oid.get_tc(data[key][0],"OutputType"))
            tmp["03"] = ("OutputCapacityUnit",oid.get_tc(data[key][1],"CapacityUnit"))
            tmp["04"] = ("OutputMaxCapacity"      , data[key][2])
            tmp["05"] = ("OutputRemainingCapacity", data[key][3])
            tmp["06"] = ("OutputStatus",oid.get_tc(data[key][4],"SubUnitStatus"))
            tmp["07"] = ("OutputName"        ,data[key][5])
            tmp["08"] = ("OutputVendorName"  ,data[key][6])
            tmp["09"] = ("OutputModel"       ,data[key][7])
            tmp["10"] = ("OutputVersion"     ,data[key][8])
            tmp["11"] = ("OutputSerialNumber",data[key][9])
            tmp["12"] = ("OutputDescription" ,data[key][10])
            tmp["13"] = ("OutputSecurity",oid.get_tc(data[key][11],"PresentOnOff"))
            tmp["14"] = ("OutputDimUnit" ,oid.get_tc(data[key][12],"MediaUnit"))
            tmp["15"] = ("OutputMaxDimFeedDir" ,data[key][13])
            tmp["16"] = ("OutputMaxDimXFeedDir",data[key][14])
            tmp["17"] = ("OutputMinDimFeedDir" ,data[key][15])
            tmp["18"] = ("OutputMinDimXFeedDir",data[key][16])
            tmp["19"] = ("OutputStackingOrder",oid.get_tc(data[key][17],"StackingOrder"))
            tmp["20"] = ("OutputPageDeliveryOrientation",oid.get_tc(data[key][18],"PageDeliveryOrientation"))
            tmp["21"] = ("OutputBursting",oid.get_tc(data[key][19],"PresentOnOff")) # cut Fanfold
            tmp["22"] = ("OutputDecollating",oid.get_tc(data[key][20],"PresentOnOff")) # mail bin
            tmp["23"] = ("OutputPageCollated",oid.get_tc(data[key][21],"PresentOnOff")) # collate
            tmp["24"] = ("OutputOffsetStacking",oid.get_tc(data[key][22],"PresentOnOff")) #
            mibdic[key] = tmp
        return mibdic

    # MediaPath : 13
    def get_printer_MediaPath(self,mib,printer):
        key = oid.get_key_oid('prtMediaPathEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer))
        data = self.get_index_mib2(mib,key,printer,count,2,11)
        if not data:
           return None
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["02"] = ("MediaPathMaxSpeedPrintUnit", oid.get_tc(data[key][0],"MediaPathSpeed"))
            tmp["03"] = ("MediaPathMediaSizeUnit",oid.get_tc(data[key][1],"MediaUnit"))
            tmp["04"] = ("MediaPathMaxSpeed"         ,data[key][2])
            tmp["05"] = ("MediaPathMaxMediaFeedDir"  ,data[key][3])  # -1: unlimited, -2:unknown
            tmp["06"] = ("MediaPathMaxMediaXFeedDir" ,data[key][4])
            tmp["07"] = ("MediaPathMinMediaFeedDir"  ,data[key][5])
            tmp["08"] = ("MediaPathMinMediaXFeedDir" ,data[key][6])
            tmp["09"] = ("MediaPathType" ,oid.get_tc(data[key][7],"MediaPathType"))
            tmp["10"] = ("MediaPathDescription" ,data[key][8])
            tmp["11"] = ("MediaPathStatus" , oid.get_tc(data[key][9],"SubUnitStatus"))
            mibdic[key] = tmp
        return mibdic

    # Channel : 14
    def get_printer_Channel(self,mib,printer):
        key = oid.get_key_oid('prtChannelEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer),32)
        data = self.get_index_mib2(mib,key,printer,count,2,9)
        if not data:
           return None
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["2"] = ("ChannelType", oid.get_tc(data[key][0],"ChannelType"))
            tmp["3"] = ("ChannelProtocolVersion"         ,data[key][1])
            tmp["4"] = ("ChannelCurrentJobCntlLangIndex" ,data[key][2])  # interpreters index ::PJL
            tmp["5"] = ("ChannelDefaultPageDescLangIndex",data[key][3])  # interpreters index ::PDL
            tmp["6"] = ("ChannelState" ,oid.get_tc(data[key][4],"ChannelState"))
            tmp["7"] = ("ChannelIfIndex",data[key][5])  # IF oid or 0
            tmp["8"] = ("ChannelStatus",oid.get_tc(data[key][6],"SubUnitStatus"))
            if data[key][7] != '':
               tmp["9"] = ("ChannelInformation", data[key][7]) # new
            mibdic[key] = tmp
        return mibdic

    # Interpreter : 15
    def get_printer_Interpreter(self,mib,printer):
        key = oid.get_key_oid('prtInterpreterEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer))
        data = self.get_index_mib2(mib,key,printer,count,2,12)
        if not data:
           return None
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            if data[key][0] == 1 and data[key][3]:
                tmp["02"] = ("InterpreterLangFamily",data[key][3])
            else:
                tmp["02"] = ("InterpreterLangFamily",oid.get_tc(data[key][0],"LangFamily"))
            tmp["03"] = ("InterpreterLangLevel",   data[key][1])
            tmp["04"] = ("InterpreterLangVersion", data[key][2])
            tmp["05"] = ("InterpreterDescription", data[key][3])
            tmp["06"] = ("InterpreterVersion",     data[key][4])
            tmp["07"] = ("InterpreterDefaultOrientation",oid.get_tc(data[key][5],"Orientation"))
            tmp["08"] = ("InterpreterFeedAddressability",data[key][6])  # 10000 * tc_printer.MediaUnit
            tmp["09"] = ("InterpreterXFeedAddressability",data[key][7]) # 10000 * tc_printer.MediaUnit
            tmp["10"] = ("InterpreterDefaultCharSetIn"  ,data[key][8]) # CodedCharSet
            tmp["11"] = ("InterpreterDefaultCharSetOut" ,data[key][9]) # CodedCharSet
            tmp["12"] = ("InterpreterTwoWay",oid.get_tc(data[key][10],"YesNo"))
            mibdic[key] = tmp
        return mibdic

    ## Marker : 10
    def get_printer_Marker(self,mib,printer):
        key = oid.get_key_oid('prtMarkerEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer),32)
        data = self.get_index_mib2(mib,key,printer,count,2,15)
        if not data:
           return None
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["02"] = ("MarkerMarkTech",oid.get_tc(data[key][0],"MarkerMarkTech"))
            tmp["03"] = ("MarkerCounterUnit",oid.get_tc(data[key][1],"MarkerCounterUnit"))
            tmp["04"] = ("MarkerLifeCount"    ,data[key][2])
            tmp["05"] = ("MarkerPowerOnCount" ,data[key][3])
            tmp["06"] = ("MarkerProcessColorants" ,data[key][4])
            tmp["07"] = ("MarkerSpotColorants"    ,data[key][5])
            tmp["08"] = ("MarkerAddressabilityUnit"   ,oid.get_tc(data[key][6],"MediaUnit"))
            tmp["09"] = ("MarkerAddressabilityFeedDir",data[key][7]) # -1:other, -2:unknown
            tmp["10"] = ("MarkerAddressabilityXFeedDir",data[key][8]) # -1:other, -2:unknown
            tmp["11"] = ("MarkerNorthMargin"           ,data[key][9])
            tmp["12"] = ("MarkerSouthMargin"           ,data[key][10])
            tmp["13"] = ("MarkerWestMargin"            ,data[key][11])
            tmp["14"] = ("MarkerEastMargin"            ,data[key][12])
            tmp["15"] = ("MarkerStatus",oid.get_tc(data[key][13],"SubUnitStatus"))
            mibdic[key] = tmp
        return mibdic
    # 11
    def get_printer_MarkerSupplies(self,mib,printer):
        key = oid.get_key_oid('prtMarkerSuppliesEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer),32)
        data = self.get_index_mib2(mib,key,printer,count,2,9)
        if not data:
           return None
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["2"] = ("MarkerSuppliesMarkerIndex" ,data[key][0])  # marker index
            tmp["3"] = ("MarkerSuppliesColorantIndex",data[key][1]) # MarkerColorantMarkerIndex
            tmp["4"] = ("MarkerSuppliesClass",oid.get_tc(data[key][2],"MarkerSuppliesClass"))
            tmp["5"] = ("MarkerSuppliesType" ,oid.get_tc(data[key][3],"MarkerSuppliesType"))
            tmp["6"] = ("MarkerSuppliesDescription",data[key][4])
            tmp["7"] = ("MarkerSuppliesSupplyUnit" ,oid.get_tc(data[key][5],"MarkerSuppliesSupplyUnit"))
            tmp["8"] = ("MarkerSuppliesMaxCapacity",data[key][6])
            tmp["9"] = ("MarkerSuppliesLevel"      ,data[key][7])
            mibdic[key] = tmp
        return mibdic
    # 12
    def get_printer_MarkerColorant(self,mib,printer):
        key = oid.get_key_oid('prtMarkerColorantEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer))
        data = self.get_index_mib2(mib,key,printer,count,2,5)
        if not data:
           return None
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["2"] = ("MarkerColorantMarkerIndex" ,data[key][0])  # marker index
            tmp["3"] = ("MarkerColorantRole",oid.get_tc(data[key][1],"MarkerColorantRole"))
            tmp["4"] = ("MarkerColorantValue"    ,data[key][2]) # color name
            tmp["5"] = ("MarkerColorantTonality" ,data[key][3])# tone
            mibdic[key] = tmp
        return mibdic
    ## Console : 16
    def get_printer_ConsoleDisplay(self,mib,printer):
        key = oid.get_key_oid('prtConsoleDisplayBufferEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer))
        data = self.get_index_mib2(mib,key,printer,count,2,2)
        if not data:
           return None
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            keys = string.split(data[key][0])
            txt = ''
            for ii in keys:
                if ii.startswith("0x"):
                    txt += str(ii[2:])
                else:
                    txt += str(ii)
            tmp["2"] = ("ConsoleDisplayBufferText" ,txt)  # marker index
            mibdic[key] = tmp
        return mibdic
    # 17
    def get_printer_ConsoleLight(self,mib,printer):
        key = oid.get_key_oid('prtConsoleLightEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer))
        data = self.get_index_mib2(mib,key,printer,count,2,5)
        if not data:
           return None
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["2"] = ("ConsoleOnTime" ,data[key][0])
            tmp["3"] = ("ConsoleOffTime",data[key][1])
            tmp["4"] = ("ConsoleColor"  ,oid.get_tc(data[key][2],"ConsoleColor"))
            tmp["5"] = ("ConsoleDescription",data[key][3])
            mibdic[key] = tmp
        return mibdic

    ## Finisher : 30
    def get_printer_findevice(self,mib,printer):
        key = oid.get_key_oid('finDeviceEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer))
        data = self.get_index_mib2(mib,key,printer,count,2,10)
        if not data:
           return None
        mibdic = {}
        for key in data.keys():
            tmp = {}
            tmp["02"] = ("finDeviceType",oid.get_tc(data[key][0],"FinDeviceType"))
            tmp["03"] = ("finDevicePresentOnOff",oid.get_tc(data[key][1],"PresentOnOff"))
            tmp["04"] = ("finDeviceCapacityUnit",oid.get_tc(data[key][2],"CapacityUnit"))
            tmp["05"] = ("finDeviceMaxCapacity"     ,data[key][3])
            tmp["06"] = ("finDeviceCurrentCapacity" ,data[key][4])
            tmp["07"] = ("finDeviceAssociatedMediaPaths",data[key][5])
            tmp["08"] = ("finDeviceAssociatedOutputs"   ,data[key][6])
            tmp["09"] = ("finDeviceStatus" ,oid.get_tc(data[key][7],"SubUnitStatus"))
            tmp["10"] = ("finDeviceDescription",data[key][8])
            mibdic[key] = tmp
        return mibdic
    # 31
    def get_printer_finSupply(self,mib,printer):
        key = oid.get_key_oid('finSupplyEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer))
        data = self.get_index_mib2(mib,key,printer,count,2,9)
        if not data:
           return None
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["2"] = ("finSupplyDeviceIndex", data[key][0]) # finDeviceIndex
            tmp["3"] = ("finSupplyClass"      ,oid.get_tc(data[key][1],"MarkerSuppliesClass"))
            tmp["4"] = ("finSupplyType"       ,oid.get_tc(data[key][2],"MarkerSuppliesType"))
            tmp["5"] = ("finSupplyDescription",data[key][3])  # PrtLocalizedDescriptionStringTC,
            tmp["6"] = ("finSupplyUnit"       ,oid.get_tc(data[key][4],"MarkerSuppliesSupplyUnit"))
            tmp["7"] = ("finSupplyMaxCapacity",data[key][5])
            tmp["8"] = ("finSupplyCurrentLevel",data[key][6])
            tmp["9"] = ("finSupplyColorName"   ,data[key][7])
            mibdic[key] = tmp
        return mibdic
    # 32
    def get_printer_finMediaInput(self,mib,printer):
        key = oid.get_key_oid('finSupplyMediaInputEntry',oid.OID_printer)
        if key is None:
            return None
        count = self.mib_index_count(mib,'%s.2.%s.' % (key,printer))
        data = self.get_index_mib2(mib,key,printer,count,2,15)
        if not data:
           return None
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["02"] = ("finSupplyMediaInputDeviceIndex",data[key][0]) # finDeviceIndex
            tmp["03"] = ("finSupplyMediaInputSupplyIndex",data[key][1]) # Integer32,
            tmp["04"] = ("finSupplyMediaInputType"       ,oid.get_tc(data[key][2],"InputType"))
            tmp["05"] = ("finSupplyMediaInputDimUnit"    ,oid.get_tc(data[key][3],"MediaUnit"))
            tmp["06"] = ("finSupplyMediaInputMediaDimFeedDir" ,data[key][4])
            tmp["07"] = ("finSupplyMediaInputMediaDimXFeedDir",data[key][5])
            tmp["08"] = ("finSupplyMediaInputStatus"     ,oid.get_tc(data[key][6],"SubUnitStatus"))
            tmp["09"] = ("finSupplyMediaInputMediaName"  ,data[key][7])
            tmp["10"] = ("finSupplyMediaInputName"       ,data[key][8])
            tmp["11"] = ("finSupplyMediaInputDescription",data[key][9]) # PrtLocalizedDescriptionStringTC,
            tmp["12"] = ("finSupplyMediaInputSecurity"   ,oid.get_tc(data[key][10],"PresentOnOff"))
            tmp["13"] = ("finSupplyMediaInputMediaWeight",data[key][11])
            tmp["14"] = ("finSupplyMediaInputMediaThickness",data[key][12])
            tmp["15"] = ("finSupplyMediaInputMediaType"     ,data[key][13])
            mibdic[key] = tmp
        return mibdic
    # 33
    def get_printer_finAttribute(self,mib,printer): # ,finDeviceIndex):
        ## printer,finDeviceIndex,finDeviceAttributeTypeIndex,finDeviceAttributeInstanceIndex
        keys = data.keys()
        keys.sort()
        mibdic = {}
        for key in keys:
            tmp = {}
            tmp["1"] = ("finDeviceAttributeTypeIndex",oid.get_tc(data[key][0],"FinAttributeType"))
            tmp["2"] = ("finDeviceAttributeInstanceIndex" ,data[key][1]) # Integer32,
            tmp["3"] = ("finDeviceAttributeValueAsInteger",data[key][2]) # Integer32,
            #tmp["4"] = ("finDeviceAttributeValueAsOctets" ,data[key][3])
            mibdic[key] = tmp
        return mibdic

