-- CGI library.
--

local stdout = io.stdout
local getenv = os.getenv
local setmetatable = setmetatable
local require = require
local string = string
local table = table
local type = type
local stderr = io.stderr
local os = os
local _G = _G

module(..., package.seeall)
local lp = require "cgi.lp"
local urlcode = require "cgi.urlcode"

---------------------------------------------------------------------------
-- Returns a temporary file in a directory using a name generator
-- @param dir Base directory for the temporary file
-- @param namefunction Name generator function
---------------------------------------------------------------------------
local _tmpfiles = {}
function tmpfile(dir, namefunction)
    dir = dir or tmp_path
    namefunction = namefunction or tmpname
    local tempname = namefunction()
    local filename = dir.."/"..tempname
    local file, err = io.open(filename, "wb+")
    if file then
        table.insert(_tmpfiles, {name = filename, file = file})
    end
    return file, err
end
local function reset ()
    -- clean temporary files
    table.foreachi(_tmpfiles, function (i, v)
        v.file:close()
        local _, err = os.remove(v.name)
        if err then
            error(err)
        end
    end)
end


local post = require "cgi.post"

-- global variable
log = loadfile('cgi.logging.config')()

--
-- Define variables and build the cgi.POST, cgi.GET tables.
--
local function getparams()
    -- Fill in the POST table.
    local POST = {}
    local servervariable = SAPI.Request.servervariable
    if servervariable("REQUEST_METHOD") == "POST" then
        post.parsedata {
            read = SAPI.Request.getpostdata,
            discardinput = ap and ap.discard_request_body,
            content_type = servervariable("CONTENT_TYPE"),
            content_length = servervariable("CONTENT_LENGTH"),
            maxinput = 5000000,
            maxfilesize = 5000000,
            args = POST,
        }
    end
    -- Fill in the QUERY table.
    local QUERY = {}
    urlcode.parsequery(servervariable("QUERY_STRING"), QUERY)
    return POST, QUERY
end
-- Default path for temporary files
tmp_path = getenv("TEMP") or getenv ("TMP") or "/tmp"

-- Default function for temporary names
-- @returns a temporay name using os.tmpname
tmpname = function()
    local tempname = os.tmpname()
    -- Lua os.tmpname returns a full path in Unix, but not in Windows
    -- so we strip the eventual prefix
    tempname = string.gsub(tempname, "(/tmp/)", "")
    return tempname
end



-- global variable
request = nil
-- global variable
response = nil
function init(env, action_dir)
    action_dir = action_dir or 'lua'
    request = Request.new()
    response = Response.new()
    -- load actions.
    require "lfs"
    local currentdir = lfs.currentdir()
    lfs.chdir(action_dir)
    for file in lfs.dir('action') do
        if string.match(file, "\.lua") then
            local action_file_path = string.format("action/%s", file:sub(0, string.len(file) - 4))
            cgi.log:debug('load file. ' .. action_file_path)
            local p = require(action_file_path)
            cgi.log:debug('loaded file. ' .. action_file_path)
            cgi.log:debug(p._NAME)
            cgi.log:debug(p._PACKAGE)
            for k,v in pairs(p) do
            end
        end
    end
    lfs.chdir(currentdir)
end
function destroy()
    cgi.request.session:close()
    reset()
end

local Session = {}
function Session.new()
    require("cgi.session")
    cgi.session.setsessiondir('./session/')
    local id = cgi.session.open()
    local self = {
        id = id,
        data = cgi.session.data or {}
    }
    return setmetatable(self, {__index = Session})
end
function Session:close()
    cgi.session.data = self.data
    cgi.session.close(self.id)
end

function get_action(subapp_name, action_name, env)
    local func = nil
    local packagename = string.format("action/%s", subapp_name or 'root')
    action_name = action_name or 'index'
    if packagename then
        if package.loaded[packagename] and type(package.loaded[packagename][action_name]) == 'function' then
            func = package.loaded[packagename][action_name]
        elseif type(package.loaded[packagename]) == 'function' then
            func = package.loaded[packagename]
        end
    end
    return func
end
local function get_path_infos(path_info)
    local path_infos = {}
    if not path_info or string.len(path_info) < 2 then
        return path_infos
    end
    string.gsub(path_info ,"/?([^\/]+)/?",
        function (x)
            table.insert(path_infos, x)
        end
    )
    return path_infos
end

Request = {}
function Request.new()
    local POST, QUERY = getparams()
    local session = Session.new()
    local self = {
        query_string = getenv('QUERY_STRING'),
        server_name = getenv('SERVER_NAME'),
        server_port = getenv('SERVER_PORT'),
        path_info = getenv('PATH_INFO'),
        path_infos = get_path_infos(getenv('PATH_INFO')),
        script_filename = getenv('SCRIPT_FILENAME'),
        script_name = getenv('SCRIPT_NAME'),
        app_name = string.gsub(getenv('SCRIPT_NAME'), "[^\/]+$", ""),
        session = session,
        POST = POST,
        QUERY = QUERY
    }
    return setmetatable(self, {__index = Request})
end

Response = {}
function Response.new()
    local self = {
        script_filename = getenv('SCRIPT_FILENAME'),
        script_name = getenv('SCRIPT_NAME'),
    }
    return setmetatable(self, {__index = Response})
end
function Response:put(...)
    stdout:write(...)
end
function Response:forward(result)
    local filename, env = result.forward, _G
        if result.env then
        for k, v in pairs(result.env) do
            env[k] = v
        end
    end
    local dir = string.match(self.script_filename, '^(.*)/[^/]+$')
    self:put("Content-Type: text/html;charset=UTF-8;\n\n")
    cgi.log:debug('forward')
    lp.include(dir .. '/' .. filename, env)
    cgi.log:debug('forwarded')
end
function Response:redirect(result)
    local redirect_url, params = result.redirect, result.params
    local querystring = ''
    if params then
        local encodedparams = {}
        for k, v in pairs(params) do
            table.insert(encodedparams, string.format('%s=%s', k, cgi.urlcode.escape(v)))
        end
        querystring = '?' .. table.concat(encodedparams, '&amp;')
    end
    self:put("Status: 302 Moved Temporary.\n")
    self:put(string.format("location: %s%s%s\n\n", self.script_name, redirect_url, querystring))
end
function Response:file_not_found()
    self:put("Status: 404 file not found.\n")
    self:put("Content-Type: text/html; charset=UTF-8\n\n")
    self:put("<html><head><title>404 not found.</title></head><body>404 not found.<a href=\"" .. self.script_name .. "/auth/login\">ログインする</a></body></html>\n")
end
function Response:error500()
    self:put("Status: 500 internal server error.\n")
    self:put("Content-Type: text/html; charset=UTF-8\n\n")
    self:put("<html><head><title>500 Internal Server Error</title></head><body>エラーが発生しました。<a href=\"" .. self.script_name .. "/auth/login\">ログインする</a></body></html>\n")
end

function action(env)
    local request, response = cgi.request, cgi.response

    local subapp_name = table.remove(request.path_infos, 1)
    local action_name = table.remove(request.path_infos, 1)

--    if subapp_name ~= 'auth' and not request.session.data.loginuser then
--        cgi.log:debug('timeout? data: ', request.session.data)
--        local targetpath = request.path_info
--        cgi.log:debug('targetpath: ', targetpath)
--        cgi.destroy()
--        response:redirect({redirect = '/auth/login', params = {targetpath = targetpath}})
--        os.exit(-1)
--    end
    local action = cgi.get_action(subapp_name, action_name, env)
    cgi.log:info('got action', action)

    local result = {}
    env.errors = {}
    if action then
        result = action(request, response)
    else
        cgi.log:info('no action. ', request.url)
        response:file_not_found(request.url)
        cgi.destroy()
        os.exit(-1)
    end

    if not result then
        cgi.log:debug('ok.')
        cgi.log:debug(result)
        -- do nothing.
    elseif result.error then
        cgi.log:debug('print error page.')
        response:error500(request.url)
    elseif result.forward then
        cgi.log:debug('forward to ' .. result.forward)
        response:forward(result)
    elseif result.redirect then
        cgi.log:debug('redirect to ' .. result.redirect)
        response:redirect(result)
    end
    cgi.destroy()
end
-- vim: set ft=lua ts=4 sw=4 sts=4 expandtab :
