/* Copyright(C) 2004,2005,2006 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "senna_in.h"
#include "str.h"
#include "inv.h"
#include "store.h"
#include <string.h>

static int len_sum = 0;
static int img_sum = 0;
static int simple_sum = 0;
static int skip_sum = 0;

sen_store *
sen_store_create(const char *path)
{
  sen_store *s;
  if (!(s = SEN_MALLOC(sizeof(sen_store)))) { return NULL; }
  s->vgram = sen_sym_create(path, sizeof(sen_id) * 2, 0, sen_enc_none);
  if (!s->vgram) {
    SEN_FREE(s);
    return NULL;
  }
  return s;
}

sen_store *
sen_store_open(const char *path)
{
  sen_store *s;
  if (!(s = SEN_MALLOC(sizeof(sen_store)))) { return NULL; }
  s->vgram = sen_sym_open(path);
  if (!s->vgram) {
    SEN_FREE(s);
    return NULL;
  }
  return s;
}

sen_store_buf *
sen_store_buf_open(size_t len)
{
  sen_store_buf *b;
  if (!(b = SEN_MALLOC(sizeof(sen_store_buf)))) { return NULL; }
  b->len = len;
  b->tvs = b->tvp = SEN_MALLOC(sizeof(sen_id) * len);
  if (!b->tvp) { SEN_FREE(b); return NULL; }
  b->tve = b->tvs + len;
  b->vps = b->vpp = SEN_MALLOC(sizeof(sen_store_vnode) * len * 2);
  if (!b->vpp) { SEN_FREE(b->tvp); SEN_FREE(b); return NULL; }
  b->vpe = b->vps + len;
  return b;
}

sen_rc
sen_store_buf_add(sen_store_buf *b, sen_id tid)
{
  uint8_t dummybuf[8], *dummyp;
  if (b->tvp < b->tve) { *b->tvp++ = tid; }
  dummyp = dummybuf;
  SEN_B_ENC(tid, dummyp);
  simple_sum += dummyp - dummybuf;
  return sen_success;
}

typedef struct {
  sen_id vid;
  sen_id tid;
} vgram_key;

sen_rc
sen_store_update(sen_store *store, sen_id rid, sen_store_buf *b, sen_set *terms)
{
  sen_inv_updspec **u;
  if (b && b->tvs < b->tvp) {
    sen_id *t0, *tn;
    for (t0 = b->tvs; t0 < b->tvp - 1; t0++) {
      sen_store_vnode *v, **vp;
      sen_set_at(terms, t0, (void **) &u);
      vp = &(*u)->vnodes;
      for (tn = t0 + 1; tn < b->tvp; tn++) {
	for (v = *vp; v && v->tid != *tn; v = v->cdr) ;
	if (!v) {
	  if (b->vpp < b->vpe) {
	    v = b->vpp++;
	  } else {
	    // todo;
	    break;
	  }
	  v->car = NULL;
	  v->cdr = *vp;
	  *vp = v;
	  v->tid = *tn;
	  v->vid = 0;
	  v->freq = 0;
	  v->len = tn - t0;
	}
	v->freq++;
	if (v->vid) {
	  vp = &v->car;
	} else {
	  break;
	}
      }
    }
    {
      sen_set *th = sen_set_open(sizeof(sen_id), sizeof(int), 0);
      if (!th) { return sen_memory_exhausted; }
      if (t0 == b->tvp) { SEN_LOG(sen_log_debug, "t0 == tvp"); }
      for (t0 = b->tvs; t0 < b->tvp; t0++) {
	sen_id vid, vid0 = *t0, vid1 = 0;
	sen_store_vnode *v, *v2 = NULL, **vp;
	sen_set_at(terms, t0, (void **) &u);
	vp = &(*u)->vnodes;
	for (tn = t0 + 1; tn < b->tvp; tn++) {
	  for (v = *vp; v; v = v->cdr) {
	    if (!v->vid && (v->freq < 2 || v->freq * v->len < 4)) {
	      *vp = v->cdr;
	      v->freq = 0;
	    }
	    if (v->tid == *tn) { break; }
	    vp = &v->cdr;
	  }
	  if (v) {
	    if (v->freq) {
	      v2 = v;
	      vid1 = vid0;
	      vid0 = v->vid;
	    }
	    if (v->vid) {
	      vp = &v->car;
	      continue;
	    }
	  }
	  break;
	}
	if (v2) {
	  if (!v2->vid) {
	    vgram_key key;
	    key.vid = vid1;
	    key.tid = v2->tid;
	    v2->vid = sen_sym_get(store->vgram, (char *)&key);
	  }
	  vid = *t0 = v2->vid * 2 + 1;
	  memset(t0 + 1, 0, sizeof(sen_id) * v2->len);
	  t0 += v2->len;
	} else {
	  vid = *t0 *= 2;
	}
	{
	  int *tf;
	  sen_set_get(th, &vid, (void **) &tf);
	  (*tf)++;
	}
      }
      if (!th->n_entries) { SEN_LOG(sen_log_debug, "th->n_entries == 0"); }
      {
	int j = 0;
	int skip = 0;
	sen_set_eh *ehs, *ehp, *ehe;
	sen_set_sort_optarg arg;
	uint8_t *ps = SEN_MALLOC(b->len * 2), *pp, *pe;
  if (!ps) {
    sen_set_close(th);
    return sen_memory_exhausted;
  }
	pp = ps;
	pe = ps + b->len * 2;
	arg.mode = sen_sort_descending;
	arg.compar = NULL;
	arg.compar_arg = (void *)(intptr_t)sizeof(sen_id);
	arg.compar_arg0 = NULL;
	ehs = sen_set_sort(th, 0, &arg);
	if (!ehs) {
	  SEN_FREE(ps);
    sen_set_close(th);
    return sen_memory_exhausted;
	
  }
	SEN_B_ENC(th->n_entries, pp);
	for (ehp = ehs, ehe = ehs + th->n_entries; ehp < ehe; ehp++, j++) {
	  int *id = (int *)SEN_SET_INTVAL(*ehp);
	  SEN_B_ENC(*SEN_SET_INTKEY(*ehp), pp);
	  *id = j;
	}
	for (t0 = b->tvs; t0 < b->tvp; t0++) {
	  if (*t0) {
	    int *id;
	    if (!sen_set_at(th, t0, (void **) &id)) {
	      SEN_LOG(sen_log_error, "lookup error (%d)", *t0);
	    }
	    SEN_B_ENC(*id, pp);
	  } else {
	    skip++;
	  }
	}
	len_sum += b->len;
	img_sum += pp - ps;
	skip_sum += skip;
	SEN_FREE(ehs);
	SEN_FREE(ps);
      }
      sen_set_close(th);
    }
  }
  return sen_success;
}

sen_rc
sen_store_buf_close(sen_store_buf *b)
{
  if (!b) { return sen_invalid_argument; }
  if (b->tvs) { SEN_FREE(b->tvs); }
  if (b->vps) { SEN_FREE(b->vps); }
  SEN_FREE(b);
  return sen_success;
}

sen_rc
sen_store_close(sen_store *store)
{
  if (!store) { return sen_invalid_argument; }
  SEN_LOG(sen_log_debug, "len=%d img=%d skip=%d simple=%d", len_sum, img_sum, skip_sum, simple_sum);
  sen_sym_close(store->vgram);
  SEN_FREE(store);
  return sen_success;
}
