/* 
 * Copyright (c) 2006-2007 NTT DATA CORPORATION.
 * All rights reserved.
 *
 * derived from original pg_senna.c developed by Daisuke Maki
 * (See below for copyright notice)
 *
 */
/* 
 * Portions of this code were written based on pg_senna.c:
 * 
 * $Id: pg_senna.c 14 2006-01-02 11:03:54Z daisuke $
 *
 * Copyright (c) 2005 Daisuke Maki <dmaki@cpan.org>
 * All rights reserved.
 *
 * I used pg_rast.c as a sample to learn from. Much thanks goes to the
 * pg_rast developers (See below for copyright notice)
 *
 * Development was funded by Brazil, Ltd. <http://b.razil.jp>
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without a
 * written agreement is hereby granted, provided that the above
 * copyright notice and this paragraph and the following two
 * paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING
 * LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 * DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS
 * IS" BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */
/* 
 * Portions of this code were written based on pg_rast.c:
 *
 * Copyright (c) 2005 Satoshi Nagayasu <snaga@snaga.org>
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without a
 * written agreement is hereby granted, provided that the above
 * copyright notice and this paragraph and the following two
 * paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING
 * LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 * DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS
 * IS" BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include "postgres.h"
#include <mb/pg_wchar.h>
#include "access/genam.h"
#include "access/heapam.h"
#include "pgsenna2.h"
#include "access/xlogutils.h"
#include "catalog/index.h"
#include "catalog/catalog.h"
#include "storage/smgr.h"
#include "commands/vacuum.h"
#include "executor/executor.h"
#include "miscadmin.h"
#include "optimizer/cost.h"
#include "optimizer/restrictinfo.h"
#include "nodes/relation.h"
#include "storage/itemptr.h"
#include "utils/guc.h"
#include "funcapi.h"

#ifdef POSTGRES82
#include "access/reloptions.h"

PG_MODULE_MAGIC;
#endif

PG_FUNCTION_INFO_V1(pgs2insert);
PG_FUNCTION_INFO_V1(pgs2beginscan);
PG_FUNCTION_INFO_V1(pgs2gettuple);
PG_FUNCTION_INFO_V1(pgs2getmulti);
PG_FUNCTION_INFO_V1(pgs2rescan);
PG_FUNCTION_INFO_V1(pgs2endscan);
PG_FUNCTION_INFO_V1(pgs2markpos);
PG_FUNCTION_INFO_V1(pgs2restrpos);
PG_FUNCTION_INFO_V1(pgs2build);
PG_FUNCTION_INFO_V1(pgs2buildb);
PG_FUNCTION_INFO_V1(pgs2buildu);
PG_FUNCTION_INFO_V1(pgs2bulkdelete);
PG_FUNCTION_INFO_V1(pgs2vacuumcleanup);
PG_FUNCTION_INFO_V1(pgs2costestimate);
PG_FUNCTION_INFO_V1(pgs2contain);
PG_FUNCTION_INFO_V1(pgs2nop);
PG_FUNCTION_INFO_V1(pgs2getscore);
PG_FUNCTION_INFO_V1(pgs2getscore_simple);
PG_FUNCTION_INFO_V1(pgs2getnhits);
#ifdef POSTGRES82
PG_FUNCTION_INFO_V1(pgs2options);
#endif
PG_FUNCTION_INFO_V1(pgs2indexcache);

typedef struct _index_info{
  Oid relid;
  Oid namespace;
  char name[NAMEDATALEN];
  sen_index *index;
  int using;
  IndexScanDesc s;
  struct _index_info *previous;
  struct _index_info *next;
} index_info;

typedef struct {
  FmgrInfo filterFn;
} support_funcs;

typedef struct {
  support_funcs funcs;
  double indtuples;
  index_info *index;
} build_stat;

typedef struct {
  sen_records *records;
  index_info *index;
  sen_recordh *mark;
} scan_stat;

static index_info *index_cache;
static index_info *last_used_cache;
static scan_stat *curr_scan_stat;
static int last_nhits = 0;
static int max_n_index_cache = 16;

inline static void
sen_check_init(void)
{
  static int initialized = 0;
  int i;
  sen_rc rc;
  index_info *ii;
  const char *option;

  if (initialized) { return ; }
  if ((rc = sen_init())) { elog(ERROR, "pgsenna2: sen_init() failed %d", rc); }
  option = GetConfigOption("ludia.max_n_index_cache");
  if (option) {
    max_n_index_cache = atoi(option);
    if(max_n_index_cache <= 0) {
      elog(ERROR, "pgsenna2: value of max_n_index_cache is invalid: %d", max_n_index_cache);
    }
  } else {
    elog(WARNING, "pgsenna2: value of max_n_index_cache = %d", max_n_index_cache);
  }
  index_cache = (index_info*)malloc(max_n_index_cache * sizeof(*index_cache));
  last_used_cache = NULL;
  for (i = max_n_index_cache, ii = index_cache; i; i--, ii++) {
    ii->namespace = 0;
    ii->relid = 0;
    ii->name[0] = '\0';
    ii->index = NULL;
    ii->using = 0;
    ii->s = NULL;
    ii->previous = NULL;
    ii->next = NULL;
  }
  initialized++;
}

inline static void
index_list_update(index_info *ii) {
  if (last_used_cache == ii) {
    return;
  }
  if (ii->previous) {
    ((index_info *)ii->previous)->next = ii->next;
  }
  if (ii->next) {
    ((index_info *)ii->next)->previous = ii->previous;
  }
  if (last_used_cache) {
    last_used_cache->previous = ii;
  }
  ii->next = last_used_cache;
  ii->previous = NULL;
  last_used_cache = ii;
}

/* todo: SizeOfIptrData may be more suitable.  */
#define KEY_SIZE (sizeof(ItemPointerData))

inline static index_info *
index_info_open(Relation r, int createp, int flags)
{
  int i;
  index_info *ii, *cand;
  char path[MAXPGPATH];
  Oid namespace = RelationGetNamespace(r);
  Oid relid = RelationGetRelid(r);
  char *name = RelationGetRelationName(r);
  sen_check_init();
  ii = last_used_cache;
  for (i = max_n_index_cache - 1; i; i--) {
    if(!ii) {
      break;
    } else {
      if (ii->index && 
          namespace == ii->namespace &&
          relid == ii->relid && 
          !strcmp(ii->name, name)) {
        if (createp) { /* for REINDEX (and CLUSTER) */
          break;
        } else {
          index_list_update(ii);
          return ii;
        }
      } else {
        ii = ii->next;
      }
    }
  }
  cand = ii;  // cand is oldest or NULL
  if (!cand) {
    for (i = max_n_index_cache, ii = index_cache; i; i--, ii++) {
      if (!ii->using) {
        cand = ii;
        break;
      }
    }
  }
  if (!cand) {
    elog(ERROR, "pgsenna2: n of indices in use reached max(%d)", max_n_index_cache);
  }
  index_list_update(cand);

  sen_index_close(cand->index);
  cand->namespace = namespace;
  cand->relid = relid;
  strcpy(cand->name, name);
  {
    char *pathname;
    RelationOpenSmgr(r);
    pathname = relpath(r->rd_smgr->smgr_rnode);
    snprintf(path, MAXPGPATH, "%s/%s", DataDir, pathname);
    RelationCloseSmgr(r);
    pfree(pathname);
  }
  if (createp) {
    sen_encoding encoding;
    switch (GetDatabaseEncoding()) {
    case PG_UTF8:
      encoding = sen_enc_utf8;
      break;
    case PG_EUC_JP:
      encoding = sen_enc_euc_jp;
      break;
    case PG_SJIS:
      encoding = sen_enc_sjis;
      break;
    default:
      encoding = sen_enc_default;
    }
    sen_index_remove(path);
    cand->index = sen_index_create(path,
                                   KEY_SIZE,
                                   flags, 0, encoding);
    if (!cand->index) {
      elog(ERROR, "pgsenna2: index create failed (%s)", path);
    }
  } else {
    cand->index = sen_index_open(path);
    if (!cand->index) {
      elog(ERROR, "pgsenna2: index open failed (%s)", path);
    }
  }
  cand->using = 1;
  return cand;
}

inline static void
index_info_close(index_info *ii)
{
  /* ii->using = 0; */
}

inline static void *
scan_stat_open(sen_records *records, index_info *ii)
{
  scan_stat *ss = palloc(sizeof(scan_stat));
  ss->records = records;
  ss->index = ii;
  ss->mark = NULL;
  curr_scan_stat = ss;
  last_nhits = sen_records_nhits(records);
  return (void *) ss;
}

inline static void
scan_stat_close(void *opaque)
{
  scan_stat *ss = (scan_stat *) opaque;
  if (ss) {
    if (ss->records) { sen_records_close(ss->records); }
    if (ss->index) { index_info_close(ss->index); }
    pfree(ss);
  }
  curr_scan_stat = NULL;
}

inline static void
init_support_funcs(support_funcs *funcs, Relation index)
{
  fmgr_info_copy(&funcs->filterFn,
                 index_getprocinfo(index, 1, PGS2_FILTER_PROC),
                 CurrentMemoryContext);
}

inline static void
do_insert(Relation r, ItemPointer tid, index_info *ii,
          Datum *values, bool *isnull, support_funcs *funcs)
{
  char *c;
  text *t;
  sen_rc rc = sen_success;
  Datum value;
  int i, natts = RelationGetNumberOfAttributes(r);
  for (i = 0; i < natts; i ++) {
    if (!isnull[i]) {
      value = FunctionCall1(&funcs->filterFn, PointerGetDatum(values[i]));
      t = DatumGetTextP(value);
      c = text2cstr(t);
      if (c && *c) {
        rc = sen_index_upd(ii->index, (void *)tid, NULL, 0, c, strlen(c));
      } else {
        if (!(sen_sym_get(ii->index->keys, (void *)tid))) { rc = sen_other_error; }
      }
      pfree(c);
      if (t != (text *) value) {
        sen_log("pgsenna2: palloced when untoasted (%p)", t);
        pfree(t);
      }
      if (value != values[i]) {
        sen_log("pgsenna2: palloced in filter function (%p)", value);
        pfree((text *)value);
      }
      if (rc) {
        elog(ERROR, "pgsenna2: insert failed (%d)", rc);
      }
    }
  }
}

static void
buildCallback(Relation index,
              HeapTuple htup,
              Datum *values,
              bool *isnull,
              bool tupleIsAlive,
              void *state)
{
  build_stat *bs = (build_stat *) state;
  do_insert(index, &htup->t_self, bs->index, values, isnull, &bs->funcs);
  bs->indtuples += 1;
}

Datum
pgs2build(PG_FUNCTION_ARGS)
{
  Relation  heap = (Relation) PG_GETARG_POINTER(0);
  Relation  index = (Relation) PG_GETARG_POINTER(1);
  IndexInfo  *indexInfo = (IndexInfo *) PG_GETARG_POINTER(2);
#ifdef POSTGRES82
  IndexBuildResult *result;
#endif
  build_stat bs;
  double    reltuples;
  init_support_funcs(&bs.funcs, index);
  bs.indtuples = 0;
  bs.index = index_info_open(index, 1, SEN_INDEX_NORMALIZE);
  reltuples = IndexBuildHeapScan(heap, index, indexInfo,
                                 buildCallback, (void *) &bs);
#ifdef POSTGRES82
  result = (IndexBuildResult *) palloc(sizeof(IndexBuildResult));
  result->heap_tuples = reltuples;
  result->index_tuples = bs.indtuples;
  index_info_close(bs.index);
  PG_RETURN_POINTER(result);
#else
  IndexCloseAndUpdateStats(heap, reltuples, index, bs.indtuples);
  index_info_close(bs.index);
  PG_RETURN_VOID();
#endif
}

Datum
pgs2buildb(PG_FUNCTION_ARGS)
{
  Relation  heap = (Relation) PG_GETARG_POINTER(0);
  Relation  index = (Relation) PG_GETARG_POINTER(1);
  IndexInfo  *indexInfo = (IndexInfo *) PG_GETARG_POINTER(2);
#ifdef POSTGRES82
  IndexBuildResult *result;
#endif
  build_stat bs;
  double    reltuples;
  init_support_funcs(&bs.funcs, index);
  bs.indtuples = 0;
  bs.index = index_info_open(index, 1, SEN_INDEX_NGRAM|SEN_INDEX_NORMALIZE);
  reltuples = IndexBuildHeapScan(heap, index, indexInfo,
                                 buildCallback, (void *) &bs);
#ifdef POSTGRES82
  result = (IndexBuildResult *) palloc(sizeof(IndexBuildResult));
  result->heap_tuples = reltuples;
  result->index_tuples = bs.indtuples;
  index_info_close(bs.index);
  PG_RETURN_POINTER(result);
#else
  IndexCloseAndUpdateStats(heap, reltuples, index, bs.indtuples);
  index_info_close(bs.index);
  PG_RETURN_VOID();
#endif
}

Datum
pgs2buildu(PG_FUNCTION_ARGS)
{
  Relation  heap = (Relation) PG_GETARG_POINTER(0);
  Relation  index = (Relation) PG_GETARG_POINTER(1);
  IndexInfo  *indexInfo = (IndexInfo *) PG_GETARG_POINTER(2);
#ifdef POSTGRES82
  IndexBuildResult *result;
#endif
  build_stat bs;
  double    reltuples;
  init_support_funcs(&bs.funcs, index);
  bs.indtuples = 0;
  const char *option;
  int flags;

  option = GetConfigOption("ludia.sen_index_flags");
  if (option == NULL) {
    elog(ERROR, "pgsenna2: value of sen_index_flags is invalid: NULL.");
  }
  flags = atoi(option);
  if (flags < 0) {
    elog(ERROR, "pgsenna2: value of sen_index_flags is invalid: %d.", flags);
  }
  bs.index = index_info_open(index, 1, flags);
  reltuples = IndexBuildHeapScan(heap, index, indexInfo,
                                 buildCallback, (void *) &bs);
#ifdef POSTGRES82
  result = (IndexBuildResult *) palloc(sizeof(IndexBuildResult));
  result->heap_tuples = reltuples;
  result->index_tuples = bs.indtuples;
  index_info_close(bs.index);
  PG_RETURN_POINTER(result);
#else
  IndexCloseAndUpdateStats(heap, reltuples, index, bs.indtuples);
  index_info_close(bs.index);
  PG_RETURN_VOID();
#endif
}

Datum
pgs2insert(PG_FUNCTION_ARGS)
{
  Relation  r = (Relation) PG_GETARG_POINTER(0);
  Datum     *values = (Datum *) PG_GETARG_POINTER(1);
  bool     *isnull = (bool *) PG_GETARG_POINTER(2);
  ItemPointer ht_ctid = (ItemPointer) PG_GETARG_POINTER(3);
  support_funcs    funcs;
  index_info *ii;
  init_support_funcs(&funcs, r);
  /*
   * currently, pgsenna2 is not marked "amconcurrent" in pg_am,
   * caller should have acquired exclusive lock on index
   * relation.
   */
  ii = index_info_open(r, 0, 0);
  do_insert(r, ht_ctid, ii, values, isnull, &funcs);
  index_info_close(ii);
  PG_RETURN_BOOL(true);
}

Datum
pgs2bulkdelete(PG_FUNCTION_ARGS)
{
#ifdef POSTGRES82
  IndexVacuumInfo *info = (IndexVacuumInfo *) PG_GETARG_POINTER(0);
  //IndexBulkDeleteResult *stats = (IndexBulkDeleteResult *) PG_GETARG_POINTER(1);
  IndexBulkDeleteCallback callback = (IndexBulkDeleteCallback) PG_GETARG_POINTER(2);
  void *callback_state = (void *) PG_GETARG_POINTER(3);
  Relation r = info->index;
#else
  Relation r = (Relation) PG_GETARG_POINTER(0);
  IndexBulkDeleteCallback callback = (IndexBulkDeleteCallback) PG_GETARG_POINTER(1);
  void *callback_state = (void *) PG_GETARG_POINTER(2);
#endif
  double tuples_removed = 0;
  IndexBulkDeleteResult *result;
  index_info *ii = index_info_open(r, 0, 0);
  sen_rc rc;
  sen_id id;
  ItemPointerData tid;
  unsigned int i, nrecords = sen_sym_size(ii->index->keys);
  sen_log("pgs2bulkdelete n=%d", nrecords);
  for (id = 1, i = 0; i < nrecords; id++) {
    if (!sen_sym_key(ii->index->keys, id, (void *) &tid, KEY_SIZE)) {
      break;
    }
    if (sen_sym_at(ii->index->keys, (void *) &tid) == id) {
      if (callback(&tid, callback_state)) {
        rc = sen_index_del(ii->index, &tid);
        if (rc) { sen_log("sen_index_del failed(%d)", rc); }
    /*
        rc = sen_sym_del(ii->index->keys, &tid);
        if (rc) { sen_log("sen_sym_del failed(%d)", rc); }
    */
        tuples_removed += 1;
      }
    }
  }
  result = (IndexBulkDeleteResult *) palloc0(sizeof(IndexBulkDeleteResult));
  result->num_pages = 1;
  result->num_index_tuples = sen_sym_size(ii->index->keys);
  result->tuples_removed = tuples_removed;
  PG_RETURN_POINTER(result);
}

Datum
pgs2vacuumcleanup(PG_FUNCTION_ARGS)
{
  //IndexVacuumInfo *info = (IndexVacuumInfo *) PG_GETARG_POINTER(0);
  IndexBulkDeleteResult *stats = (IndexBulkDeleteResult *) PG_GETARG_POINTER(1);

  // under construction...
  
  PG_RETURN_POINTER(stats);
}

Datum
pgs2beginscan(PG_FUNCTION_ARGS)
{
  Relation  r = (Relation) PG_GETARG_POINTER(0);
  int      nkeys = PG_GETARG_INT32(1);
  ScanKey    key = (ScanKey) PG_GETARG_POINTER(2);
  IndexScanDesc s;
  s = RelationGetIndexScan(r, nkeys, key);
  PG_RETURN_POINTER(s);
}

Datum
pgs2gettuple(PG_FUNCTION_ARGS)
{
  IndexScanDesc s = (IndexScanDesc) PG_GETARG_POINTER(0);
  ScanDirection dir = (ScanDirection) PG_GETARG_INT32(1);
  scan_stat *ss = s->opaque;
  sen_rc rc;
  int res;

  if (dir != 1) { sen_log("dir(%d) assigned in pgs2gettuple()", dir); }

  if (!ss || !ss->records) {
    elog(ERROR, "pgsenna2: inconsistent scan");
  }

  if (s->kill_prior_tuple) {
    rc = sen_index_del(ss->index->index, &(s->currentItemData));
    if (rc) { sen_log("sen_index_del failed(%d)", rc); }
    /*
    rc = sen_sym_del(ss->index->index->keys, &(s->currentItemData));
    if (rc) { sen_log("sen_sym_del failed(%d)", rc); }
    */
  }

  res = sen_records_next(ss->records, &s->xs_ctup.t_self, KEY_SIZE, NULL);
  if (res) { memcpy(&s->currentItemData, &s->xs_ctup.t_self, KEY_SIZE); }
  PG_RETURN_BOOL(res ? true : false);
}

Datum
pgs2getmulti(PG_FUNCTION_ARGS)
{
  IndexScanDesc s = (IndexScanDesc) PG_GETARG_POINTER(0);
  ItemPointer tids = (ItemPointer) PG_GETARG_POINTER(1);
  int32 max_tids = PG_GETARG_INT32(2);
  int32 *returned_tids = (int32 *) PG_GETARG_POINTER(3);
  int32 i;
  scan_stat *ss = s->opaque;
  bool more = true;
  if (!ss || !ss->records) {
    elog(ERROR, "pgsenna2: inconsistent scan");
  }
  for (i = 0; i < max_tids; i++) {
    if (!sen_records_next(ss->records, &tids[i], KEY_SIZE, NULL)) {
      more = false;
      break;
    }
    memcpy(&s->currentItemData, &tids[i], KEY_SIZE);
  }
  *returned_tids = i;
  PG_RETURN_BOOL(more);
}

Datum
pgs2rescan(PG_FUNCTION_ARGS)
{
  IndexScanDesc s = (IndexScanDesc) PG_GETARG_POINTER(0);
  ScanKey    key = (ScanKey) PG_GETARG_POINTER(1);
  int i;
  sen_rc rc;
  sen_query *q;
  sen_records *r;
  index_info *ii;
  sen_encoding encoding;
  sen_sel_operator op;
  const char *option;
  int max_n_sort_result = 10000;

  if (!key) { elog(ERROR, "pgsenna2: access method does not support scan without scankey."); }

  ItemPointerSetInvalid(&s->currentItemData);
  ItemPointerSetInvalid(&s->currentMarkData);

  scan_stat_close(s->opaque);

  ii = index_info_open(s->indexRelation, 0, 0);
  rc = sen_index_info(ii->index, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &encoding);
  if (rc) {
    elog(ERROR, "pgsenna2: index_info failed(%d)", rc);
  }

  sen_log("nok=%d ign=%d", s->numberOfKeys, s->ignore_killed_tuples);

  if (key && s->numberOfKeys > 0) {
    memmove(s->keyData, key, s->numberOfKeys * sizeof(ScanKeyData));
  }
  r = sen_records_open(sen_rec_document, sen_rec_none, 0);
  if (!r) {
    elog(ERROR, "pgsenna2: sen_records_open failed");
  }

  r->ignore_deleted_records = s->ignore_killed_tuples ? 1 : 0;

  for (i = 0, op = sen_sel_or; i < s->numberOfKeys; i++, op = sen_sel_and) {
    char *c;
    sen_log("sk_flags=%d,atn=%d,strategy=%d,subtype=%d,argument=%x",
            s->keyData[i].sk_flags,
            s->keyData[i].sk_attno,
            s->keyData[i].sk_strategy,
            s->keyData[i].sk_subtype,
            s->keyData[i].sk_argument);
    if(s->keyData[i].sk_argument != 0) {
      c = text2cstr(DatumGetTextP(s->keyData[i].sk_argument));
      if ((q = sen_query_open(c, strlen(c), sen_sel_or,
                              PGS2_MAX_N_EXPRS, encoding))) {
        rc = sen_query_exec(ii->index, q, r, op);
        sen_query_close(q);
      }
      pfree(c);
    }
    if (rc) {
      elog(ERROR, "pgsenna2: sen_query_exec failed(%d)", rc);
    }
  }
  option = GetConfigOption("ludia.max_n_sort_result");
  if (option) {
    max_n_sort_result = atoi(option);
    if (max_n_sort_result <= 0) {
      elog(ERROR, "pgsenna2: value of max_n_sort_result is invalid: %d", max_n_sort_result);
    }
  } else {
      elog(WARNING, "pgsenna2: value of max_n_sort_result = %d", max_n_sort_result);
  }
  sen_records_sort(r, max_n_sort_result, NULL);
  s->opaque = scan_stat_open(r, ii);
  ii->s = s;
  PG_RETURN_VOID();
}

Datum
pgs2endscan(PG_FUNCTION_ARGS)
{
  IndexScanDesc s = (IndexScanDesc) PG_GETARG_POINTER(0);
  scan_stat *ss = (scan_stat *) s->opaque;
  ss->index->s = NULL;
  scan_stat_close(ss);
  PG_RETURN_VOID();
}

Datum
pgs2markpos(PG_FUNCTION_ARGS)
{
  IndexScanDesc s = (IndexScanDesc) PG_GETARG_POINTER(0);
  scan_stat *ss = (scan_stat *) s->opaque;
  ss->mark = ss->records->curr_rec;
  s->currentMarkData = s->currentItemData;
  PG_RETURN_VOID();
}

Datum
pgs2restrpos(PG_FUNCTION_ARGS)
{
  IndexScanDesc s = (IndexScanDesc) PG_GETARG_POINTER(0);
  scan_stat *ss = (scan_stat *) s->opaque;
  ss->records->curr_rec = ss->mark;
  s->currentItemData = s->currentMarkData;
  PG_RETURN_VOID();
}

Datum 
pgs2costestimate(PG_FUNCTION_ARGS)
{
  PlannerInfo *root = (PlannerInfo *) PG_GETARG_POINTER(0);
  IndexOptInfo *index = (IndexOptInfo *) PG_GETARG_POINTER(1);
  List *indexQuals = (List *) PG_GETARG_POINTER(2);
#ifdef POSTGRES82
  //  RelOptInfo *outer_rel = (RelOptInfo *) PG_GETARG_POINTER(3);
  Cost *indexStartupCost = (Cost *) PG_GETARG_POINTER(4);
  Cost *indexTotalCost = (Cost *) PG_GETARG_POINTER(5);
  Selectivity *indexSelectivity = (Selectivity *) PG_GETARG_POINTER(6);
  double *indexCorrelation = (double *) PG_GETARG_POINTER(7);
#else
  Cost *indexStartupCost = (Cost *) PG_GETARG_POINTER(3);
  Cost *indexTotalCost = (Cost *) PG_GETARG_POINTER(4);
  Selectivity *indexSelectivity = (Selectivity *) PG_GETARG_POINTER(5);
  double *indexCorrelation = (double *) PG_GETARG_POINTER(6);
#endif
  double numIndexTuples = 0.0;
  double numIndexPages = 0.0;
  QualCost index_qual_cost;
  double qual_op_cost;
  double qual_arg_cost;
  List *selectivityQuals;
  /*
    index_info *ii = index_info_open(index, 0, 0);
    unsigned int nrecords = sen_sym_size(ii->index->keys);
  */

  if (index->indpred != NIL)
    {
      List       *strippedQuals;
      List       *predExtraQuals;

      strippedQuals = get_actual_clauses(indexQuals);
      predExtraQuals = list_difference(index->indpred, strippedQuals);
      selectivityQuals = list_concat(predExtraQuals, indexQuals);
    }
  else
    selectivityQuals = indexQuals;

  *indexSelectivity = clauselist_selectivity(root, selectivityQuals,
                                             index->rel->relid,
                                             JOIN_INNER);
  if (*indexSelectivity <= 1.0) { *indexSelectivity = 0.0; }

  if (numIndexTuples <= 0.0)
    numIndexTuples = *indexSelectivity * index->rel->tuples;

  if (numIndexTuples > index->tuples)
    numIndexTuples = index->tuples;

  *indexTotalCost = numIndexPages;

  cost_qual_eval(&index_qual_cost, indexQuals);
  qual_op_cost = cpu_operator_cost * list_length(indexQuals);
  qual_arg_cost = index_qual_cost.startup +
    index_qual_cost.per_tuple - qual_op_cost;
  /* if (qual_arg_cost < nrecords) { qual_arg_cost = 0.0; } */
  qual_arg_cost = 0.0;
  *indexStartupCost = qual_arg_cost;
  *indexTotalCost += qual_arg_cost;
  *indexTotalCost += numIndexTuples * (cpu_index_tuple_cost + qual_op_cost);
#ifdef POSTGRES82
  *indexTotalCost += -DEFAULT_RANDOM_PAGE_COST;
#endif
  *indexCorrelation = 1.0;
  sen_log("cost=(%f,%f,%f)", *indexStartupCost, *indexTotalCost, *indexSelectivity);
  PG_RETURN_VOID();
}

typedef struct _cons {
  int8_t type ;
  int8_t op;
  int8_t weight;
  int8_t mode;
  int option;
  struct _cons *cdr;
  union {
    struct {
      const char *start;
      unsigned int len;
    } token;
    struct {
      struct _cons *car;
      struct _cons **tail;
    } expr;
  } u;
} cons;

struct _sen_query {
  const char *str;
  const char *cur;
  const char *str_end;
  char *buffer;
  unsigned int max_size;
  sen_sel_operator default_op;
  sen_sel_mode default_mode;
  int escalation_threshold;
  int escalation_decaystep;
  int weight_offset;
  sen_encoding encoding;
  cons *expr;
  int max_exprs;
  int cur_expr;
  cons cons_pool[1]; /* dummy */
};

static bool exec_query_noindex(sen_query *q, char *ct)
{
  bool res = false;
  cons *token = q->expr->u.expr.car;
  if (token) {
    sen_sel_operator t_op;
    int len;
    int max_len = 16;
    char *cquery = palloc(sizeof(char) * max_len);
    for (t_op = sen_sel_or;; t_op = token->op) {
      len = (token->u.token.len + 1) * sizeof(char);
      if (len > max_len) {
        max_len = len;
        cquery = repalloc(cquery, max_len);
      }
      memcpy(cquery, token->u.token.start, len);
      cquery[token->u.token.len] = '\0';
      switch (t_op) {
      case sen_sel_or :
        res = res || strstr(ct, cquery);
        break;
      case sen_sel_and :
        res = res && strstr(ct, cquery);
        break;
      case sen_sel_but :
        res = res && !strstr(ct, cquery);
        break;
      default :
        break;
      }
      if (!(token = token->cdr)) {
        break;
      }
    }
  }
  return res;
}

Datum 
pgs2contain(PG_FUNCTION_ARGS)
{
  text *t = PG_GETARG_TEXT_P(0);
  text *q = PG_GETARG_TEXT_P(1);
  const char *option;

  option = GetConfigOption("ludia.enable_seqscan");
  if (option) {
    if (!strcmp(option, "on")) {
      char *ct = text2cstr(t);
      char *cq = text2cstr(q);
      bool res = false;
      sen_encoding encoding;
      sen_query *sq;
      switch (GetDatabaseEncoding()) {
      case PG_UTF8:
        encoding = sen_enc_utf8;
        break;
      case PG_EUC_JP:
        encoding = sen_enc_euc_jp;
        break;
      case PG_SJIS:
        encoding = sen_enc_sjis;
        break;
      default:
        encoding = sen_enc_default;
      }
      if ((sq = sen_query_open(cq, strlen(cq), sen_sel_or,
                               PGS2_MAX_N_EXPRS, encoding))) {
        res = exec_query_noindex(sq, ct);
        sen_query_close(sq);
      }
      pfree(ct);
      pfree(cq);
      PG_RETURN_BOOL(res);
    }
    ereport(ERROR, (errmsg("pgsenna2: sequencial scan disabled.")));
  }
  ereport(ERROR, (errmsg("pgsenna2: value of enable_seqscan is invalid.")));
  PG_RETURN_BOOL(false);
}

Datum 
pgs2nop(PG_FUNCTION_ARGS)
{
  PG_RETURN_POINTER(PG_GETARG_POINTER(0));
}

Datum
pgs2getscore(PG_FUNCTION_ARGS)
{
  int i, result;
  index_info *ii;
  scan_stat *ss = NULL;
  ItemPointer tid = (ItemPointer)PG_GETARG_POINTER(0);
  text *t = PG_GETARG_TEXT_P(1);
  char *indexname = text2cstr(t);
  for (i = max_n_index_cache, ii = index_cache; i; i--, ii++) {
    if (ii->index && !strcmp(ii->name, indexname)) { break; }
  }
  if (!i || !ii->s || !(ss = ii->s->opaque) || !ss->records) {
    elog(ERROR, "pgsenna2: index (%s) is not available", indexname);
  }
  result = sen_records_find(ss->records, (void *)tid);
  pfree(indexname);
  PG_RETURN_INT32(result);
}

Datum
pgs2getscore_simple(PG_FUNCTION_ARGS)
{
  int result = 0;
  ItemPointer tid = (ItemPointer)PG_GETARG_POINTER(0);
  if (curr_scan_stat) {
    result = sen_records_find(curr_scan_stat->records, (void *)tid);
  }
  PG_RETURN_INT32(result);
}

Datum
pgs2getnhits(PG_FUNCTION_ARGS)
{
  PG_RETURN_INT32(last_nhits);
}

#ifdef POSTGRES82
Datum
pgs2options(PG_FUNCTION_ARGS)
{
  elog(ERROR, "pgsenna2: pgs2options is called");
  /*
  Datum      reloptions = PG_GETARG_DATUM(0);
  bool       validate = PG_GETARG_BOOL(1);
  bytea      *result;

  result = default_reloptions(reloptions, validate,
                              PGS2_MIN_FILLFACTOR,
                              PGS2_DEFAULT_FILLFACTOR);
  if (result)
    PG_RETURN_BYTEA_P(result);
  */
  PG_RETURN_NULL();
}
#endif

Datum
pgs2indexcache(PG_FUNCTION_ARGS)
{
  int n = 0;

  FuncCallContext     *funcctx;
  int                  call_cntr;
  int                  max_calls;
  TupleDesc            tupdesc;
  AttInMetadata       *attinmeta;
  index_info **last_used_cache_array;
  
  if (SRF_IS_FIRSTCALL()) {
    index_info *tmp_used_cache;
    MemoryContext   oldcontext;
    funcctx = SRF_FIRSTCALL_INIT();
    oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);
    if (get_call_result_type(fcinfo, NULL, &tupdesc) != TYPEFUNC_COMPOSITE)
      ereport(ERROR,
              (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
               errmsg("function returning record called in context "
                      "that cannot accept type record")));
    attinmeta = TupleDescGetAttInMetadata(tupdesc);
    funcctx->attinmeta = attinmeta;

    last_used_cache_array = (index_info **)palloc(max_n_index_cache * sizeof(index_info *));
    tmp_used_cache = last_used_cache;
    while(tmp_used_cache) {
      last_used_cache_array[n] = tmp_used_cache;
      tmp_used_cache = tmp_used_cache->next;
      n++;
      if(n > max_n_index_cache) {
        elog(WARNING, "pgsenna2: warning happened at index cache");
        break;
      }
    }
    funcctx->user_fctx = (void *)last_used_cache_array;
    funcctx->max_calls = n;
    MemoryContextSwitchTo(oldcontext);
  }

  funcctx = SRF_PERCALL_SETUP();

  last_used_cache_array = (index_info **) funcctx->user_fctx;
  
  call_cntr = funcctx->call_cntr;
  max_calls = funcctx->max_calls;
  attinmeta = funcctx->attinmeta;
 
  if (call_cntr < max_calls) {
    int i;
    char       **values;
    HeapTuple    tuple;
    Datum        result;
    
    values = (char **) palloc(12 * sizeof(char *));
    for (i = 0; i < 6; i++) {
      values[i] = (char *) palloc(16 * sizeof(char));
    }
    last_used_cache_array = (index_info **) funcctx->user_fctx;
    snprintf(values[0], 16, "%d", last_used_cache_array[call_cntr]->relid);
    snprintf(values[1], 16, "%d", last_used_cache_array[call_cntr]->namespace);
    snprintf(values[2], 16, "%s", last_used_cache_array[call_cntr]->name);
    snprintf(values[3], 16, "%d", (int)last_used_cache_array[call_cntr]->previous);
    snprintf(values[4], 16, "%d", (int)last_used_cache_array[call_cntr]);
    snprintf(values[5], 16, "%d", (int)last_used_cache_array[call_cntr]->next);
    tuple = BuildTupleFromCStrings(attinmeta, values);
    result = HeapTupleGetDatum(tuple);
    for (i = 0; i < 6; i++) {
      pfree(values[i]);
    }
    pfree(values);

    SRF_RETURN_NEXT(funcctx, result);
  } else {
    SRF_RETURN_DONE(funcctx);
  }
}
