/*
 *  m2m_thread.c:
 *     pthread(POSIX thread) routines.
 *
 *  Copyright (C) Taichi Nakamura <pdf30044@biglobe.ne.jp> - Feb 2000
 *
 *
 *  This file is part of m2m, a free MPEG2-Program-Stream player.
 *  It's a frontend of mpeg2dec.
 *    
 *  m2m is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  m2m is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 
 *
 */
#ifndef _LARGEFILE64_SOURCE
#define _LARGEFILE64_SOURCE
#endif
#ifndef _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 199506L
#endif
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE  500
#endif

#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <signal.h>
#include <stdint.h>
#include "m2m.h"

static m2m_buf_t *wk_buf=NULL;
static m2m_buf_t *load_buf=NULL;
pthread_mutex_t bmutex= PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t rmutex= PTHREAD_MUTEX_INITIALIZER;

inline void m2m_flush_buffer( int flg )
{
  if ( flg == 0 ) pthread_mutex_lock( &bmutex );
  if ( load_buf == NULL ) return;
  if ( wk_buf ){
    m2m_buf_t * last;
    for ( last= wk_buf ; last->n ; last=last->n ) ;
    load_buf->p = last;
    last->n = load_buf;
  }else{
    wk_buf = load_buf;
  }
  load_buf=NULL;
  if ( flg == 0 ) pthread_mutex_unlock( &bmutex );
}

inline void m2m_free_buffer( m2m_buf_t *in )
{
  pthread_mutex_lock( &bmutex );
  in->p=NULL;
  in->n=wk_buf;
  if ( wk_buf ) wk_buf->p=in;
  wk_buf=in;
  pthread_mutex_unlock( &bmutex );
}

m2m_buf_t * m2m_new_buffer(int flg )
{
    m2m_buf_t * ret ;
    if ( flg == 0 ) pthread_mutex_lock( &bmutex );
    ret = wk_buf;
    if ( ret ){
        wk_buf=wk_buf->n;
        if ( wk_buf ) wk_buf->p=NULL;
        ret->p=NULL;
        ret->n=NULL;
        if ( flg == 0 ) pthread_mutex_unlock( &bmutex );
    }else{
        if ( flg == 0 ) pthread_mutex_unlock( &bmutex );
        ret = (m2m_buf_t *)malloc( sizeof(m2m_buf_t) );
        memset( ret , 0x0, sizeof(m2m_buf_t) );
    }
    return ret;
}

static inline m2m_buf_t * m2m_check_buffer( int is_video )
{
  m2m_buf_t *ret=NULL;
  for ( ret = load_buf ; ret ; ret=ret->n){
    if ( is_video ){
        if ( ret->type == 0x0e0 ) break;
    }else{
        if ( ret->type == 0x0c0  
          || ret->type == 0x080   
          || ret->type == 0x0a0 ) break;
    }
  }
  if ( ret ){
    if ( ret == load_buf ) load_buf=load_buf->n;
    if ( ret->p ) ret->p->n = ret->n ;
    if ( ret->n ) ret->n->p = ret->p ;
    ret->p = NULL;
    ret->n = NULL;
  }
  return ret;
}

static inline void m2m_add_last_buffer( m2m_buf_t * in )
{
  if ( load_buf ){
    m2m_buf_t * last;
    for ( last= load_buf ; last->n ; last=last->n ) ;
    in->p = last;
    in->n = NULL;
    last->n = in;
  }else{
    in->p = NULL;
    in->n = NULL;
    load_buf= in;
  }
}

static inline m2m_buf_t * m2m_read_buffer( int flg )
{
    extern int read_sector(uint8_t []);
    extern void DE_CSS(uint8_t []);
    uint32_t pes_ts;
    uint8_t *pes;
    int bytes_read,hdrlen,peslen ;
    m2m_buf_t * ret = m2m_new_buffer( flg );
    uint8_t * _local_buf=ret->buf;

    /* read for checking */
    for( bytes_read= 0 ; bytes_read == 0 ; bytes_read = 0 )
    {
        bytes_read= read_sector(_local_buf);
        if( bytes_read == 0 )
        {
return NULL;
/*
            if ( m2m_inc_index() == 0 && ! IS_LOOP )
            {
                pthread_mutex_lock( &play_stat_mutex );
                play_status = STATUS_STOP ;
                while( play_status == STATUS_STOP )
                    pthread_cond_wait( &play_stat_cond,&play_stat_mutex );
                pthread_mutex_unlock( &play_stat_mutex );
                pthread_cond_signal( &play_stat_cond );
            }
            continue;
*/
        }
        if( bytes_read != READ_BLOCK_SIZE )
            kill(getpid(),SIGINT);
        if ( (_local_buf[4]&0xf0)==0x40 ){ //mpeg2
            pes = _local_buf + 14 + (_local_buf[13] & 7) ;
            if ( pes[0] != 0x00 || pes[1] != 0x00 || pes[2] != 0x01 )
            {
                PACK_DUMP(_local_buf,"not PES(2):");
                continue;
            }
            if ( pes[3] == 0x0bb )
            {
                pes += ( pes[4] << 8 ) + pes[5] + 6 ;
                if ( pes[0] != 0x00 || pes[1] != 0x00 || pes[2] != 0x01 )
                {
                        PACK_DUMP(_local_buf,"not PES(3):");
                        continue;
                }
            }
            peslen = ( pes[4] << 8 ) + pes[5] +6 ;
            hdrlen = pes[8] + 6 + 3 ;
            DE_CSS( _local_buf ) ;
            pes_ts= ( pes[7] & 0x040 )?(GET_DTS(_local_buf)):
                    ( pes[7] & 0x080 )?(GET_PTS(_local_buf)):0;
        }else
        if ( (_local_buf[4]&0xf0)==0x20 ){ //mpeg1
            pes = _local_buf + 12;
            if ( pes[0] != 0x00 || pes[1] != 0x00 || pes[2] != 0x01 )
            {
                PACK_DUMP(_local_buf,"not PES(2):");
                continue;
            }
            peslen = 2048-12;
            if ( pes[3] == 0x0bb )
            {
                pes += ( pes[4] << 8 ) + pes[5] + 6 ;
                if ( pes[0] != 0x00 || pes[1] != 0x00 || pes[2] != 0x01 )
                {
                        PACK_DUMP(_local_buf,"not PES(3):");
                        continue;
                }
            }

            peslen = ( pes[4] << 8 ) + pes[5] +6 ;
            hdrlen = 6;
            while( pes[hdrlen] == 0xff ) hdrlen++;
            if ( (pes[hdrlen]&0xc0) == 0x40 ) hdrlen+=2;
            switch( pes[hdrlen] >>4 ){
            case 0: hdrlen++; break;
            case 2:
               pes_ts=(pes[hdrlen+1]<<24l)
                     |(pes[hdrlen+2]<<16l)
                     |(pes[hdrlen+3]<<8l)
                     |(pes[hdrlen+4]);
               hdrlen+=5;
               break;
            case 3:
               pes_ts=(pes[hdrlen+6]<<24l)
                     |(pes[hdrlen+7]<<16l)
                     |(pes[hdrlen+8]<<8l)
                     |(pes[hdrlen+9]);
               hdrlen+=10;
               break;
            }
            if ( hdrlen > peslen || peslen-hdrlen > 2048 ){
                fprintf( stderr,"%d %d:",hdrlen,peslen );
                PACK_DUMP((pes+hdrlen),"mp1-1:bad");
                exit(2);
            }
        }else{
            fprintf( stderr,"%d %d:",hdrlen,peslen );
            PACK_DUMP((pes+hdrlen),"unknown mpeg type");
            exit(3);
        }

        pthread_mutex_lock( &play_stat_mutex );
        if( play_status == STATUS_TO_SEEK )
        {
            extern uint64_t to_seek_pos;
            if( do_seek( pes[3]&0x0f0,pes[hdrlen]&0x0f0,to_seek_pos,flg ) )
            {
                pthread_mutex_unlock( &play_stat_mutex );
                pthread_cond_signal( &play_stat_cond );
                continue;
            }
        }
        pthread_mutex_unlock( &play_stat_mutex );
        pthread_cond_signal( &play_stat_cond );

        ret->type= pes[3]&0x0f0 ;
        if ( ret->type == 0x0e0    /* video */
          || ret->type == 0x0c0 )  /* mpeg Audio */
        {
            ret->ch  = pes[3]&0x0f ;
        }
        else
        if ( pes[3] == 0xbd    /* External Audio */
          && (pes[hdrlen]&0x0f0) == 0x080 ) /* AC3-Track */
        {
            ret->type= 0x080;
            ret->ch  = pes[hdrlen] & 0x0f;
            hdrlen += 4;
        }
        else
        if ( (pes[3] == 0xbd || pes[3] == 0xbf )    /* is External Audio */
         && (pes[hdrlen]&0x0f0) == 0x0a0 )
        {
            ret->type= 0x0a0;
            ret->ch  = pes[hdrlen] & 0x0f;
            while( ++hdrlen < peslen-1 ){
                if( pes[hdrlen] == 0x01 && pes[hdrlen+1] == 0x80 ){
                    hdrlen+=2;
                    break;
                }
            }
        }
        else
        {
            continue;
        }
        ret->p=NULL;
        ret->n=NULL;
        ret->pos=pes+hdrlen;
        ret->len=peslen-hdrlen;
        ret->ts=pes_ts;
        break;
    }
    return ret;
}

inline m2m_buf_t * m2m_next_buffer( int is_video )
{
  static int flg=0;
  int cnt;
  m2m_buf_t * ret;
  m2m_buf_t * last;

  pthread_mutex_lock( &bmutex );
  ret=m2m_check_buffer( is_video );
  if ( ret == NULL && flg ){
    pthread_mutex_unlock( &bmutex );
    while( flg ){
        sched_yield();
        //usleep( 1000000/600 );
        continue;
    }
    pthread_mutex_lock( &bmutex );
    ret=m2m_check_buffer( is_video );
  }
  if ( ret == NULL ){
    for( ;; ){
#define BCNT 200
      cnt=BCNT;
      for ( last= load_buf ; last && cnt ; last=last->n ) cnt--;
      flg=1;
/*
      if ( cnt < 50 || (is_video && load_buf && load_buf->type != 0xe0 )){
        pthread_mutex_unlock( &bmutex );
        //flg=0;
        sched_yield();
        pthread_mutex_lock( &bmutex );
        //flg=1;
      }
*/
      if ( cnt ==0 ) pthread_mutex_unlock( &bmutex );
//printf("%d %02x %d \r",is_video,load_buf?load_buf->type:9,BCNT-cnt );
      ret = m2m_read_buffer( cnt );
      if ( cnt ==0 ) pthread_mutex_lock( &bmutex );
      flg=0;
if ( ret == NULL ) break;
      if ( is_video ){
        if ( ret->type == 0x0e0 ) break;
      }else{
        if ( ret->type == 0x0c0  
          || ret->type == 0x080   
          || ret->type == 0x0a0 ) break;
      }
      m2m_add_last_buffer( ret );
    }
  }
  pthread_mutex_unlock( &bmutex );
  return ret;
}
