/*
 *  m2m_caster.c:
 *     pthread(POSIX thread) routines.
 *
 *  Copyright (C) Taichi Nakamura <pdf30044@biglobe.ne.jp> - Feb 2000
 *
 *
 *  This file is part of m2m, a free MPEG2-Program-Stream player.
 *  It's a frontend of mpeg2dec.
 *    
 *  m2m is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  m2m is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 
 *
 */
#ifndef _LARGEFILE64_SOURCE
#define _LARGEFILE64_SOURCE
#endif
#ifndef _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 199506L
#endif
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE  500
#endif
#ifndef _BSD_SOURCE
#define _BSD_SOURCE
#endif

#include <features.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <time.h>
#include <signal.h>
#include <pthread.h>
#include <string.h>
#include <stdint.h>
#include <ifo.h>

#include "m2m.h"
#include "m2m_descripter.h"
#include "m2m_ifo.h"
#include "m2m_playlist.h"
#include "m2m_command.h"

static int sd=-1;
static int pipe_in=-1;
static int pipe_out=-1;
m2m_listener_handle_t *m2m_listeners=NULL;

static inline int recv_hello(m2m_listener_handle_t *listener )
{
  m2m_listener_handle_t *target;
  void * id;
  int cmd;
  int iprm;
  char *sprm=NULL;
  char buf[2048];
  int status = m2m_attach_command (listener->desc[0],listener->desc[1],&id,&cmd ,&iprm,&sprm );
  if ( status < 0 ){
    fprintf(stderr,"recv_hello:m2m_attach_command err.%d",status );
    if ( sprm ) free(sprm);
    return status;
  }
  switch ( cmd ){
  case M2M_CMD_HELLO_NEW_ID:
    sprintf(buf,"OK %d\r\n",listener);
    m2m_sendline(listener->desc[0],listener->desc[1],buf);
    m2m_sendline(listener->desc[0],listener->desc[1],"END\r\n");
    if ( sprm ) free(sprm);
    return cmd;
  case M2M_CMD_ADM_GET_LINSTENER_LIST:
  case M2M_CMD_GET_LINSTENER_LIST:
    for(target=m2m_listeners;target;target=target->n){
      if ( cmd == M2M_CMD_GET_LINSTENER_LIST &&
           listener->addr.sin_addr.s_addr != target->addr.sin_addr.s_addr ){
        continue;
      }
      sprintf(buf,"%ld\r\n",target);
      m2m_sendline(listener->desc[0],listener->desc[1],buf);
    }
    m2m_sendline(listener->desc[0],listener->desc[1],"END\r\n");
    if ( sprm ) free(sprm);
    return cmd;
  }
  for(target=m2m_listeners;target && target != id;target=target->n);
  if ( target == NULL ){
    fprintf(stderr,"recv_hello:ignore target. %d",id );
    return -1;
  }
  switch ( cmd ){
  case M2M_CMD_GET_PLAYLIST:
    {
      m2m_playlist_entry_t *cur;
      for(cur=target->playlist;cur;cur=cur->next){
        sprintf(buf,"%s\r\n",cur->name);
        m2m_sendline(listener->desc[0],listener->desc[1],buf);
      }
    }
    break;
  case M2M_CMD_JUMP_TITLE:
    {
      m2m_playlist_entry_t *cur=NULL;
      if ( sprm == NULL ) break;
      for(cur=target->playlist;cur;cur=cur->next){
          if ( strcmp(cur->name,sprm) == 0 ) break;
      }
      if ( cur == NULL &&
         ( sprm[0] == '-' || sprm[0] == '+' ||
          (sprm[0] >= '0' && sprm[0] <= '9')) ){
        int idx;
        for(idx=0,cur=target->playlist;cur && cur != target->cur;idx++,cur=cur->next);
        if ( cur == NULL ) idx=0;
        if ( sprm[0] == '-' || sprm[0] == '+' ) idx += atol(sprm);
        else idx=atol(sprm);
        for(cur=target->playlist;cur && idx>0;idx--,cur=cur->next);
        if ( idx < 0 ) cur=NULL;
      }
      if ( cur ){
        target->cur=cur;
        target->play_status=2;
      }
    }
    break;
  case M2M_CMD_START:
    target->play_status=1;
    break;
  case M2M_CMD_PAUSE:
    target->play_status=0;
    break;
  case M2M_CMD_ATTACH_DVD:
    m2m_add_playlist_from_ifo( target, sprm );
    break;
  case M2M_CMD_GET_CURRENT_INFO:
    if ( target->cur->descripter->type != 1 ){
      m2m_filedesc_t *fd=(m2m_filedesc_t *)(target->cur->descripter);
      sprintf(buf,"%ld  %ld %ld %ld %ld %ld %ld\r\n",target->play_status,
           (uint32_t)(fd->desc.st>>32),
           (uint32_t)(fd->desc.st&0x0ffffffff ),
           (uint32_t)(fd->desc.ed>>32),
           (uint32_t)(fd->desc.ed&0x0ffffffff ),
           (uint32_t)(fd->pos>>32),
           (uint32_t)(fd->pos&0x0ffffffff )
      );
    }else{
      m2m_dvddesc_t *dd=(m2m_dvddesc_t *)(target->cur->descripter);
      sprintf(buf,"%ld  %ld %ld %ld %ld %ld %ld\r\n",target->play_status,
           (uint32_t)(dd->desc.st>>32),
           (uint32_t)(dd->desc.st&0x0ffffffff ),
           (uint32_t)(dd->desc.ed>>32),
           (uint32_t)(dd->desc.ed&0x0ffffffff ),
           (uint32_t)(dd->ifo->pos>>32),
           (uint32_t)(dd->ifo->pos&0x0ffffffff )
      );
    }
    m2m_sendline(listener->desc[0],listener->desc[1],buf);
    break;
  }
  m2m_sendline(listener->desc[0],listener->desc[1],"END\r\n");
  if ( sprm ) free(sprm);
  return cmd;

}

static inline int m2m_init_caster( char ip_addr[] , unsigned short port )
{
  struct sockaddr_in addr;
  int status;

  sd = socket(PF_INET, SOCK_STREAM , 0);
  if ( sd <= -1 ){ perror("socket"); return -1; }
  memset(&addr, 0, sizeof(addr));
  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr=INADDR_ANY;
  addr.sin_port = htons(port);
  status = bind(sd, (struct sockaddr *) &addr, sizeof(addr));
  if ( status <= -1 ){ perror("bind"); return -2; }
  status = listen( sd,2 );
  if ( status <= -1 ){ perror("ilisten"); return -3; }
  return 0;
}

static inline m2m_listener_handle_t * m2m_accept( void )
{
  int len;
  m2m_listener_handle_t *ret;

  ret=(m2m_listener_handle_t *)malloc( sizeof(m2m_listener_handle_t) );
  if ( ret == NULL ) return NULL;
  memset( ret,0x0,sizeof(m2m_listener_handle_t) );

  memset( &(ret->addr), 0x0, sizeof(ret->addr) );
  len=sizeof(ret->addr);
  ret->desc[0] = ret->desc[1] = accept( sd,(struct sockaddr *)&(ret->addr),&len );
  if ( ret->desc[0] <= -1 ){ perror("accept"); return NULL; }
  if( recv_hello(ret) != 0 ){
    close(ret->desc[0]);
    free(ret);
    return NULL;
  }
  ret->p=ret->n=NULL;
  return ret;

/*
  sd = socket(PF_INET, SOCK_STREAM , 0);
  if ( sd  == -1 ){ perror("socket"); return -1; }
  memset(&addr, 0, sizeof(addr));
  addr.sin_port = htons(port);
  addr.sin_family = AF_INET;
  inet_pton( AF_INET,ip_addr,&addr.sin_addr );
  status = connect( sd, (struct sockaddr *)&addr, sizeof(addr));
  if ( status == -1 ){ perror("conect" ); return -5; }
  return 0;
*/
}

static inline int m2m_write_sector_sock(int fd,char in_buf[] , int len )
{
  //return send(fd, in_buf, len, 0);
  return write(fd, in_buf, len);
}

static inline void m2m_cast( m2m_listener_handle_t *listener )
{
  struct timeval st;
  struct timeval ed;
  int fd=listener->desc[1];
  int i=0;
  gettimeofday( &st,NULL );
  while (1)
  {
    m2m_playlist_entry_t * cur;
    if ( listener->playlist == NULL ){
      m2m_add_playlist_from_properties(listener);
    }
    m2m_get_playlist_length(listener);
    for( listener->cur=listener->playlist ; listener->cur; )
    {
      int (*read_func)( struct m2m_desctipter_s *,uint8_t b[],int * );
      struct m2m_desctipter_s *descripter;
      uint8_t buf[2048];
      int len=sizeof(buf);
      while ( listener->play_status == 0 ){
        usleep(10000);
        sched_yield();
      }
      descripter = listener->cur->descripter;
      descripter->seek( descripter , descripter->st );
      read_func=descripter->read;
      while( len == sizeof(buf) )
      {
        read_func( descripter , buf,&len );
        if ( m2m_write_sector_sock(fd,buf, len) < 0 ) return;
        if ( ++i > 250 ){
          while ( listener->play_status == 0 ){
            usleep(10000);
            sched_yield();
          }
          if ( listener->play_status == 2 ) break;
/*
          gettimeofday( &ed,NULL );
          fprintf( stderr,"send:%8.1lf KByte/sec\r", i*2/(ed.tv_sec-st.tv_sec+(ed.tv_usec-st.tv_usec)/1000000.0));
          st=ed;
*/
          i=0;
        }
      }
      memset( buf,0xff,sizeof(buf) );
      buf[0]=buf[1]=0x0; buf[2]=0x01; buf[3]=0x0f2; buf[4]=0x08; buf[5]=0x0;
      if ( m2m_write_sector_sock(fd,buf,2048) < 0 ) return;
      if ( listener->play_status == 2 ) listener->play_status = 1 ;
      else listener->cur=listener->cur->next;
    }
  }
}

static inline void m2m_cast_close(){
  close(sd);
  sd=-1;
}

static void m2m_sighanhler( int signum )
{
fprintf( stderr,"catched signal %d\n",signum );
  m2m_cast_close();
  exit(1);
}

static inline int handle_args( int argc,char *argv[] )
{
  int i;
  int show_prop=0;
  int help=0;
  int isPipe=0;
  char * prop=NULL;

  for ( i=1 ; i< argc ; i++ )
  {
      if( argv[i][0] != '-' ) continue;
      if( argv[i][1] == 0x0 && i+2 < argc )
      {
          pipe_in=atol(argv[i+1]);
          pipe_out=atol(argv[i+2]);
          i+=2;
          isPipe=1;
      }else
      if ( strcmp(argv[i],"--conf") == 0 && i+1 < argc && argv[i+1][0] != '-' )
      {
          i++;
          prop=argv[i];
      }else
      if ( strcmp(argv[i],"--showconf") == 0 )
      {
          show_prop=1;
      }else
      if ( strcmp(argv[i],"--help") == 0 || strcmp(argv[i],"-h") == 0 )
      {
          help=1;
      }
  }
  m2m_load_properties( prop );
  if ( show_prop ) m2m_write_properties( 1 );
  if ( help ) {
    printf( "Usage:%s options... \n",argv[0] );
    printf( "options:\n" );
    printf( "  --conf config_file        : chose a config.\n" );
    printf( "  --showconf                : show config.\n" );
    exit(0);
  }
  return isPipe;
}

static void * m2m_new_caster( void *_listener )
{
  m2m_listener_handle_t *listener=(m2m_listener_handle_t *)_listener;
  m2m_cast( listener );
  if ( listener->p ) listener->p->n = listener->n;
  if ( listener->n ) listener->n->p = listener->p;
  if ( listener == m2m_listeners ) m2m_listeners=listener->n;
  close( listener->desc[0]);
  if ( listener->desc[0] != listener->desc[1] ){
    close( listener->desc[1]);
  }
  while( listener->playlist )
  {
    m2m_remove_list(listener,listener->playlist->name);
  }
  free( listener );
fprintf( stderr,"leave listener:\n" );
}

int main( int argc,char *argv[] )
{
  if ( handle_args(argc,argv ) )
  {
    m2m_listener_handle_t listener;
    memset( &listener,0x0,sizeof(m2m_listener_handle_t) );
    listener.desc[0]=pipe_in;
    listener.desc[1]=pipe_out;
    if ( m2m_listeners ){
      (m2m_listeners->p=&listener)->n=m2m_listeners;
    }
    m2m_listeners=&listener;
    if( recv_hello( &listener) != 0 ) return -1;
    while( recv_hello( &listener) != M2M_CMD_START ) ;
    m2m_cast( &listener);
    return 0;
  }else
  if ( m2m_init_caster( m2m_get_property("CAST_IP") ,
                        atoi(m2m_get_property("CAST_PORT"))) )
  {
    perror("init:");
    return -1;
  }

  signal( SIGPIPE,SIG_IGN );
  signal( SIGINT,m2m_sighanhler );
  while(1){
     m2m_listener_handle_t *listener;
     int fd;
     pthread_t thr;
     pthread_attr_t attr;
     pthread_attr_init( &attr );
     pthread_attr_setscope( &attr , PTHREAD_SCOPE_SYSTEM );
     listener= m2m_accept( );
     if ( listener == NULL ) continue;
     if ( m2m_listeners ){
       (m2m_listeners->p=listener)->n=m2m_listeners;
     }
     m2m_listeners=listener;
    while( 1){
        int cmd=recv_hello( listener);
        if ( cmd == M2M_CMD_START || cmd == M2M_CMD_PAUSE ) break;
    }
fprintf( stderr,"new listener:%s\n",inet_ntoa(listener->addr.sin_addr) );
     pthread_create( &thr , &attr  , m2m_new_caster , (void *)listener );
  }
  return 0;
}

