/*
 *  m2m_main.c:
 *     main and GUI.
 *
 *  Copyright (C) Taichi Nakamura <pdf30044@biglobe.ne.jp> - Mar 2000
 *
 *
 *  This file is part of m2m, a free MPEG2-Program-Stream player.
 *  It's a frontend of mpeg2dec.
 *    
 *  m2m is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  m2m is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 
 *
 */

#ifndef _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 199506L
#endif
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE  500
#endif

#include <pthread.h>
#include <sys/time.h>
#include <sys/soundcard.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <locale.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtk/gtkenums.h>
#include "pics/menu.xpm"
#include "pics/play.xpm"
#include "pics/step.xpm"
#include "pics/rew.xpm"
#include "pics/prev.xpm"
#include "pics/next.xpm"
#include "pics/loop.xpm"
#include "pics/no_loop.xpm"
#include "pics/window.xpm"
#include "config.h"
#include "m2m.h"
#include "video_out/video_out.h"
#include <mpeg2dec/mpeg2.h>
#include <mpeg2dec/mm_accel.h>
#include <a52dec/a52.h>
#include <a52dec/audio_out.h>

#define FABS(X) ((X)>=0.0)?(X):-(X)
#define TOL 10.0

play_status_t play_status=STATUS_STOP;
int m2m_brightness =135;
int m2m_contrast =127;
int m2m_ad_sync =100;
int m2m_ad_late =500;
int m2m_ad_skip =50;
static float last_adjust_value=0.0;
static GtkAdjustment *adjust_time=NULL;
static GtkAdjustment *adjust_brightness=NULL;
static GtkAdjustment *adjust_contrast=NULL;
static GtkAdjustment *adjust_audio_no=NULL;
static GtkAdjustment *adjust_ad_sync=NULL;
static GtkAdjustment *adjust_ad_late=NULL;
static GtkAdjustment *adjust_ad_skip=NULL;
#ifdef M2M_OLD_PULLDOWN
static GtkOptionMenu *pg_pulldown=NULL;
#else
static GtkLabel *pg_echo=NULL;
#endif
static GtkWidget *handles[5];
char **file_name=NULL;
char *device_name=NULL;
static char * default_device_name="/dev/dvd";
int title_no=1;
int audio_no=1;
int audio_rate=0;
#ifdef HAVE_ESOUND
char *esd_host=NULL;
#endif
uint32_t commandline_params=0;
mpeg2dec_t mpeg2dec;
char *vo_driver=NULL;
char *ao_driver=NULL;
vo_instance_t * vo_instance=NULL;

pthread_mutex_t play_stat_mutex=PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t play_stat_cond=PTHREAD_COND_INITIALIZER;


/* public function */
void sync_adjust_value(float v)
{
    if( adjust_time )
        gtk_adjustment_set_value( adjust_time, last_adjust_value=v );
}
void set_brightness(int v)
{
    if( adjust_brightness )
        gtk_adjustment_set_value( adjust_brightness, (float)(m2m_brightness=v) );
}
void set_contrast(int v)
{
    if( adjust_contrast )
        gtk_adjustment_set_value( adjust_contrast, (float)(m2m_contrast=v) );
}

void chapter_echo( int chapter_no , int number_of_chapter )
{
    char str[32];
#ifdef M2M_OLD_PULLDOWN
    GtkWidget *menu=NULL;
    while ( pg_pulldown == NULL ) sched_yield();
    menu=gtk_option_menu_get_menu( pg_pulldown ) ;
    sprintf( str,"%02d/%02d ",chapter_no,number_of_chapter );
    gtk_label_set_text( GTK_LABEL(pg_pulldown->button.child),str );
    gtk_menu_set_active( GTK_MENU(menu) , chapter_no-1 );
#else
    sprintf( str,"%02d/%02d ",chapter_no,number_of_chapter );
    gtk_label_set_text( pg_echo,str );
#endif

}

/* private functions */
static void destroy( GtkWidget *widget, gpointer   data )
{
    gtk_main_quit();
}

static gint delete_event( GtkWidget *widget, GdkEvent *event, gpointer data )
{
    destroy( widget, data );
    return TRUE ;
}

static void loop_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    static GtkWidget *loop_l=NULL;
    GdkBitmap *mask;
    GdkPixmap *pixmap;
    GDK_THREADS_ENTER();
    if ( loop_l ) gtk_container_remove( GTK_CONTAINER(widget),loop_l );
    loop_l=NULL;
    if ( IS_LOOP )
    {
        RESET_COMMANDLINE_PARAMS( IS_LOOP );
    }
    else
    {
        SET_COMMANDLINE_PARAMS( IS_LOOP );
    }
    if ( IS_LOOP )
    {
        pixmap=gdk_pixmap_create_from_xpm_d( widget->window,
                                             &mask,NULL,loop_xpm );
        loop_l=gtk_pixmap_new( pixmap , mask );
        gtk_widget_show( loop_l );
        gtk_container_add( GTK_CONTAINER(widget),loop_l );
    }
    else
    {
        pixmap=gdk_pixmap_create_from_xpm_d( widget->window,
                                             &mask,NULL,no_loop_xpm );
        loop_l=gtk_pixmap_new( pixmap , mask );
        gtk_widget_show( loop_l );
        gtk_container_add( GTK_CONTAINER(widget),loop_l );
    }
    gtk_widget_show( widget );
    GDK_THREADS_LEAVE();
}

static void window_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    vo_instance->next_window_mode(vo_instance);
}

static void tool_cb( GtkWidget *widget, int no )
{
    static char *show_names[]={
            "menu bar" ,
            "Show controller buttons" ,
            "Show program select buttons" ,
            "Show indicator" ,
            "Show advanced controlls"
    };
    static char *hide_names[]={
            "menu bar" ,
            "Hide controller buttons" ,
            "Hide program select buttons" ,
            "Hide indicator" ,
            "Hide advanced controlls"
    };
    static GtkWidget *labels[]={NULL,NULL,NULL,NULL,NULL};
    static int show_flag[]={1,1,1,1,0};
    if ( labels[no] ) gtk_container_remove( GTK_CONTAINER(widget),labels[no] );
    labels[no]=NULL;
    labels[no]=gtk_label_new( show_flag[no]?show_names[no]:hide_names[no] );
    gtk_widget_show( labels[no] );
    gtk_container_add( GTK_CONTAINER(widget),labels[no] );
    show_flag[no]=1-show_flag[no];
    if ( show_flag[no] ) gtk_widget_show( handles[no] );
    else                 gtk_widget_hide( handles[no] );
}
static void tool1_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    tool_cb( widget, 1 );
}
static void tool2_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    tool_cb( widget, 2 );
}
static void tool3_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    tool_cb( widget, 3 );
}
static void tool4_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    tool_cb( widget, 4 );
}

static void map_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    char **xpm[]={ menu_xpm,play_xpm,step_xpm,prev_xpm,rew_xpm,next_xpm,loop_xpm,window_xpm,NULL };
    char *label[]={ NULL,NULL   , NULL   , NULL   , NULL  , NULL   ,NULL,NULL };
    int n=1+(int)data;
    GtkWidget *inner_widget;
    if ( n<0 )
    {
        tool_cb( widget, -n );
        tool_cb( widget, -n );
        return;
    }
    if ( xpm[n] == loop_xpm )
    {
        loop_cb( widget, event, data );
        loop_cb( widget, event, data );
        return;
    }
    if ( xpm[n] )
    {
        GdkBitmap *mask;
        GdkPixmap *pixmap;
        pixmap=gdk_pixmap_create_from_xpm_d(widget->window,&mask,NULL,xpm[n]);
        inner_widget=gtk_pixmap_new( pixmap , mask );
    }
    else
    {
        inner_widget=gtk_label_new( label[n] );
    }
    gtk_widget_show( inner_widget );
    gtk_container_add( GTK_CONTAINER(widget),inner_widget );
    gtk_widget_show( widget );
}

static void menu_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
	change_to_menu_program();
}

static void b1_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    pthread_mutex_lock( &play_stat_mutex );
    switch( play_status )
    {
        case STATUS_STOP:
            play_status = STATUS_PLAY;
            break;
        case STATUS_PLAY:
            play_status = STATUS_STOP;
            break;
        case STATUS_TO_SEEK:
            break;
    }
    pthread_mutex_unlock( &play_stat_mutex );
    pthread_cond_signal( &play_stat_cond );
}

static void b2_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    if ( play_status != STATUS_PLAY )
    {
        uint32_t ats=ts_a;
        play_status = STATUS_PLAY ;
	pthread_cond_signal( &play_stat_cond );
        while( ts_a == ats )
            sched_yield();
        play_status = STATUS_STOP ;
	pthread_cond_signal( &play_stat_cond );
    }
}

static void b3_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    last_adjust_value=change_program_by_offset( -1 );
}

static void b4_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    last_adjust_value=change_program_by_offset( 0 );
}

static void b5_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    last_adjust_value=change_program_by_offset( 1 );
}

static void pgpd_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    int i = (int)gtk_object_get_data( GTK_OBJECT (widget), "no" );
    last_adjust_value=change_program_by_index( i );
}
static void pgbtn_cb( GtkWidget *widget, GdkEvent *event, gpointer   data )
{
    int i ,idx;
    GList *child = gtk_container_children( GTK_CONTAINER(widget->parent) );
    for ( i=0,idx=-1 ; child ; i++ ) {
        if ( child->data == widget ) idx=i;
        child = g_list_remove_link( child,child );
    }
    if ( idx >= 0 )
        last_adjust_value=change_program_by_index( idx );
}

static void an_cb( GtkObject *obj, gpointer   data )
{
    audio_no = ( int ) adjust_audio_no->value ;
}
static void br_cb( GtkObject *obj, gpointer   data )
{
   m2m_brightness = ( int ) adjust_brightness->value ;
}
static void cn_cb( GtkObject *obj, gpointer   data )
{
    m2m_contrast = ( int ) adjust_contrast->value ;
}
static void ad_sync_cb( GtkObject *obj, gpointer   data )
{
    m2m_ad_sync = ( int ) adjust_ad_sync->value ;
}
static void ad_late_cb( GtkObject *obj, gpointer   data )
{
    m2m_ad_late = ( int ) adjust_ad_late->value ;
}
static void ad_skip_cb( GtkObject *obj, gpointer   data )
{
    m2m_ad_skip = ( int ) adjust_ad_skip->value ;
}

static void sc_cb( GtkObject *obj, gpointer   data )
{
    if ( play_status == STATUS_TO_SEEK )
        return;
    if ( FABS(last_adjust_value - adjust_time->value) > TOL )
        last_adjust_value=change_program_by_lba( (long)adjust_time->value );
    else
        last_adjust_value = adjust_time->value;
}

static int adjustor_sync( void *arg )
{
    sync_adjust_value(get_current_value());
    return TRUE;
}

static int panel_create( int argc, char *argv[] )
{
    int i;
    long ed;
    GtkWidget *window;
    GtkWidget *form1;
    GtkWidget *form2;
    GtkWidget *form3;
    GtkWidget *spin;
    GtkWidget *scale;
    GtkWidget *button;
    GtkWidget *menubar;
    GtkWidget *submenu;
    GtkWidget *item;
    GtkTooltips *tip;
    GtkAccelGroup *accel;
    ed=get_total_vob_length();

fprintf(stderr,"ed=%ld\n",ed );
    gtk_set_locale();
    gtk_init(&argc, &argv);
    gtk_rc_parse("~/.gtkrc");
    tip = gtk_tooltips_new();
    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

    accel = gtk_accel_group_new();
    gtk_window_add_accel_group( GTK_WINDOW(window),accel );

    //gtk_widget_set_usize( GTK_WIDGET ( window ) , 500 , 60 );
    gtk_signal_connect( GTK_OBJECT (window), "delete_event",
                        GTK_SIGNAL_FUNC (delete_event), NULL);
    gtk_signal_connect( GTK_OBJECT (window), "destroy_event",
                        GTK_SIGNAL_FUNC (destroy), NULL);
    gtk_signal_connect( GTK_OBJECT (window), "destroy",
                        GTK_SIGNAL_FUNC (destroy), NULL);
    gtk_container_set_border_width (GTK_CONTAINER (window), 0);
    handles[1]=gtk_handle_box_new();
    form1=gtk_vbox_new(0,0);
    form2=gtk_hbox_new(0,0);
    gtk_container_set_border_width (GTK_CONTAINER (form1), 0);
    gtk_container_set_border_width (GTK_CONTAINER (form2), 4);
    gtk_container_add (GTK_CONTAINER (window), form1);

    adjust_audio_no = (GtkAdjustment *)gtk_adjustment_new(1,
                                                 1, 9, 1, 1, 1 );
    adjust_brightness = (GtkAdjustment *)gtk_adjustment_new( m2m_brightness,
                                                 0, 255, 1, 5, 10 );
    adjust_contrast = (GtkAdjustment *)gtk_adjustment_new( m2m_contrast,
                                                 0, 256, 1, 5, 10 );
    adjust_ad_sync = (GtkAdjustment *)gtk_adjustment_new(m2m_ad_sync,
                                                 0, 999, 1, 5, 10 );
    adjust_ad_late = (GtkAdjustment *)gtk_adjustment_new(m2m_ad_late,
                                                 0, 999, 1, 5, 10 );
    adjust_ad_skip = (GtkAdjustment *)gtk_adjustment_new( m2m_ad_skip,
                                                 0, 999, 1, 5, 10 );
    gtk_signal_connect( GTK_OBJECT (adjust_audio_no), "value_changed",
                        GTK_SIGNAL_FUNC (an_cb), NULL);
    gtk_signal_connect( GTK_OBJECT (adjust_brightness), "value_changed",
                        GTK_SIGNAL_FUNC (br_cb), NULL);
    gtk_signal_connect( GTK_OBJECT (adjust_contrast), "value_changed",
                        GTK_SIGNAL_FUNC (cn_cb), NULL);
    gtk_signal_connect( GTK_OBJECT (adjust_ad_sync), "value_changed",
                        GTK_SIGNAL_FUNC (ad_sync_cb), NULL);
    gtk_signal_connect( GTK_OBJECT (adjust_ad_late), "value_changed",
                        GTK_SIGNAL_FUNC (ad_late_cb), NULL);
    gtk_signal_connect( GTK_OBJECT (adjust_ad_skip), "value_changed",
                        GTK_SIGNAL_FUNC (ad_skip_cb), NULL);
    adjust_time = (GtkAdjustment *)gtk_adjustment_new(0, 0, ed, 1, 5, 10);
    gtk_signal_connect( GTK_OBJECT (adjust_time), "value_changed",
                        GTK_SIGNAL_FUNC (sc_cb), NULL);
    if ( device_name && ! DISABLE_MENU )
    {
        button = gtk_button_new();
        gtk_signal_connect( GTK_OBJECT (button), "map_event",
                            GTK_SIGNAL_FUNC (map_cb), (gpointer)-1 );
        gtk_signal_connect( GTK_OBJECT (button), "clicked",
                            GTK_SIGNAL_FUNC (menu_cb), NULL );
        gtk_box_pack_start( GTK_BOX (form2), button,FALSE,FALSE,0 );
        gtk_widget_show ( button );
        gtk_tooltips_set_tip(tip,button,_("menu"),"");
    }
    for( i=0 ; ; i++ )
    {
        void (*cb[])(GtkWidget*,GdkEvent*,gpointer)=
                 {b1_cb,b2_cb,b3_cb,b4_cb,b5_cb,loop_cb,window_cb,NULL};
        if ( cb[i] == NULL ) break;
        button = gtk_button_new();
        gtk_signal_connect( GTK_OBJECT (button), "map_event",
                            GTK_SIGNAL_FUNC (map_cb), (gpointer)i );
        gtk_signal_connect( GTK_OBJECT (button), "clicked",
                            GTK_SIGNAL_FUNC (cb[i]), NULL );
        gtk_box_pack_start( GTK_BOX (form2), button,FALSE,FALSE,0 );
        gtk_widget_show ( button );
        switch(i)
        {
            case 0: gtk_tooltips_set_tip(tip,button,_("play/stop"),""); break;
            case 1: gtk_tooltips_set_tip(tip,button,_("step")     ,""); break;
            case 2: gtk_tooltips_set_tip(tip,button,_("prev")     ,""); break;
            case 3: gtk_tooltips_set_tip(tip,button,_("rewind")   ,""); break;
            case 4: gtk_tooltips_set_tip(tip,button,_("next")     ,""); break;
            case 5: gtk_tooltips_set_tip(tip,button,_("loop")     ,""); break;
            case 6: gtk_tooltips_set_tip(tip,button,_("disp mode"),""); break;
        }
        if( i == 3 )
        {
#ifdef M2M_OLD_PULLDOWN
            GtkMenu *menu;
            int pg,npg;
            pg_pulldown = GTK_OPTION_MENU(button = gtk_option_menu_new()) ;
            menu=GTK_MENU(gtk_menu_new());
            npg=get_number_of_chapter();
            for( pg=0 ; pg<npg ; pg++ )
            {
                char str[32];
                sprintf( str,"%02d/%02d ",pg+1,npg );
                item=gtk_menu_item_new_with_label(str);
                gtk_object_set_data( GTK_OBJECT (item), "no",(gpointer)pg );
                gtk_signal_connect( GTK_OBJECT (item), "activate",
                                    GTK_SIGNAL_FUNC (pgpd_cb), NULL );
                gtk_menu_append( menu,item );
                gtk_widget_show ( item );
            }
            gtk_option_menu_set_menu(GTK_OPTION_MENU(button),GTK_WIDGET(menu));
            gtk_widget_show ( GTK_WIDGET(menu) );
            gtk_box_pack_start( GTK_BOX (form2), button,FALSE,FALSE,0 );
            gtk_widget_show ( button );
            gtk_tooltips_set_tip(tip,button,_("program"),"");
#else
            pg_echo = GTK_LABEL(gtk_label_new("00/00"));
            gtk_widget_show ( GTK_WIDGET(pg_echo) );
            gtk_box_pack_start( GTK_BOX(form2),GTK_WIDGET(pg_echo),
                                FALSE,FALSE,0);
#endif
        }
    }
    spin = gtk_spin_button_new( GTK_ADJUSTMENT(adjust_audio_no),0,0 );
    gtk_spin_button_set_wrap( GTK_SPIN_BUTTON(spin),FALSE );
    gtk_box_pack_start (GTK_BOX (form2), spin, FALSE,FALSE,0 );
    gtk_widget_show ( spin );
    gtk_tooltips_set_tip(tip,spin,_("audio track"),"");
    spin = gtk_spin_button_new( GTK_ADJUSTMENT(adjust_brightness),0,0 );
    gtk_spin_button_set_wrap( GTK_SPIN_BUTTON(spin),FALSE );
    gtk_box_pack_start (GTK_BOX (form2), spin, FALSE,FALSE,0 );
    gtk_widget_show ( spin );
    gtk_tooltips_set_tip(tip,spin,_("brightness"),"");
    spin = gtk_spin_button_new( GTK_ADJUSTMENT(adjust_contrast),0,0 );
    gtk_spin_button_set_wrap( GTK_SPIN_BUTTON(spin),FALSE );
    gtk_box_pack_start (GTK_BOX (form2), spin, FALSE,FALSE,0 );
    gtk_widget_show ( spin );
    gtk_tooltips_set_tip(tip,spin,_("contrast"),"");
    gtk_widget_show( form2 );
    gtk_container_add( GTK_CONTAINER(handles[1]),form2 );
    gtk_widget_show( handles[1] );

    handles[2]=gtk_handle_box_new();
    form2=gtk_scrolled_window_new( NULL,NULL );
    gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW(form2),
                             GTK_POLICY_ALWAYS,GTK_POLICY_NEVER );
    form3=gtk_hbox_new(0,0);
    gtk_container_set_border_width (GTK_CONTAINER (form3), 0);
    for( i=0 ; i<get_number_of_chapter() ; i++ )
    {
       extern vob_file_t *vobfiles;
       char str[32];
       sprintf( str,"%02d",i+1 );
       button=gtk_button_new_with_label(str);
       gtk_signal_connect( GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (pgbtn_cb), NULL );
       gtk_tooltips_set_tip(tip,button,vobfiles[i].name,"");
       gtk_box_pack_start (GTK_BOX (form3), button, FALSE,FALSE,0 );
       gtk_widget_show ( button );
    }
    gtk_widget_show( form3 );
    gtk_scrolled_window_add_with_viewport( GTK_SCROLLED_WINDOW(form2),form3 );
    gtk_widget_show( form2 );
    gtk_container_add( GTK_CONTAINER(handles[2]),form2 );
    gtk_widget_show( handles[2] );

    handles[3]=gtk_handle_box_new();
    scale = gtk_hscale_new( GTK_ADJUSTMENT(adjust_time) );
    gtk_widget_set_usize( GTK_WIDGET ( scale ) , 480 , 24 );
    gtk_scale_set_draw_value( GTK_SCALE( scale ) , FALSE );
    gtk_scale_set_value_pos( GTK_SCALE(scale), GTK_POS_LEFT );
    gtk_widget_show( scale );
    gtk_container_add( GTK_CONTAINER(handles[3]),scale );
    gtk_widget_show( handles[3] );

    handles[4]=gtk_handle_box_new();
    form2=gtk_hbox_new(0,0);
    gtk_container_set_border_width (GTK_CONTAINER (form2), 0);
    gtk_container_add( GTK_CONTAINER(handles[4]),form2 );
    spin = gtk_spin_button_new( GTK_ADJUSTMENT(adjust_ad_sync),0,0 );
    gtk_spin_button_set_wrap( GTK_SPIN_BUTTON(spin),FALSE );
    gtk_box_pack_start (GTK_BOX (form2), spin, FALSE,FALSE,0 );
    gtk_widget_show ( spin );
    gtk_tooltips_set_tip(tip,spin,_("sync"),"");
    spin = gtk_spin_button_new( GTK_ADJUSTMENT(adjust_ad_late),0,0 );
    gtk_spin_button_set_wrap( GTK_SPIN_BUTTON(spin),FALSE );
    gtk_box_pack_start (GTK_BOX (form2), spin, FALSE,FALSE,0 );
    gtk_widget_show ( spin );
    gtk_tooltips_set_tip(tip,spin,_("late"),"");
    spin = gtk_spin_button_new( GTK_ADJUSTMENT(adjust_ad_skip),0,0 );
    gtk_spin_button_set_wrap( GTK_SPIN_BUTTON(spin),FALSE );
    gtk_box_pack_start (GTK_BOX (form2), spin, FALSE,FALSE,0 );
    gtk_widget_show ( spin );
    gtk_tooltips_set_tip(tip,spin,_("skip"),"");
    gtk_widget_show( form2 );

    handles[0]=gtk_handle_box_new();
    menubar=gtk_menu_bar_new();
    submenu=gtk_menu_new();
    item=gtk_menu_item_new_with_label( "Quit" );
    gtk_menu_append( GTK_MENU(submenu),item );
    gtk_widget_add_accelerator( item,"activate",accel,'Q',GDK_CONTROL_MASK,1 );
    gtk_signal_connect( GTK_OBJECT (item), "activate",
                        GTK_SIGNAL_FUNC (destroy), NULL);
    gtk_widget_show(item);
    item=gtk_menu_item_new_with_label( "File" );
    gtk_widget_show(item);
    gtk_menu_item_set_submenu( GTK_MENU_ITEM(item),submenu );
    gtk_menu_bar_append( GTK_MENU_BAR(menubar),item );
    submenu=gtk_menu_new();
    item=gtk_menu_item_new_with_label( "Preferences..." );
    gtk_menu_append( GTK_MENU(submenu),item );
    //gtk_signal_connect( GTK_OBJECT (item), "activate",
    //                    GTK_SIGNAL_FUNC (pref_cb), NULL);
    gtk_widget_show(item);
    item=gtk_menu_item_new_with_label( "Edit" );
    gtk_widget_show(item);
    gtk_menu_item_set_submenu( GTK_MENU_ITEM(item),submenu );
    gtk_menu_bar_append( GTK_MENU_BAR(menubar),item );
    submenu=gtk_menu_new();
    item=gtk_menu_item_new( );
    gtk_menu_append( GTK_MENU(submenu),item );
    gtk_signal_connect( GTK_OBJECT (item), "activate",
                        GTK_SIGNAL_FUNC (tool1_cb), NULL );
    gtk_signal_connect( GTK_OBJECT (item), "map_event",
                        GTK_SIGNAL_FUNC (map_cb), (gpointer)-2 );
    gtk_widget_show(item);
    item=gtk_menu_item_new( );
    gtk_menu_append( GTK_MENU(submenu),item );
    gtk_signal_connect( GTK_OBJECT (item), "activate",
                        GTK_SIGNAL_FUNC (tool2_cb), NULL );
    gtk_signal_connect( GTK_OBJECT (item), "map_event",
                        GTK_SIGNAL_FUNC (map_cb), (gpointer)-3 );
    gtk_widget_show(item);
    item=gtk_menu_item_new( );
    gtk_menu_append( GTK_MENU(submenu),item );
    gtk_signal_connect( GTK_OBJECT (item), "activate",
                        GTK_SIGNAL_FUNC (tool3_cb), NULL );
    gtk_signal_connect( GTK_OBJECT (item), "map_event",
                        GTK_SIGNAL_FUNC (map_cb), (gpointer)-4 );
    gtk_widget_show(item);
    item=gtk_menu_item_new( );
    gtk_menu_append( GTK_MENU(submenu),item );
    gtk_signal_connect( GTK_OBJECT (item), "activate",
                        GTK_SIGNAL_FUNC (tool4_cb), NULL );
    gtk_signal_connect( GTK_OBJECT (item), "map_event",
                        GTK_SIGNAL_FUNC (map_cb), (gpointer)-5 );
    gtk_widget_show(item);
    item=gtk_menu_item_new_with_label( "Tool" );
    gtk_widget_show(item);
    gtk_menu_item_set_submenu( GTK_MENU_ITEM(item),submenu );
    gtk_menu_bar_append( GTK_MENU_BAR(menubar),item );
    submenu=gtk_menu_new();
    item=gtk_menu_item_new_with_label( "About..." );
    gtk_menu_append( GTK_MENU(submenu),item );
    //gtk_signal_connect( GTK_OBJECT (item), "activate",
    //                    GTK_SIGNAL_FUNC (about_cb), NULL);
    gtk_widget_show(item);
    item=gtk_menu_item_new_with_label( "Help" );
    gtk_widget_show(item);
    gtk_menu_item_set_submenu( GTK_MENU_ITEM(item),submenu );
    gtk_menu_bar_append( GTK_MENU_BAR(menubar),item );
    gtk_widget_show( menubar );
    gtk_container_add( GTK_CONTAINER(handles[0]),menubar );
    gtk_widget_show( handles[0] );

    gtk_box_pack_start (GTK_BOX (form1), handles[0], FALSE,FALSE,0 );
    gtk_box_pack_start (GTK_BOX (form1), handles[1], FALSE,FALSE,0 );
    gtk_box_pack_start (GTK_BOX (form1), handles[2], TRUE,FALSE,3 );
    gtk_box_pack_start (GTK_BOX (form1), handles[3], TRUE,FALSE,3 );
    gtk_box_pack_start (GTK_BOX (form1), handles[4], TRUE,TRUE,3 );
    gtk_range_set_update_policy( GTK_RANGE(scale), GTK_UPDATE_DELAYED );
    gtk_widget_show( form1 );
    if ( ! HIDE_CONTROLL )
        gtk_widget_show( window );
    gtk_timeout_add(1000,adjustor_sync,NULL );
    return 0;
}

static void m2m_usage(char * command )
{
    printf( "Version " PACKAGE "/" VERSION "\n" );
    printf( "Usage: %s [options] mpeg2_file.. \n",command );
    printf( "       %s [options] [device_option [title_option]] \n",command );
    printf( "Option:\n" );
    printf( "  --device=device \n" );
    printf( "  -d device         : DVD device file\n" );
    printf( "                      default is /dev/dvd \n" );
    printf( "  --title=title   \n" );
    printf( "  -t title          : DVD title number \n" );
    printf( "                      default is 1 \n" );
    printf( "  --audio=stream  \n" );
    printf( "  -a title          : select audio stream number \n" );
    printf( "                      default is 1 \n" );
    printf( "  --rate=audiorate  : set rate of audio device(for downcast)\n" );
    printf( "                      default is source(no downcasting) \n" );
    printf( "  --fullscreen      : fullscreen (need XVideo-Extension) \n" );
    printf( "  --bg              : X11 background(desktop) \n" );
    printf( "  --autostart       : to automate start.\n" );
    printf( "  --hidecontroller  : to hide a controller.\n" );
    printf( "  --disable-menu    : to disable a title/chapter menu.\n" );
    printf( "  --loop            : doing loop a play \n" );
    printf( "  --vo=driver       : select video out driver \n" );
    {
        vo_driver_t * drivers = vo_drivers();
        int i;
        for ( i=0 ; drivers[i].name ; i++ ){
           printf( "                          %s \n",drivers[i].name );
        }
    }
    printf( "  --ao=driver       : select audio out driver \n" );
    {
        ao_driver_t * drivers = ao_drivers();
        int i;
        for ( i=0 ; drivers[i].name ; i++ ){
           printf( "                          %s \n",drivers[i].name );
        }
    }

#ifdef HAVE_LIBASOUND
    printf( "  --alsa            : use ALSA audio driver \n" );
#endif
#ifdef HAVE_ESOUND
    printf( "  --esd             : use Esound audio driver \n" );
    printf( "  --esd=hostname    : use Esound and connect to sound server \n" );
#endif
    printf( "  --help            : help message \n" );
}

static void replace_intvalue( char *p , int *val )
{
    int new_val,i;
    for ( new_val=0,i=0 ; p[i] >= '0' && p[i] <= '9' ; i++ ) 
        new_val = new_val * 10 + p[i] - '0' ;
    if ( new_val > 0 && val ) *val=new_val;
}

static int m2m_init( int argc,char *argv[] )
{
    int arg;
    int do_usage=0;

    setlocale (LC_ALL, "");
    bindtextdomain (PACKAGE, LOCALEDIR);
    textdomain (PACKAGE);

    printf(" mpeg2dec: (C) 1999 Aaron Holtzman <aholtzma@ess.engr.uvic.ca>\n");
    printf(" m2m : (C) 2000 Taichi Nakamura <pdf30044@biglobe.ne.jp>\n");

    for ( arg=1 ; arg<argc ; arg++ )
    {
        if ( argv[arg][0] != '-' ){
            if ( file_name == NULL ) file_name = argv+arg ;
            continue;
        }
        if ( strcmp(argv[arg],"-d") == 0 && arg+1 < argc )
            device_name = argv[++arg] ;
        else
        if( strncmp(argv[arg],"--device=",9) == 0 && argv[arg][9] != 0x0 )
            device_name = argv[arg]+9;
        else
        if ( strcmp(argv[arg],"-t") == 0 && arg+1 < argc )
            replace_intvalue( argv[++arg] , &title_no );
        else
        if( strncmp(argv[arg],"--title=",8) == 0 )
            replace_intvalue( argv[arg]+8 , &title_no );
        else
        if ( strcmp(argv[arg],"-a") == 0  && arg+1 < argc )
            replace_intvalue( argv[++arg] , &audio_no );
        else
        if( strncmp(argv[arg],"--audio=",8) == 0 )
            replace_intvalue( argv[arg]+8 , &audio_no );
        else
        if( strncmp(argv[arg],"--rate=",7) == 0 )
            replace_intvalue( argv[arg]+7 , &audio_rate );
        else
#ifdef HAVE_ESOUND
        if( strncmp(argv[arg],"--esd=",6) == 0 ){
            SET_COMMANDLINE_PARAMS( IS_ESD );
            esd_host = argv[arg]+6;
        }else
#endif
        if( strcmp(argv[arg],"--fullscreen") == 0 )
            SET_COMMANDLINE_PARAMS( IS_FULLSCREEN );
        if( strcmp(argv[arg],"--bg") == 0 )
            SET_COMMANDLINE_PARAMS( IS_BG );
        if( strcmp(argv[arg],"--autostart") == 0 )
            SET_COMMANDLINE_PARAMS( IS_AUTOSTART );
        if( strcmp(argv[arg],"--hidecontroller") == 0 )
            SET_COMMANDLINE_PARAMS( (HIDE_CONTROLL|IS_AUTOSTART) );
        if( strcmp(argv[arg],"--disable-menu") == 0 )
            SET_COMMANDLINE_PARAMS( DISABLE_MENU );
        if( strcmp(argv[arg],"--loop") == 0 )
            SET_COMMANDLINE_PARAMS( IS_LOOP );
        if( strncmp(argv[arg],"--vo=",5) == 0 && argv[arg][5] != 0x0 )
            vo_driver = argv[arg]+5;
        if( strncmp(argv[arg],"--ao=",5) == 0 && argv[arg][5] != 0x0 )
            ao_driver = argv[arg]+5;
#ifdef HAVE_LIBASOUND
        if( strcmp(argv[arg],"--alsa") == 0 )
            SET_COMMANDLINE_PARAMS( IS_ALSA );
#endif
#ifdef HAVE_ESOUND
        if( strcmp(argv[arg],"--esd") == 0 )
            SET_COMMANDLINE_PARAMS( IS_ESD );
#endif
        if( strcmp(argv[arg],"--help") == 0 )
            do_usage=1;
    }


    if ( (!file_name && !device_name) || do_usage )
    {
        m2m_usage( argv[0] );
        if ( do_usage ) return -1;
        device_name=default_device_name;
        printf( "using default device \'%s\'\n",default_device_name );
    }

    open_file();

    {
        int i;
        uint32_t accel = mm_accel () | MM_ACCEL_MLIB;
        vo_driver_t * drivers = vo_drivers();
        vo_accel (accel);
        vo_instance = NULL;
        for ( i=0 ; vo_driver && drivers[i].name ; i++ ){
            if ( strcmp(vo_driver,drivers[i].name)==0 ){
                vo_instance = vo_open (drivers[i].open);
if ( vo_instance ) fprintf( stderr,"select video driver=%s\n",drivers[i].name);
                break;
            }
        }
        if ( vo_instance == NULL ) vo_instance = vo_open (drivers[0].open);
        if ( vo_instance == NULL ) return -1;
        mpeg2_init( &mpeg2dec , accel, vo_instance );
    }

    //start_thread;
    m2m_thread_control( M2M_VIDEO_THREAD_ID,1 );
    m2m_thread_control( M2M_AUDIO_THREAD_ID,1 );
    if ( IS_AUTOSTART )
    {
        play_status = STATUS_PLAY ;
        pthread_cond_signal( &play_stat_cond );
    }

    return 0;
}

int main( int   argc, char *argv[] )
{
    if ( m2m_init( argc,argv ) )
        return 1;
    panel_create( argc, argv );
    gtk_main ();
    return 0;
}



#define ARCH_X86
#ifdef ARCH_X86
static uint32_t x86_accel (void)
{
    uint32_t eax, ebx, ecx, edx;
    int AMD;
    uint32_t caps;

#define cpuid(op,eax,ebx,ecx,edx)	\
    asm ("cpuid"			\
	 : "=a" (eax),			\
	   "=b" (ebx),			\
	   "=c" (ecx),			\
	   "=d" (edx)			\
	 : "a" (op)			\
	 : "cc")

    asm ("pushfl\n\t"
	 "popl %0\n\t"
	 "movl %0,%1\n\t"
	 "xorl $0x200000,%0\n\t"
	 "pushl %0\n\t"
	 "popfl\n\t"
	 "pushfl\n\t"
	 "popl %0"
         : "=a" (eax),
	   "=b" (ebx)
	 :
	 : "cc");

    if (eax == ebx)		/* no cpuid */
	return 0;

    cpuid (0x00000000, eax, ebx, ecx, edx);
    if (!eax)			/* vendor string only */
	return 0;

    AMD = (ebx == 0x68747541) && (ecx == 0x444d4163) && (edx == 0x69746e65);

    cpuid (0x00000001, eax, ebx, ecx, edx);
    if (! (edx & 0x00800000))	/* no MMX */
	return 0;

    caps = MM_ACCEL_X86_MMX;
    if (edx & 0x02000000)	/* SSE - identical to AMD MMX extensions */
	caps = MM_ACCEL_X86_MMX | MM_ACCEL_X86_MMXEXT;

    cpuid (0x80000000, eax, ebx, ecx, edx);
    if (eax < 0x80000001)	/* no extended capabilities */
	return caps;

    cpuid (0x80000001, eax, ebx, ecx, edx);

    if (edx & 0x80000000)
	caps |= MM_ACCEL_X86_3DNOW;

    if (AMD && (edx & 0x00400000))	/* AMD MMX extensions */
	caps |= MM_ACCEL_X86_MMXEXT;

    return caps;
}
#endif

uint32_t mm_accel (void)
{
#ifdef ARCH_X86
    static int got_accel = 0;
    static uint32_t accel;

    if (!got_accel) {
	got_accel = 1;
	accel = x86_accel ();
    }

    return accel;
#else
    return 0;
#endif
}

