import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.LayoutManager;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.UIManager;

import util.GeneralFileFilter;
import util.Log;

/**
 * ファイルの文字コード変換ツール
 * @author Masayasu Fujiwara
 */
public class Main extends JFrame {
	public Main(String name) {
		super(name);
	}
	public static void main(String[] args) {
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} catch (Exception e) {
			Log.err(Main.class, e);
		}
		final Main dialog = new Main("文字コード変換");
		dialog.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		LayoutManager fieldLayout = new FlowLayout(FlowLayout.CENTER, 10, 5);
		dialog.setLayout(fieldLayout);

		JPanel codeFieldPanel = new JPanel(fieldLayout);
		
		final JTextField fromCodeField = new JTextField(5);
		fromCodeField.setText("SJIS");
		codeFieldPanel.add(fromCodeField);

		codeFieldPanel.add(new JLabel("→"));

		final JTextField toCodeField = new JTextField(5);
		toCodeField.setText("UTF-8");
		codeFieldPanel.add(toCodeField);

		JPanel fieldPanel = new JPanel(fieldLayout);
		fieldPanel.add(codeFieldPanel);
		
		JPanel filePanel = new JPanel(fieldLayout);
		final JTextField fileField = new JTextField(20);
		filePanel.add(fileField);
		JButton fileButton = new JButton("選択");
		filePanel.add(fileButton);

		JPanel optionPanel = new JPanel(fieldLayout);
		JPanel optionListPanel = new JPanel(new GridLayout(0, 1, 0, 0));
		final JCheckBox sub = new JCheckBox("サブディレクトリも変換", true);
		final JCheckBox backup = new JCheckBox("バックアップ（.bak）", true);
		final JCheckBox extension = new JCheckBox("Javaソースのみ変換（.java）", true);
		
		optionListPanel.add(sub);
		optionListPanel.add(backup);
		optionListPanel.add(extension);
		
		optionPanel.add(optionListPanel);
		
		fileButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				JFileChooser chooser = new JFileChooser();
				chooser.setDialogType(JFileChooser.OPEN_DIALOG);
				chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
				int returnVal = chooser.showDialog(dialog, "選択");
				if (returnVal == JFileChooser.APPROVE_OPTION) {
					File file = chooser.getSelectedFile();
					try {
						fileField.setText(file.getCanonicalPath());
					} catch (IOException e1) {
						JOptionPane.showMessageDialog(dialog, "選択したファイルが不正です。", "警告", JOptionPane.ERROR_MESSAGE);
					}
				}
			}
		});

		JPanel buttonPanel = new JPanel(new GridLayout(0, 1, 10, 10));
		JButton transformButton = new JButton("変換");

		JButton cancel = new JButton("終了");
		cancel.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				System.exit(0);
			}
		});
		buttonPanel.add(transformButton);
		buttonPanel.add(cancel);
		transformButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				try {
					FileFilter filter = null;
					if (extension.isSelected()) {
						filter = new GeneralFileFilter("java");
					}
					File file = new File(fileField.getText());
					if (file.isDirectory()) {
						dialog.convertDir(file, fromCodeField.getText(), toCodeField.getText(), filter, backup.isSelected(), sub.isSelected());
					} else {
						if (filter == null || filter.accept(file)) {
							dialog.convertFile(file, fromCodeField.getText(), toCodeField.getText(), filter, backup.isSelected());
						} else {
							JOptionPane.showMessageDialog(dialog, "選択したファイルが不正です。", "警告", JOptionPane.ERROR_MESSAGE);							
						}
					}
				} catch (IOException ex) {
					JOptionPane.showMessageDialog(dialog, "選択したファイルが不正です。", "警告", JOptionPane.ERROR_MESSAGE);
				} catch (Exception ex) {
					JOptionPane.showMessageDialog(dialog, "入力が不正です。", "警告", JOptionPane.ERROR_MESSAGE);					
				}
			}
		});
		
		dialog.add(new JLabel("変換前後の文字コードを入力してください。", SwingConstants.CENTER), 0);
		dialog.add(fieldPanel);
		dialog.add(new JLabel("変換するファイルまたはディレクトリを選択してください。", SwingConstants.CENTER));
		dialog.add(filePanel);
		dialog.add(optionPanel);
		dialog.add(buttonPanel);
		dialog.setSize(320, 230);

		dialog.setVisible(true);
		dialog.setFocusable(true);
	}
	
	/**
	 * ディレクトリ内のファイルの文字コードを変換します。
	 * @param dir 変換するファイルまたはディレクトリ
	 * @param code 変換前の文字コード
	 * @param code2 変換後の文字コード
	 * @param filter 
	 * @param isBackup 
	 * @param isSubdir サブディレクトリの変換の有無
	 * @throws IOException 入出力エラー
	 */
	void convertDir(File dir, String code, String code2, FileFilter filter, boolean isBackup, boolean isSubdir) throws IOException {
		for (File file : dir.listFiles()) {
			if (file.isDirectory() && isSubdir) {
				convertDir(file, code, code2, filter, isBackup, isSubdir);
			} else if (file.isFile()){
				convertFile(file, code, code2, filter, isBackup);
			}
		}
	}
	
	/**
	 * ファイルの文字コードを変換します。
	 * @param file
	 * @param code
	 * @param code2
	 * @param filter
	 * @param isBackup
	 * @throws IOException
	 */
	void convertFile(File file, String code, String code2, FileFilter filter, boolean isBackup) throws IOException {
		if (file.getPath().endsWith(".bak")) {
			return;
		}
		String filepath = file.getCanonicalPath();
		BufferedReader bi = null;
		BufferedWriter bo = null;
		File tmp = new File(filepath + ".tmp");
		boolean exception = false;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(file), code));
			bo = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(tmp), code2));
			while(bi.ready()) {
				String line = bi.readLine();
				bo.write(line);
				bo.write('\n');
				bo.flush();
			}
		} catch (UnsupportedEncodingException e) {
			JOptionPane.showMessageDialog(this, e.getMessage() + "には対応していません。", "警告", JOptionPane.ERROR_MESSAGE);
			exception = true;
		} finally {
			if (bi != null) {
				bi.close();
			}
			if (bo != null) {
				bo.close();
			}
			if (exception) {
				tmp.delete();
				return;
			}
		}
		File backup = new File(filepath + ".bak");
		int count = 1;
		while (backup.exists()) {
			backup = new File(filepath+ "("+ count+ ")" + ".bak");
			count++;
		}
		if (file.renameTo(backup) && tmp.renameTo(new File(filepath))) {;
			System.out.println("Convert: "+ filepath);
			if (!isBackup) {
				backup.delete();
			}
		} else {
			throw new IOException("リネームに失敗しました。"+ filepath);
		}
	}
}
