# -*- coding: utf-8 -*-

#########################################################################
## - Application Name: Machikane-Red
## - Version: 3.1811
## - Date: 2019-01-04
## - Copyright: (c) 2018-2019 Mitsuhiro Tsuda.
## - License: Machikane-Red (version 3.1811) is released
##            under the GNU General Public License (GPL).
##            See the copyright notice LICENSE.
#########################################################################

import os
import io
import datetime
import re
import json
from gluon.sanitizer import sanitize

from PIL import Image
import pyvips


mr_datahelper = local_import('mr_datahelper', reload=MR_CONF['RELOAD'])
mr_helper = local_import('mr_helper', reload=MR_CONF['RELOAD'])


def __precheck():
    """
    該当しなければログインを通過
    2018-11
    """
    if auth.user.id < 5 and not ( request.env.remote_addr in configuration.get('myconfig.allowances') or request.env.remote_addr.startswith(configuration.get('myconfig.local_ip')) ):
        redirect(URL('staff','user',args=['logout']))

    if session.auth.user is not None and session.groupid is None:
        session.groupid = db(db.auth_membership.user_id==session.auth.user.id).select().first().group_id

    # 制限
    if session.groupid > MR_CONF['GROUP_EDITOR']:
        redirect(URL('staff','user',args=['logout']))


@auth.requires_login()
def user():
    """
    ユーザー認証
    2018-11
    """
    auth.settings.controller = 'staff'
    return dict(form=auth())


@auth.requires_login()
def index():
    """
    トップ,ページ
    2018-11
    """
    __precheck()

    # 引数（ページID）の取得
    pageid = 1  #デフォルト
    if len(request.args) > 0:
        pageid = request.args(0, cast=int) or 1

    # ページセクション一覧
    da = mr_datahelper.l_d_sections(dbm, pageid)

    # ページがpageidであるページセクション（複数）の取得
    dom_div = DIV(_class='m-content')

    title = da['title'] or T('Undefined')

    for d_dom in da['array']:

        dom_section = TAG.section(
                        *[H2(mr_helper.__helper_br(v['value'])) for v in d_dom['title']]
                        ,DIV(
                            *[P(mr_helper.__helper_br(v['value']),_class='') for v in d_dom['description']]
                            ,_class='m-div-text'
                        )
                        ,DIV(
                            SPAN(T(d_dom['date'][0]['label']),': ',XML(d_dom['date'][0]['value']) or XML('&mdash;'))
                            ,XML('&nbsp'),'|',XML('&nbsp;'),SPAN(T(d_dom['creator'][0]['label']),': ',XML(d_dom['creator'][0]['value']) or XML('&mdash;'))
                            ,_class='m-div-info'
                        )
                        ,_class='m-page-section'
                    )
        dom_div.append(dom_section)

    # サイドメニュー索引の作成
    dom_sidemenu = __get_sidemenu()

    return dict(sidemenu=dom_sidemenu, title=title, dom=dom_div)


def __form_search():
    """
    検索フォーム（最小）
    2018-11
    """
    form = FORM(
                DIV(
                    DIV(
                        INPUT(
                            _type='search'
                            ,_placeholder=T('Search words')
                            ,_name='q'
                            ,_class='form-control'
                        )
                        ,_class='col col-8'
                    )
                    ,DIV(
                        TAG.button(
                            I(_class='fa fa-search')
                            ,T('Search')
                            ,_type='submit'
                            ,_name='dosearch'
                            ,_class='btn btn-primary'
                        )
                        ,_class='col col-4'
                    )
                    ,_class='formgroup row m-search-box'
                )
            )

    return form


@auth.requires_login()
def list():
    """
    リソース・リスト（コレクション）
    2018-11
    """
    __precheck()

    # 引数（コレクションID）の取得
    collid = 0  #デフォルト
    if len(request.args) > 0:
        collid = request.args(0, cast=int) or 0

    l_query = []

    form = __form_search()

    if form.accepts(request, session, keepvalues=True):
        #response.flash = T('form accepted')
        l_query = sanitize(form.vars.get('q')).split()

    elif form.errors:
        response.flash = T('form has errors')
    else:
        #response.flash = T('please fill the form')
        pass

    # リソース一覧
    da = mr_datahelper.l_d_resources(dbm, collid, l_query)

    session.imgs = {}

    # コレクションがcollidであるリソース（複数）の取得
    dom_div = DIV(_class='m-content')

    title = da['title'] or T('Undefined')

    for d_dom in da['array']:

        dom_resource = DIV(
                        DIV(
                            __get_dom_thumbnail_files(session, d_dom)
                            ,_class='col'
                        )
                        ,DIV(
                            H4(A(d_dom['title']['value']
                                ,_href=URL('item',args=[d_dom['subid']])
                                )
                            )
                            ,DIV(
                                P(mr_helper.__helper_br(d_dom['description']['value']),_class='col col-12')
                                ,_class='row'
                            )
                            ,_class='col'
                        )
                        ,_class='row m-resource-block'
                    )

        dom_div.append(dom_resource)

    # サイドメニュー索引の作成
    dom_sidemenu = __get_sidemenu()

    return dict(sidemenu=dom_sidemenu, title=title, dom=dom_div, form=form, da=da, q=l_query)


@auth.requires_login()
def item():
    """
    リソース・アイテム
    2018-11
    """
    __precheck()

    # 引数（リソースID）の取得
    resourceid = 1  #デフォルト
    if len(request.args) > 0:
        resourceid = request.args(0, cast=int) or 1

    # リソースアイテム
    da = mr_datahelper.l_d_resource_items(dbm, resourceid)

    session.imgs = {}

    # リソースがresourceidであるリソースアイテム（複数）の取得
    dom_div = DIV(_class='m-content')

    title = da['title'] or T('Undefined')

    for d_dom in da['array']:

        dom_resource = DIV(
                        H2(d_dom['title']['value'])
                        ,DIV(
                            P(mr_helper.__helper_br(d_dom['description']['value']),_class='col col-12')
                            ,TABLE(*[
                                TR(TD(d_dom[ky]['label'], _class='m-label'),TD(mr_helper.__helper_br(d_dom[ky]['value']))) for ky in d_dom['keyset']
                                if ky not in ('title','description') and d_dom[ky]['value'] is not None
                            ],_class='col col-12')
                            ,_class='row'
                        )
                        ,__get_dom_image_annotes(session, d_dom, ['area'], True)
                        ,_class='m-resource-item'
                    )
        dom_div.append(dom_resource)

    # サイドメニュー索引の作成
    dom_sidemenu = __get_sidemenu()

    return dict(sidemenu=dom_sidemenu, title=title, dom=dom_div)


def __form_edit(a_id = 0, enable_format = False):
    """
    ビューア編集フォーム
    2018-11
    """

    aopts = [OPTION(x[0],_value=x[1]) for x in ([T('Annotation'),2],)]
    if enable_format:
        aopts = [OPTION(x[0],_value=x[1]) for x in ([T('Annotation'),2],[T('Segment'),1])]

    form = FORM(
                DIV(
                    DIV(
                        TEXTAREA(
                            _rows=2
                            ,_name='atext'
                            ,_class='form-control'
                        )
                        ,DIV(
                            INPUT(
                                _type='text'
                                ,_name='aarea'
                                ,_class='form-control col col-7'
                            )
                            ,INPUT(
                                _type='text'
                                ,_name='arot'
                                ,_value='0'
                                ,_class='form-control col col-3'
                            )
                            ,INPUT(
                                _type='hidden'
                                ,_name='aid'
                                ,_value=str(a_id)
                            )
                            ,_class="row m-edit-control"
                        )
                    )
                    ,DIV(
                        SELECT(
                            aopts
                            ,_name='atype'
                            ,value=2
                            ,_class='form-control col col-4'
                        )
                        ,TAG.button(
                            T('Entry')
                            ,_type='submit'
                            ,_name='doentry'
                            ,_class='btn m-primary col col-3'
                        )
                        ,TAG.button(
                            T('Cancel')
                            ,_name='docancel'
                            ,_class='btn col col-3'
                        )
                        ,_class="row m-edit-control"
                    )
                    ,_class='formgroup m-entry-box'
                )
            )

    return form


@auth.requires_login()
def viewer():
    """
    画像表示
    2018-11
    """
    __precheck()

    var_a_id = 0

    var_target = tuple(request.args)
    (var_r_id,var_f_id,var_a_id) = session.imgs[var_target]

    var_manifest = None

    var_folder = None
    var_name = None
    var_format = None

    var_on_annote = False

    if request.args and len(request.args) > 2:
        if request.args[-1] == 'manifest.json':
            # IIIF Manifest
            var_manifest = '/'.join(request.args)
        else:
            var_folder = sanitize(request.args(0))
            var_name = sanitize(request.args(1))
            var_format = sanitize(request.args(2))
        if 'annote' in request.args:
            var_on_annote = True

    form = __form_edit(var_a_id, (var_format=='tif' or var_format=='jp2'))

    if form.accepts(request, session, keepvalues=True):
        #response.flash = T('form accepted')

        # 範囲の取得
        var_area = sanitize(form.vars.get('aarea')).split(',')
        var_arot = int(form.vars.get('arot') or 0.0)
        var_text = sanitize(form.vars.get('atext'))
        #var_slug = sanitize(form.vars.get('aslug'))
        var_type = 'rect'
        if int(form.vars.get('atype') or 2) == 1:
            var_type = 'area'

        d = {}
        d.setdefault('mr_annote_area', [int(v) for v in var_area])
        d.setdefault('mr_annote_rotation', var_arot)
        d.setdefault('mr_text', var_text)
        #d.setdefault('mr_slug', var_slug)
        d.setdefault('mr_annote_type', var_type)    # area,rect
        if var_type == 'rect':
            d.setdefault('mr_annote_data', [int(v) for v in var_area])
        d.setdefault('mr_resource_id', var_r_id)
        d.setdefault('mr_file_id', var_f_id)
        d.setdefault('mr_open', True)

        try:
            dbm.mr_annotes.update_or_insert(dbm.mr_annotes.id==var_a_id,**dbm.mr_annotes._filter_fields(d))
            dbm.commit()
        except:
            dbm.rollback()
        response.flash = T('susscee entry')

    elif form.errors:
        response.flash = T('form has errors')
    else:
        #response.flash = T('please fill the form')
        pass


    # プレビュー・リスト（このバージョンではアノテーション）
    da = mr_datahelper.l_d_file(dbm, var_f_id)

    preview = UL()

    for d in da['array'][0]['annotes']:
        if not var_on_annote and d['annotetype'] == 'rect' and d['ids'][1] == var_f_id:

            li = LI(
                    SPAN(
                        d['title']
                    )
                    ,_id='n%d' % d['subid']
                    ,_class=''
                    ,**{ '_data-rect': ','.join( [str(v) for v in d['annotedata']] ) }
                )
            preview.append(li)


    return dict(vmanifest=var_manifest, vfolder=var_folder, vname=var_name, vformat=var_format, form=form, preview=preview, idset=session.imgs[var_target])


# ========================================
# IIIF Manifest
# 2018-11
# ========================================

@auth.requires_login()
def iiif():
    """
    IIIF Presentation API 2.1
    2018-11
    """
    __precheck()

    # 引数（ファイルID/アノテーションID）の取得
    resourceid = 1  # Default
    fileid = 0  # Default
    annoteid = 0  # Default
    req_api_type = 'manifest.json'
    if len(request.args) > 1:
        domain = sanitize(request.args(0))
        if domain == 'annote':
            annoteid = request.args(1, cast=int) or 0
            if len(request.args) > 2:
                req_api_type = sanitize(request.args(-1))
        elif domain == 'file':
            fileid = request.args(1, cast=int) or 0
            if len(request.args) > 2:
                req_api_type = sanitize(request.args(-1))
        elif domain == 'resource':
            resourceid = request.args(1, cast=int) or 1
            if len(request.args) > 2:
                req_api_type = sanitize(request.args(-1))
        else:
            domain = 'resource'
            resourceid = request.args(1, cast=int) or 1
            if len(request.args) > 2:
                req_api_type = sanitize(request.args(-1))

    if req_api_type.startswith('manifest'):

        if domain == 'annote':
            da = mr_datahelper.l_d_annote(dbm, annoteid)
        elif domain == 'file':
            da = mr_datahelper.l_d_file(dbm, fileid)
        else:
            # get Dataset of Resource-Items
            da = mr_datahelper.l_d_resource_items(dbm, resourceid)

        # get Manifest JSON
        json_ld = __get_iiif_manifest(da['array'])

        return response.json(json_ld)

    elif req_api_type.startswith('annotationlist'):

        da = mr_datahelper.l_d_file(dbm, fileid)

        # get Annotation List JSON
        json_ld = __get_iiif_annotationlist(da['array'])

        return response.json(json_ld)

    else:
        return response.json({})


def __get_iiif_manifest(array):
    """
    Construction of IIIF Presentation API 2.1
    2018-11
    """
    # Context
    d_context = {
                    "@context": "http://iiif.io/api/presentation/2/context.json"
                    ,"@type": "sc:Manifest"
                    ,"@id": "%s/iiif/%s/manifest.json" % (MR_CONF['SERVER_URL'],'test')
                    ,"label": "test"
                    ,"sequences":[
                        {
                            "@type": "sc:Sequence"
                            ,"canvases": [
                                {
                                    "@type": "sc:Canvas"
                                }
                            ]
                        }
                    ]
                 }

    if len(array) == 1:

        d_d = array[0]

        l_d_metadata = []
        for ky in d_d:
            if ky and (not ky in ('description','title','keyset','subid','files','annotes','ids','contains','role')):
                l_d_metadata.append(d_d[ky])

        if 'role' in d_d:
            if d_d['role'] == 'annote':
                d_context['@id'] = "%s/default/iiif/annote/%s/manifest.json" % (MR_CONF['SERVER_URL'],str(d_d['subid']))
            elif d_d['role'] == 'file':
                d_context['@id'] = "%s/default/iiif/file/%s/manifest.json" % (MR_CONF['SERVER_URL'],str(d_d['subid']))
        else:
            d_context['@id'] = "%s/default/iiif/resource/%s/manifest.json" % (MR_CONF['SERVER_URL'],str(d_d['subid']))

        d_context['label'] = d_d['title']['value'] or ''
        d_context['description'] = d_d['description']['value'] or ''
        d_context['metadata'] = l_d_metadata

        # Sequences
        d_context['sequences'] = []
        for i,r in enumerate(d_d['files']):
            d_sequense = {"@type": "sc:Sequence"}
            d_context['sequences'].append(d_sequense)

            # Canvases
            d_sequense['canvases'] = []
            if r:
                d_canvas = {"@type": "sc:Canvas"}
                d_sequense['canvases'].append(d_canvas)

                canves_id = "%s/default/iiif/%s/%s/canvas/image%s" % (MR_CONF['SERVER_URL'],d_d['role'],str(d_d['subid']),str(i+1))

                d_canvas['@id'] = canves_id
                d_canvas['label'] = r['title']

                if 'area' in d_d:
                    d_canvas['width'] = d_d['area']['value'][2]
                    d_canvas['height'] = d_d['area']['value'][3]
                else:
                    d_canvas['width'] = r['width']
                    d_canvas['height'] = r['height']

                # get default @id
                default_id = ''
                if ('format' in r) and (r['format'] in ('tif','jp2')):
                    default_id = '%s%s.%s/full/full/0/default.jpg' % (MR_CONF['IIIF_SERVER_URL'],'%2F'.join(r['href'][:-1]),r['format'])
                else:
                #elif ('format' in r) and (r['format'] in ('jpg')):
                    default_id = "%s/resources/%s.%s" % (MR_CONF['SERVER_URL'],'/'.join(r['href'][:-1]),r['format'])

                # get image @id
                resource_id = ''
                if ('format' in r) and (r['format'] in ('tif','jp2')):
                    if 'area' in d_d:
                        resource_id = '%s%s.%s/%s/full/%d/default.jpg' % (MR_CONF['IIIF_SERVER_URL'],'%2F'.join(r['href'][:-1]),r['format'],','.join([str(v) for v in d_d['area']['value']]),int(d_d['rotation']['value']))
                    else:
                        resource_id = '%s%s.%s/full/full/0/default.jpg' % (MR_CONF['IIIF_SERVER_URL'],'%2F'.join(r['href'][:-1]),r['format'])
                elif ('format' in r) and (r['format'] in ('fzp','fzp3','dzi','zoomify')):
                    resource_id = "%s/resources/%s" % (MR_CONF['SERVER_URL'],'/'.join(r['snapimage']))
                else:
                #elif ('format' in r) and (r['format'] in ('jpg')):
                    resource_id = "%s/resources/%s.%s" % (MR_CONF['SERVER_URL'],'/'.join(r['href'][:-1]),r['format'])

                # get thumbnail image @id
                thumbnail_id = ''
                if ('format' in r) and (r['format'] in ('tif','jp2')):
                    if 'area' in d_d:
                        thumbnail_id = '%s%s.%s/%s/%d,/%d/default.jpg' % (MR_CONF['IIIF_SERVER_URL'],'%2F'.join(r['href'][:-1]),r['format'],','.join([str(v) for v in d_d['area']['value']]),MR_CONF['IMG_THUMBNAIL_SIZE'],int(d_d['rotation']['value']))
                    else:
                        thumbnail_id = '%s%s.%s/full/%d,/0/default.jpg' % (MR_CONF['IIIF_SERVER_URL'],'%2F'.join(r['href'][:-1]),r['format'],MR_CONF['IMG_THUMBNAIL_SIZE'])

                elif ('format' in r) and (r['format'] in ('fzp','fzp3','dzi','zoomify')):
                    thumbnail_id = "%s/resources/%s" % (MR_CONF['SERVER_URL'],'/'.join(r['thumbnail']))

                elif ('format' in r) and (r['format'] in ('jpg')):
                    thumbnail_id = "%s/resources/%s" % (MR_CONF['SERVER_URL'],'/'.join(r['thumbnail']))

                # Thumbnail
                d_canvas['thumbnail'] = { "@type": "dctypes:Image"
                            ,"@id": thumbnail_id
                            ,"format": "image/jpeg"
                            }
                # Images
                d_canvas['images'] = []
                if r:
                    d_image = {"@type": "oa:Annotation"
                            ,"motivation": "sc:painting"
                            ,"on": canves_id
                            }
                    d_canvas['images'].append(d_image)

                    # Resource
                    d_resource = {"@type": "dctypes:Image"
                            ,"@id": resource_id
                            ,"format": "image/jpeg"
                            }
                    d_image['resource'] = d_resource;

                    if ('format' in r) and (r['format'] in ('tif','jp2')):

                        if 'area' in d_d:
                            # default
                            d_default = {"@type": "dctypes:Image"
                                ,"@id": default_id
                                }
                            d_resource['default'] = d_default

                            # Service
                            d_service = {"@context": "http://iiif.io/api/image/2/context.json"
                                ,"@id": '%s%s.%s' % (MR_CONF['IIIF_SERVER_URL'],'%2F'.join(r['href'][:-1]),r['href'][-1])
                                }
                            d_default['service'] = d_service

                        else:
                            # Service
                            d_service = {"@context":    "http://iiif.io/api/image/2/context.json"
                                ,"@id": '%s%s.%s' % (MR_CONF['IIIF_SERVER_URL'],'%2F'.join(r['href'][:-1]),r['href'][-1])
                                }
                            d_resource['service'] = d_service

                    if 'area' in d_d:
                        # Selector
                        d_selector = {
                            "@context": "http://iiif.io/api/presentation/2/context.json"
                            ,"@type": "iiif:ImageApiSelector"
                            ,"region": "%s" % ','.join([str(v) for v in d_d['area']['value']])
                            }
                        d_image['selector'] = d_selector

                # アノテーションの場合は補正
                if d_d['role'] == 'annote':
                    file_subid = d_d['files'][0]['subid']
                else:
                    file_subid = d_d['subid']

                # Annotation link
                d_canvas['otherContent'] = []
                d_annote = {"@type": "sc:AnnotationList"
                            ,"@id": "%s/default/iiif/file/%s/annotationlist.json" % (MR_CONF['SERVER_URL'],str(file_subid))
                            }
                if d_d['role'] == 'file':
                    d_canvas['otherContent'].append(d_annote)

    return d_context


def __get_iiif_annotationlist(array):
    """
    Construction of IIIF Presentation API 2.1
    2018-11
    """
    # Context
    d_context = {
                    "@context": "http://iiif.io/api/presentation/2/context.json"
                    ,"@type": "sc:AnnotationList"
                    ,"@id": "%s/iiif/%s/annotationlist.json" % (MR_CONF['SERVER_URL'],'test')
                    ,"resources":[
                        {
                            "@type": "oa:Annotation"
                            ,"motivation": "sc:painting"
                            ,"resource": [
                                {
                                    "@type": "cnt:ContentAsText"
                                    ,"chars": ""
                                    ,"format": "text/plain"
                                    ,"language": "ja"
                                }
                            ]
                            ,"on": {
                                "@type": "oa:SpecificResource"
                                ,"full": "%s/iiif/%s/canvas/image" % (MR_CONF['SERVER_URL'],'test')
                                ,"selector": {
                                    "@type": "oa:FragmentSelector"
                                    ,"value": "xywh=0,0,0,0"
                                }
                                ,"within": {
                                    "@id": "%s/iiif/%s/manifest.json" % (MR_CONF['SERVER_URL'],'test')
                                    ,"@type": "sc:Manifest"
                                }
                            }
                        }
                    ]
                 }

    if len(array) == 1:

        d_d = array[0]

        l_d_metadata = []
        for ky in d_d:
            if ky and (not ky in ('description','title','keyset','subid','files')):
                l_d_metadata.append(d_d[ky])

        d_context['@id'] = "%s/default/iiif/%s/%s/annotationlist.json" % (MR_CONF['SERVER_URL'],d_d['role'],str(d_d['subid']))

        d_manifest = "%s/default/iiif/%s/%s/manifest.json" % (MR_CONF['SERVER_URL'],d_d['role'],str(d_d['subid']))

        d_full = "%s/default/iiif/%s/%s/canvas/image1" % (MR_CONF['SERVER_URL'],d_d['role'],str(d_d['subid']))

        d_context['resources'] = []
        for i,r in enumerate(d_d['annotes']):
            if r['annotetype'] == 'area':
                continue

            if r:
                # Canvases
                d_annote = {
                    "@type": "oa:Annotation"
                    ,"motivation": "sc:painting"
                    ,"resource": {
                        "@type": "cnt:ContentAsText"
                        ,"chars": r['title'] or ''
                        ,"format": "text/plain"
                        ,"language": "ja"
                    }
                    ,"on": {
                        "@type": "oa:SpecificResource"
                        ,"full": d_full
                        ,"selector": {
                            "@type": "oa:FragmentSelector"
                            ,"value": "xywh=%s" % ','.join([str(v) for v in r['annotedata']])
                        }
                        ,"within": {
                            "@type": "sc:Manifest"
                            ,"@id": d_manifest
                        }
                    }
                }
                d_context['resources'].append(d_annote)

    return d_context


# ========================================
# Test
# 2018-11
# ========================================

@auth.requires_login()
def test_elementset():
    """
    要素セットを返す（評価用）
    2018-11
    """
    # 引数の取得
    classid = 0
    if len(request.args) > 0:
        classid = request.args(0, cast=int) or 0

    return mr_datahelper.d__elementset(dbm, classid)


@auth.requires_login()
def test_datahelper():
    """
    データを返す（評価用）
    2018-11
    """
    # 引数の取得
    _id = 1
    if len(request.args) > 0:
        _id = request.args(0, cast=int) or 1

    #return mr_datahelper.l_d_sections(dbm, _id)
    return mr_datahelper.l_d_resources(dbm, _id)
    #return mr_datahelper.l_d_resource_items(dbm, _id)


# ========================================
# Side menu
# 2018-11
# ========================================

def __get_sidemenu(a_offset=0, a_limit=20):
    """
    Side menuを取得する
    2018-11
    """
    # サイドメニュー索引の作成
    dom_div = DIV(_class='m-leftside')

    ul = UL(_class='list-group list-group-flush m-collection')

    li = LI(A(T('Home'),_href=URL('index')),_class="list-group-item")
    ul.append(li)

    for r in dbm((dbm.mr_collections._id>1)&(dbm.mr_collections.mr_open==True)).select(orderby=[dbm.mr_collections.mr_order,dbm.mr_collections.id]):
        li = LI(A('%s' % r.mr_title,_href=URL('list',args=[str(r.id)])),_class="list-group-item")
        ul.append(li)

    for r in dbm((dbm.mr_pages._id>1)&(dbm.mr_pages.mr_open==True)).select(orderby=[dbm.mr_pages.mr_order,dbm.mr_pages.id]):
        li = LI(A('%s' % r.mr_title,_href=URL('index',args=[str(r.id)]),_class=""),_class="list-group-item")
        ul.append(li)

    dom_div.append(ul)

    return dom_div


# ========================================
# Thumbnail image
# 2018-11
# ========================================

def __get_dom_icon_manifest(d_dom):
    """
    Manifest link
    2018-11
    """
    a_manifest = ''

    if 'format' in d_dom:
        if 'tif' in d_dom['format'] or 'jp2' in d_dom['format']:
            a_manifest = SPAN(A(
                IMG(
                    _src=URL('static','mr_images/iiif_logo.png')
                    ,_width='32'
                )
                ,_href=URL('iiif',args=[d_dom['role'],d_dom['subid'],'manifest.json'])
                ,_target='_blank'
                ,_title='IIIF-Manifest'
                )
                ,_class="m-manifest-link")
        elif 'jpg' in d_dom['format'] or 'fzp' in d_dom['format'] or 'fzp3' in d_dom['format'] or 'dzi' in d_dom['format'] or 'zoomify' in d_dom['format']:
            a_manifest = SPAN(A(
                IMG(
                    _src=URL('static','mr_images/iiif_logo.png')
                    ,_width='32'
                )
                ,_href=URL('iiif',args=[d_dom['role'],d_dom['subid'],'manifest.json'])
                ,_target='_blank'
                ,_title='IIIF-Manifest'
                )
                ,_class="m-manifest-link")

    return a_manifest


def __get_dom_image_annotes(_session, d_d, _atypes=['area'], enable_link=False):
    """
    Annotation imageのHTMLを取得する
    2018-11
    """
    dom_file_ul = None

    if len(d_d['files']) > 0:

        dom_file_ul = UL(_class='m-items')

        for d_dom in d_d['files']:

            # Manifest link
            a_manifest = __get_dom_icon_manifest(d_dom)

            # Snap image
            img = IMG(_src='/%s/resources/%s' % (MR_CONF['APP_NAME'],'/'.join(d_dom['snapimage']))
                        ,_width=240
                        ,_alt=d_dom['title']
                        ,_class='m-snapimage'
                    )
            href = URL('viewer',args=d_dom['href'])
            if 'pdf' in d_dom['href']:
                href = '/%s/resources/%s.pdf' % (MR_CONF['IMAGE_NAME'],'/'.join(d_dom['href'][:-1]))

            _session.imgs[tuple(d_dom['href'])] = d_dom['ids']

            dom_file_li= LI(
                            DIV(
                                DIV(
                                    A(img
                                        ,_href=href
                                        ,_target='_blank'
                                        ,_title=d_dom['title']
                                    )
                                    ,a_manifest
                                    ,_label=d_dom['title']
                                    ,_class='m-image m-item-snapimage'
                                    ,**{'_data-id':d_dom['subid']}
                                )
                                ,P(
                                    d_dom['title']
                                )
                            )
                            ,_class='m-item m-item-wid240'
                        )

            dom_file_ul.append(dom_file_li)

    if len(d_d['annotes']) > 0:

        for d_dom in d_d['annotes']:

            # 対象のアノテーション形式（デフォルト'area'）以外はスルー（出力抑制）
            if d_dom['annotetype'] not in _atypes: continue

            # Manifest link
            a_manifest = __get_dom_icon_manifest(d_dom)

            # Snap image
            if 'tif' in d_dom['format'] or 'jp2' in d_dom['format']:
                img_src = '%s%s.%s/%s/240,/%d/default.jpg' % (MR_CONF['IIIF_SERVER_URL'],'%2F'.join(d_dom['href'][:-1]),d_dom['format'],','.join([str(v) for v in d_dom['annotearea']]),int(d_dom['annoterotation']))
            elif 'jpg' in d_dom['format']:
                img_src='/%s/resources/%s' % (MR_CONF['APP_NAME'],'/'.join(d_dom['thumbnail']))
            else:
                continue

            img = IMG(_src=img_src
                        ,_alt=d_dom['title']
                        ,_class='m-thumbnail'
                    )
            href = URL('viewer',args=d_dom['href'])
            var_args = d_dom['href']
            if 'tif' in d_dom['format'] or 'jp2' in d_dom['format']:
                manifest_id = "iiif/annote/%s/manifest.json" % (str(d_dom['subid']))
                var_args = ['iiif','annote',str(d_dom['subid']),'manifest.json']
                href = URL('viewer',args=[manifest_id])

            if enable_link:
                img_obj = A(img
                            ,_href=href
                            ,_target='_blank'
                            ,_title=d_dom['title']
                        )
            else:
                img_obj = img

            _session.imgs[tuple(var_args)] = d_dom['ids']

            dom_file_li= LI(
                            DIV(
                                DIV(img_obj
                                    ,a_manifest
                                    ,_label=d_dom['title']
                                    ,_class='m-image m-item-thumbnail'
                                    ,**{'_data-id':d_dom['subid']}
                                )
                                ,P(
                                    d_dom['title']
                                )
                            )
                            ,_class='m-item m-item-wid240'
                        )

            dom_file_ul.append(dom_file_li)

    return dom_file_ul or ''


def __get_dom_thumbnail_annotes(_session, d_d, enable_link=False):
    """
    Annotation imageのHTMLを取得する
    2018-11
    """
    dom_file_ul = None

    if len(d_d['annotes']) > 0:

        dom_file_ul = UL(_class='m-items')

        for d_dom in d_d['annotes']:

            # Manifest link
            a_manifest = __get_dom_icon_manifest(d_dom)

            # Thumbnail image
            if 'tif' in d_dom['format'] or 'jp2' in d_dom['format']:
                img_src = '%s%s.%s/%s/%d,/%d/default.jpg' % (MR_CONF['IIIF_SERVER_URL'],'%2F'.join(d_dom['href'][:-1]),d_dom['format'],','.join([str(v) for v in d_dom['annotearea']]),MR_CONF['IMG_THUMBNAIL_SIZE'],int(d_dom['annoterotation']))
            elif 'jpg' in d_dom['format']:
                img_src='/%s/resources/%s' % (MR_CONF['APP_NAME'],'/'.join(d_dom['thumbnail']))

            img = IMG(_src=img_src
                        ,_alt=d_dom['title']
                        ,_class='m-thumbnail'
                    )
            href = URL('viewer',args=d_dom['href'])
            var_args = d_dom['href']
            if 'tif' in d_dom['format'] or 'jp2' in d_dom['format']:
                manifest_id = "iiif/annote/%s/manifest.json" % (str(d_dom['subid']))
                var_args = ['iiif','annote',str(d_dom['subid']),'manifest.json']
                href = URL('viewer',args=[manifest_id])

            if enable_link:
                img_obj = A(img
                            ,_href=href
                            ,_target='_blank'
                            ,_title=d_dom['title']
                        )
            else:
                img_obj = img

            _session.imgs[tuple(var_args)] = d_dom['ids']

            dom_file_li= LI(
                            DIV(
                                DIV(img_obj
                                    ,a_manifest
                                    ,_label=d_dom['title']
                                    ,_class='m-image m-item-thumbnail'
                                    ,**{'_data-id':d_dom['subid']}
                                )
                                ,P(
                                    d_dom['title']
                                )
                            )
                            ,_class='m-item m-item-wid240'
                        )

            dom_file_ul.append(dom_file_li)

    return dom_file_ul or ''


def __get_dom_thumbnail_files(_session, d_d):
    """
    Thumbnail imageのHTMLを取得する
    2018-11
    """
    dom_file_ul = None

    if len(d_d['files']) > 0:

        dom_file_ul = UL(_class='m-items')

        for d_dom in d_d['files']:

            # Thumbnail image
            img = IMG(_src='/%s/resources/%s' % (MR_CONF['APP_NAME'],'/'.join(d_dom['thumbnail']))
                        ,_alt=d_dom['title']
                        ,_class='m-thumbnail'
                    )

            href = URL('item',args=[d_d['subid']])

            _session.imgs[tuple(d_dom['href'])] = d_dom['ids']

            dom_file_li= LI(
                            DIV(
                                DIV(
                                    A(img
                                        ,_href=href
                                    )
                                    ,_label=d_dom['title']
                                    ,_class='m-image m-item-thumbnail'
                                    ,**{'_data-id':d_dom['subid']}
                                )
                                ,P(
                                    d_dom['title']
                                )
                            )
                            ,_class='m-item m-item-wid160'
                        )

            dom_file_ul.append(dom_file_li)

    return dom_file_ul or ''


def __get_dom_snapimage_files(_session, d_d):
    """
    Snap imageのHTMLを取得する
    2018-11
    """
    dom_file_ul = None

    if len(d_d['files']) > 0:

        dom_file_ul = UL(_class='m-items')

        for d_dom in d_d['files']:

            # Manifest link
            a_manifest = __get_dom_icon_manifest(d_dom)

            # Snap image
            img = IMG(_src='/%s/resources/%s' % (MR_CONF['APP_NAME'],'/'.join(d_dom['snapimage']))
                        ,_alt=d_dom['title']
                        ,_width='320'
                        ,_class='m-snapimage'
                    )

            href = URL('viewer',args=d_dom['href'])
            a_title = T('click to open viewer')

            if 'pdf' in d_dom['href']:
                href = '/%s/resources/%s.pdf' % (MR_CONF['IMAGE_NAME'],'/'.join(d_dom['href'][:-1]))
                a_title = T('click to open pdf')

            _session.imgs[tuple(d_dom['href'])] = d_dom['ids']

            dom_file_li= LI(
                            DIV(
                                DIV(
                                    A(img
                                        ,_href=href
                                        ,_target='_blank'
                                        ,_title=a_title
                                    )
                                    ,a_manifest
                                    ,_label=d_dom['title']
                                    ,_class='m-image m-item-snapimage'
                                    ,**{'_data-id':d_dom['subid']}
                                )
                                ,P(
                                    d_dom['title']
                                )
                            )
                            ,_class='m-item m-item-wid320'
                        )

            dom_file_ul.append(dom_file_li)

    return dom_file_ul or ''


# ========================================
# Editor
# 2018-11
# ========================================

def __mr_on_validation(form):
    """
    編集後処理
    2018-11
    """
    pass


def __mr_on_create_class(form):
    """
    クラス作成後処理
    2018-11
    """
    if 'mr_class_id' in form.vars:
        return

    var_new_id = form.vars.id

    dbm.mr_class_elements.insert(mr_class_id = var_new_id
                                ,mr_definition_name = 'title'
                                ,mr_title = 'Title'
                                ,mr_description = 'Title'
                                ,mr_open = True
                                )
    dbm.mr_class_elements.insert(mr_class_id = var_new_id
                                ,mr_definition_name = 'description'
                                ,mr_title = 'Description'
                                ,mr_description = 'Description'
                                ,mr_open = True
                                )

    if ('mr_target' in form.vars) and (form.vars.mr_target == 'page'):

        dbm.mr_class_elements.insert(mr_class_id = var_new_id
                                ,mr_definition_name = 'creator'
                                ,mr_title = 'Creator'
                                ,mr_description = 'You'
                                ,mr_open = True
                                )
        dbm.mr_class_elements.insert(mr_class_id = var_new_id
                                ,mr_definition_name = 'created'
                                ,mr_title = 'Created'
                                ,mr_description = '20xx/xx/xx'
                                ,mr_open = True
                                )

    dbm.commit()


def __mr_on_create_section(form):
    """
    セクション作成後処理
    2018-11
    """
    if 'mr_section_id' in form.vars:
        return

    var_new_id = form.vars.id

    # Page section
    var_class_id = form.vars.mr_class_id

    var_rows = dbm(
                    dbm.mr_class_elements.mr_class_id==var_class_id
                ).select(
                    orderby=[dbm.mr_class_elements.mr_order,dbm.mr_class_elements.id]
                )
    for var_row in var_rows:
        dbm.mr_section_items.insert(mr_section_id=var_new_id
                                    ,mr_class_element_id=var_row.id
                                    ,mr_text=XML('<!-- // %s -->' % var_row.mr_description)
                                    ,mr_order=var_row.mr_order
                                    ,mr_open=var_row.mr_open
                                    )
    dbm.commit()


def __mr_on_create_resource(form):
    """
    リソース作成後処理
    2018-11
    """
    if 'mr_resource_id' in form.vars:
        return

    var_new_id = form.vars.id

    # Resource
    var_class_id = form.vars.mr_class_id

    var_rows = dbm(
                    dbm.mr_class_elements.mr_class_id==var_class_id
                ).select(
                    orderby=[dbm.mr_class_elements.mr_order,dbm.mr_class_elements.id]
                )
    for var_row in var_rows:
        dbm.mr_resource_items.insert(mr_resource_id=var_new_id
                                    ,mr_class_element_id=var_row.id
                                    ,mr_text=XML('<!-- // %s -->' % var_row.mr_description)
                                    ,mr_order=var_row.mr_order
                                    ,mr_open=var_row.mr_open
                                    )
    dbm.commit()


@auth.requires_login()
def editcollections():
    """
    コレクション編集用
    2018-11
    """
    __precheck()

    var_name = 'collection'

    var_db = dbm.mr_collections
    var_fields = [dbm.mr_collections.id
                ,dbm.mr_collections.mr_parent_id
                ,dbm.mr_collections.mr_slug
                ,dbm.mr_collections.mr_title
                ,dbm.mr_collections.mr_description
                ,dbm.mr_collections.mr_order
                ,dbm.mr_collections.mr_open
                ]

    grid = SQLFORM.grid(
                var_db
                ,fields=var_fields
                ,paginate=20
                ,searchable=True
                ,editable=True
                ,details=True
                ,deletable=True
                ,create=True
                #,csv=False
                ,exportclasses=dict(
                    #csv=(ExporterCSV, 'CSV')
                    csv_with_hidden_cols=False #(ExporterCSV, 'CSV (hidden cols)')
                    #,json=(ExporterJSON, 'JSON')
                    ,xml=False #(ExporterXML, 'XML')
                    ,html=False #(ExporterHTML, 'HTML')
                    #,tsv=(ExporterTSV, 'TSV (Excel compatible)')
                    ,tsv_with_hidden_cols=False #(ExporterTSV, 'TSV (Excel compatible, hidden cols)')
                )
                ,ui='web2py'
                ,formstyle='table3cols'
                #,buttons_placement = 'right'
                ,user_signature=False
                )

    return dict(grid=grid,dbname=var_name)


@auth.requires_login()
def editpages():
    """
    ページ編集用
    2018-11
    """
    __precheck()

    var_name = 'page'

    var_db = dbm.mr_pages
    var_fields = [dbm.mr_pages.id
                ,dbm.mr_pages.mr_slug
                ,dbm.mr_pages.mr_title
                ,dbm.mr_pages.mr_text
                ,dbm.mr_pages.mr_order
                ,dbm.mr_pages.mr_open
                ]

    grid = SQLFORM.grid(
                var_db
                ,fields=var_fields
                ,paginate=20
                ,searchable=True
                ,editable=True
                ,details=True
                ,deletable=True
                ,create=True
                #,csv=False
                ,exportclasses=dict(
                    #csv=(ExporterCSV, 'CSV')
                    csv_with_hidden_cols=False #(ExporterCSV, 'CSV (hidden cols)')
                    #,json=(ExporterJSON, 'JSON')
                    ,xml=False #(ExporterXML, 'XML')
                    ,html=False #(ExporterHTML, 'HTML')
                    #,tsv=(ExporterTSV, 'TSV (Excel compatible)')
                    ,tsv_with_hidden_cols=False #(ExporterTSV, 'TSV (Excel compatible, hidden cols)')
                )
                ,ui='web2py'
                ,formstyle='table3cols'
                #,buttons_placement = 'right'
                ,user_signature=False
                )

    return dict(grid=grid,dbname=var_name)


@auth.requires_login()
def editfiles():
    """
    ファイル編集用
    2018-11
    """
    __precheck()

    var_name = 'file'

    var_db = dbm.mr_files
    var_fields = [dbm.mr_files.id
                ,dbm.mr_files.mr_resource_id
                ,dbm.mr_files.mr_title
                ,dbm.mr_files.mr_license
                ,dbm.mr_files.mr_original_file_name
                ,dbm.mr_files.mr_imw
                ,dbm.mr_files.mr_imh
                ,dbm.mr_files.mr_public_file_format
                ,dbm.mr_files.mr_order
                ,dbm.mr_files.mr_open
                ]

    grid = SQLFORM.grid(
                var_db
                ,fields=var_fields
                ,paginate=20
                ,searchable=True
                ,editable=True
                ,details=True
                ,deletable=True
                ,create=True
                #,csv=False
                ,exportclasses=dict(
                    #csv=(ExporterCSV, 'CSV')
                    csv_with_hidden_cols=False #(ExporterCSV, 'CSV (hidden cols)')
                    #,json=(ExporterJSON, 'JSON')
                    ,xml=False #(ExporterXML, 'XML')
                    ,html=False #(ExporterHTML, 'HTML')
                    #,tsv=(ExporterTSV, 'TSV (Excel compatible)')
                    ,tsv_with_hidden_cols=False #(ExporterTSV, 'TSV (Excel compatible, hidden cols)')
                )
                ,ui='web2py'
                ,formstyle='table3cols'
                #,buttons_placement = 'right'
                ,user_signature=False
                )

    return dict(grid=grid,dbname=var_name)


@auth.requires_login()
def editannotes():
    """
    アノテーション編集用
    2018-11
    """
    __precheck()

    var_name = 'annote'

    var_db = dbm.mr_annotes
    var_fields = [dbm.mr_annotes.id
                ,dbm.mr_annotes.mr_resource_id
                ,dbm.mr_annotes.mr_file_id
                ,dbm.mr_annotes.mr_slug
                ,dbm.mr_annotes.mr_text
                ,dbm.mr_annotes.mr_annote_type
                ,dbm.mr_annotes.mr_annote_area
                ,dbm.mr_annotes.mr_annote_data
                #,dbm.mr_annotes.mr_order
                ,dbm.mr_annotes.mr_open
                ]

    grid = SQLFORM.grid(
                var_db
                ,fields=var_fields
                ,paginate=20
                ,searchable=True
                ,editable=True
                ,details=True
                ,deletable=True
                ,create=True
                #,csv=False
                ,exportclasses=dict(
                    #csv=(ExporterCSV, 'CSV')
                    csv_with_hidden_cols=False #(ExporterCSV, 'CSV (hidden cols)')
                    #,json=(ExporterJSON, 'JSON')
                    ,xml=False #(ExporterXML, 'XML')
                    ,html=False #(ExporterHTML, 'HTML')
                    #,tsv=(ExporterTSV, 'TSV (Excel compatible)')
                    ,tsv_with_hidden_cols=False #(ExporterTSV, 'TSV (Excel compatible, hidden cols)')
                )
                ,ui='web2py'
                ,formstyle='table3cols'
                #,buttons_placement = 'right'
                ,user_signature=False
                )

    return dict(grid=grid,dbname=var_name)


@auth.requires_login()
def editclasses():
    """
    クラス編集用
    2018-11
    """
    __precheck()

    var_name = 'class'

    var_db = dbm.mr_classes
    var_links = ['mr_class_elements']
    var_fields = [dbm.mr_classes.id]

    grid = SQLFORM.smartgrid(
                var_db
                #,fields=var_fields
                ,linked_tables=var_links
                ,paginate=20
                ,searchable=True
                ,editable=True
                ,details=True
                ,deletable=True
                ,create=True
                #,csv=False
                ,exportclasses=dict(
                    #csv=(ExporterCSV, 'CSV')
                    csv_with_hidden_cols=False #(ExporterCSV, 'CSV (hidden cols)')
                    #,json=(ExporterJSON, 'JSON')
                    ,xml=False #(ExporterXML, 'XML')
                    ,html=False #(ExporterHTML, 'HTML')
                    #,tsv=(ExporterTSV, 'TSV (Excel compatible)')
                    ,tsv_with_hidden_cols=False #(ExporterTSV, 'TSV (Excel compatible, hidden cols)')
                )
                ,ui='web2py'
                ,formstyle='table3cols'
                #,buttons_placement = 'right'
                ,user_signature=False
                ,onvalidation=__mr_on_validation
                ,oncreate=__mr_on_create_class
            )

    return dict(grid=grid,dbname=var_name)


@auth.requires_login()
def editsections():
    """
    セクション編集用
    2018-11
    """
    __precheck()

    var_name = 'section'

    var_db = dbm.mr_sections
    var_links = ['mr_section_items']
    var_fields = [dbm.mr_sections.id]

    grid = SQLFORM.smartgrid(
                var_db
                #,fields=var_fields
                ,linked_tables=var_links
                ,paginate=20
                ,searchable=True
                ,editable=True
                ,details=True
                ,deletable=True
                ,create=True
                #,csv=False
                ,exportclasses=dict(
                    #csv=(ExporterCSV, 'CSV')
                    csv_with_hidden_cols=False #(ExporterCSV, 'CSV (hidden cols)')
                    #,json=(ExporterJSON, 'JSON')
                    ,xml=False #(ExporterXML, 'XML')
                    ,html=False #(ExporterHTML, 'HTML')
                    #,tsv=(ExporterTSV, 'TSV (Excel compatible)')
                    ,tsv_with_hidden_cols=False #(ExporterTSV, 'TSV (Excel compatible, hidden cols)')
                )
                ,ui='web2py'
                ,formstyle='table3cols'
                #,buttons_placement = 'right'
                ,user_signature=False
                ,onvalidation=__mr_on_validation
                ,oncreate=__mr_on_create_section
            )

    return dict(grid=grid,dbname=var_name)


@auth.requires_login()
def editresources():
    """
    リソース編集用
    2018-11
    """
    __precheck()

    var_name = 'resource'

    var_db = dbm.mr_resources
    var_links = ['mr_resource_items']
    var_fields = [dbm.mr_resources.id]

    grid = SQLFORM.smartgrid(
                var_db
                #,fields=var_fields
                ,linked_tables=var_links
                ,paginate=20
                ,searchable=True
                ,editable=True
                ,details=True
                ,deletable=True
                ,create=True
                #,csv=False
                ,exportclasses=dict(
                    #csv=(ExporterCSV, 'CSV')
                    csv_with_hidden_cols=False #(ExporterCSV, 'CSV (hidden cols)')
                    #,json=(ExporterJSON, 'JSON')
                    ,xml=False #(ExporterXML, 'XML')
                    ,html=False #(ExporterHTML, 'HTML')
                    #,tsv=(ExporterTSV, 'TSV (Excel compatible)')
                    ,tsv_with_hidden_cols=False #(ExporterTSV, 'TSV (Excel compatible, hidden cols)')
                )
                ,ui='web2py'
                ,formstyle='table3cols'
                #,buttons_placement = 'right'
                ,user_signature=False
                ,onvalidation=__mr_on_validation
                ,oncreate=__mr_on_create_resource
            )

    return dict(grid=grid,dbname=var_name)


# ========================================
# Uploader
# 2018-11
# ========================================

def __list_files(a_offset=0,a_limit=20):
    """
    最近のファイルのリスト
    2018-11
    """

    ITEM_LIMIT = 20

    rows = dbm(dbm.mr_files.mr_open==True
                ).select(
                    limitby=(a_offset,ITEM_LIMIT)
                    ,orderby=~dbm.mr_files.mr_create_date|~dbm.mr_files.id
                )
    count = len(rows)

    #pager = None
    #dom_pager = ''

    dom_ul=UL()

    if rows:

        for r in rows:

            # Metadata
            _title = r.mr_title
            if len(_title) == 0:
                _title = r.mr_original_file_name

            # Thumbnail
            img=XML('&nbsp;')
            if r.mr_ims is not None and '.jpg' in r.mr_ims:
                img = IMG(_src='/%s/resources/%s/%s' % (MR_CONF['APP_NAME'],r.mr_subfolder,r.mr_ims)
                        ,_alt=_title
                        ,_class='m-thumbnail'
                        )

            # Link
            a_href = URL('viewer',args=[r.mr_subfolder,r.mr_iml,r.mr_public_file_format])
            if r.mr_public_file_format == 'pdf':
                a_href = '/%s/resources/%s/%s.pdf' % (MR_CONF['IMAGE_NAME'],r.mr_subfolder, r.mr_iml)

            dom_li= LI(
                        DIV(
                            DIV(
                                img
                                ,_label=_title
                                ,_class='m-image-item-thumbnail col col-5'
                                ,**{'_data-id':r.id}
                            )
                            ,DIV(
                                TABLE(
                                    TR(TD(T('Title'),_class="m-label"),TD(_title))
                                    ,TR(TD(T('Identifier'),_class="m-label"),TD(r.mr_identifier))
                                    ,TR(TD(T('Size (px)'),_class="m-label"),TD('%d,%d' % (r.mr_imw,r.mr_imh)))
                                    ,TR(TD(T('Format'),_class="m-label"),TD(r.mr_public_file_format))
                                    ,TR(TD(T('Uploaded date'),_class="m-label"),TD(r.mr_create_date))
                                    ,TR(TD(T('Original'),_class="m-label"),TD(r.mr_original_file_name))
                                    ,TR(TD(T('File license'),_class="m-label"),TD(r.mr_license))
                                    ,TR(TD(T('Note'),_class="m-label"),TD(r.mr_note))
                                )
                                ,_class='col col-7'
                            )
                            ,_class='m-image-item row'
                        )
                        ,_class='m-item'
                    )
            dom_ul.append(dom_li)

    return dom_ul


@auth.requires_login()
def uploader():
    """
    File uploader
    2018-11
    """
    __precheck()

    # 一覧
    dom_ul = __list_files(0)

    return dict(title=T('Uploader'), dom=dom_ul)


@auth.requires_login()
def upload_callback():
    """
    Dropzoneのための関数
    2018-11
    """
    __precheck()

    res = None
    if 'file' in request.vars:

        f = request.vars.file

        f_name = sanitize(request.vars.get('name')) or f.filename
        f_identifier = sanitize(request.vars.get('ident',''))
        f_title = sanitize(request.vars.get('title',''))
        f_collection = int(request.vars.get('collection') or 0)
        f_rotate = int(request.vars.get('rotate') or 0)*90
        f_format = int(request.vars.get('format') or 0)
        f_license = int(request.vars.get('license') or 0)
        f_entry = int(request.vars.get('entry') or 0)

        f_ext = ''
        if f_name.lower().endswith('.jpg'): f_ext = 'jpg'
        elif f_name.lower().endswith('.pdf'): f_ext = 'pdf'
        elif f_name.lower().endswith('.tif'): f_ext = 'tif'
        elif f_name.lower().endswith('.bmp'): f_ext = 'bmp'
        elif f_name.lower().endswith('.png'): f_ext = 'png'
        else: return json.dumps(res, separators=(',', ':'))

        o_format = 'jpg'
        #if f_format == 1: o_format = 'pdf' # auto
        if f_format == 2: o_format = 'tif'
        if f_format == 3: o_format = 'jp2'
        if f_ext == 'pdf': o_format = 'pdf'

        a_license = MR_CONF['LICENSES'][f_license]

        #a_username = None
        a_date = datetime.datetime.now()
        #a_subfolder = a_date.strftime('%Y%m%d')

        r_id = 1

        a_id = dbm.mr_files.insert(
                            mr_resource_id = r_id
                            ,mr_original_file=dbm.mr_files.mr_original_file.store(f.file, f.filename)
                            ,mr_original_file_name=f_name
                            ,mr_original_file_format=f_ext
                            ,mr_identifier=f_identifier
                            ,mr_title=f_title
                            ,mr_license=a_license
                            ,mr_public_file_format=o_format
                            ,mr_rotation=f_rotate
                            ,mr_create_date=a_date
                            ,mr_open=True
                            )
        dbm.commit()

        a_file = dbm.mr_files[a_id]

        t_filenames = re.split('[\.\\/]',a_file.mr_original_file)
        #a_subfolder = t_filenames[0]+'.'+t_filenames[1]+'/'+t_filenames[2][0:2]
        a_subfolder = t_filenames[2][0:2]
        a_filename = t_filenames[2:-1]

        if f_ext in ('jpg','tif'):
            new_name = __make_resource(a_file, a_subfolder, o_format, f_rotate)
        elif f_ext in ('jp2'):
            new_name = __make_resource(a_file, a_subfolder, o_format, f_rotate)
        elif f_ext in ('pdf'):
            new_name = __copy_resource(a_file, a_subfolder, o_format)
        else:
            new_name = __copy_resource(a_file, a_subfolder)

        res = dict(files=[{
            "filename": f.filename
            ,"format": o_format
        }])

    return json.dumps(res, separators=(',', ':'))


def __convert(a_in, a_out, a_out_format):
    """
    Tiled Pyramid TIFF作成にはpyvipsが必要
    例：__convert('folder/sample.jpg','sample','tif')
    2018-11
    """
    try:
        # save as Tiled Pyramid TIFF
        tile_size = MR_CONF['IMG_TILE_SIZE']
        if (type(tile_size) is not int) or tile_size<128 or tile_size>512:
           tile_size = 256
        im = pyvips.Image.new_from_file(a_in)
        im.write_to_file('%s.tif' % a_out, Q=95, compression="jpeg", tile=True, pyramid=True, tile_width=tile_size)
    except:
        # save alternative as JPEG
        try:
            im = Image.open(a_in)
            im.save('%s.jpg' % a_out, 'jpeg', quality=95)
        except:
            pass


def __make_resource(a_file, store_subfolder='', make_format='jpg', image_rotate=0):
    """
    2018-11
    """
    t_filenames = a_file.mr_original_file.split('.')
    a_srcfolder = '%s.%s/%s' % (t_filenames[0],t_filenames[1],t_filenames[2][0:2])
    a_subfolder = t_filenames[2][0:2]
    a_newname = ( '%s.%s' % ('_'.join(t_filenames[2:-1]),make_format),
            '%s_%s.jpg' % ('_'.join(t_filenames[2:-1]),str(MR_CONF['IMG_SNAP_SIZE'])),
            '%s_%s.jpg' % ('_'.join(t_filenames[2:-1]),str(MR_CONF['IMG_THUMBNAIL_SIZE'])),
            '_'.join(t_filenames[2:-1]),
            make_format,)

    im = Image.open(os.path.join(MR_CONF['FILEBASE_PATH'], 'uploads', a_srcfolder, a_file.mr_original_file))
    if image_rotate>0:
        im = im.rotate(-image_rotate,False,True)
    (wid, hei) = im.size
    a_file.update_record(mr_imw=wid, mr_imh=hei, mr_subfolder=store_subfolder)

    # サブフォルダの作成
    a_path = (
            os.path.normpath(os.path.join(MR_CONF['RESOURCEBASE_PATH'], 'resources', store_subfolder)),
            os.path.normpath(os.path.join(MR_CONF['IMAGEBASE_PATH'], 'resources', store_subfolder)),
            os.path.normpath(os.path.join(MR_CONF['IMAGEBASE_PATH'], 'resources', store_subfolder)),
        )
    if not os.path.exists(a_path[0]): os.makedirs(a_path[0])
    if not os.path.exists(a_path[1]): os.makedirs(a_path[1])
    #if not os.path.exists(a_path[2]): os.makedirs(a_path[2])

    if make_format == 'tif':
        # Tiled Pyramid TIFF (IIIF/IIPSrv)
        # save to 'RESOURCEBASE_PATH'
        __convert(os.path.join(MR_CONF['FILEBASE_PATH'], 'uploads', a_srcfolder, a_file.mr_original_file), os.path.join(a_path[0], a_newname[3]), a_newname[4])
    elif make_format == 'jp2':
        # JPEG2000 (Experimental)
        # save to 'RESOURCEBASE_PATH'
        tile_size = MR_CONF['IMG_TILE_SIZE']
        if (type(tile_size) is not int) or tile_size<128 or tile_size>512:
           tile_size = 256
        im.save(os.path.join(a_path[0], '%s.%s' % (a_newname[3], a_newname[4])), 'jpeg2000', tile_size=(tile_size,tile_size))
    else:
        # JPEG (Default)
        # save to 'IMAGEBASE_PATH'
        im.save(os.path.join(a_path[1], a_newname[0]), 'jpeg', quality=95)
    a_file.update_record(mr_iml=a_newname[3], mr_imlw=wid, mr_imlh=hei)

    # リサイズ画像の作成 M,S
    im.thumbnail((MR_CONF['IMG_SNAP_SIZE'],MR_CONF['IMG_SNAP_SIZE']), Image.ANTIALIAS)
    (widM, heiM) = im.size
    im.save(os.path.join(a_path[1], a_newname[1]), 'jpeg')
    a_file.update_record(mr_imm=a_newname[1], mr_immw=widM, mr_immh=heiM)

    im.thumbnail((MR_CONF['IMG_THUMBNAIL_SIZE'],MR_CONF['IMG_THUMBNAIL_SIZE']), Image.ANTIALIAS)
    (widS, heiS) = im.size
    im.save(os.path.join(a_path[2], a_newname[2]), 'jpeg')
    a_file.update_record(mr_ims=a_newname[2], mr_imsw=widS, mr_imsh=heiS)

    return a_newname


def __getImageName(name, prefix='', postfix='', sep=''):
    """
    2018-11
    """
    return ''.join(prefix, sep, name[:name.rfind('.')], sep, postfix, name[name.rfind('.'):])


def __copy_resource(a_file, store_subfolder='', make_format='pdf'):
    """
    2018-11
    """
    import shutil

    filenames = a_file.mr_original_file.split('.')
    a_fileformat = filenames[-1].lower()
    #a_subfolder = filenames[0]+'.'+filenames[1]+'/'+filenames[2][0:2]
    a_subfolder = filenames[2][0:2]
    a_newname = ( '%s.%s' % ('_'.join(filenames[2:-1]), a_fileformat),
            '%s_%s.jpg' % ('_'.join(filenames[2:-1]),str(MR_CONF['IMG_SNAP_SIZE'])),
            '%s_%s.jpg' % ('_'.join(filenames[2:-1]),str(MR_CONF['IMG_THUMBNAIL_SIZE'])),
            '_'.join(filenames[2:-1]),
            make_format,)

    (filename, stream) = dbm.mr_files.mr_original_file.retrieve(a_file.mr_original_file)

    a_path = os.path.normpath(os.path.join(MR_CONF['IMAGEBASE_PATH'], 'resources', store_subfolder))
    if not os.path.exists(a_path): os.makedirs(a_path)

    b_path = os.path.join(a_path, a_newname[0])
    shutil.copyfileobj(stream, open(b_path,'wb'))

    a_file.update_record(mr_iml=a_newname[3], mr_subfolder=store_subfolder)

    if a_fileformat == 'pdf':
        # PDF
        # save to 'IMAGEBASE_PATH'
        im = pyvips.Image.new_from_file('%s[0]' % b_path)

        # リサイズ画像の作成 M,S
        th1 = im.thumbnail_image(MR_CONF['IMG_SNAP_SIZE'])
        th1.write_to_file(os.path.join(a_path, a_newname[1]), Q=75)
        a_file.update_record(mr_imm=a_newname[1], mr_immw=th1.width, mr_immh=th1.height)

        th2 = th1.thumbnail_image(MR_CONF['IMG_THUMBNAIL_SIZE'])
        th2.write_to_file(os.path.join(a_path, a_newname[2]), Q=75)
        a_file.update_record(mr_ims=a_newname[2], mr_imsw=th2.width, mr_imsh=th2.height)

    return a_newname
