/* -*- mode:objc; coding:utf-8; tab-width:8; c-basic-offset:2; indent-tabs-mode:nil -*- */
/*
  Copyright (c) 2004 MacUIM Project
  http://www.digital-genes.com/~yatsu/macuim/

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.
*/

#import "CandidateController.h"

static CandidateController *sharedController;

@implementation CandidateController
 
+ (CandidateController *)sharedController
{
  return sharedController;
}

/**
 * Initialize Cocoa
 */
- (id)init
{
  self = [super init];
  NSApplicationLoad();
  if (![NSBundle loadNibNamed:@"CandidateWindow" owner:self]) {
    NSLog(@"failed to load CandidateWindow nib");
  }
  sharedController = self;

  origSize = [window frame].size;
  candIndex = -1;

  //printf("origsize: width=%f height=%f\n",
  //       origSize.width, origSize.height);

  [window setFrame:NSMakeRect([window frame].origin.x,
                              [window frame].origin.y,
                              origSize.width, 37)
          display:NO];

  return self;
}

/**
 * Set a callback for a CandidateController
 */
- (void)setCallBack:(CallBackType)callBack
{
  _callBack = callBack;
}

/**
 * Show a CandidateController and make it activate
 */
- (void)showWindow:(int)qdX:(int)qdY:(int)height
{
  lineHeight = height;

  //printf("CandidateController::showWindow:\n");

  [window setAutodisplay:NO];

  //[table setHidden:YES];

  [self replaceWindow:qdX:qdY];

  //[table setHidden:NO];

  if (candIndex >= 0) {
    NSIndexSet *indexSet =
      [[NSIndexSet alloc] initWithIndex:candIndex];
    [table selectRowIndexes:indexSet byExtendingSelection:nil];
    [indexSet release];
  }

  if ([window isVisible] == NO) {
    [window makeFirstResponder:table];
    [window orderFront:nil];
    [window setLevel:NSFloatingWindowLevel];
    //[window setAlphaValue:0.8];
    //[[window contentView] setNeedsDisplay:YES];
  }

  [window setAutodisplay:YES];
  //[window setViewsNeedDisplay:YES];
}

/**
 * Hide a candidates-window
 */
- (void)hideWindow
{
  NSPoint origin = [window frame].origin;

  if ([window isVisible] == NO)
    return;

  //printf("CandidateController::hideWindow\n");

  [window orderOut:nil];

  [[table tableColumnWithIdentifier:@"candidate"] setWidth:36];
  [window setFrame:NSMakeRect(origin.x, origin.y,
                              origSize.width, 37)
          display:NO];
}

/**
 * Request the NSTableView to reload candidates
 */
- (void)reloadData
{
  [table reloadData];
}

/**
 * Initialize
 */
- (void)awakeFromNib
{
  headArray = [[NSMutableArray alloc] init];
  candArray = [[NSMutableArray alloc] init];
}

/**
 * Get a number of rows in the TableView
 */
- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
#if 0
  int i = 0;

  if ([candArray count] == 0)
    return 0;

  while (1) {
    NSString *str;
    if (i == [candArray count])
      break;
    str = [candArray objectAtIndex:i];
    if (!str || [str length] == 0)
      break;
    i++;
  }

  return i;
#else
  return [candArray count];
#endif
}

/**
 * Get data
 */
- (id)tableView:(NSTableView *)tableView
 objectValueForTableColumn:(NSTableColumn *)tableColumn
            row:(int)rowIndex
{
  id colID = [tableColumn identifier];

  if ([colID isEqual:@"head"])
    return [headArray objectAtIndex:rowIndex];
  else if ([colID isEqual:@"candidate"])
    return [candArray objectAtIndex:rowIndex];

  return nil;
}

- (void)tableView:(NSTableView *)tableView
  willDisplayCell:(id)cell
   forTableColumn:(NSTableColumn *)tableColumn
              row:(int)row
{
  [cell setDrawsBackground:YES];
  if (row % 2)
    [cell setBackgroundColor:[NSColor colorWithCalibratedWhite:0.95
                                      alpha:1.0]];
  else
    [cell setBackgroundColor:[NSColor whiteColor]];
}

- (UniCharPtr)getCandidate:(int)index
{
  return nil;
}

- (void)addCandidate:(UniCharPtr)head:(int)headLen
                    :(UniCharPtr)cand:(int)candLen
{
  NSString *headStr;
  NSString *candStr;

  if (head && headLen > 0)
    headStr = [[NSString alloc] initWithCharacters:head + headLen - 1
                                length:1];
  else
    headStr = [[NSString alloc] initWithString:@""];;

  if (cand && candLen > 0)
    candStr = [[NSString alloc] initWithCharacters:cand
                                length:candLen];
  else
    candStr = [[NSString alloc] initWithString:@""];;

  [headArray addObject:headStr];
  [candArray addObject:candStr];
}

/**
 * Clear candidates
 */
- (void)clearCandidate
{
  //printf("CandidateController::clearCandidate\n");

  [headArray removeAllObjects];
  [candArray removeAllObjects];
}

/**
 * Select a candidate
 */
- (void)selectCandidate:(int)index
{
  NSIndexSet *indexSet;

  indexSet = [[NSIndexSet alloc] initWithIndex:index];

  candIndex = index;

  [table selectRowIndexes:indexSet byExtendingSelection:nil];
  [table scrollRowToVisible:index];

  [indexSet release];
}

/**
 * deselect a candidate
 */
- (void)deselectCandidate
{
  [table deselectAll:nil];

  candIndex = -1;
}

/**
 * Set a page label
 */
- (void)setPage:(int)index:(int)max
{
  NSString *label;

  if (index > 0)
    label = [NSString stringWithFormat:@"%d / %d",
                      index, max];
  else
    label = [NSString stringWithFormat:@"- / %d", max];

  [textfield setStringValue:label];
}

/**
 * Button press action
 */
- (IBAction)candClicked:(id)sender
{
  //printf("CandidateController::candClicked()\n");

  [window orderOut:nil];

  (*_callBack)([sender clickedRow]);
}

- (void)replaceWindow:(int)replyX:(int)replyY
{
  NSTableColumn *col = [table tableColumnWithIdentifier:@"candidate"];
  NSSize mainSize = [[NSScreen mainScreen] frame].size;
  NSRect rect = [window frame];
  float columnWidth = 0.0;
  int x, y, i;

  for (i = 0; i < [candArray count]; i++) {
    NSMutableAttributedString *text =
      [[NSAttributedString alloc] initWithString:[candArray objectAtIndex:i]
                                  attributes:[NSDictionary dictionaryWithObjectsAndKeys:
                                                             [[col dataCell] font],
                                                           NSFontAttributeName,
                                                           nil]];
    if (columnWidth < [text size].width)
      columnWidth = [text size].width;
    [text release];
  }

  if (columnWidth > 700)
    columnWidth = 700;
  if (columnWidth + 4 > [col width])
    [col setWidth:columnWidth + 4];

  i = 0;
  while (1) {
    NSString *str;
    if (i == [candArray count])
      break;
    str = [candArray objectAtIndex:i];
    if (!str || [str length] == 0)
      break;
    i++;
  }


  if ([[window contentView] frame].size.height < 17 * i + 20) {
    [window setContentSize:NSMakeSize([table frame].size.width + 2,
                                      17 * i + 20)];
  }
  else if ([[window contentView] frame].size.width <
          [table frame].size.width + 2) {
    [window setContentSize:NSMakeSize([table frame].size.width + 2,
                                      [[window contentView] frame].size.height)];
  }

  //[window setContentSize:NSMakeSize([table frame].size.width + 2,
  //                                  17 * i + 20)];

  [table setFrameSize:NSMakeSize([table frame].size.width,
                                 17 * i)];

  rect = [window frame];

  x = replyX;
  y = mainSize.height - replyY - rect.size.height;

  if (x < 0)
    x = 0;
  if (x + rect.size.width > mainSize.width)
    x = mainSize.width - rect.size.width;
  if (mainSize.height - replyY - origSize.height < 0)
    y = mainSize.height - replyY + lineHeight + 3;
  if (y + rect.size.height > mainSize.height)
    y = mainSize.height - rect.size.height;

  //printf("CandidateController::replaceWindow: x=%d y=%d origin.x=%d origin.y=%d\n",
  //       x, y, (int) rect.size.width, (int) rect.size.height);

  if (x != (int) rect.origin.x || y != (int) rect.origin.y)
    [window setFrameOrigin:NSMakePoint(x, y)];

  /*
  [window setFrame:NSMakeRect(x, y,
                              columnWidth + 30,
                              17 * i + 38)
          display:NO];
  */
}

@end

/**
 * Carbon entry point and C-callable wrapper functions
 */
OSStatus
initializeBundle(OSStatus (*callBack)(int))
{
  CandidateController *candWin;
  NSAutoreleasePool *localPool;
    
  localPool = [[NSAutoreleasePool alloc] init];        
  candWin = [[CandidateController alloc] init];
  [candWin setCallBack:callBack];
  [localPool release];

  return noErr;
}

/**
 * Move candidates-window to front
 * This function called by a Carbon function.
 */
OSStatus
orderWindowFront(SInt16 inQDX, SInt16 inQDY, SInt16 inLineHeight)
{
  NSAutoreleasePool *localPool;
        
  localPool = [[NSAutoreleasePool alloc] init];        
  [[CandidateController sharedController] reloadData];
  [[CandidateController sharedController] showWindow:inQDX:inQDY:inLineHeight];
  [localPool release];

  return noErr;
}

/**
 * Move candidates-window to back
 * This function called by a Carbon function.
 */
OSStatus
orderWindowBack()
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CandidateController sharedController] hideWindow];
  [localPool release];

  return noErr;
}

/**
 * Get a candidate string
 * This function called by a Carbon function.
 */
UniCharPtr
getCandidate(UInt32 inIndex)
{
  NSAutoreleasePool *localPool;
  UniCharPtr str = nil;

  localPool = [[NSAutoreleasePool alloc] init];        
  str = [[CandidateController sharedController] getCandidate:inIndex];
  [localPool release];

  return str;
}

/**
 * Add a candidate
 * This function called by a Carbon function.
 */
OSStatus
addCandidate(UniCharPtr inHead, int inHeadLen,
             UniCharPtr inCand, int inCandLen)
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CandidateController sharedController]
    addCandidate:inHead:inHeadLen:inCand:inCandLen];
  [localPool release];

  return noErr;
}

/**
 * Clear candidates
 * This function called by a Carbon function.
 */
OSStatus
clearCandidate()
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CandidateController sharedController] clearCandidate];
  [localPool release];

  return noErr;
}

/**
 * Select a candidate
 * This function called by a Carbon function.
 */
OSStatus
selectCandidate(int inIndex)
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CandidateController sharedController] selectCandidate:inIndex];
  [localPool release];

  return noErr;
}

/**
 * deselect a candidate
 * This function called by a Carbon function.
 */
OSStatus
deselectCandidate(int inIndex)
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CandidateController sharedController] deselectCandidate];
  [localPool release];

  return noErr;
}

/**
 * Set a page label
 * This function called by a Carbon function.
 */
OSStatus
setPage(int inIndex, int inMax)
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CandidateController sharedController] setPage:inIndex:inMax];
  [localPool release];

  return noErr;
}
