/* -*- mode:c; coding:utf-8; tab-width:8; c-basic-offset:2; indent-tabs-mode:nil -*- */
/*
  Copyright (c) 2004 MacUIM Project
  http://www.digital-genes.com/~yatsu/macuim/

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.
*/

#define TARGET_API_MAC_CARBON 1

#include <Carbon/Carbon.h>

#include <sys/types.h>
#include <unistd.h>
#include <pwd.h>

#include "MUIM.h"
#include "MUIMInputEvents.h"
#include "MUIMScript.h"
#include "UIMCallback.h"
#include "CandidateCarbon.h"
#include "KeycodeToUKey.h"
#include "Preference.h"


#define ENABLE_HELPER  1

//#define DEBUG_IM  1


const short kMENU_Pencil = kBaseResourceID + 1;

const short kICNx_Pencil = kBaseResourceID + 1;

enum
{
  kShowHideKeyboardPaletteMenuItem = 1,
  kShowHideSendEventPaletteMenuItem = 2,
  kConvertToLowercaseMenuItem = 4,
  kConvertToUppercaseMenuItem = 5
};

enum
{
  kShowHideKeyboardPaletteMenuCommand = 'SHKP',
  kShowHideSendEventPaletteMenuCommand = 'SHDP',
  kConvertToLowercaseMenuCommand = 'CLOW',
  kConvertToUppercaseMenuCommand = 'CUPP'
};

const short kSTRx_MenuItems = kBaseResourceID + 1;

enum
{
  kShowKeyboardPaletteMenuItemString = 1,
  kHideKeyboardPaletteMenuItemString = 2,
  kShowSendEventPaletteMenuItemString = 3,
  kHideSendEventPaletteMenuItemString = 4
};

MUIMSessionHandle gActiveSession;
Boolean gDisableFocusedContext = FALSE;

int gUimFD;
CFSocketRef gUimSock;
CFRunLoopSourceRef gUimRun;
CFSocketContext gSockContext;

static MenuRef gPencilMenu;

static CFStringRef gShowKeyboardPaletteMenuItemString;
static CFStringRef gHideKeyboardPaletteMenuItemString;
static CFStringRef gShowSendEventPaletteMenuItemString;
static CFStringRef gHideSendEventPaletteMenuItemString;

static pascal OSStatus MUIMPencilMenuEventHandler(EventHandlerCallRef
                                                 inEventHandlerCallRef,
                                                 EventRef inEventRef,
                                                 void *inUserData);

static int ConvertKeyVal(int inKey);

static int ConvertModifier(int inMod);


ComponentResult
MUIMInitialize(ComponentInstance inComponentInstance,
               MenuRef *outTextServiceMenu)
{
  ComponentResult result;
  short refNum;
  EventTypeSpec menuEventSpec;
  Handle iconData = NULL;
  Handle menuIconSuite;
  Str255 menuText;

  result = noErr;
  refNum = -1;

  gActiveSession = NULL;
  gPencilMenu = NULL;
  gDisableFocusedContext = FALSE;

  refNum = OpenComponentResFile((Component) inComponentInstance);
  result = ResError();
  if ((result == noErr) && (refNum == -1))
    result = resFNotFound;

  if (result == noErr) {
    gPencilMenu = GetMenu(kMENU_Pencil);
    if (gPencilMenu)
      *outTextServiceMenu = gPencilMenu;
    else
      result = resNotFound;
  }

  if (result == noErr)
    ChangeMenuAttributes(gPencilMenu, kMenuAttrUsePencilGlyph, 0);

  if (result == noErr) {
    menuEventSpec.eventClass = kEventClassCommand;
    menuEventSpec.eventKind = kEventProcessCommand;
    result = InstallMenuEventHandler(gPencilMenu,
                                     NewEventHandlerUPP
                                     (MUIMPencilMenuEventHandler), 1,
                                     &menuEventSpec, nil, nil);
  }

  if (result == noErr)
    result = NewIconSuite(&menuIconSuite);
  if (result == noErr) {
    iconData = GetResource('ics8', kICNx_Pencil);
    if (iconData == nil)
      result = resNotFound;
    else
      DetachResource(iconData);
  }
  if (result == noErr)
    result = AddIconToSuite(iconData, menuIconSuite, 'ics8');
  if (result == noErr) {
    iconData = GetResource('ics4', kICNx_Pencil);
    if (iconData == nil)
      result = resNotFound;
    else
      DetachResource(iconData);
  }
  if (result == noErr)
    result = AddIconToSuite(iconData, menuIconSuite, 'ics4');
  if (result == noErr) {
    iconData = GetResource('ics#', kICNx_Pencil);
    if (iconData == nil)
      result = resNotFound;
    else
      DetachResource(iconData);
  }
  if (result == noErr)
    result = AddIconToSuite(iconData, menuIconSuite, 'ics#');

  if (result == noErr) {
    menuText[0] = 5;
    menuText[1] = 1;
    *(Handle *) (&menuText[2]) = menuIconSuite;
    SetMenuTitle(gPencilMenu, menuText);
  }

  if (result == noErr) {
    GetIndString(menuText, kSTRx_MenuItems,
                 kShowKeyboardPaletteMenuItemString);
    gShowKeyboardPaletteMenuItemString =
      CFStringCreateWithPascalString(NULL, menuText,
                                     kTextEncodingMacRoman);
    GetIndString(menuText, kSTRx_MenuItems,
                 kHideKeyboardPaletteMenuItemString);
    gHideKeyboardPaletteMenuItemString =
      CFStringCreateWithPascalString(NULL, menuText,
                                     kTextEncodingMacRoman);
    GetIndString(menuText, kSTRx_MenuItems,
                 kShowSendEventPaletteMenuItemString);
    gShowSendEventPaletteMenuItemString =
      CFStringCreateWithPascalString(NULL, menuText,
                                     kTextEncodingMacRoman);
    GetIndString(menuText, kSTRx_MenuItems,
                 kHideSendEventPaletteMenuItemString);
    gHideSendEventPaletteMenuItemString =
      CFStringCreateWithPascalString(NULL, menuText,
                                     kTextEncodingMacRoman);
  }

  if (refNum != -1)
    CloseComponentResFile(refNum);

  /*
  if (result == noErr)
    result = MUIMInitializeMessageReceiving();
  */

  DEBUG_PRINT("MUIMInitialize()\n");

  {
    char *home = getenv("HOME");
    if (home == NULL) {
      struct passwd *pw = NULL;
      uid_t uid = getuid();
      pw = getpwuid(uid);
      if (pw == NULL)
        setenv("HOME", "/", 0);
      else
        setenv("HOME", pw->pw_dir, 0);
    }
  }

  // for Uim debug
  //setenv("LIBUIM_VERBOSE", "5", 1);

  uim_init();

  gUimFD = -1;
  gUimSock = NULL;
  gUimRun = NULL;

  return result;
}

void
MUIMTerminate(ComponentInstance inComponentInstance)
{
    DEBUG_PRINT("MUIMTerminate()\n");

#ifdef ENABLE_HELPER
    UIMHelperClose();
#endif

    uim_quit();

    gActiveSession = NULL;
    gDisableFocusedContext = FALSE;
    gPencilMenu = NULL;
}

#pragma mark -

ComponentResult
MUIMSessionOpen(ComponentInstance inComponentInstance,
                MUIMSessionHandle *outSessionHandle)
{
  ComponentResult result = noErr;
  CFStringRef appID = CFSTR(kAppID);
  CFPropertyListRef imVal;
  const char *imName;

#ifdef DEBUG_IM
  UInt32 i;
#endif

  // Load the value
  imVal = CFPreferencesCopyAppValue(CFSTR("InputMethod"), appID);
  if (imVal && CFGetTypeID(imVal) == CFStringGetTypeID())
    imName = CFStringGetCStringPtr((CFStringRef) imVal, kCFStringEncodingASCII);
  else
    imName = kDefaultIM;

  DEBUG_PRINT("MUIMSessionOpen() imName='%s'\n", imName);

  if (*outSessionHandle == nil) {
    *outSessionHandle =
      (MUIMSessionHandle) NewHandle(sizeof(MUIMSessionRecord));
    DEBUG_PRINT("MUIMSessionOpen() NewHandle=%p\n", *outSessionHandle);
  }

  if (*outSessionHandle) {
    (**outSessionHandle)->fComponentInstance = inComponentInstance;
    (**outSessionHandle)->fLastUpdateLength = 0;
    (**outSessionHandle)->fSegments = NULL;
    (**outSessionHandle)->fSegmentCount = 0;

    (**outSessionHandle)->fUC =
      uim_create_context(*outSessionHandle, "UTF-8", NULL, imName,
                         uim_iconv, UIMCommitString);

    if (!((**outSessionHandle)->fUC))
      (**outSessionHandle)->fNumIM = 0;
    else
      (**outSessionHandle)->fNumIM = uim_get_nr_im((**outSessionHandle)->fUC);

    if ((**outSessionHandle)->fUC == NULL) {
      DEBUG_PRINT("uim_create_context() failed\n");
      result = memFullErr;
    }

#ifdef DEBUG_IM
    for (i = 0; i < (**outSessionHandle)->fNumIM; i++) {
      DEBUG_PRINT("MUIMSessionOpen() IM %lu: '%s' '%s'\n",
                  uim_get_im_name((**outSessionHandle)->fUC, i),
                  uim_get_im_language((**outSessionHandle)->fUC, i));
    }
#endif

#ifdef ENABLE_HELPER
    UIMCheckHelper(*outSessionHandle);
#endif

    uim_set_preedit_cb((**outSessionHandle)->fUC,
                       UIMPreeditClear,
                       UIMPreeditPushback,
                       UIMPreeditUpdate);
#ifdef ENABLE_HELPER
    uim_set_prop_list_update_cb((**outSessionHandle)->fUC,
                                UIMUpdatePropList);
    uim_set_prop_label_update_cb((**outSessionHandle)->fUC,
                                 UIMUpdatePropLabel);

    uim_prop_list_update((**outSessionHandle)->fUC);
#endif
    uim_set_candidate_selector_cb((**outSessionHandle)->fUC,
                                  UIMCandAcivate,
                                  UIMCandSelect,
                                  UIMCandShiftPage,
                                  UIMCandDeactivate);

    (**outSessionHandle)->fMode =
      uim_get_current_mode((**outSessionHandle)->fUC);

    (**outSessionHandle)->fLastUpdateLength = 0;
    (**outSessionHandle)->fFixBuffer = NULL;
    (**outSessionHandle)->fFixLen = 0;
    (**outSessionHandle)->fOldPreedit = NULL;
    (**outSessionHandle)->fOldPreeditLen = 0;

    (**outSessionHandle)->fBundleRef = NULL;

    (**outSessionHandle)->fWindowOpened = false;
  }
  else
    result = memFullErr;

  return result;
}

void
MUIMSessionClose(MUIMSessionHandle inSessionHandle)
{
  UInt32 i;

  DEBUG_PRINT("MUIMSessionClose() inSessionHandle=%p\n",
              inSessionHandle);

  if (inSessionHandle) {
    for (i = 0; i < (*inSessionHandle)->fSegmentCount; i++)
      free((*inSessionHandle)->fSegments[i].fBuffer);

    free((*inSessionHandle)->fSegments);
    (*inSessionHandle)->fSegments = NULL;
    (*inSessionHandle)->fSegmentCount = 0;

    DisposeHandle((Handle) inSessionHandle);
  }
}

ComponentResult
MUIMSessionActivate(MUIMSessionHandle inSessionHandle)
{
  OSStatus result = noErr;

  gActiveSession = inSessionHandle;
  gDisableFocusedContext = FALSE;

  long keyboardID = GetScriptVariable(GetScriptManagerVariable(smLastScript),
                                      smScriptKeys);
  SetScriptVariable (smJapanese, smScriptKeys, keyboardID);

  DEBUG_PRINT("MUIMSessionActivate() gActiveSession=%p\n",
              gActiveSession);

#ifdef ENABLE_HELPER
  UIMCheckHelper(inSessionHandle);

  uim_prop_list_update((*inSessionHandle)->fUC);
  uim_prop_label_update((*inSessionHandle)->fUC);

  uim_helper_client_focus_in((*inSessionHandle)->fUC);
#endif

  return result;
}

ComponentResult
MUIMSessionDeactivate(MUIMSessionHandle inSessionHandle)
{
  DEBUG_PRINT("MUIMSessionDeactivate() gActiveSession=%p\n",
              gActiveSession);

#ifdef ENABLE_HELPER
  uim_helper_client_focus_out((*inSessionHandle)->fUC);
#endif

  /*
  if (inSessionHandle == gActiveSession) {
    gActiveSession = nil;
    gDisableFocusedContext = TRUE;
  }
  */

  return noErr;
}

ComponentResult
MUIMSessionEvent(MUIMSessionHandle inSessionHandle, EventRef inEventRef)
{
  Boolean handled;
  UInt32 eventClass;
  UInt32 eventKind;

  handled = FALSE;

  eventClass = GetEventClass(inEventRef);
  eventKind = GetEventKind(inEventRef);

  // kEventRawKeyDown = 1
  // kEventRawKeyRepeat = 2
  // kEventRawKeyUp = 3
  // kEventRawKeyModifiersChanged = 4
  // kEventHotKeyPressed = 5
  // kEventHotKeyReleased = 6

  //DEBUG_PRINT("MUIMSessionEvent() eventClass=%lu eventKind=%lu\n",
  //            eventClass, eventKind);

  // kEventClassKeyboard:
  //   kEventRawKeyDown - A key was pressed
  //   kEventRawKeyRepeat - Sent periodically as a key is held down by the user
  //   kEventRawKeyUp - A key was released
  //   kEventRawKeyModifiersChanged - The keyboard modifiers (bucky bits) have changed

  if (eventClass == kEventClassKeyboard &&
      (eventKind == kEventRawKeyDown ||
       eventKind == kEventRawKeyRepeat)) {
    UInt32 keyCode;
    unsigned char charCode;
    UInt32 modifiers;

    GetEventParameter(inEventRef, kEventParamKeyCode, typeUInt32, nil,
                      sizeof(keyCode), nil, &keyCode);

    GetEventParameter(inEventRef, kEventParamKeyMacCharCodes, typeChar, nil,
                      sizeof(charCode), nil, &charCode);

    GetEventParameter(inEventRef, kEventParamKeyModifiers, typeUInt32, nil,
                      sizeof(modifiers), nil, &modifiers);

    DEBUG_PRINT("MUIMSessionEvent() keycode=0x%lx, char=%c, charCode=0x%x, modifiers=0x%lx\n",
                keyCode, charCode, charCode, modifiers);

    if (!(modifiers & cmdKey))
       handled = MUIMHandleInput(inSessionHandle, keyCode, charCode, modifiers);
  }

  return handled;
}

/*
ComponentResult
MUIMSessionFix(MUIMSessionHandle inSessionHandle)
{
  ComponentResult result = noErr;

  DEBUG_PRINT("MUIMSessionFix()\n");

  return result;
}
*/

#pragma mark -

MUIMSessionHandle
MUIMGetActiveSession(void)
{
  DEBUG_PRINT("MUIMGetActiveSession()\n");

  return gActiveSession;
}

Boolean
MUIMHandleInput(MUIMSessionHandle inSessionHandle, UInt32 inKeycode,
                unsigned char inCharCode, UInt32 inModifiers)
{
  Boolean handled;
  Boolean isScriptKey;
  int key = 0, mod = 0;
  int rv;
  int i;

  handled = FALSE;
  isScriptKey = FALSE;

  DEBUG_PRINT("MUIMHandleInput() inCharCode=%02x inModifiers=%lx\n",
              inCharCode, inModifiers);

  /* Check for special keys first */
  for (i = 0; KeycodeToUKey[i].ukey; i++) {
    if (KeycodeToUKey[i].keycode == inKeycode) {
      key = KeycodeToUKey[i].ukey;
      break;
    }
  }
  if (key == UKey_Private1 || key == UKey_Private2)
    isScriptKey = TRUE;
  
  /* Then convert normal keys */
  if (key == 0)
    key = ConvertKeyVal(inCharCode);

  mod = ConvertModifier(inModifiers);

  DEBUG_PRINT("MUIMHandleInput() key=0x%x mod=0x%x\n", key, mod);

  rv = uim_press_key((*inSessionHandle)->fUC, key, mod);

  uim_release_key((*inSessionHandle)->fUC, key, mod);

  if (!rv || isScriptKey)
    handled = TRUE;

  DEBUG_PRINT("MUIMHandleInput() uim_press_key handled=%s\n",
              handled ? "true" : "false");

  return handled;
}

void
MUIMUpdateShowHideKeyboardPaletteMenuItem(Boolean inIsHidden)
{
  DEBUG_PRINT("MUIMUpdateShowHideKeyboardPaletteMenuItem() inIsHidden=%d\n",
              inIsHidden);

  if (inIsHidden)
    SetMenuItemTextWithCFString(gPencilMenu,
                                kShowHideKeyboardPaletteMenuItem,
                                gShowKeyboardPaletteMenuItemString);
  else
    SetMenuItemTextWithCFString(gPencilMenu,
                                kShowHideKeyboardPaletteMenuItem,
                                gHideKeyboardPaletteMenuItemString);
}

void
MUIMUpdateShowHideSendEventPaletteMenuItem(Boolean inIsHidden)
{
  DEBUG_PRINT("MUIMUpdateShowHideSendEventPaletteMenuItem() inIsHidden=%d\n",
              inIsHidden);

  if (inIsHidden)
    SetMenuItemTextWithCFString(gPencilMenu,
                                kShowHideSendEventPaletteMenuItem,
                                gShowSendEventPaletteMenuItemString);
  else
    SetMenuItemTextWithCFString(gPencilMenu,
                                kShowHideSendEventPaletteMenuItem,
                                gHideSendEventPaletteMenuItemString);
}

#pragma mark -

static pascal OSStatus
MUIMPencilMenuEventHandler(EventHandlerCallRef inEventHandlerCallRef,
                           EventRef inEventRef, void *inUserData)
{
  OSStatus result;
  HICommand command;

  DEBUG_PRINT("MUIMPencilMenuEventHandler()\n");

  result =
    GetEventParameter(inEventRef, kEventParamDirectObject, typeHICommand,
                      nil, sizeof(command), nil, &command);
  if (result == noErr) {
    switch (command.commandID) {

    case kShowHideKeyboardPaletteMenuCommand:
      break;

    case kShowHideSendEventPaletteMenuCommand:
      break;

    case kConvertToLowercaseMenuCommand:
      break;

    case kConvertToUppercaseMenuCommand:
      break;

    default:
      result = eventNotHandledErr;
      break;
    }
  }
  else
    result = eventNotHandledErr;
  return result;
}

static int
ConvertKeyVal(int inKey)
{
  DEBUG_PRINT("ConvertKeyVal() inKey=%02x\n", inKey);

  if (inKey >= 0x01 && inKey <= 0x1a)
    return inKey + 0x60;
  return inKey;
}

static int
ConvertModifier(int inMod)
{
  int modifier = 0;

  DEBUG_PRINT("ConvertModifier() inMod=%lx\n", inMod);

  if (inMod & shiftKey)
    modifier += UMod_Shift;
  if (inMod & controlKey)
    modifier += UMod_Control;

  return modifier;
}
