/* -*- mode:objc; coding:utf-8; tab-width:8; c-basic-offset:2; indent-tabs-mode:nil -*- */
/*
  Copyright (c) 2004 MacUIM Project
  http://www.digital-genes.com/~yatsu/macuim/

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.
*/

#import <sys/types.h>
#import <sys/socket.h>
#import "MacUIMHelperExtra.h"


static MacUIMHelperExtra *sharedExtra;


static void
uimDisconnect();


@implementation MacUIMHelperExtra

+ (MacUIMHelperExtra *)sharedExtra
{
  return sharedExtra;
}

- initWithBundle:(NSBundle *)bundle
{
  self = [super initWithBundle:bundle];
  if (!self)
    return nil;

  sharedExtra = self;

  menu = [[NSMenu alloc] initWithTitle:@""];
  //[menu setAutoenablesItems:NO];

  view = [[MacUIMHelperView alloc] initWithFrame:[[self view] frame]
                                   menuExtra:self];
  [self setView:view];

  uim_helper_client_get_prop_list();

  uimFD = uim_helper_init_client_fd(uimDisconnect);

  if (uimFD >= 0) {
    NSNotificationCenter *noc;

    uimHandle = [[NSFileHandle alloc]
                  initWithFileDescriptor:uimFD];

    [uimHandle waitForDataInBackgroundAndNotify];

    noc = [NSNotificationCenter defaultCenter];
    [noc addObserver:self
         selector:@selector(helperRead:)
         name:@"NSFileHandleDataAvailableNotification"
         object:uimHandle];
  }

  //label = [[NSMutableString alloc] initWithString:@"? "];
  labels = [[NSMutableArray alloc] init];

  clicked = NO;

  return self;
}

- (void)willUnload
{
  [super willUnload];
}

- (void)dealloc
{
  [view release];
  [menu release];

  [self helperClose];

  [uimHandle release];

  [super dealloc];
}

- (NSImage *)image
{
  return [self createImage:NO];
}

- (NSImage *)alternateImage
{
  return [self createImage:YES];
}

- (NSImage *)createImage:(BOOL)alter
{
  NSImage *image = nil;

  clicked = alter;

  if ([labels count] <= 1) {
    [view setFrameSize:NSMakeSize(kMenuBarWidth, [view frame].size.height)];
    [self setLength:kMenuBarWidth];
    image = [[[NSImage alloc] initWithSize:NSMakeSize(kMenuBarWidth, kMenuBarHeight)]
              autorelease];
  }
  else {
    [view setFrameSize:NSMakeSize(kMenuBarWidth2, [view frame].size.height)];
    [self setLength:kMenuBarWidth2];
    image = [[[NSImage alloc] initWithSize:NSMakeSize(kMenuBarWidth2, kMenuBarHeight)]
              autorelease];
  }

  [self renderFrame:image];
  [self renderText:image];

  return image;
}

- (void)renderFrame:(NSImage *)image
{
  NSBezierPath *framePath;
  NSColor *color;

  if (clicked == NO)
    color = [NSColor blackColor];
  else
    color = [NSColor whiteColor];

  [image lockFocus];
  [color set];

  if ([labels count] <= 1) {
    framePath =
      [NSBezierPath bezierPathWithRect:NSMakeRect(0.5, 3.5,
                                                  kMenuBarWidth - 1.0, kMenuBarHeight - 7.0)];
    [framePath stroke];
  }
  else {
    /*
    framePath =
      [NSBezierPath bezierPathWithRect:NSMakeRect(0.5, 3.5,
                                                  kMenuBarWidth2 / 2.0, kMenuBarHeight - 7.0)];
    [framePath stroke];

    framePath =
      [NSBezierPath bezierPathWithRect:NSMakeRect(kMenuBarWidth2 / 2.0, 3.5,
                                                  kMenuBarWidth2 / 2.0 - 0.5, kMenuBarHeight - 7.0)];
    [framePath stroke];
    */
    framePath =
      [NSBezierPath bezierPathWithRect:NSMakeRect(0.5, 3.5,
                                                  kMenuBarWidth2 - 1.0, kMenuBarHeight - 7.0)];
    [framePath stroke];

    framePath =
      [NSBezierPath bezierPathWithRect:NSMakeRect(kMenuBarWidth2 / 2.0, 3.5,
                                                  0.1, kMenuBarHeight - 7.0)];
    [framePath stroke];
  }

  [image unlockFocus];
}

- (void)renderText:(NSImage *)image
{
  NSMutableAttributedString *text;
  NSMutableString *label = nil;
  NSColor *color;
  int i;

  if (clicked == NO)
    color = [NSColor blackColor];
  else
    color = [NSColor whiteColor];

  [image lockFocus];

  [[NSGraphicsContext currentContext] setShouldAntialias:YES];

  if ([labels count] < 1)
    label = [[NSMutableString alloc] initWithString:@"?"];

  i = 0;
  do {
    if (!label)
      label = [[NSMutableString alloc] initWithString:[labels objectAtIndex:i]];

    //[label appendString:@" "];

    if ([labels count] <= 1) {
      text = [[NSAttributedString alloc] initWithString:label
                                         attributes:[NSDictionary dictionaryWithObjectsAndKeys:
                                                                    [NSFont systemFontOfSize:12],
                                                                  NSFontAttributeName,
                                                                  color,
                                                                  NSForegroundColorAttributeName,
                                                                  nil]];
      [text drawAtPoint:NSMakePoint(0.5 + 0.5 +
                                    (kMenuBarWidth - 1.0 -
                                     ceil([text size].width + 0.5)) / 2.0,
                                    (kMenuBarHeight - ceil([text size].height)) / 2.0)];
    }
    else {
      text = [[NSAttributedString alloc] initWithString:label
                                         attributes:[NSDictionary dictionaryWithObjectsAndKeys:
                                                                    [NSFont systemFontOfSize:11],
                                                                  NSFontAttributeName,
                                                                  color,
                                                                  NSForegroundColorAttributeName,
                                                                  nil]];
      [text drawAtPoint:NSMakePoint(0.5 + 0.5 +
                                    (i > 0 ? (kMenuBarWidth2 - 1.0) / 2.0 : 0)
                                    + (kMenuBarWidth2 / 2.0 - 0.5 -
                                     ceil([text size].width + 0.5)) / 2.0,
                                    (kMenuBarHeight - ceil([text size].height)) / 2.0)];
    }

    [text release];
    [label release];
    label = nil;

  } while (++i < [labels count]);

  [image unlockFocus];
}

- (NSMenu *)menu
{
  if ([menu numberOfItems] > 0) {
    while ([menu numberOfItems])
      [menu removeItemAtIndex:0];
  }

  [[menu addItemWithTitle:@"Open MacUIM Preferences"
         action:@selector(openSystemPrefs:) keyEquivalent:@""]
    setTarget:self];

  return menu;
}

- (void)openSystemPrefs:(id)sender
{
  [NSTask launchedTaskWithLaunchPath:@"/usr/bin/open"
          arguments:[NSArray arrayWithObject:@"/Library/PreferencePanes/MacUIM.prefPane"]];
}

- (void)helperRead:(NSNotification *)notification
{
  char *tmp;

  uim_helper_read_proc(uimFD);
  while ((tmp = uim_helper_get_message())) {
    [self helperParse:tmp];
    free(tmp);
  }

  if (uimFD >= 0)
    [uimHandle waitForDataInBackgroundAndNotify];
}

- (void)helperParse:(char *)str
{
  NSString *nsstr;
  NSArray *array;

  if (!str || strlen(str) == 0)
    return;

  nsstr = [[NSString alloc] initWithUTF8String:str];

  if (nsstr) {
    array = [nsstr componentsSeparatedByString:@"\n"];
    if (array && [array count] > 0) {
      NSString *first = [array objectAtIndex:0];
      if (first) {
        if ([first compare:@"prop_list_update"
                   options:NSCaseInsensitiveSearch
                   range:NSMakeRange(0, strlen("prop_list_update"))]
            == NSOrderedSame) {
          [self propListUpdate:array];
        }
        else if ([first compare:@"prop_label_update"
                        options:NSCaseInsensitiveSearch
                        range:NSMakeRange(0, strlen("prop_label_update"))]
                 == NSOrderedSame) {
          [self propLabelUpdate:array];
        }
      }
    }
    [nsstr release];
  }
}

- (void)helperClose
{
  if (uimFD >= 0)
    uim_helper_close_client_fd(uimFD);
}

- (void)helperDisconnect
{
  [[NSNotificationCenter defaultCenter] removeObserver:self];

  uimFD = -1;
}

- (void)propListUpdate:(NSArray *)lines
{
  //int i;

  if (!lines || [lines count] < 2)
    return;

  /*
  for (i = 0; i < [lines count]; i++)
    printf("propListUpdate: %d '%s'\n",
           i, [[lines objectAtIndex:i] UTF8String]);
  */
}

- (void)propLabelUpdate:(NSArray *)lines
{
  int i;
  NSString *line;
  NSArray *cols;
  NSString *charset = nil;

  if (!lines)
    return;

  line = [lines objectAtIndex:1];

  cols = [line componentsSeparatedByString:@"="];
  if (!cols || [cols count] < 2)
    return;

  charset = [cols objectAtIndex:1];

  //uim_helper_client_get_prop_list();

  [labels removeAllObjects];

  for (i = 2; i < [lines count]; i++) {
    line = [lines objectAtIndex:i];
    if (!line || [line compare:@""] == NSOrderedSame)
      break;

    /* XXX
    if (charset) {
      // Convert
    }
    */
    cols = [line componentsSeparatedByString:@"\t"];
    if (cols && [cols count] >= 2) {
      NSMutableString *label = [[NSMutableString alloc] initWithString:[cols objectAtIndex:0]];
      //printf("propLabelUpdate: label='%s'\n", [label UTF8String]);
      [labels addObject:label];
    }
  }

  [view setNeedsDisplay:YES];
}

@end

static void
uimDisconnect()
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[MacUIMHelperExtra sharedExtra] helperDisconnect];
  [localPool release];
}
