# coding: UTF-8

# Ruby 1.8.2 および Ruby 1.9 以上の両方をサポートするため、機能の吸収を行うモジュールです。
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: ruby_version_absorber.rb 570 2012-03-02 04:33:13Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)

if RUBY_VERSION < '1.9.0' then
	begin
		require 'rubygems'
	rescue LoadError
	end
	require 'kconv'

	# String クラスに擬似的な 1.9.0 互換の encoding および encode メソッドを追加します。
	# ただし、mail2weblog が必要とするレベルの互換性のみの保証で、完全な互換性は持ちません。
	class String
		@encoding = nil

		# エンコーディングを取得します。
		def encoding
			if @encoding != nil then
				return @encoding
			else
				case Kconv.guess(self)
				when Kconv::JIS
					return "ISO-2022-JP"
				when Kconv::SJIS
					return "Shift_JIS"
				when Kconv::EUC
					return "EUC-JP"
				when Kconv::ASCII
					return "ASCII"
				when Kconv::UTF8
					return "UTF-8"
				when Kconv::UTF16
					return "UTF-16BE"
				when Kconv::UNKNOWN
					return nil
				when Kconv::BINARY
					return nil
				else
					return nil
				end
			end
		end

		# エンコードを変更します。
		def encode(to_encoding, from_encoding = nil, options = nil)
			if (from_encoding == nil)
				if @encoding == nil then
					f_encoding = Kconv::AUTO
				else
					f_encoding = @encoding
				end
			else
				f_encoding = get_kconv_encoding(from_encoding)
			end

			t_encoding = get_kconv_encoding(to_encoding)
			if t_encoding != Kconv::UNKNOWN then
				result = Kconv::kconv(self, t_encoding, f_encoding)
			else
				result = self.clone
			end
			result.set_encoding(to_encoding)
			return result
		end

		def get_kconv_encoding(encoding)
			if encoding != nil then
				case encoding.upcase
				when "ISO-2022-JP"
					return Kconv::JIS
				when "SHIFT_JIS", "WINDOWS-31J"
					return Kconv::SJIS
				when "EUC-JP"
					return Kconv::EUC
				when "ASCII"
					return Kconv::ASCII
				when "UTF-8"
					return Kconv::UTF8
				when "UTF-16BE"
					return Kconv::UTF16
				else
					return Kconv::UNKNOWN
				end
			end
		end
		private :get_kconv_encoding

		def set_encoding(encoding)
			@encoding = encoding
		end
	end

	class Hash

		def assoc(key)
			each do |k, v|
				if k == key then
					return [k, v]
				end
			end

			return nil
		end

		def rassoc(value)
			each do |k, v|
				if v == value then
					return [k, v]
				end
			end

			return nil
		end

	end

end

def File::basename(filename, suffix = "")
	suffix = suffix.gsub('.', '\.').gsub('*', '[^\.]+')

	pre_encoding = nil
	if filename.encoding.to_s.upcase != 'UTF-8' then
		pre_encoding = filename.encoding
		filename = filename.encode('UTF-8')
	end

	if %r!^(.+/)?(.+?)(#{suffix})?$! =~ filename then
		result = $2
		if pre_encoding != nil then
			return result.encode(pre_encoding, 'UTF-8')
		else
			return result
		end
	else
		return ""
	end
end

def File::extname(filename)
	pre_encoding = nil
	if filename.encoding.to_s.upcase != 'UTF-8' then
		pre_encoding = filename.encoding
		filename = filename.encode('UTF-8')
	end

	if /^.+(\.[^\.]+)$/ =~ filename then
		result = $1
		if pre_encoding != nil then
			return result.encode(pre_encoding, 'UTF-8')
		else
			return result
		end
	else
		return ""
	end
end
