# coding: UTF-8

require 'base64'
require 'time'

require 'mailutils/mail'
require 'mailutils/mail_address'
require 'mailutils/mail_attachment'
require 'mailutils/mail_encoder'
require 'mailutils/mime'

#=メールの解析と送信用メールデータを作成するメールプロセッサー
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: mail_processor.rb 741 2012-06-17 06:48:41Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
#
# - メール種別（Content-Type）の対応状況
# -- サポート対象
# --- テキストメール（text/plain & text/html）をサポートします。
# --- 添付ファイル付きメール（multipart/mixed）をサポートします。
# --- 関連ファイル付きメール（multipart/related）をサポートします。
# --- text/plain と text/html の複合メール（multipart/alternative）をサポートします。
# --- レポートメール（multipart/report）をサポート（受け取り判別だけで中身は無視）します。
# -- 未サポート
# --- 絵文字入りメール
# --- その他、サポート対象以外
# -- 備考
# --- 画像入り text/html を作成する場合は img タグの src 属性に cid:content_id を自前で指定してください。
#
# - エンコードの対応状況
# -- タイトル／送信元／送信先は、Ｂ符号化およびＱ符号化の両方をサポートします。
# -- 本文は Content-Transfer-Encoding ヘッダーの値が 7bit/8bit/base64/quoted-printable のものをサポートします。
# -- 添付ファイルは Content-Transfer-Encoding ヘッダーの値が base64 のものをサポートします。
class MailProcessor

	# メールを解析します。
	def MailProcessor.parse(mail, to_encoding = nil)
		i = 0
		buffers = mail.split(/\r+\n|\r|\n/)

		# ヘッダーを解析する
		(header, line) = parse_header(buffers, 0)

		# 添付ファイルが無い場合
		if %r!text/plain! =~ header['content-type'] || nil == header['content-type'] then
			content = MailProcessor.parse_text_plain(header, line, buffers, to_encoding)
			(subject, from_address, to_addresses) = MailProcessor.get_mail_info(header, to_encoding)
			return Mail.new(subject, from_address, to_addresses, content, [])
		# 添付ファイルがある、もしくはHTML メールの場合
		elsif %r!multipart/(mixed|alternative|related);.*boundary="?(.+?)("| |$)!m =~ header['content-type'] then
			multipart_type = $1
			boundary = $2
			(main_header, content, attachments, html_header, html_content, related_attachments) = MailProcessor.parse_multipart(multipart_type, header, line, boundary, buffers, to_encoding)
			if main_header == nil && content == nil && html_content == nil then
				raise ArgumentError.new('サポートしていないメールを引数に指定しました（コンテンツを持っていないメール）。')
			end
			(subject, from_address, to_addresses) = MailProcessor.get_mail_info(header, to_encoding)
			return Mail.new(subject, from_address, to_addresses, content, attachments, html_content, related_attachments)
		elsif %r!multipart/report! =~ header['content-type'] then
			(subject, from_address, to_addresses) = MailProcessor.get_mail_info(header, to_encoding)
			result = Mail.new(subject, from_address, to_addresses, '')
			result.is_report = true
			return result
		end

		raise ArgumentError.new("サポートしていないメールを引数に指定しました（未対応な Content-Type[#{header['content-type']}] を持つメール）。")
	end

	# 送信用メールデータを作成します。
	def MailProcessor.make(mail, date = nil, to_header_encoding = "ISO-2022-JP", to_content_encoding = "UTF-8", mime = Mime.new)
		result = make_header(mail, date, to_header_encoding)

		if mail.attachments == nil || mail.attachments.size > 0 then
			result << MailProcessor.make_multipart_mixed(mail, to_header_encoding, to_content_encoding, mime)
		elsif mail.related_attachments == nil || mail.related_attachments.size > 0 then
			result << MailProcessor.make_multipart_related(mail, to_header_encoding, to_content_encoding, mime)
		elsif mail.has_content && mail.has_html_content then
			result << MailProcessor.make_multipart_alternative(mail, to_content_encoding, mime)
		elsif mail.has_content then
			result << MailProcessor.make_text_plain(mail, to_content_encoding)
		elsif mail.has_html_content then
			result << MailProcessor.make_text_html(mail, to_content_encoding)
		end

		return result
	end

	def MailProcessor.load_attachment(filename, content_id = nil)
		File.open(filename, 'rb') do |file|
			return MailAttachment.new(File.basename(filename), file.read, content_id)
		end
	end

	def MailProcessor.load_attachments(pattern, flags = 0)
		result = []
		Dir.glob(pattern, flags) do |filename|
			result.push(load_attachment(filename))
		end
		return result
	end

	def MailProcessor.decode_header(content, to_encoding = nil)
		content = MailProcessor.__normal_multiline_header(content)

		while %r!^(.*?)=\?(.+?)\?([bq])\?(.+?)\?=(.*?)$!i =~ content
			# ＢもしくはＱ符号化されている
			left = $1
			from_encoding = $2
			encoding_type = $3
			encoded_content = $4
			right = $5

			content = left

			decoded_content = ''
			if encoding_type.upcase == "B" then
				decoded_content = Base64::decode64(encoded_content)
			elsif encoding_type.upcase == "Q" then
				decoded_content = __encode(encoded_content.unpack("M")[0], to_encoding, from_encoding)
			end

			if to_encoding != nil then
				content << __encode(decoded_content, to_encoding, from_encoding).chomp
			else
				content << __encode(decoded_content, from_encoding, from_encoding).chomp
			end

			content << right
		end

		return content.gsub(/\r+\n|\r|\n/, '')
	end

	def MailProcessor.__normal_multiline_header(content)
		if content =~ /\n/ then
			contents = content.split(/\r+\n|\r|\n/)
			result = contents[0]
			contents.each_index do |i|
				if i != 0 then
					if contents[i - 1] =~ %r!=\?.+?\?[bq]\?.+?\?=$!i && contents[i] =~ %r!^[\t ]+?(=\?.+?\?[bq]\?.+?\?=.*)$!i then
						result << $1
					else
						result << contents[i]
					end
				end
			end
			return result
		else
			return content
		end
	end

	def MailProcessor.encode_header(content, to_encoding = "ISO-2022-JP")
		if content.encoding == nil || content.encoding.to_s == "ASCII" then
			return content
		else
			return "=?#{to_encoding}?B?#{Base64::encode64(__encode(content, to_encoding, content.encoding)).chomp}?="
		end
	end

	def MailProcessor.parse_header(buffers, start_line)
		header = {}

		i = start_line
		while i < buffers.size
			line = buffers[i].chomp

			if /^([a-zA-Z][a-zA-Z0-9\-]+): *(.+)$/ =~ line then
				header_name = $1.downcase
				header_value = $2
				if header.has_key?(header_name) then
					header[header_name] << header_value
				else
					header[header_name] = header_value
				end
			elsif line == '' then
				break
			else
				header[header_name] << "\n"
				header[header_name] << line
			end

			i = i + 1
		end

		return [header, i]
	end

	def MailProcessor.parse_text_plain(header, line, buffers, to_encoding)
		i = line + 1
		content = ''
		while i < buffers.size
			content = content + buffers[i] + "\n"

			i = i + 1
		end

		return MailProcessor.decode_content(header, content, to_encoding)
	end

	def MailProcessor.parse_multipart(multipart_type, header, line, boundary, buffers, to_encoding)
		main_header = nil
		main_content = nil
		attachments = []
		html_header = nil
		html_content = nil
		related_attachments = []

		i = line + 1
		while i < buffers.size
			if buffers[i] == "--#{boundary}--" then
				break
			elsif buffers[i] == "--#{boundary}" then
				if i + 1 == buffers.size || buffers[i + 1] == '' then
					break
				else
					(part_header, part_line) = MailProcessor.parse_header(buffers, i + 1)
					i = part_line + 1
					if %r!^ *([a-z]+/[a-z]+)! =~ part_header['content-type'] then
						type = $1
					end

					content = ''
					while i < buffers.size && /^--#{boundary}/ !~ buffers[i]
						if %r!text/.+! =~ type then
							content << buffers[i] + "\n"
						else
							content << buffers[i]
						end

						i = i + 1
					end

					case multipart_type
					when 'alternative'
						if %r!text/plain! =~ type then
							main_header = part_header
							main_content = MailProcessor.decode_content(part_header, content, to_encoding)
						elsif %r!text/html! =~ type then
							html_header = part_header
							html_content = MailProcessor.decode_content(part_header, content, to_encoding)
						end
					when 'mixed', 'related'
						if part_header['content-type'] != nil && %r!multipart/(.+);.+boundary="?(.+?)("| |$)!m =~ part_header['content-type'] then
							new_multipart_type = $1
							new_part_boundary = $2
							(multipart_header, multipart_content, multipart_attachments, multipart_html_header, multipart_html_content, multipart_related_attachments) =
								MailProcessor.parse_multipart(new_multipart_type, part_header, line, new_part_boundary, buffers, to_encoding)
							main_header = multipart_header if multipart_header != nil
							main_content = multipart_content if multipart_content != nil
							html_header = multipart_html_header if multipart_html_header != nil
							html_content = multipart_html_content if multipart_html_content != nil
							multipart_attachments.each do |attachment|
								attachments.push(attachment)
							end
							multipart_related_attachments.each do |attachment|
								related_attachments.push(attachment)
							end
						elsif main_content == nil && %r!text/plain! =~ type then
							main_header = part_header
							main_content = MailProcessor.decode_content(part_header, content, to_encoding)
						elsif html_content == nil && %r!text/html! =~ type then
							html_header = part_header
							html_content = MailProcessor.decode_content(part_header, content, to_encoding)
						else
							if part_header.has_key?('content-id') then
								related_attachments.push(MailAttachment.new(
									MailProcessor.decode_filename(part_header, to_encoding),
									MailProcessor.decode_filecontents(part_header, content),
									MailProcessor.decode_contentid(part_header, to_encoding)))
							else
								attachments.push(MailAttachment.new(
									MailProcessor.decode_filename(part_header, to_encoding),
									MailProcessor.decode_filecontents(part_header, content)))
							end
						end
					else
						# ここに来ることは想定していないが…来たとしたら添付ファイルとして扱う
						if part_header['content-type'] != nil && %r!multipart/(.+);.+boundary="?(.+?)("| |$)!m =~ part_header['content-type'] then
							multipart_type = $1
							part_boundary = $2
							(multipart_header, multipart_content, multipart_attachments) = MailProcessor.parse_multipart(multipart_type, part_header, line, part_boundary, buffers, to_encoding)
							multipart_attachments.each do |attachment|
								attachments.push(attachment)
							end
						end
					end
				end
			else
				i = i + 1
			end
		end

		return [main_header, main_content, attachments, html_header, html_content, related_attachments]
	end

	def MailProcessor.make_header(mail, date, to_header_encoding)
		result = ""

		# タイトル
		result << "Subject: #{MailProcessor.encode_header(mail.subject, to_header_encoding)}\n"

		# From
		result << "From: #{MailProcessor.make_address_content(mail.from_address, to_header_encoding)}\n"

		# To
		result << "To: #{MailProcessor.make_addresses_content(mail.to_addresses, to_header_encoding)}\n"

		# 時刻
		if date != nil then
			result << "Date: #{date.rfc2822}\n"
		end

		return result
	end

	def MailProcessor.make_basic_content(mail, to_content_encoding, mime, multipart_prefix = "MailUtils")
		result = ""

		if mail.has_content && mail.has_html_content then
			result << MailProcessor.make_multipart_alternative(mail, to_content_encoding, mime, multipart_prefix)
		elsif mail.has_content then
			result << MailProcessor.make_text_plain(mail, to_content_encoding)
		elsif mail.has_html_content
			result << MailProcessor.make_text_html(mail, to_content_encoding)
		end

		return result
	end

	def MailProcessor.make_text_plain(mail, to_content_encoding)
		result = ""

		result << %Q!Content-Type: text/plain; charset="#{to_content_encoding}";\n!
		result << "Content-Transfer-Encoding: base64\n"
		result << "\n"
		result << Base64::encode64(mail.content)
		result << "\n"

		return result
	end

	def MailProcessor.make_text_html(mail, to_content_encoding)
		result = ""

		content = ""
		content << mail.html_content

		result << %Q!Content-Type: text/html; charset="#{to_content_encoding}";\n!
		result << "Content-Transfer-Encoding: base64\n"
		result << "\n"
		result << Base64::encode64(content)
		result << "\n"

		return result
	end

	def MailProcessor.make_multipart_alternative(mail, to_content_encoding, mime, multipart_prefix = "MailUtils")
		result = ""

		boundary = "--#{multipart_prefix}Alternative"

		result << %Q!Content-Type: multipart/alternative; boundary="#{boundary}"\n!
		result << "\n"
		result << "--#{boundary}\n"

		result << MailProcessor.make_text_plain(mail, to_content_encoding)
		result << "--#{boundary}\n"

		result << MailProcessor.make_text_html(mail, to_content_encoding)
		result << "--#{boundary}--\n"
	end

	def MailProcessor.make_multipart_related(mail, to_header_encoding, to_content_encoding, mime, multipart_prefix = "MailUtils")
		result = ""

		boundary = "--#{multipart_prefix}Related"

		result << %Q!Content-Type: multipart/related; boundary="#{boundary}"\n!
		result << "\n"
		result << "--#{boundary}\n"

		result << make_basic_content(mail, to_content_encoding, mime, multipart_prefix)
		result << "--#{boundary}\n"

		# 関連ファイル
		mail.related_attachments.each_with_index do |related_attachment, i|
			result << make_related_attachment(related_attachment, to_header_encoding, mime)
			if i + 1 != mail.related_attachments.size then
				result << "--#{boundary}\n"
			else
				result << "--#{boundary}--\n"
			end
		end

		return result
	end

	def MailProcessor.make_multipart_mixed(mail, to_header_encoding, to_content_encoding, mime, multipart_prefix = "MailUtils")
		result = ""

		# 境界の定義
		# ※バウンダリの決定方針：本文は Base64 でエンコーディングしてしまうため境界にハイフンを入れた時点で問題ない
		boundary = "--#{multipart_prefix}Mixed"

		result << %Q!Content-Type: multipart/mixed; boundary="#{boundary}"\n!
		result << "\n"
		result << "--#{boundary}\n"

		if mail.related_attachments != nil && mail.related_attachments.size > 0 then
			result << MailProcessor.make_multipart_related(mail, to_header_encoding, to_content_encoding, mime)
		else
			# 本文
			result << make_basic_content(mail, to_content_encoding, mime, multipart_prefix)
		end
		result << "--#{boundary}\n"

		# 添付ファイル
		mail.attachments.each_with_index do |attachment, i|
			result << make_attachment(attachment, to_header_encoding, mime)
			if i + 1 != mail.attachments.size then
				result << "--#{boundary}\n"
			else
				result << "--#{boundary}--\n"
			end
		end

		return result
	end

	def MailProcessor.make_attachment(attachment, to_header_encoding, mime)
		result = ""

		result << %Q!Content-Type: #{mime.mime(attachment.filename)}; name="#{MailProcessor.decode_header(File.basename(attachment.filename), to_header_encoding)}"\n!
		result << "Content-Transfer-Encoding: base64\n"
		result << %Q!Content-Disposition: attachment; filename="#{MailProcessor.decode_header(attachment.filename, to_header_encoding)}"\n!
		result << "\n"
		result << Base64::encode64(attachment.content)
		result << "\n"

		return result
	end

	def MailProcessor.make_related_attachment(related_attachment, to_header_encoding, mime)
		result = ""

		result << %Q!Content-Type: #{mime.mime(related_attachment.filename)}; name="#{MailProcessor.decode_header(File.basename(related_attachment.filename), to_header_encoding)}"\n!
		result << "Content-Transfer-Encoding: base64\n"
		result << %Q!Content-ID: <#{related_attachment.content_id}>\n!
		result << "\n"
		result << Base64::encode64(related_attachment.content)
		result << "\n"

		return result
	end

	def MailProcessor.get_mail_info(header, to_encoding = nil)
		# Subject
		subject = MailProcessor.decode_header(header['subject'], to_encoding)

		# From
		from_address = MailProcessor.parse_mail_address(header['from'], to_encoding)

		# To
		to_addresses = []
		header['to'].split(',').each do |to|
			to_addresses.push(MailProcessor.parse_mail_address(to, to_encoding))
		end

		return [subject, from_address, to_addresses]
	end

	def MailProcessor.parse_mail_address(buffer, to_encoding = nil)
		if /^ *"?(.*?)"? *<(.+)> *$/ =~ buffer then
			return MailAddress.new($2, MailProcessor.decode_header($1, to_encoding))
		else
			return MailAddress.new(buffer)
		end
	end

	def MailProcessor.decode_content(header, content, to_encoding = nil)
		if %r!charset="?([0-9a-zA-Z_\-]+)"?! =~ header['content-type'] then
			from_encoding = $1
		else
			from_encoding = nil
		end

		if header['content-transfer-encoding'] != nil then
			case header['content-transfer-encoding'].downcase
			when '7bit'
			when '8bit'
			when 'base64'
				content = Base64::decode64(content)
			when 'quoted-printable'
				content = content.unpack('M')[0]
			else
				raise EncodingError.new('7bit/8bit/base64/quoted-printable 以外の Content-Transfer-Encoding を持つメールはサポートしていません。')
			end
		end

		if from_encoding != nil then
			if to_encoding != nil then
				return __encode(content, to_encoding, from_encoding).gsub(/\r+\n/, "\n")
			else
				return __encode(content, from_encoding, from_encoding).gsub(/\r+\n/, "\n")
			end
		else
			return content.gsub(/\r+\n/, "\n")
		end
	end

	def MailProcessor.decode_filename(header, to_encoding)
		if /name="(.+?)"/ =~ header['content-type'] then
			return MailProcessor.decode_header($1, to_encoding)
		else
			return nil
		end
	end

	def MailProcessor.decode_filecontents(header, contents)
		if header['content-transfer-encoding'] == nil || 'base64' != header['content-transfer-encoding'].downcase then
			raise EncodingError.new("base64 以外の Content-Transfer-Encoding を持つ添付ファイルはサポートしていません。(#{header['content-transfer-encoding']})")
		end

		return Base64::decode64(contents)
	end

	def MailProcessor.decode_contentid(header, to_encoding)
		if /^\s*<?(.+?)>?\s*$/ =~ header['content-id'] then
			return MailProcessor.decode_header($1, to_encoding)
		else
			return header['content-id']
		end
	end

	def MailProcessor.make_address_content(address, to_encoding = nil)
		if address.name != nil then
			if to_encoding != nil then
				return %Q!"#{MailProcessor.encode_header(address.name, to_encoding)}" <#{address.address}>!
			else
				return %Q!"#{MailProcessor.encode_header(address.name)}" <#{address.address}>!
			end
		else
			return address.address
		end
	end

	def MailProcessor.make_addresses_content(addresses, to_encoding = nil)
		result = ''

		i = 0
		while i < addresses.size
			if i != 0 then
				result << ', '
			end

			result << MailProcessor.make_address_content(addresses[i])

			i = i + 1
		end

		return result
	end

	def MailProcessor.make_address_list(addresses)
		result = []
		addresses.each do |address|
			result.push(address.address)
		end
		return result
	end

	def MailProcessor.__encode(content, to_encoding, from_encoding)
		if to_encoding.kind_of?(MailEncoder) then
			to_encoding.encode(content, from_encoding)
		else
			content.encode(to_encoding, from_encoding)
		end
	end

end
