# coding: UTF-8

require 'exception/m2w_timeout'
require 'mail_encoder/simple_mail_encoder'
require 'mailutils/mail_processor'

M2W_TRANSFORM_TARGET_COMMAND = [SimplePost_CommandPlugin, Post_CommandPlugin, Get_CommandPlugin, Edit_CommandPlugin, Delete_CommandPlugin]

#=メールを基にブログへのコマンドを発行するブログコマンド
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: m2w_command.rb 965 2013-04-12 01:44:25Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011 - 2013. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
#
# 発行するコマンドの種類は、メールタイトル（Subject ヘッダー）で判断します。
#
# 想定するメールタイトルのフォーマットは以下の通りです。
#
# header:コマンド名:パラメータ
#
# hedaer は、コンストラクタで指定した任意の文字列です。
# 本クラスは、メールタイトルが上記フォーマットにマッチしないメールを無視します。
#
# 該当するメールであることが分かった場合は、コマンド名に応じた処理を実行します。
# コマンドの詳細は、コマンドを実行する各クラスを参照してください。
#
# コマンドが成功したと認識した場合、本クラスはメールボックスから該当のメールを削除します。
class Mail2WeblogCommand

	attr_accessor :start_time

	def initialize(weblog, mail_parser, mail_sender, plugin_manager, processing_time, logger = nil)
		@weblog = weblog
		@weblog.logger = logger
		@mail_parser = mail_parser
		@mail_sender = mail_sender
		@plugin_manager = plugin_manager
		@processing_time = processing_time
		@logger = logger

		@start_time = Time.now.to_i
	end

	def run(m)
		if @processing_time >= 0 && Time.now.to_i - @start_time >= @processing_time then
			raise Mail2WeblogTimeout.new("動作制限時間に到達。", "動作制限時間に到達しました。")
		end

		begin
			mail = MailProcessor.parse(m.pop, SimpleMailEncoder.new(M2W_SYSTEM_INNER_ENCODING))

			if @mail_sender.is_own_mail(mail.from_address) then
				@logger.info("応答メールアドレスからメールが送信されてきました。") if @logger
				m.delete
				return true
			end

			if mail.is_report then
				@logger.info("メール送信に失敗し、エラーレポートが #{mail.from_address.address} から返ってきました。") if @logger
				m.delete
				return true
			end
		rescue => e
			@logger.error("メール解析中に予測不能なエラーが発生しました。") if @logger
			@logger.error(e) if @logger
			@logger.error(m.pop) if @logger
			m.delete
			return true
		end

		begin
			@logger.info("Processing ... subject = #{mail.subject}, from = #{mail.from_address.address}") if @logger

			process_mail(mail)

			m.delete
		rescue => e
			@logger.error("ブログコマンド実行中に予測不能なエラーが発生しました。") if @logger
			@logger.error(e) if @logger
			@logger.error(m.pop) if @logger
			m.delete
			@mail_sender.sendmail_as_command("error", nil, {}, "ブログコマンド実行中に予測不能なエラーが発生しました。", mail.from_address, [])
		end

		return true
	end

	def process_mail(mail)
		response = nil

		command_name = __get_command_name(mail)

		command_adapter = @plugin_manager.get_command_adapter_plugin(command_name, 'mail')
		command = @plugin_manager.get_command_plugin(command_name)

		if command_adapter != nil && command != nil then
			command_adapter.mail_parser = @mail_parser if command_adapter.respond_to?("mail_parser=")
			command_adapter.mail_sender = @mail_sender if command_adapter.respond_to?("mail_sender=")
			command_adapter.plugin_manager = @plugin_manager if command_adapter.respond_to?("plugin_manager=")

			command.weblog = @weblog if command.respond_to?("weblog=")
			command.plugin_manager = @plugin_manager if command.respond_to?("plugin_manager=")
			command.logger = @logger

			request = command_adapter.create_request(command, mail)
			response = command_adapter.create_response(command, request, command.run(request))
			if response.success then
				@logger.info("Succeeded") if @logger
			else
				@logger.info("Failed") if @logger
			end
			@mail_sender.send_notices(response.notices)
		else
			@logger.error("不明なコマンド(#{command_name})を実行しようとしました。") if @logger
			@mail_sender.sendmail_as_command("error", nil, {}, "不明なコマンド(#{command_name})を実行しようとしました。", mail.from_address, [])
		end

		return response
	end

	def __get_command_name(mail)
		subject_parsed_result = @mail_parser.parse_subject(mail.escaped_prefix_subject)
		return subject_parsed_result ? subject_parsed_result[0] : 'simple_post'
	end

end
