# coding: UTF-8

require 'm2w_command'
require 'm2w_mail_parser'
require 'm2w_mail_sender'
require 'm2w_plugin_manager'
require 'mui/mail_command_processor'

class Mail2WeblogRunner

	attr_reader :mail_parser
	attr_reader :mail_sender
	attr_reader :mail_command_processor

	def initialize(
			weblog = M2W_WEBLOG,
			mail_parser = nil,
			mail_sender = nil,
			mail_command_processor = nil,
			plugin_manager = Mail2WeblogPluginManager.new(M2W_COMMAND_PLUGIN_CONF, M2W_FORMAT_PLUGIN_CONF, M2W_WEBLOG_PLUGIN_CONF),
			processing_time = M2W_SYSTEM_PROCESSING_TIME,
			logger = nil
			)
		@weblog = weblog
		@logger = M2W_LOGGER if !logger

		if mail_parser == nil then
			@mail_parser = Mail2WeblogMailParser.new(
				M2W_SUBJECT_HEADER,
				M2W_SUBJECT_SEPARATOR_REGEX,
				{})
		else
			@mail_parser = mail_parser
		end

		if mail_sender == nil then
			@mail_sender = Mail2WeblogMailSender.new(
				M2W_SMTP_SERVER_ADDRESS,
				M2W_SMTP_SERVER_PORT,
				M2W_SMTP_SECURITY_TYPE,
				M2W_SMTP_SERVER_USER,
				M2W_SMTP_SERVER_PASSWORD,
				M2W_POP3_BEFORE_SMTP_SERVER_ADDRESS,
				M2W_POP3_BEFORE_SMTP_SERVER_PORT,
				M2W_POP3_BEFORE_SMTP_SERVER_USER,
				M2W_POP3_BEFORE_SMTP_SERVER_PASSWORD,
				M2W_SUBJECT_HEADER,
				M2W_SUBJECT_SEPARATOR,
				MailProcessor.parse_mail_address(M2W_REPLY_ADDRESS),
				M2W_REPLY,
				M2W_LOGGER)
			@mail_sender.enable_ssl_smtp if M2W_SMTP_SERVER_ENABLE_SSL
			@mail_sender.enable_ssl_pop3_before_smtp(OpenSSL::SSL::VERIFY_NONE) if M2W_POP3_BEFORE_SMTP_SERVER_ENABLE_SSL
		else
			@mail_sender = mail_sender
		end
		@mail_sender.logger = logger if logger

		if mail_command_processor == nil then
			@mail_command_processor = MailCommandProcessor.new(
				M2W_POP3_SERVER_ADDRESS,
				M2W_POP3_SERVER_PORT,
				M2W_POP3_SERVER_USER,
				M2W_POP3_SERVER_PASSWORD,
				M2W_LOGGER)
			@mail_command_processor.enable_ssl(OpenSSL::SSL::VERIFY_NONE) if M2W_POP3_SERVER_ENABLE_SSL
		else
			@mail_command_processor = mail_command_processor
		end
		@mail_command_processor.logger = logger if logger

		@plugin_manager = plugin_manager
		@plugin_manager.logger = logger if logger

		@mail_command = Mail2WeblogCommand.new(
			@weblog,
			@mail_parser,
			@mail_sender,
			@plugin_manager,
			processing_time,
			M2W_LOGGER)
		@mail_command.logger = logger if logger
		@mail_command_processor.command = @mail_command

		@processing_time = processing_time
	end

	def run(continuous_processing, mail_receiver_sleep_time)
		@mail_command.start_time = Time.now.to_i

		while true
			@mail_command_processor.run

			if !(continuous_processing && @processing_time >= 0 && @processing_time >= Time.now.to_i - @mail_command.start_time) then
				break
			end

			if continuous_processing && @processing_time >= 0 && mail_receiver_sleep_time >= 0 then
				sleep(mail_receiver_sleep_time)
			end
		end
	end

	def command()
		require 'optparse'

		# デフォルト値を設定する
		config = {
			:command => 'post',
			:attachments => [],
			:encoding => M2W_SYSTEM_INNER_ENCODING,
		}

		# 必須オプションを設定する
		required = [:command]

		# 引数を解析する
		OptionParser.new do |opts|
			begin
				# オプション情報を設定する
				opts = OptionParser.new
				opts.on('-c COMMAND', '--command COMMAND', "[任意]コマンドを指定する（デフォルト値：#{config[:command]}）。") { |v| config[:command] = v }
				opts.on('-a ATTACHMENT_FILENAME', '--attach ATTACHMENT_FILENAME', "[任意]添付ファイル名を指定する。複数添付する場合は複数回指定する。") { |v| config[:attachments].push(v) }
				opts.on('-e ENCODING', '--encoding ENCODING', "[任意]ファイルのエンコーディグを指定する（デフォルト値：#{config[:encoding]}）。") { |v| config[:encoding] = v }

				opts.parse!(ARGV)

				# 必須オプションをチェックする
				for field in required
					raise ArgumentError.new("必須オプション（#{field}）が不足しています。") if config[field].nil?
				end
			rescue => e
				puts opts.help
				puts
				puts e.message
				exit 1
			end
		end

		attachments = []
		config[:attachments].each do |filename|
			attachments.push(MailProcessor.load_attachment(filename))
		end

		# 処理する
		command_name = config[:command]
		content = STDIN.read.encode(M2W_SYSTEM_INNER_ENCODING, config[:encoding])

		mail = Mail.new('', MailAddress.new("dummy"), [MailAddress.new("dummy")], content, attachments)

		command_adapter = @plugin_manager.get_command_adapter_plugin(command_name, 'mail')
		command = @plugin_manager.get_command_plugin(command_name)

		if command_adapter != nil && command != nil then
			command_adapter.mail_parser = @mail_parser if command_adapter.respond_to?("mail_parser=")
			command_adapter.mail_sender = @mail_sender if command_adapter.respond_to?("mail_sender=")
			command_adapter.plugin_manager = @plugin_manager if command_adapter.respond_to?("plugin_manager=")

			command.weblog = @weblog if command.respond_to?("weblog=")
			command.plugin_manager = @plugin_manager if command.respond_to?("plugin_manager=")
			command.logger = @logger

			request = command_adapter.create_request(command, mail)
			response = command_adapter.create_response(command, request, command.run(request))
			if response.success then
				puts "成功しました。"
				@logger.info("Succeeded") if @logger
			else
				puts "失敗しました。"
				@logger.info("Failed") if @logger
			end
		else
			@logger.error("不明なコマンド(#{command_name})を実行しようとしました。") if @logger
			puts "不明なコマンド(#{command_name})を実行しようとしました。"
		end

		return response
	end

end
