# coding: UTF-8

require 'blogger/blogger'
require 'exception/m2w_error'
require 'plugin/weblog/lib/standard'
require 'plugin/weblog/meta_weblog/lib/meta_weblog_post_common'

require 'style/formatter'

require 'xmlrpc_support/xmlrpc_writer'

require 'time'
require 'uri'

M2W_META_WEBLOG_DEFAULT_ENCODING = 'UTF-8'
M2W_META_WEBLOG_ATTACHMENT_ROOT_PATH = 'attrs'
M2W_META_WEBLOG_THUMBNAIL_ROOT_PATH = 'thumbs'

#=metaWeblog API を使った標準のウェブログプラグイン
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: meta_weblog.rb 1032 2013-07-30 06:13:56Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011 - 2013. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class MetaWeblog_WeblogPlugin < Standard_WeblogPlugin
	include MetaWeblogPostCommon

	attr_accessor :plugin_manager

	attr_reader :endpoint
	attr_reader :username
	attr_reader :password
	attr_reader :attachment_root_path
	attr_reader :thumbnail_root_path
	attr_reader :blogid

	attr_writer :logger

	def initialize(
			endpoint = M2W_WEBLOG_ENDPOINT,
			username = M2W_WEBLOG_USERNAME,
			password = M2W_WEBLOG_PASSWORD,
			blogid = M2W_WEBLOG_BLOGID,
			attachment_root_path = M2W_META_WEBLOG_ATTACHMENT_ROOT_PATH,
			thumbnail_root_path = M2W_META_WEBLOG_THUMBNAIL_ROOT_PATH)
		@endpoint = endpoint
		@username = username
		@password = password
		@blogid = blogid.to_s
		@attachment_root_path = attachment_root_path
		@thumbnail_root_path = thumbnail_root_path
	end

	def create_client
		uri = URI.parse(@endpoint)
		result = XMLRPC::Client.new(uri.host, uri.path)
		if M2W_SYSTEM_COMMUNICATION_TIMEOUT != nil then
			result.timeout = M2W_SYSTEM_COMMUNICATION_TIMEOUT
		end
		return result
	end

	def create_formatter
		return Formatter.new(@plugin_manager)
	end

	# 文字列を日付に直します。
	def parse_date(date_string, default_date, errors, warnings)
		begin
			if @plugin_manager != nil then
				return @plugin_manager.parse_weblog_date(date_string, default_date)
			elsif date_string != nil
				return Time.parse(date_string)
			end
		rescue ArgumentError => e
			@logger.error("ブログ投稿期日の指定が不正でうまく認識できませんでした(ブログ投稿期日指定 = #{date_string})。") if @logger
			@logger.error(e) if @logger
			warnings.push(WeblogPluginWarning.new(
				"ブログ投稿期日指定不正(ブログ投稿期日指定 = #{date_string})。",
				"ブログ投稿期日が解析できませんでした。指定は無視して投稿時日付と解釈します(ブログ投稿期日指定 = #{date_string})。"))
		end

		return default_date
	end

	# 日付を文字列に直します。
	def get_date(date)
		date = date.clone
		date.gmtime
		return "#{date.strftime('%Y%m%dT%H:%M:%S')}Z"
	end

	# 添付ファイルのデフォルトのパスを取得します。
	def get_attachment_default_filepath(postid, dateCreated, attachment_index, attachment_filename)
		return "#{@attachment_root_path}#{postid}_#{attachment_index}#{File.extname(attachment_filename)}"
	end

	# 添付ファイルのパスを取得します。
	def get_attachment_filepath(postid, dateCreated, attachment_index, attachment_filename)
		default = get_attachment_default_filepath(postid, dateCreated, attachment_index, attachment_filename)
		if @plugin_manager != nil then
			return @plugin_manager.get_weblog_filepath('attachment_url', postid, dateCreated, attachment_index, attachment_filename, default)
		else
			return default
		end
	end

	# サムネイルのデフォルトのパスを取得します。
	def get_thumbnail_default_filepath(postid, dateCreated, attachment_index, thumbnail_filename)
		return "#{@thumbnail_root_path}#{postid}_#{attachment_index}#{File.extname(thumbnail_filename)}"
	end

	# サムネイルのパスを取得します。
	def get_thumbnail_filepath(postid, dateCreated, attachment_index, attachment_filename)
		default = get_thumbnail_default_filepath(postid, dateCreated, attachment_index, attachment_filename)
		if @plugin_manager != nil then
			return @plugin_manager.get_weblog_filepath('thumbnail_url', postid, dateCreated, attachment_index, attachment_filename, default)
		else
			return default
		end
	end

	# ブログ名を取得します。
	def blog_name(client = nil)
		client = create_client() if client == nil
		return __get_blog(__get_blogs(client))
	end



	def get_article(client, postid)
		@logger.debug("Start  metaWeblog.getPost ... postid = #{postid}") if @logger && @logger.debug?
		post = client.call("metaWeblog.getPost", postid, @username, @password)
		@logger.debug("Finish metaWeblog.getPost ... result = #{@post.to_s}") if @logger && @logger.debug?

		if post['description'] == nil then
			raise WeblogPluginError.new(
				"本文取得エラー",
				"本文の取得に失敗しました。"
			)
		end

		if post['link'] == nil then
			raise WeblogPluginError.new(
				"リンク取得エラー",
				"リンクの取得に失敗しました。"
			)
		end

		return [post, post['description'], post['link'], post['title']]
	end



	def post_temp_article(client)
		@logger.debug("Start  metaWeblog.newPost ... blogid = #{@blogid}") if @logger && @logger.debug?
		postid = client.call(
			"metaWeblog.newPost",
			@blogid,
			@username,
			@password,
			{'title' => M2W_TEMP_POST_TITLE, 'description' => M2W_TEMP_POST_DESCRIPTION},
			0)
		postid = parse_postid(postid).to_s
		@logger.debug("Finish metaWeblog.newPost ... postid = #{postid}") if @logger && @logger.debug?
		return postid
	end

	def repost_formatted_article(client, postid, header, dateCreated)
		@logger.debug("Start  metaWeblog.editPost ... postid = #{postid}, header = #{header.keys.to_s}") if @logger && @logger.debug?
		result = client.call(
			"metaWeblog.editPost",
			postid,
			@username,
			@password,
			get_header_for_post(header),
			1)
		@logger.debug("Finish metaWeblog.editPost ... result = #{result}") if @logger && @logger.debug?
	end

	def delete_temp_article(client, postid)
		@logger.debug("Start  blogger.deletePost ... postid = #{postid}") if @logger && @logger.debug?
		result = client.call(
			"blogger.deletePost",
			1,
			postid,
			@username,
			@password,
			1)
		@logger.debug("Finish blogger.deletePost ... result = #{result}") if @logger && @logger.debug?
	end

	def parse_postid(postid)
		return postid
	end



	def edit_article(client, postid, header, dateCreated)
		@logger.debug("Start  metaWeblog.editPost ... header = #{header.keys.to_s}") if @logger && @logger.debug?
		result = client.call(
			"metaWeblog.editPost",
			postid,
			@username,
			@password,
			get_header_for_post(header),
			1)
		@logger.debug("Finish metaWeblog.editPost ... result = #{result}") if @logger && @logger.debug?
	end



	def delete_article(client, postid)
		@logger.debug("Start  blogger.deletePost ... postid = #{postid}") if @logger && @logger.debug?
		result = client.call(
			"blogger.deletePost",
			"",
			postid,
			@username,
			@password,
			0)
		@logger.debug("Finish blogger.deletePost ... result = #{result}") if @logger && @logger.debug?

		return result
	end



	def __get_blogs(client = nil)
		begin
			return Blogger.new(client, @username, @password, M2W_SYSTEM_INNER_ENCODING).get_users_blogs()
		rescue => e
			@logger.error("ブログ情報の取得に失敗しました。") if @logger
			@logger.error(e) if @logger
			raise WeblogPluginError.new(
				"ブログ情報取得エラー",
				"ブログ情報の取得に失敗しました。"
			)
		end
	end

	def __get_blog(blogs)
		i = blogs.index { |blog| blog.blogid == @blogid }
		blog = i ? blogs[i] : nil
		if blog then
			return blog.blog_name
		elsif blogs.length == 1 then
			return blogs[0].blog_name
		else
			raise WeblogPluginError.new(
				"ブログ情報取得エラー",
				"ブログ情報の取得に失敗しました。"
			)
		end
	end

end
