# coding: utf-8
'''\
Module for RemoteStation for Linux
'''

__author__ = 'M.Decimal at Specurio Studio.'
__version__ = '0.1'
__date__ = '2009-12-08'
__credits__ = """\
RemoteStation PC-OP-RS1 is a product for trasmitting ir signals made by
Japanese peripheral equipment vender Buffalo,Inc.
I used pexpect module coded by Noah Spurrier (http://www.noah.org/wiki/Pexpect)
for getting info from privilege protected file. 
This script was coded on Pydev, powerful python IDE by Fabio Zadrozny, and now
becoming Apatana open source product.
Of course, Pydev is plug-in for, THE GREATE IDE, Eclipse provided by Eclipse
Foundation.
"""


import logging
import os
import gettext
import getpass
import pexpect
import re
import core

from core import DeviceBase
from exception import SetupDeviceError


# decision file whether script is in develop environment
IN_DEVELOP_ENV = False
DEVELOP_TOP_DIR = os.path.ancdirname(__file__, 2)
if os.path.exists(os.path.join(DEVELOP_TOP_DIR, 'setup.py')) :
    IN_DEVELOP_ENV = True


#======================================================================
# Module variable settings
# 
DRIVER_INFO_FILE = '/proc/tty/driver/usbserial'
DRIVER_INFO_PATTERN = r'^(?P<number>\d+):\s+module:ftdi_sio\s+name:"FTDI USB Serial Device"\s+vendor:0411\s+product:00b3.+$'
HELPER_GET_USBSERIAL_DRIVER = ['/bin/cat', DRIVER_INFO_FILE]
DRIVER_SELCTOR_TEMPLATE = r'/dev/ttyUSB%(number)s'


#======================================================================
# L10N
#
# setting gettext
if IN_DEVELOP_ENV :
    _ = gettext.translation('managers.rsdevice',
                            os.path.join(DEVELOP_TOP_DIR, 'locale'),
                            ).ugettext
else :
    _ = gettext.translation('managers.rsdevice', fallback=True).ugettext



#======================================================================
# Logging SETTINGS
#
LOGGER = logging.getLogger('managers.device')


class Device(DeviceBase):
    '''Device class for Linux
    
    For detail, see DeviceBase class in core module
    '''

    @staticmethod
    def get_devices():
        '''Get device path ( will be /dev/ttyUSB[\d])
        '''
        devlist = []

        #======================================================================
        # Check driver information file with sudo on pexpect, because normally
        # that file is root protected.
        #======================================================================
        LOGGER.info(_('scanning device information file'))
        LOGGER.info(_('- executing %s on "sudo"') % HELPER_GET_USBSERIAL_DRIVER)
        
        result = ''
        #----------------------------------------------------------------------
        # Expected words against execution "sudo" command
        expected_words = [
            r'[pP]assword:\s*',
            r'Sorry,\stry\sagain.\s*',
            r'sudo: \d+ incorrect password attempts?',
            r'\S+ is not in the sudoers file.  This incident will be reported.',
            r'%s: .*: No such file or directory' % HELPER_GET_USBSERIAL_DRIVER[0],
        ]

        child = pexpect.spawn('sudo', HELPER_GET_USBSERIAL_DRIVER, env={'LANGUAGE':'en-US'})
        while True :
            try :
                #--------------------------------------------------------------
                # Check the response 
                index = child.expect(expected_words)

                # Prompted password
                if index == 0 :
                    password = getpass.getpass('Enter password for sudo : ')
                    child.sendline(password)

                # Input password was incorrect
                elif index == 1 :
                    LOGGER.warning(_('Sorry, input password correctly.'))
                
                # Inputting password over time
                elif index == 2 :
                    LOGGER.warning(_('Password input failed, exit...'))
                    raise SetupDeviceError(_('authentication failed for scanning device infomation file'))
                
                # Sudo did not allow command execution.
                elif index == 3 :
                    raise SetupDeviceError(_('user is not in sudoers file.'))
                
                # There are not driver information file
                elif index == 4 :
                    raise SetupDeviceError(_('driver is not built in your kernel, run "/sbin/modprobe ftdi_sio vendor=0x0411 product=0x00b3"'))
                
            #------------------------------------------------------------------ 
            # Sudo exit normally
            except pexpect.EOF:
                LOGGER.debug(_('sudo ended normally'))
                child.close()
                result = child.before.strip()
                break
                        
            #------------------------------------------------------------------ 
            # Sudo or command timeouted.
            except pexpect.TIMEOUT :
                LOGGER.debug(_('sudo was timeouted'))
                child.close()
                raise SetupDeviceError(_('Timeout occured when executing helper program for searching RemoteStation device.'))

            #------------------------------------------------------------------ 
            # something else happened to sudo command or pexpect
            except :
                raise

        
        #---------------------------------------------------------------------- 
        # Helper command returns nothing.
        if result == '' :
            LOGGER.info(_('could not get any line in device information file'))
            raise SetupDeviceError(_('driver is built in your kernel but no device for usbserial'))
        
        #=======================================================================
        # Analyse response and make device list
        #=======================================================================
        result = result.strip().split('\r\n')
        LOGGER.debug('Raw text : %s' % result )
        re_obj = re.compile(DRIVER_INFO_PATTERN, re.I)
        for item in result :
            match = re_obj.match(item)
            if match is not None :
                devpath = DRIVER_SELCTOR_TEMPLATE % { 'number':match.group('number') }
                devlist.append(devpath)

        LOGGER.info( 'got %d RemoteStation device' % len(devlist))
        return devlist

    @staticmethod
    def is_exists(name):
        '''Check whether device.name is really exists
        '''
        return os.path.exists(name)

