/*******************************************************************************
 * ManjyuRfc3986
 * Copyright (C) 2012 Toshiki IGA
 * 
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2012 Toshiki IGA and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *      Toshiki IGA - initial API and implementation
 *******************************************************************************/
/*******************************************************************************
 * Copyright 2012 Toshiki IGA and others.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *******************************************************************************/
package org.manjyu.rfc3986;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;

/**
 * RFC 3986 Utility class.
 * 
 * Uniform Resource Identifier (URI): Generic Syntax
 * 
 * <pre>
 *         foo://example.com:8042/over/there?name=ferret#nose
 *         \_/   \______________/\_________/ \_________/ \__/
 *          |           |            |            |        |
 *       scheme     authority       path        query   fragment
 *          |   _____________________|__
 *         / \ /                        \
 *         urn:example:animal:ferret:nose
 * </pre>
 * 
 * @see <a href="http://www.ietf.org/rfc/rfc3986.txt">RFC 3986</a>
 * @author ToshikiIga
 */
public class ManjyuRfc3986Util {
	public static final char[] GEN_DELIMS = new char[] { ':', '/', '?', '#', '[', ']', '@' };
	public static final char[] SUB_DELIMS = new char[] { '!', '$', '&', '\'', '(', ')', '*', '+', ',', ';', '=' };

	public static final char[] UNRESERVED_CHARACTERS = new char[] { 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J',
			'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd', 'e',
			'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
			'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '-', '.', '_', '~' };

	/**
	 * Hide constructor.
	 */
	private ManjyuRfc3986Util() {
	}

	/**
	 * Check is char is RFC 3986 Reserved character or not.
	 * 
	 * @param checkChar
	 * @return true:reserved character. false:non reserved character.
	 */
	public static boolean isReservedCharacter(final char checkChar) {
		for (char lookup : GEN_DELIMS) {
			if (lookup == checkChar) {
				return true;
			}
		}
		for (char lookup : SUB_DELIMS) {
			if (lookup == checkChar) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check is char is RFC 3986 Unreserved character or not.
	 * 
	 * @param checkChar
	 * @return
	 */
	public static boolean isUnreservedCharacter(final char checkChar) {
		for (char lookup : UNRESERVED_CHARACTERS) {
			if (lookup == checkChar) {
				return true;
			}
		}

		return false;
	}

	public static boolean isPassiveModeSpecialCharacter(final char checkChar) {
		if (checkChar < 0x20 || checkChar == 0x7f) {
			// Control char.
			return true;
		}

		if (isUnreservedCharacter(checkChar)) {
			return false;
		}

		if (checkChar <= 0x7f) {
			// Other ASCII char.
			return true;
		}

		return false;
	}

	/**
	 * Encode string with Percent encoding.
	 * 
	 * UTF-8 based Encoding.
	 * Encode Unreserved Characters to percent.
	 *  
	 * @param input
	 * @return
	 * @throws IOException
	 */
	public static String encodePercentEncoding(final String input) {
		try {
			final StringReader reader = new StringReader(input);
			final StringWriter writer = new StringWriter();

			for (;;) {
				final int iRead = reader.read();
				if (iRead < 0) {
					break;
				}

				char cRead = (char) iRead;

				if (isUnreservedCharacter(cRead) == false) {
					// Encode here.
					final byte[] bytes = new String(String.valueOf(cRead)).getBytes("UTF-8");
					for (byte lookup : bytes) {
						writer.write('%');
						writer.write(String.format("%02X", lookup));
					}
				} else {
					writer.write(cRead);
				}
			}

			writer.flush();
			return writer.toString();
		} catch (IOException ex) {
			throw new IllegalArgumentException(ex);
		}
	}

	/**
	 * Encode string with Percent encoding.
	 * 
	 * UTF-8 based Encoding.
	 * Encode Unreserved Characters to percent.
	 *  
	 * @param input
	 * @return
	 * @throws IOException
	 */
	public static String encodePassivePercentEncoding(final String input) {
		try {
			final StringReader reader = new StringReader(input);
			final StringWriter writer = new StringWriter();

			for (;;) {
				final int iRead = reader.read();
				if (iRead < 0) {
					break;
				}

				char cRead = (char) iRead;

				if (isReservedCharacter(cRead) || isPassiveModeSpecialCharacter(cRead)) {
					// Encode here.
					final byte[] bytes = new String(String.valueOf(cRead)).getBytes("UTF-8");
					for (byte lookup : bytes) {
						writer.write('%');
						writer.write(String.format("%02X", lookup));
					}
				} else {
					writer.write(cRead);
				}
			}

			writer.flush();
			return writer.toString();
		} catch (IOException ex) {
			throw new IllegalArgumentException(ex);
		}
	}

	public static String decodePercentEncoding(final String input) {
		try {
			final StringReader reader = new StringReader(input);
			final StringWriter writer = new StringWriter();

			final ByteArrayOutputStream percentOutputStream = new ByteArrayOutputStream();
			boolean isInPercentEncoding = false;
			for (;;) {
				final int iRead = reader.read();
				if (iRead < 0) {
					break;
				}

				final char cRead = (char) iRead;

				if (cRead == '%') {
					isInPercentEncoding = true;

					// Decode here.

					final int iRead2 = reader.read();
					if (iRead2 < 0) {
						throw new IllegalArgumentException("Unexpected end of String.");
					}
					final int iRead3 = reader.read();
					if (iRead3 < 0) {
						throw new IllegalArgumentException("Unexpected end of String.");
					}

					final int charHexValue = Integer.parseInt(
							String.valueOf((char) iRead2) + String.valueOf((char) iRead3), 16);
					percentOutputStream.write((byte) charHexValue);
				} else {
					if (isInPercentEncoding) {
						// flush
						if (percentOutputStream.size() > 0) {
							percentOutputStream.flush();
							writer.write(new String(percentOutputStream.toByteArray(), "UTF-8"));
							percentOutputStream.reset();
						}
					}

					writer.write(cRead);
				}
			}

			if (isInPercentEncoding) {
				// flush
				if (percentOutputStream.size() > 0) {
					percentOutputStream.flush();
					writer.write(new String(percentOutputStream.toByteArray(), "UTF-8"));
					percentOutputStream.reset();
				}
			}

			writer.flush();
			return writer.toString();
		} catch (IOException ex) {
			throw new IllegalArgumentException(ex);
		}
	}

	public String getSchemeFromUri(final String uri) {
		// TODO Decode Percent encoding
		// TODO Decode &amp; and so on

		throw new IllegalArgumentException("Not implemented!");
	}

	public String getAuthorityFromUri(final String uri) {
		throw new IllegalArgumentException("Not implemented!");
	}

	public String getPathFromUri(final String uri) {
		throw new IllegalArgumentException("Not implemented!");
	}

	public String getQueryFromUri(final String uri) {
		throw new IllegalArgumentException("Not implemented!");
	}

	public String getFragmentFromUri(final String uri) {
		throw new IllegalArgumentException("Not implemented!");
	}
}
