/*******************************************************************************
 * Copyright (c) 2005 Koji Hisano <hisano@users.sourceforge.net>
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 *     Koji Hisano - initial API and implementation
 *******************************************************************************/
package jp.sf.mapswidgets;

/**
 * Instance of this class represents 2-dimensional bounding box.
 * <p>
 * If the Bounds is in the latitude/longitude coordinate system, the x coordinates represent longitude and the y coordinates represent latitude.
 * If the latitude/longitude bounds crosses the International Date Line, the "minimum" coordinates refer to the top left coordinates rather than the mathematical minimum of the two coordinates.
 * </p>
 * <p>
 * The top left corner of the box is specified by its minX and minY values, and the bottom right corner of the box is specified by its maxX and maxY.
 * </p>
 * <p>
 * Application code does <em>not</em> need to explicitly release the
 * resources managed by each instance when those instances are no longer
 * required, and thus no <code>dispose()</code> method is provided.
 * </p>
 * <p>
 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GBounds_code_">Class Reference &gt; GBounds</a>].
 * </p>
 *
 * @see Point
 * @see Size
 */
public final class Bounds implements Cloneable {
	private double minX;
	private double minY;
	private double maxX;
	private double maxY;

	/**
	 * Construct a new instance of this class given the 
	 * minX, minY, maxX and maxY values.
	 *
	 * @param minX the x coordinate of the top left corner of the bounds
	 * @param minY the y coordinate of the top left corner of the bounds
	 * @param maxX the x coordinate of the bottom right corner of the bounds
	 * @param maxY the y coordinate of the bottom right corner of the bounds
	 */
	public Bounds(double minX, double minY, double maxX, double maxY) {
		setMinX(minX);
		setMinY(minY);
		setMaxX(maxX);
		setMaxY(maxY);
	}

	public String toString() {
		return getMinX() + "," + getMinY() + "," + getMaxX() + "," + getMaxY();
	}
	
	public Bounds clone() {
		try {
			return (Bounds)super.clone();
		} catch (CloneNotSupportedException e) {
			throw new InternalError(e.getMessage());
		}
	}

	public int hashCode() {
		return toString().hashCode();
	}
	
	public boolean equals(Object tested) {
		if (tested instanceof Bounds) {
			Bounds bounds = (Bounds)tested;
		    return minX == bounds.minX && minY == bounds.minY && maxX == bounds.maxX && maxY == bounds.maxY;
		}
		return false;
	}

	/**
	 * Get the x coordinate of the top left of the bounds.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GBounds_code_">Class Reference &gt; GBounds &gt; minX</a>].
	 * </p>
	 * 
	 * @return the x coordinate of the top left of the bounds
	 */
	public double getMinX() {
		return minX;
	}

	/**
	 * Set the x coordinate of the top left of the bounds.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GBounds_code_">Class Reference &gt; GBounds &gt; minX</a>].
	 * </p>
	 * 
	 * @param minX the x coordinate of the top left of the bounds
	 */
	public void setMinX(double minX) {
		this.minX = minX;
	}

	/**
	 * Get the y coordinate of the top left of the bounds.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GBounds_code_">Class Reference &gt; GBounds &gt; minY</a>].
	 * </p>
	 * 
	 * @return the y coordinate of the top left of the bounds
	 */
	public double getMinY() {
		return minY;
	}

	/**
	 * Set the y coordinate of the top left of the bounds.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GBounds_code_">Class Reference &gt; GBounds &gt; minY</a>].
	 * </p>
	 * 
	 * @param minY the y coordinate of the top left of the bounds
	 */
	public void setMinY(double minY) {
		this.minY = minY;
	}

	/**
	 * Get the x coordinate of the bottom right of the bounds.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GBounds_code_">Class Reference &gt; GBounds &gt; maxX</a>].
	 * </p>
	 * 
	 * @return the x coordinate of the bottom right of the bounds
	 */
	public double getMaxX() {
		return maxX;
	}

	/**
	 * Set the x coordinate of the bottom right of the bounds.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GBounds_code_">Class Reference &gt; GBounds &gt; maxX</a>].
	 * </p>
	 * 
	 * @param maxX the x coordinate of bottom right left of the bounds
	 */
	public void setMaxX(double maxX) {
		this.maxX = maxX;
	}

	/**
	 * Get the y coordinate of the bottom right of the bounds.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GBounds_code_">Class Reference &gt; GBounds &gt; maxY</a>].
	 * </p>
	 * 
	 * @return the y coordinate of the bottom right of the bounds
	 */
	public double getMaxY() {
		return maxY;
	}

	/**
	 * Set the y coordinate of the bottom right of the bounds.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GBounds_code_">Class Reference &gt; GBounds &gt; maxY</a>].
	 * </p>
	 * 
	 * @param maxY the y coordinate of bottom right left of the bounds
	 */
	public void setMaxY(double maxY) {
		this.maxY = maxY;
	}
}
