/*******************************************************************************
 * Copyright (c) 2005 Koji Hisano <hisano@users.sourceforge.net>
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 *     Koji Hisano - initial API and implementation
 *******************************************************************************/
package jp.sf.mapswidgets;

import org.eclipse.swt.internal.SWTEventListener;

/**
 * Classes which implement this interface provide methods
 * that deal with the events that are generated in a Google
 * Maps control.
 * <p>
 * After creating an instance of a class that implements
 * this interface it can be added to a control using the
 * <code>addGoogleMapsListener</code> method and removed using
 * the <code>removeGoogleMapsListener</code> method. 
 * </p>
 * <p>
 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events</a>].
 * </p>
 *
 * @see GoogleMapsAdapter
 */
public interface GoogleMapsListener extends SWTEventListener {
	/**
	 * Called when the user clicks the overlay on the map.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events &gt; click</a>].
	 * </p>
	 * 
	 * @param clicked the clicked overlay
	 */
	void clicked(Overlay clicked);
	
	/**
	 * Called when the user clicks the map.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events &gt; click</a>].
	 * </p>
	 * 
	 * @param point the clicked point
	 */
	void clicked(Point point);
	
	/**
	 * Called when the map is moving.
	 * <p>
	 * This event is triggered continuously as the map is dragged.
	 * </p>
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events &gt; move</a>].
	 * </p>
	 */
	void moved();

	/**
	 * Called at the beginning of a continuous pan/drag movement.
	 * <p>
	 * This event is not triggered when the map moves discretely.
	 * </p>
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events &gt; movestart</a>].
	 * </p>
	 */
	void moveStarted();
	
	/**
	 * Called at the end of a discrete or continuous map movement.
	 * <p>
	 * This event is triggered once at the end of a continuous pan.
	 * </p>
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events &gt; moveend</a>].
	 * </p>
	 */
	void moveEnded();
	
	/**
	 * Called after the map zoom level changes.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events &gt; zoom</a>].
	 * </p>
	 * 
	 * @param oldZoomLevel the old zoom level
	 * @param newZoomLevel the new zoom level
	 */
	void zoomed(int oldZoomLevel, int newZoomLevel);
	
	/**
	 * Called after the map type (Map, Hybrid, or Satellite) changes.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events &gt; maptypechanged</a>].
	 * </p>
	 */
	void mapTypeChanged();
	
	/**
	 * Called after the info window is displayed.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events &gt; infowindowopen</a>].
	 * </p>
	 */
	void windowOpend();
	
	/**
	 * Called after the info window is closed.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events &gt; infowindowclose</a>].
	 * </p>
	 */
	void windowClosed();
	
	/**
	 * Called after an overlay is added to the map.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events &gt; addoverlay</a>].
	 * </p>
	 * 
	 * @param added the added overlay
	 */
	void overlayAdded(Overlay added);
	
	/**
	 * Called after an overlay is removed from the map.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events &gt; removeoverlay</a>].
	 * </p>
	 *  
	 * @param removed the removed overlay
	 */
	void overlayRemoved(Overlay removed);
	
	/**
	 * Called after all overlays are cleared from the map.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GMap_code_">Class Reference &gt; GMap &gt; Events &gt; clearoverlays</a>].
	 * </p>
	 */
	void allOverlaysRemoved();
}
