/*******************************************************************************
 * Copyright (c) 2005 Koji Hisano <hisano@users.sourceforge.net>
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 *     Koji Hisano - initial API and implementation
 *******************************************************************************/
package jp.sf.mapswidgets;

/**
 * Instances of this class represents a 2-dimensional size measurement.
 * <p>
 * If a Size represents a latitude/longitude span, width is the number of longitude degrees, and height is the number of latitude degrees.
 * </p>
 * <p>
 * Application code does <em>not</em> need to explicitly release the
 * resources managed by each instance when those instances are no longer
 * required, and thus no <code>dispose()</code> method is provided.
 * </p>
 * <p>
 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GSize_code_">Class Reference &gt; GSize</a>].
 * </p>
 *
 * @see Bounds
 * @see Point
 */
public final class Size implements Cloneable {
	private double width;
	private double height;

	public Size(double width, double height) {
		setWidth(width);
		setHeight(height);
	}

	public String toString() {
		return getWidth() + "," + getHeight();
	}

	public Size clone() {
		try {
			return (Size)super.clone();
		} catch (CloneNotSupportedException e) {
			throw new InternalError(e.getMessage());
		}
	}
	
	public int hashCode() {
		return toString().hashCode();
	}
	
	public boolean equals(Object tested) {
		if (tested instanceof Size) {
			Size size = (Size)tested;
		    return height == size.height && width == size.width;
		}
		return false;
	}

	/**
	 * Get the width.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GSize_code_">Class Reference &gt; GSize &gt; width</a>].
	 * </p>
	 * 
	 * @return the width
	 */
	public double getWidth() {
		return width;
	}

	/**
	 * Set the width.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GSize_code_">Class Reference &gt; GSize &gt; height</a>].
	 * </p>
	 * 
	 * @param width the width
	 */
	public void setWidth(double width) {
		this.width = width;
	}

	/**
	 * Get the height.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GSize_code_">Class Reference &gt; GSize &gt; height</a>].
	 * </p>
	 * 
	 * @return the height
	 */
	public double getHeight() {
		return height;
	}

	/**
	 * Set the height.
	 * <p>
	 * See Google Maps API documentation [<a href="http://www.google.com/apis/maps/documentation/#GSize_code_">Class Reference &gt; GSize &gt; height</a>].
	 * </p>
	 * 
	 * @param height the height
	 */
	public void setHeight(double height) {
		this.height = height;
	}

	String getExpression() {
		return "new GSize(" + getWidth() + ", " + getHeight() + ")";
	}
}
