/*
 * Copyright 2000-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.marevol.utils.ynd.search;

import java.beans.IntrospectionException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLEncoder;

import org.apache.commons.betwixt.BindingConfiguration;
import org.apache.commons.betwixt.io.BeanReader;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.xml.sax.SAXException;

import com.marevol.utils.ynd.search.model.v1.web.ResultSet;

public class WebSearch
{
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(WebSearch.class);

    private String webSearchServiceUrl;

    public WebSearch()
    {
        this(WebConstants.YAHOO_WEB_SEARCH_SERVICE_URL_V1);
    }

    public WebSearch(String url)
    {
        webSearchServiceUrl = url;
    }

    public ResultSet search(String appid, String query, int results, int start) throws IOException
    {
        return search(appid, query, "all", results, start, "any", true, true, null, null, null);
    }

    /**
     *  パラメータ     値   説明
     *  appid（必須    string  アプリケーションID。詳細はこちらをご覧ください。
     *  query (必須）  string  （UTF-8エンコードされた）検索クエリーです。このクエリーはYahoo!検索の全言語をサポートし、またメタキーワードも含みます。
     *  type           all（デフォルト）, any, phrase     指定検索の種類:
     *        allは全クエリー文字を含む検索結果を返します。
     *        anyはクエリー文字のうちいずれかを含む検索結果を返します。
     *        phraseはクエリー文字を文章として含む検索結果を返します。
     *  results       integer: 10（デフォルト）, 50（最大）  返却結果の数です。
     *  start         integer: 1（デフォルト）   返却結果の先頭位置（1 - based）です。最終位置（start + results - 1）は、1000を超えられません。
     *  format        any（デフォルト）, html, msword, pdf, ppt, rss, txt, xls   検索するファイルの種類を指定します。
     *  adult_ok      値なし（デフォルト）, 1   アダルトコンテンツの検索結果を含めるかどうかを指定します。1の場合はアダルトコンテンツを含みます。
     *  similar_ok    値なし（デフォルト）, 1   同じコンテンツを別の検索結果とするかどうかを指定します。1の場合は同じコンテンツを含みます。
     *  language      string: ja（デフォルト）   languageで書かれた結果になります。「サポートしている言語」をご参照ください。
     *  country       string: 値なし（デフォルト）  ウェブサイトが位置する国の国コードです。「サポートしている国・地域」をご参照ください。
     *  site          string: 値なし（デフォルト）  検索するドメイン（例えば www.yahoo.co.jp）を制限します。30ドメインまで指定することができます
     *  （site=www.yahoo.co.jp&site=www.asahi.com）。
     * @throws IOException 
     */
    public ResultSet search(String appid, String query, String type, int results, int start, String format,
            boolean adultOk, boolean similarOk, String language, String country, String site) throws IOException
    {
        if (query == null)
        {
            return null;
        }

        StringBuffer url = new StringBuffer(webSearchServiceUrl);
        url.append("?appid=");
        url.append(appid);
        url.append("&query=");
        try
        {
            url.append(URLEncoder.encode(query, "UTF-8"));
        }
        catch (UnsupportedEncodingException e)
        {
            log.error("Unsupported Encoding Exception", e);
            url.append(query);
        }

        if (type != null)
        {
            url.append("&type=");
            url.append(type);
        }
        url.append("&results=");
        url.append(results);
        url.append("&start=");
        url.append(start);
        if (format != null)
        {
            url.append("&format=");
            url.append(format);
        }
        if (adultOk)
        {
            url.append("&adult_ok=1");
        }
        else
        {
            url.append("&adult_ok=0");
        }
        if (similarOk)
        {
            url.append("&similar_ok=1");
        }
        else
        {
            url.append("&similar_ok=0");
        }
        if (language != null)
        {
            url.append("&language=");
            url.append(language);
        }
        if (country != null)
        {
            url.append("&country=");
            url.append(country);
        }
        if (site != null)
        {
            url.append("&site=");
            url.append(site);
        }

        if (log.isDebugEnabled())
        {
            log.debug("search(String, String...) -  : url=" + url.toString());
        }

        URL ywsUrl = new URL(url.toString());
        URLConnection con = ywsUrl.openConnection();
        return load(con.getInputStream());
    }

    public ResultSet load(InputStream is)
    {
        BindingConfiguration config = new BindingConfiguration();
        config.setMapIDs(false);
        ResultSet resultSet = null;
        BeanReader reader = new BeanReader();
        try
        {
            reader.registerBeanClass(ResultSet.class);
            resultSet = (ResultSet) reader.parse(new InputStreamReader(is, "UTF-8"));
        }
        catch (IntrospectionException e)
        {
            log.error("Introspection Exception occurs." + e.toString());
        }
        catch (IOException e)
        {
            log.error("I/O Exception occurs." + e.toString());
        }
        catch (SAXException e)
        {
            log.error("SAX Exception occurs." + e.toString());
        }
        return resultSet;
    }
}
