package org.maskat.core;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * maskatリクエスト
 * <pre>
 * MaskatBuilderによって組み上げられる。
 * 
 * リクエストXMLが
 * &lt;maskat&gt;
 *   &lt;superStars&gt;
 *     &lt;superStar&gt;
 *       &lt;age&gt;22&lt;/age&gt;
 *       &lt;sex&gt;女&lt;/sex&gt;
 *     &lt;/superStar&gt;
 *     &lt;superStar&gt;
 *       &lt;age&gt;18&lt;/age&gt;
 *       &lt;sex&gt;男&lt;/sex&gt;
 *     &lt;/superStar&gt;
 *   &lt;superStars&gt;
 * &lt;/maskat&gt;
 * 
 * の場合
 * 
 * ("maskat", "")
 *    ("superStars", "")
 *       ("superStar", "")
 *          ("age", "22")
 *          ("sex", "女")
 *       ("superStar", "")
 *          ("age", "18")
 *          ("sex", "男")
 * と階層構造に組み立てられる。 ※(キー, 値)
 * </pre>
 */
public class MaskatRequest {
	
	private List children = new ArrayList();
	
	private final String key;
	
	private final String value;
	
	/**
	 * コンストラクタ
	 * @param key キー
	 * @param value 値
	 */
	MaskatRequest(String key, String value) {
		this.key = key;
		this.value = value;
	}
	
	/**
	 * 値を取得
	 * @return 値
	 */
	public String getValue() {
		return value;
	}
	
	/**
	 * キー取得
	 * @return キー
	 */
	private String getKey() {
		return key;
	}
	
	/**
	 * 子リクエスト追加
	 * @param request
	 */
	void add(MaskatRequest request) {
		children.add(request);
	}
	
	/**
	 * 子リクエスト取得
	 * <pre>
	 * ダイレクトアクセスが可能で、
	 * ("maskat", "")
	 *    ("superStars", "")
	 *       ("superStar", "")
	 *          ("age", "22")
	 *          ("sex", "女")
	 *       ("superStar", "")
	 *          ("age", "18")
	 *          ("sex", "男")
	 * この状態(キーmaskatがカレント)からageの22にアクセスする場合、
	 * <code>String age = request.getChild("superStars/superStar/age").getValue();</code>
	 * とすることができる。
	 * <b>
	 * ダイレクトアクセスは上のように同じキー名が複数存在する場合、上に位置するものしか取得できない。
	 * ("age", "18")や("sex", "男")にはダイレクトアクセスできないので注意。
	 * </b>この場合は、MaskatRequest#iterator()を使用する。
	 * 
	 * 指定されたキーが見つからない場合は<b>null</b>が返る。
	 * </pre>
	 * 
	 *param key キー
	 * @return 子リクエスト
	 */
	public MaskatRequest getChild(String key) {
		String[] keys = key.split("/");
		if (keys.length == 1) {
			return findChild(keys[0]);
		}
		MaskatRequest node = this;
		for (int i = 0; i < keys.length; i++) {
			node = node.getChild(keys[i]);
		}
		return node;
	}
	
	/**
	 * 子検索
	 * @param key キー
	 * @return 見つかった子リクエスト
	 */
	private MaskatRequest findChild(String key) {
		for (int i = 0; i < children.size(); i++) {
			MaskatRequest child = (MaskatRequest) children.get(i);
			if (child.getKey().equals(key)) {
				return child;
			}
		}
		return null;
	}
	
	/**
	 * イテレータ取得
	 * <pre>
	 * MaskatRequest#getChild()でダイレクトアクセスできない場合などに使用する。
	 * <code>
	 * Iterator superStars = request.getChild("SuperStars").iterator();
	 * while (superStars.hasNext()) {
	 *    MaskatRequest superSter = (MaskatRequest) superStars.next();
	 *    String age = superSter.getChild("aeg").getValue();
	 *    String sex = superSter.getChild("sex").getValue();
	 * }
	 * </code>
	 * </pre>
	 * @return イテレータ
	 */
	public Iterator iterator() {
		return children.iterator();
	}
}
